r'''
# AWS Cloud Development Kit Library

The AWS CDK construct library provides APIs to define your CDK application and add
CDK constructs to the application.

## Usage

### Upgrade from CDK 1.x

When upgrading from CDK 1.x, remove all dependencies to individual CDK packages
from your dependencies file and follow the rest of the sections.

### Installation

To use this package, you need to declare this package and the `constructs` package as
dependencies.

According to the kind of project you are developing:

For projects that are CDK libraries in NPM, declare them both under the `devDependencies` **and** `peerDependencies` sections.
To make sure your library is compatible with the widest range of CDK versions: pick the minimum `aws-cdk-lib` version
that your library requires; declare a range dependency with a caret on that version in peerDependencies, and declare a
point version dependency on that version in devDependencies.

For example, let's say the minimum version your library needs is `2.38.0`. Your `package.json` should look like this:

```javascript
{
  "peerDependencies": {
    "aws-cdk-lib": "^2.38.0",
    "constructs": "^10.0.0"
  },
  "devDependencies": {
    /* Install the oldest version for testing so we don't accidentally use features from a newer version than we declare */
    "aws-cdk-lib": "2.38.0"
  }
}
```

For CDK apps, declare them under the `dependencies` section. Use a caret so you always get the latest version:

```json
{
  "dependencies": {
    "aws-cdk-lib": "^2.38.0",
    "constructs": "^10.0.0"
  }
}
```

### Use in your code

#### Classic import

You can use a classic import to get access to each service namespaces:

```python
from aws_cdk import Stack, App, aws_s3 as s3

app = App()
stack = Stack(app, "TestStack")

s3.Bucket(stack, "TestBucket")
```

#### Barrel import

Alternatively, you can use "barrel" imports:

```python
from aws_cdk import App, Stack
from aws_cdk.aws_s3 import Bucket

app = App()
stack = Stack(app, "TestStack")

Bucket(stack, "TestBucket")
```

<!--BEGIN CORE DOCUMENTATION-->

## Stacks and Stages

A `Stack` is the smallest physical unit of deployment, and maps directly onto
a CloudFormation Stack. You define a Stack by defining a subclass of `Stack`
-- let's call it `MyStack` -- and instantiating the constructs that make up
your application in `MyStack`'s constructor. You then instantiate this stack
one or more times to define different instances of your application. For example,
you can instantiate it once using few and cheap EC2 instances for testing,
and once again using more and bigger EC2 instances for production.

When your application grows, you may decide that it makes more sense to split it
out across multiple `Stack` classes. This can happen for a number of reasons:

* You could be starting to reach the maximum number of resources allowed in a single
  stack (this is currently 500).
* You could decide you want to separate out stateful resources and stateless resources
  into separate stacks, so that it becomes easy to tear down and recreate the stacks
  that don't have stateful resources.
* There could be a single stack with resources (like a VPC) that are shared
  between multiple instances of other stacks containing your applications.

As soon as your conceptual application starts to encompass multiple stacks,
it is convenient to wrap them in another construct that represents your
logical application. You can then treat that new unit the same way you used
to be able to treat a single stack: by instantiating it multiple times
for different instances of your application.

You can define a custom subclass of `Stage`, holding one or more
`Stack`s, to represent a single logical instance of your application.

As a final note: `Stack`s are not a unit of reuse. They describe physical
deployment layouts, and as such are best left to application builders to
organize their deployments with. If you want to vend a reusable construct,
define it as a subclasses of `Construct`: the consumers of your construct
will decide where to place it in their own stacks.

## Stack Synthesizers

Each Stack has a *synthesizer*, an object that determines how and where
the Stack should be synthesized and deployed. The synthesizer controls
aspects like:

* How does the stack reference assets? (Either through CloudFormation
  parameters the CLI supplies, or because the Stack knows a predefined
  location where assets will be uploaded).
* What roles are used to deploy the stack? These can be bootstrapped
  roles, roles created in some other way, or just the CLI's current
  credentials.

The following synthesizers are available:

* `DefaultStackSynthesizer`: recommended. Uses predefined asset locations and
  roles created by the modern bootstrap template. Access control is done by
  controlling who can assume the deploy role. This is the default stack
  synthesizer in CDKv2.
* `LegacyStackSynthesizer`: Uses CloudFormation parameters to communicate
  asset locations, and the CLI's current permissions to deploy stacks. This
  is the default stack synthesizer in CDKv1.
* `CliCredentialsStackSynthesizer`: Uses predefined asset locations, and the
  CLI's current permissions.

Each of these synthesizers takes configuration arguments. To configure
a stack with a synthesizer, pass it as one of its properties:

```python
MyStack(app, "MyStack",
    synthesizer=DefaultStackSynthesizer(
        file_assets_bucket_name="amzn-s3-demo-bucket"
    )
)
```

For more information on bootstrapping accounts and customizing synthesis,
see [Bootstrapping in the CDK Developer Guide](https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html).

## Nested Stacks

[Nested stacks](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html) are stacks created as part of other stacks. You create a nested stack within another stack by using the `NestedStack` construct.

As your infrastructure grows, common patterns can emerge in which you declare the same components in multiple templates. You can separate out these common components and create dedicated templates for them. Then use the resource in your template to reference other templates, creating nested stacks.

For example, assume that you have a load balancer configuration that you use for most of your stacks. Instead of copying and pasting the same configurations into your templates, you can create a dedicated template for the load balancer. Then, you just use the resource to reference that template from within other templates.

The following example will define a single top-level stack that contains two nested stacks: each one with a single Amazon S3 bucket:

```python
class MyNestedStack(cfn.NestedStack):
    def __init__(self, scope, id, *, parameters=None, timeout=None, notifications=None):
        super().__init__(scope, id, parameters=parameters, timeout=timeout, notifications=notifications)

        s3.Bucket(self, "NestedBucket")

class MyParentStack(Stack):
    def __init__(self, scope, id, *, description=None, env=None, stackName=None, tags=None, notificationArns=None, synthesizer=None, terminationProtection=None, analyticsReporting=None, crossRegionReferences=None, permissionsBoundary=None, suppressTemplateIndentation=None):
        super().__init__(scope, id, description=description, env=env, stackName=stackName, tags=tags, notificationArns=notificationArns, synthesizer=synthesizer, terminationProtection=terminationProtection, analyticsReporting=analyticsReporting, crossRegionReferences=crossRegionReferences, permissionsBoundary=permissionsBoundary, suppressTemplateIndentation=suppressTemplateIndentation)

        MyNestedStack(self, "Nested1")
        MyNestedStack(self, "Nested2")
```

Resources references across nested/parent boundaries (even with multiple levels of nesting) will be wired by the AWS CDK
through CloudFormation parameters and outputs. When a resource from a parent stack is referenced by a nested stack,
a CloudFormation parameter will automatically be added to the nested stack and assigned from the parent; when a resource
from a nested stack is referenced by a parent stack, a CloudFormation output will be automatically be added to the
nested stack and referenced using `Fn::GetAtt "Outputs.Xxx"` from the parent.

Nested stacks also support the use of Docker image and file assets.

## Accessing resources in a different stack

You can access resources in a different stack, as long as they are in the
same account and AWS Region (see [next section](#accessing-resources-in-a-different-stack-and-region) for an exception).
The following example defines the stack `stack1`,
which defines an Amazon S3 bucket. Then it defines a second stack, `stack2`,
which takes the bucket from stack1 as a constructor property.

```python
prod = {"account": "123456789012", "region": "us-east-1"}

stack1 = StackThatProvidesABucket(app, "Stack1", env=prod)

# stack2 will take a property { bucket: IBucket }
stack2 = StackThatExpectsABucket(app, "Stack2",
    bucket=stack1.bucket,
    env=prod
)
```

If the AWS CDK determines that the resource is in the same account and
Region, but in a different stack, it automatically synthesizes AWS
CloudFormation
[Exports](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-exports.html)
in the producing stack and an
[Fn::ImportValue](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-importvalue.html)
in the consuming stack to transfer that information from one stack to the
other.

## Accessing resources in a different stack and region

> **This feature is currently experimental**

You can enable the Stack property `crossRegionReferences`
in order to access resources in a different stack *and* region. With this feature flag
enabled it is possible to do something like creating a CloudFront distribution in `us-east-2` and
an ACM certificate in `us-east-1`.

```python
stack1 = Stack(app, "Stack1",
    env=Environment(
        region="us-east-1"
    ),
    cross_region_references=True
)
cert = acm.Certificate(stack1, "Cert",
    domain_name="*.example.com",
    validation=acm.CertificateValidation.from_dns(route53.PublicHostedZone.from_hosted_zone_id(stack1, "Zone", "Z0329774B51CGXTDQV3X"))
)

stack2 = Stack(app, "Stack2",
    env=Environment(
        region="us-east-2"
    ),
    cross_region_references=True
)
cloudfront.Distribution(stack2, "Distribution",
    default_behavior=cloudfront.BehaviorOptions(
        origin=origins.HttpOrigin("example.com")
    ),
    domain_names=["dev.example.com"],
    certificate=cert
)
```

When the AWS CDK determines that the resource is in a different stack *and* is in a different
region, it will "export" the value by creating a custom resource in the producing stack which
creates SSM Parameters in the consuming region for each exported value. The parameters will be
created with the name '/cdk/exports/${consumingStackName}/${export-name}'.
In order to "import" the exports into the consuming stack a [SSM Dynamic reference](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html#dynamic-references-ssm)
is used to reference the SSM parameter which was created.

In order to mimic strong references, a Custom Resource is also created in the consuming
stack which marks the SSM parameters as being "imported". When a parameter has been successfully
imported, the producing stack cannot update the value.

> [!NOTE]
> As a consequence of this feature being built on a Custom Resource, we are restricted to a
> CloudFormation response body size limitation of [4096 bytes](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/crpg-ref-responses.html).
> To prevent deployment errors related to the Custom Resource Provider response body being too
> large, we recommend limiting the use of nested stacks and minimizing the length of stack names.
> Doing this will prevent SSM parameter names from becoming too long which will reduce the size of the
> response body.

See the [adr](https://github.com/aws/aws-cdk/blob/main/packages/aws-cdk-lib/core/adr/cross-region-stack-references.md)
for more details on this feature.

### Removing automatic cross-stack references

The automatic references created by CDK when you use resources across stacks
are convenient, but may block your deployments if you want to remove the
resources that are referenced in this way. You will see an error like:

```text
Export Stack1:ExportsOutputFnGetAtt-****** cannot be deleted as it is in use by Stack1
```

Let's say there is a Bucket in the `stack1`, and the `stack2` references its
`bucket.bucketName`. You now want to remove the bucket and run into the error above.

It's not safe to remove `stack1.bucket` while `stack2` is still using it, so
unblocking yourself from this is a two-step process. This is how it works:

DEPLOYMENT 1: break the relationship

* Make sure `stack2` no longer references `bucket.bucketName` (maybe the consumer
  stack now uses its own bucket, or it writes to an AWS DynamoDB table, or maybe you just
  remove the Lambda Function altogether).
* In the `stack1` class, call `this.exportValue(this.bucket.bucketName)`. This
  will make sure the CloudFormation Export continues to exist while the relationship
  between the two stacks is being broken.
* Deploy (this will effectively only change the `stack2`, but it's safe to deploy both).

DEPLOYMENT 2: remove the resource

* You are now free to remove the `bucket` resource from `stack1`.
* Don't forget to remove the `exportValue()` call as well.
* Deploy again (this time only the `stack1` will be changed -- the bucket will be deleted).

## Durations

To make specifications of time intervals unambiguous, a single class called
`Duration` is used throughout the AWS Construct Library by all constructs
that that take a time interval as a parameter (be it for a timeout, a
rate, or something else).

An instance of Duration is constructed by using one of the static factory
methods on it:

```python
Duration.seconds(300) # 5 minutes
Duration.minutes(5) # 5 minutes
Duration.hours(1) # 1 hour
Duration.days(7) # 7 days
Duration.parse("PT5M")
```

Durations can be added or subtracted together:

```python
Duration.minutes(1).plus(Duration.seconds(60)) # 2 minutes
Duration.minutes(5).minus(Duration.seconds(10))
```

## Size (Digital Information Quantity)

To make specification of digital storage quantities unambiguous, a class called
`Size` is available.

An instance of `Size` is initialized through one of its static factory methods:

```python
Size.kibibytes(200) # 200 KiB
Size.mebibytes(5) # 5 MiB
Size.gibibytes(40) # 40 GiB
Size.tebibytes(200) # 200 TiB
Size.pebibytes(3)
```

Instances of `Size` created with one of the units can be converted into others.
By default, conversion to a higher unit will fail if the conversion does not produce
a whole number. This can be overridden by unsetting `integral` property.

```python
Size.mebibytes(2).to_kibibytes() # yields 2048
Size.kibibytes(2050).to_mebibytes(rounding=SizeRoundingBehavior.FLOOR)
```

## Secrets

To help avoid accidental storage of secrets as plain text, we use the `SecretValue` type to
represent secrets. Any construct that takes a value that should be a secret (such as
a password or an access key) will take a parameter of type `SecretValue`.

The best practice is to store secrets in AWS Secrets Manager and reference them using `SecretValue.secretsManager`:

```python
secret = SecretValue.secrets_manager("secretId",
    json_field="password",  # optional: key of a JSON field to retrieve (defaults to all content),
    version_id="id",  # optional: id of the version (default AWSCURRENT)
    version_stage="stage"
)
```

Using AWS Secrets Manager is the recommended way to reference secrets in a CDK app.
`SecretValue` also supports the following secret sources:

* `SecretValue.unsafePlainText(secret)`: stores the secret as plain text in your app and the resulting template (not recommended).
* `SecretValue.secretsManager(secret)`: refers to a secret stored in Secrets Manager
* `SecretValue.ssmSecure(param, version)`: refers to a secret stored as a SecureString in the SSM
  Parameter Store. If you don't specify the exact version, AWS CloudFormation uses the latest
  version of the parameter.
* `SecretValue.cfnParameter(param)`: refers to a secret passed through a CloudFormation parameter (must have `NoEcho: true`).
* `SecretValue.cfnDynamicReference(dynref)`: refers to a secret described by a CloudFormation dynamic reference (used by `ssmSecure` and `secretsManager`).
* `SecretValue.resourceAttribute(attr)`: refers to a secret returned from a CloudFormation resource creation.

`SecretValue`s should only be passed to constructs that accept properties of type
`SecretValue`. These constructs are written to ensure your secrets will not be
exposed where they shouldn't be. If you try to use a `SecretValue` in a
different location, an error about unsafe secret usage will be thrown at
synthesis time.

If you rotate the secret's value in Secrets Manager, you must also change at
least one property on the resource where you are using the secret, to force
CloudFormation to re-read the secret.

`SecretValue.ssmSecure()` is only supported for a limited set of resources.
[Click here for a list of supported resources and properties](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html#template-parameters-dynamic-patterns-resources).

## ARN manipulation

Sometimes you will need to put together or pick apart Amazon Resource Names
(ARNs). The functions `stack.formatArn()` and `stack.splitArn()` exist for
this purpose.

`formatArn()` can be used to build an ARN from components. It will automatically
use the region and account of the stack you're calling it on:

```python
# stack: Stack


# Builds "arn:<PARTITION>:lambda:<REGION>:<ACCOUNT>:function:MyFunction"
stack.format_arn(
    service="lambda",
    resource="function",
    arn_format=ArnFormat.COLON_RESOURCE_NAME,
    resource_name="MyFunction"
)
```

`splitArn()` can be used to get a single component from an ARN. `splitArn()`
will correctly deal with both literal ARNs and deploy-time values (tokens),
but in case of a deploy-time value be aware that the result will be another
deploy-time value which cannot be inspected in the CDK application.

```python
# stack: Stack


# Extracts the function name out of an AWS Lambda Function ARN
arn_components = stack.split_arn(arn, ArnFormat.COLON_RESOURCE_NAME)
function_name = arn_components.resource_name
```

Note that the format of the resource separator depends on the service and
may be any of the values supported by `ArnFormat`. When dealing with these
functions, it is important to know the format of the ARN you are dealing with.

For an exhaustive list of ARN formats used in AWS, see [AWS ARNs and
Namespaces](https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html)
in the AWS General Reference.

## Dependencies

### Construct Dependencies

Sometimes AWS resources depend on other resources, and the creation of one
resource must be completed before the next one can be started.

In general, CloudFormation will correctly infer the dependency relationship
between resources based on the property values that are used. In the cases where
it doesn't, the AWS Construct Library will add the dependency relationship for
you.

If you need to add an ordering dependency that is not automatically inferred,
you do so by adding a dependency relationship using
`constructA.node.addDependency(constructB)`. This will add a dependency
relationship between all resources in the scope of `constructA` and all
resources in the scope of `constructB`.

If you want a single object to represent a set of constructs that are not
necessarily in the same scope, you can use a `DependencyGroup`. The
following creates a single object that represents a dependency on two
constructs, `constructB` and `constructC`:

```python
# Declare the dependable object
b_and_c = DependencyGroup()
b_and_c.add(construct_b)
b_and_c.add(construct_c)

# Take the dependency
construct_a.node.add_dependency(b_and_c)
```

### Stack Dependencies

Two different stack instances can have a dependency on one another. This
happens when an resource from one stack is referenced in another stack. In
that case, CDK records the cross-stack referencing of resources,
automatically produces the right CloudFormation primitives, and adds a
dependency between the two stacks. You can also manually add a dependency
between two stacks by using the `stackA.addDependency(stackB)` method.

A stack dependency has the following implications:

* Cyclic dependencies are not allowed, so if `stackA` is using resources from
  `stackB`, the reverse is not possible anymore.
* Stacks with dependencies between them are treated specially by the CDK
  toolkit:

  * If `stackA` depends on `stackB`, running `cdk deploy stackA` will also
    automatically deploy `stackB`.
  * `stackB`'s deployment will be performed *before* `stackA`'s deployment.

### CfnResource Dependencies

To make declaring dependencies between `CfnResource` objects easier, you can declare dependencies from one `CfnResource` object on another by using the `cfnResource1.addDependency(cfnResource2)` method. This method will work for resources both within the same stack and across stacks as it detects the relative location of the two resources and adds the dependency either to the resource or between the relevant stacks, as appropriate. If more complex logic is in needed, you can similarly remove, replace, or view dependencies between `CfnResource` objects with the `CfnResource` `removeDependency`, `replaceDependency`, and `obtainDependencies` methods, respectively.

## Custom Resources

Custom Resources are CloudFormation resources that are implemented by arbitrary
user code. They can do arbitrary lookups or modifications during a
CloudFormation deployment.

Custom resources are backed by *custom resource providers*. Commonly, these are
Lambda Functions that are deployed in the same deployment as the one that
defines the custom resource itself, but they can also be backed by Lambda
Functions deployed previously, or code responding to SNS Topic events running on
EC2 instances in a completely different account. For more information on custom
resource providers, see the next section.

Once you have a provider, each definition of a `CustomResource` construct
represents one invocation. A single provider can be used for the implementation
of arbitrarily many custom resource definitions. A single definition looks like
this:

```python
CustomResource(self, "MyMagicalResource",
    resource_type="Custom::MyCustomResource",  # must start with 'Custom::'

    # the resource properties
    properties={
        "Property1": "foo",
        "Property2": "bar"
    },

    # the ARN of the provider (SNS/Lambda) which handles
    # CREATE, UPDATE or DELETE events for this resource type
    # see next section for details
    service_token="ARN"
)
```

### Custom Resource Providers

Custom resources are backed by a **custom resource provider** which can be
implemented in one of the following ways. The following table compares the
various provider types (ordered from low-level to high-level):

| Provider                                                             | Compute Type | Error Handling | Submit to CloudFormation |   Max Timeout   | Language | Footprint |
| -------------------------------------------------------------------- | :----------: | :------------: | :----------------------: | :-------------: | :------: | :-------: |
| [sns.Topic](#amazon-sns-topic)                                       | Self-managed |     Manual     |          Manual          |    Unlimited    |   Any    |  Depends  |
| [lambda.Function](#aws-lambda-function)                              |  AWS Lambda  |     Manual     |          Manual          |      15min      |   Any    |   Small   |
| [core.CustomResourceProvider](#the-corecustomresourceprovider-class) |  AWS Lambda  |      Auto      |           Auto           |      15min      | Node.js  |   Small   |
| [custom-resources.Provider](#the-custom-resource-provider-framework) |  AWS Lambda  |      Auto      |           Auto           | Unlimited Async |   Any    |   Large   |

Legend:

* **Compute type**: which type of compute can be used to execute the handler.
* **Error Handling**: whether errors thrown by handler code are automatically
  trapped and a FAILED response is submitted to CloudFormation. If this is
  "Manual", developers must take care of trapping errors. Otherwise, events
  could cause stacks to hang.
* **Submit to CloudFormation**: whether the framework takes care of submitting
  SUCCESS/FAILED responses to CloudFormation through the event's response URL.
* **Max Timeout**: maximum allows/possible timeout.
* **Language**: which programming languages can be used to implement handlers.
* **Footprint**: how many resources are used by the provider framework itself.

**A NOTE ABOUT SINGLETONS**

When defining resources for a custom resource provider, you will likely want to
define them as a *stack singleton* so that only a single instance of the
provider is created in your stack and which is used by all custom resources of
that type.

Here is a basic pattern for defining stack singletons in the CDK. The following
examples ensures that only a single SNS topic is defined:

```python
def get_or_create(self, scope):
    stack = Stack.of(scope)
    uniqueid = "GloballyUniqueIdForSingleton" # For example, a UUID from `uuidgen`
    existing = stack.node.try_find_child(uniqueid)
    if existing:
        return existing
    return sns.Topic(stack, uniqueid)
```

#### Amazon SNS Topic

Every time a resource event occurs (CREATE/UPDATE/DELETE), an SNS notification
is sent to the SNS topic. Users must process these notifications (e.g. through a
fleet of worker hosts) and submit success/failure responses to the
CloudFormation service.

> You only need to use this type of provider if your custom resource cannot run on AWS Lambda, for reasons other than the 15
> minute timeout. If you are considering using this type of provider because you want to write a custom resource provider that may need
> to wait for more than 15 minutes for the API calls to stabilize, have a look at the [`custom-resources`](#the-custom-resource-provider-framework) module first.
>
> Refer to the [CloudFormation Custom Resource documentation](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-custom-resources.html) for information on the contract your custom resource needs to adhere to.

Set `serviceToken` to `topic.topicArn`  in order to use this provider:

```python
topic = sns.Topic(self, "MyProvider")

CustomResource(self, "MyResource",
    service_token=topic.topic_arn
)
```

#### AWS Lambda Function

An AWS lambda function is called *directly* by CloudFormation for all resource
events. The handler must take care of explicitly submitting a success/failure
response to the CloudFormation service and handle various error cases.

> **We do not recommend you use this provider type.** The CDK has wrappers around Lambda Functions that make them easier to work with.
>
> If you do want to use this provider, refer to the [CloudFormation Custom Resource documentation](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-custom-resources.html) for information on the contract your custom resource needs to adhere to.

Set `serviceToken` to `lambda.functionArn` to use this provider:

```python
fn = lambda_.SingletonFunction(self, "MyProvider", function_props)

CustomResource(self, "MyResource",
    service_token=fn.function_arn
)
```

#### The `core.CustomResourceProvider` class

The class [`@aws-cdk/core.CustomResourceProvider`](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_core.CustomResourceProvider.html) offers a basic low-level
framework designed to implement simple and slim custom resource providers. It
currently only supports Node.js-based user handlers, represents permissions as raw
JSON blobs instead of `iam.PolicyStatement` objects, and it does not have
support for asynchronous waiting (handler cannot exceed the 15min lambda
timeout). The `CustomResourceProviderRuntime` supports runtime `nodejs12.x`,
`nodejs14.x`, `nodejs16.x`, `nodejs18.x`.

> **As an application builder, we do not recommend you use this provider type.** This provider exists purely for custom resources that are part of the AWS Construct Library.
>
> The [`custom-resources`](#the-custom-resource-provider-framework) provider is more convenient to work with and more fully-featured.

The provider has a built-in singleton method which uses the resource type as a
stack-unique identifier and returns the service token:

```python
service_token = CustomResourceProvider.get_or_create(self, "Custom::MyCustomResourceType",
    code_directory=f"{__dirname}/my-handler",
    runtime=CustomResourceProviderRuntime.NODEJS_18_X,
    description="Lambda function created by the custom resource provider"
)

CustomResource(self, "MyResource",
    resource_type="Custom::MyCustomResourceType",
    service_token=service_token
)
```

The directory (`my-handler` in the above example) must include an `index.js` file. It cannot import
external dependencies or files outside this directory. It must export an async
function named `handler`. This function accepts the CloudFormation resource
event object and returns an object with the following structure:

```js
exports.handler = async function(event) {
  const id = event.PhysicalResourceId; // only for "Update" and "Delete"
  const props = event.ResourceProperties;
  const oldProps = event.OldResourceProperties; // only for "Update"s

  switch (event.RequestType) {
    case "Create":
      // ...

    case "Update":
      // ...

      // if an error is thrown, a FAILED response will be submitted to CFN
      throw new Error('Failed!');

    case "Delete":
      // ...
  }

  return {
    // (optional) the value resolved from `resource.ref`
    // defaults to "event.PhysicalResourceId" or "event.RequestId"
    PhysicalResourceId: "REF",

    // (optional) calling `resource.getAtt("Att1")` on the custom resource in the CDK app
    // will return the value "BAR".
    Data: {
      Att1: "BAR",
      Att2: "BAZ"
    },

    // (optional) user-visible message
    Reason: "User-visible message",

    // (optional) hides values from the console
    NoEcho: true
  };
}
```

Here is an complete example of a custom resource that summarizes two numbers:

`sum-handler/index.js`:

```js
exports.handler = async (e) => {
  return {
    Data: {
      Result: e.ResourceProperties.lhs + e.ResourceProperties.rhs,
    },
  };
};
```

`sum.ts`:

```python
from constructs import Construct
from aws_cdk import CustomResource, CustomResourceProvider, CustomResourceProviderRuntime, Token

class Sum(Construct):

    def __init__(self, scope, id, *, lhs, rhs):
        super().__init__(scope, id)

        resource_type = "Custom::Sum"
        service_token = CustomResourceProvider.get_or_create(self, resource_type,
            code_directory=f"{__dirname}/sum-handler",
            runtime=CustomResourceProviderRuntime.NODEJS_18_X
        )

        resource = CustomResource(self, "Resource",
            resource_type=resource_type,
            service_token=service_token,
            properties={
                "lhs": lhs,
                "rhs": rhs
            }
        )

        self.result = Token.as_number(resource.get_att("Result"))
```

Usage will look like this:

```python
sum = Sum(self, "MySum", lhs=40, rhs=2)
CfnOutput(self, "Result", value=Token.as_string(sum.result))
```

To access the ARN of the provider's AWS Lambda function role, use the `getOrCreateProvider()`
built-in singleton method:

```python
provider = CustomResourceProvider.get_or_create_provider(self, "Custom::MyCustomResourceType",
    code_directory=f"{__dirname}/my-handler",
    runtime=CustomResourceProviderRuntime.NODEJS_18_X
)

role_arn = provider.role_arn
```

This role ARN can then be used in resource-based IAM policies.

To add IAM policy statements to this role, use `addToRolePolicy()`:

```python
provider = CustomResourceProvider.get_or_create_provider(self, "Custom::MyCustomResourceType",
    code_directory=f"{__dirname}/my-handler",
    runtime=CustomResourceProviderRuntime.NODEJS_18_X
)
provider.add_to_role_policy({
    "Effect": "Allow",
    "Action": "s3:GetObject",
    "Resource": "*"
})
```

Note that `addToRolePolicy()` uses direct IAM JSON policy blobs, *not* a
`iam.PolicyStatement` object like you will see in the rest of the CDK.

#### The Custom Resource Provider Framework

The [`@aws-cdk/custom-resources`](https://docs.aws.amazon.com/cdk/api/latest/docs/custom-resources-readme.html) module includes an advanced framework for
implementing custom resource providers.

Handlers are implemented as AWS Lambda functions, which means that they can be
implemented in any Lambda-supported runtime. Furthermore, this provider has an
asynchronous mode, which means that users can provide an `isComplete` lambda
function which is called periodically until the operation is complete. This
allows implementing providers that can take up to two hours to stabilize.

Set `serviceToken` to `provider.serviceToken` to use this type of provider:

```python
provider = customresources.Provider(self, "MyProvider",
    on_event_handler=on_event_handler,
    is_complete_handler=is_complete_handler
)

CustomResource(self, "MyResource",
    service_token=provider.service_token
)
```

See the [documentation](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-cdk-lib.custom_resources-readme.html) for more details.

## AWS CloudFormation features

A CDK stack synthesizes to an AWS CloudFormation Template. This section
explains how this module allows users to access low-level CloudFormation
features when needed.

### Stack Outputs

CloudFormation [stack outputs](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html) and exports are created using
the `CfnOutput` class:

```python
CfnOutput(self, "OutputName",
    value=my_bucket.bucket_name,
    description="The name of an S3 bucket",  # Optional
    export_name="TheAwesomeBucket"
)
```

### Parameters

CloudFormation templates support the use of [Parameters](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html) to
customize a template. They enable CloudFormation users to input custom values to
a template each time a stack is created or updated. While the CDK design
philosophy favors using build-time parameterization, users may need to use
CloudFormation in a number of cases (for example, when migrating an existing
stack to the AWS CDK).

Template parameters can be added to a stack by using the `CfnParameter` class:

```python
CfnParameter(self, "MyParameter",
    type="Number",
    default=1337
)
```

The value of parameters can then be obtained using one of the `value` methods.
As parameters are only resolved at deployment time, the values obtained are
placeholder tokens for the real value (`Token.isUnresolved()` would return `true`
for those):

```python
param = CfnParameter(self, "ParameterName")

# If the parameter is a String
param.value_as_string

# If the parameter is a Number
param.value_as_number

# If the parameter is a List
param.value_as_list
```

### Pseudo Parameters

CloudFormation supports a number of [pseudo parameters](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/pseudo-parameter-reference.html),
which resolve to useful values at deployment time. CloudFormation pseudo
parameters can be obtained from static members of the `Aws` class.

It is generally recommended to access pseudo parameters from the scope's `stack`
instead, which guarantees the values produced are qualifying the designated
stack, which is essential in cases where resources are shared cross-stack:

```python
# "this" is the current construct
stack = Stack.of(self)

stack.account # Returns the AWS::AccountId for this stack (or the literal value if known)
stack.region # Returns the AWS::Region for this stack (or the literal value if known)
stack.partition
```

### Resource Options

CloudFormation resources can also specify [resource
attributes](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-product-attribute-reference.html). The `CfnResource` class allows
accessing those through the `cfnOptions` property:

```python
raw_bucket = s3.CfnBucket(self, "Bucket")
# -or-
raw_bucket_alt = my_bucket.node.default_child

# then
raw_bucket.cfn_options.condition = CfnCondition(self, "EnableBucket")
raw_bucket.cfn_options.metadata = {
    "metadata_key": "MetadataValue"
}
```

Resource dependencies (the `DependsOn` attribute) is modified using the
`cfnResource.addDependency` method:

```python
resource_a = CfnResource(self, "ResourceA", resource_props)
resource_b = CfnResource(self, "ResourceB", resource_props)

resource_b.add_dependency(resource_a)
```

#### CreationPolicy

Some resources support a [CreationPolicy](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html) to be specified as a CfnOption.

The creation policy is invoked only when AWS CloudFormation creates the associated resource. Currently, the only AWS CloudFormation resources that support creation policies are `CfnAutoScalingGroup`, `CfnInstance`, `CfnWaitCondition` and `CfnFleet`.

The `CfnFleet` resource from the `aws-appstream` module supports specifying `startFleet` as
a property of the creationPolicy on the resource options. Setting it to true will make AWS CloudFormation wait until the fleet is started before continuing with the creation of
resources that depend on the fleet resource.

```python
fleet = appstream.CfnFleet(self, "Fleet",
    instance_type="stream.standard.small",
    name="Fleet",
    compute_capacity=appstream.CfnFleet.ComputeCapacityProperty(
        desired_instances=1
    ),
    image_name="AppStream-AmazonLinux2-09-21-2022"
)
fleet.cfn_options.creation_policy = CfnCreationPolicy(
    start_fleet=True
)
```

The properties passed to the level 2 constructs `AutoScalingGroup` and `Instance` from the
`aws-ec2` module abstract what is passed into the `CfnOption` properties `resourceSignal` and
`autoScalingCreationPolicy`, but when using level 1 constructs you can specify these yourself.

The CfnWaitCondition resource from the `aws-cloudformation` module suppports the `resourceSignal`.
The format of the timeout is `PT#H#M#S`. In the example below AWS Cloudformation will wait for
3 success signals to occur within 15 minutes before the status of the resource will be set to
`CREATE_COMPLETE`.

```python
# resource: CfnResource


resource.cfn_options.creation_policy = CfnCreationPolicy(
    resource_signal=CfnResourceSignal(
        count=3,
        timeout="PR15M"
    )
)
```

### Intrinsic Functions and Condition Expressions

CloudFormation supports [intrinsic functions](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference.html). These functions
can be accessed from the `Fn` class, which provides type-safe methods for each
intrinsic function as well as condition expressions:

```python
# my_object_or_array: Any
# my_array: Any


# To use Fn::Base64
Fn.base64("SGVsbG8gQ0RLIQo=")

# To compose condition expressions:
environment_parameter = CfnParameter(self, "Environment")
Fn.condition_and(
    # The "Environment" CloudFormation template parameter evaluates to "Production"
    Fn.condition_equals("Production", environment_parameter),
    # The AWS::Region pseudo-parameter value is NOT equal to "us-east-1"
    Fn.condition_not(Fn.condition_equals("us-east-1", Aws.REGION)))

# To use Fn::ToJsonString
Fn.to_json_string(my_object_or_array)

# To use Fn::Length
Fn.len(Fn.split(",", my_array))
```

When working with deploy-time values (those for which `Token.isUnresolved`
returns `true`), idiomatic conditionals from the programming language cannot be
used (the value will not be known until deployment time). When conditional logic
needs to be expressed with un-resolved values, it is necessary to use
CloudFormation conditions by means of the `CfnCondition` class:

```python
environment_parameter = CfnParameter(self, "Environment")
is_prod = CfnCondition(self, "IsProduction",
    expression=Fn.condition_equals("Production", environment_parameter)
)

# Configuration value that is a different string based on IsProduction
stage = Fn.condition_if(is_prod.logical_id, "Beta", "Prod").to_string()

# Make Bucket creation condition to IsProduction by accessing
# and overriding the CloudFormation resource
bucket = s3.Bucket(self, "Bucket")
cfn_bucket = my_bucket.node.default_child
cfn_bucket.cfn_options.condition = is_prod
```

### Mappings

CloudFormation [mappings](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html) are created and queried using the
`CfnMappings` class:

```python
region_table = CfnMapping(self, "RegionTable",
    mapping={
        "us-east-1": {
            "region_name": "US East (N. Virginia)"
        },
        "us-east-2": {
            "region_name": "US East (Ohio)"
        }
    }
)

region_table.find_in_map(Aws.REGION, "regionName")
```

This will yield the following template:

```yaml
Mappings:
  RegionTable:
    us-east-1:
      regionName: US East (N. Virginia)
    us-east-2:
      regionName: US East (Ohio)
```

Mappings can also be synthesized "lazily"; lazy mappings will only render a "Mappings"
section in the synthesized CloudFormation template if some `findInMap` call is unable to
immediately return a concrete value due to one or both of the keys being unresolved tokens
(some value only available at deploy-time).

For example, the following code will not produce anything in the "Mappings" section. The
call to `findInMap` will be able to resolve the value during synthesis and simply return
`'US East (Ohio)'`.

```python
region_table = CfnMapping(self, "RegionTable",
    mapping={
        "us-east-1": {
            "region_name": "US East (N. Virginia)"
        },
        "us-east-2": {
            "region_name": "US East (Ohio)"
        }
    },
    lazy=True
)

region_table.find_in_map("us-east-2", "regionName")
```

On the other hand, the following code will produce the "Mappings" section shown above,
since the top-level key is an unresolved token. The call to `findInMap` will return a token that resolves to
`{ "Fn::FindInMap": [ "RegionTable", { "Ref": "AWS::Region" }, "regionName" ] }`.

```python
# region_table: CfnMapping


region_table.find_in_map(Aws.REGION, "regionName")
```

An optional default value can also be passed to `findInMap`. If either key is not found in the map and the mapping is lazy, `findInMap` will return the default value and not render the mapping.
If the mapping is not lazy or either key is an unresolved token, the call to `findInMap` will return a token that resolves to
`{ "Fn::FindInMap": [ "MapName", "TopLevelKey", "SecondLevelKey", { "DefaultValue": "DefaultValue" } ] }`, and the mapping will be rendered.
Note that the `AWS::LanguageExtentions` transform is added to enable the default value functionality.

For example, the following code will again not produce anything in the "Mappings" section. The
call to `findInMap` will be able to resolve the value during synthesis and simply return
`'Region not found'`.

```python
region_table = CfnMapping(self, "RegionTable",
    mapping={
        "us-east-1": {
            "region_name": "US East (N. Virginia)"
        },
        "us-east-2": {
            "region_name": "US East (Ohio)"
        }
    },
    lazy=True
)

region_table.find_in_map("us-west-1", "regionName", "Region not found")
```

### Dynamic References

CloudFormation supports [dynamically resolving](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html) values
for SSM parameters (including secure strings) and Secrets Manager. Encoding such
references is done using the `CfnDynamicReference` class:

```python
CfnDynamicReference(CfnDynamicReferenceService.SECRETS_MANAGER, "secret-id:secret-string:json-key:version-stage:version-id")
```

### Template Options & Transform

CloudFormation templates support a number of options, including which Macros or
[Transforms](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html) to use when deploying the stack. Those can be
configured using the `stack.templateOptions` property:

```python
stack = Stack(app, "StackName")

stack.template_options.description = "This will appear in the AWS console"
stack.template_options.transforms = ["AWS::Serverless-2016-10-31"]
stack.template_options.metadata = {
    "metadata_key": "MetadataValue"
}
```

### Emitting Raw Resources

The `CfnResource` class allows emitting arbitrary entries in the
[Resources](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html) section of the CloudFormation template.

```python
CfnResource(self, "ResourceId",
    type="AWS::S3::Bucket",
    properties={
        "BucketName": "amzn-s3-demo-bucket"
    }
)
```

As for any other resource, the logical ID in the CloudFormation template will be
generated by the AWS CDK, but the type and properties will be copied verbatim in
the synthesized template.

### Including raw CloudFormation template fragments

When migrating a CloudFormation stack to the AWS CDK, it can be useful to
include fragments of an existing template verbatim in the synthesized template.
This can be achieved using the `CfnInclude` class.

```python
CfnInclude(self, "ID",
    template={
        "Resources": {
            "Bucket": {
                "Type": "AWS::S3::Bucket",
                "Properties": {
                    "BucketName": "amzn-s3-demo-bucket"
                }
            }
        }
    }
)
```

### Termination Protection

You can prevent a stack from being accidentally deleted by enabling termination
protection on the stack. If a user attempts to delete a stack with termination
protection enabled, the deletion fails and the stack--including its status--remains
unchanged. Enabling or disabling termination protection on a stack sets it for any
nested stacks belonging to that stack as well. You can enable termination protection
on a stack by setting the `terminationProtection` prop to `true`.

```python
stack = Stack(app, "StackName",
    termination_protection=True
)
```

You can also set termination protection with the setter after you've instantiated the stack.

```python
stack = Stack(app, "StackName")
stack.termination_protection = True
```

By default, termination protection is disabled.

### Description

You can add a description of the stack in the same way as `StackProps`.

```python
stack = Stack(app, "StackName",
    description="This is a description."
)
```

### CfnJson

`CfnJson` allows you to postpone the resolution of a JSON blob from
deployment-time. This is useful in cases where the CloudFormation JSON template
cannot express a certain value.

A common example is to use `CfnJson` in order to render a JSON map which needs
to use intrinsic functions in keys. Since JSON map keys must be strings, it is
impossible to use intrinsics in keys and `CfnJson` can help.

The following example defines an IAM role which can only be assumed by
principals that are tagged with a specific tag.

```python
tag_param = CfnParameter(self, "TagName")

string_equals = CfnJson(self, "ConditionJson",
    value={
        "f"aws:PrincipalTag/{tagParam.valueAsString}"": True
    }
)

principal = iam.AccountRootPrincipal().with_conditions({
    "StringEquals": string_equals
})

iam.Role(self, "MyRole", assumed_by=principal)
```

**Explanation**: since in this example we pass the tag name through a parameter, it
can only be resolved during deployment. The resolved value can be represented in
the template through a `{ "Ref": "TagName" }`. However, since we want to use
this value inside a [`aws:PrincipalTag/TAG-NAME`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltag)
IAM operator, we need it in the *key* of a `StringEquals` condition. JSON keys
*must be* strings, so to circumvent this limitation, we use `CfnJson`
to "delay" the rendition of this template section to deploy-time. This means
that the value of `StringEquals` in the template will be `{ "Fn::GetAtt": [ "ConditionJson", "Value" ] }`, and will only "expand" to the operator we synthesized during deployment.

### Stack Resource Limit

When deploying to AWS CloudFormation, it needs to keep in check the amount of resources being added inside a Stack. Currently it's possible to check the limits in the [AWS CloudFormation quotas](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) page.

It's possible to synthesize the project with more Resources than the allowed (or even reduce the number of Resources).

Set the context key `@aws-cdk/core:stackResourceLimit` with the proper value, being 0 for disable the limit of resources.

### Template Indentation

The AWS CloudFormation templates generated by CDK include indentation by default.
Indentation makes the templates more readable, but also increases their size,
and CloudFormation templates cannot exceed 1MB.

It's possible to reduce the size of your templates by suppressing indentation.

To do this for all templates, set the context key `@aws-cdk/core:suppressTemplateIndentation` to `true`.

To do this for a specific stack, add a `suppressTemplateIndentation: true` property to the
stack's `StackProps` parameter. You can also set this property to `false` to override
the context key setting.

## App Context

[Context values](https://docs.aws.amazon.com/cdk/v2/guide/context.html) are key-value pairs that can be associated with an app, stack, or construct.
One common use case for context is to use it for enabling/disabling [feature flags](https://docs.aws.amazon.com/cdk/v2/guide/featureflags.html). There are several places
where context can be specified. They are listed below in the order they are evaluated (items at the
top take precedence over those below).

* The `node.setContext()` method
* The `postCliContext` prop when you create an `App`
* The CLI via the `--context` CLI argument
* The `cdk.json` file via the `context` key:
* The `cdk.context.json` file:
* The `~/.cdk.json` file via the `context` key:
* The `context` prop when you create an `App`

### Examples of setting context

```python
App(
    context={
        "@aws-cdk/core:newStyleStackSynthesis": True
    }
)
```

```python
app = App()
app.node.set_context("@aws-cdk/core:newStyleStackSynthesis", True)
```

```python
App(
    post_cli_context={
        "@aws-cdk/core:newStyleStackSynthesis": True
    }
)
```

```console
cdk synth --context @aws-cdk/core:newStyleStackSynthesis=true
```

*cdk.json*

```json
{
  "context": {
    "@aws-cdk/core:newStyleStackSynthesis": true
  }
}
```

*cdk.context.json*

```json
{
  "@aws-cdk/core:newStyleStackSynthesis": true
}
```

*~/.cdk.json*

```json
{
  "context": {
    "@aws-cdk/core:newStyleStackSynthesis": true
  }
}
```

## IAM Permissions Boundary

It is possible to apply an [IAM permissions boundary](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html)
to all roles within a specific construct scope. The most common use case would
be to apply a permissions boundary at the `Stage` level.

```python
prod_stage = Stage(app, "ProdStage",
    permissions_boundary=PermissionsBoundary.from_name("cdk-${Qualifier}-PermissionsBoundary")
)
```

Any IAM Roles or Users created within this Stage will have the default
permissions boundary attached.

For more details see the [Permissions Boundary](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_iam-readme.html#permissions-boundaries) section in the IAM guide.

## Policy Validation

If you or your organization use (or would like to use) any policy validation tool, such as
[CloudFormation
Guard](https://docs.aws.amazon.com/cfn-guard/latest/ug/what-is-guard.html) or
[OPA](https://www.openpolicyagent.org/), to define constraints on your
CloudFormation template, you can incorporate them into the CDK application.
By using the appropriate plugin, you can make the CDK application check the
generated CloudFormation templates against your policies immediately after
synthesis. If there are any violations, the synthesis will fail and a report
will be printed to the console or to a file (see below).

> **Note**
> This feature is considered experimental, and both the plugin API and the
> format of the validation report are subject to change in the future.

### For application developers

To use one or more validation plugins in your application, use the
`policyValidationBeta1` property of `Stage`:

```python
# globally for the entire app (an app is a stage)
app = App(
    policy_validation_beta1=[
        # These hypothetical classes implement IPolicyValidationPluginBeta1:
        ThirdPartyPluginX(),
        ThirdPartyPluginY()
    ]
)

# only apply to a particular stage
prod_stage = Stage(app, "ProdStage",
    policy_validation_beta1=[
        ThirdPartyPluginX()
    ]
)
```

Immediately after synthesis, all plugins registered this way will be invoked to
validate all the templates generated in the scope you defined. In particular, if
you register the templates in the `App` object, all templates will be subject to
validation.

> **Warning**
> Other than modifying the cloud assembly, plugins can do anything that your CDK
> application can. They can read data from the filesystem, access the network
> etc. It's your responsibility as the consumer of a plugin to verify that it is
> secure to use.

By default, the report will be printed in a human readable format. If you want a
report in JSON format, enable it using the `@aws-cdk/core:validationReportJson`
context passing it directly to the application:

```python
app = App(
    context={"@aws-cdk/core:validationReportJson": True}
)
```

Alternatively, you can set this context key-value pair using the `cdk.json` or
`cdk.context.json` files in your project directory (see
[Runtime context](https://docs.aws.amazon.com/cdk/v2/guide/context.html)).

If you choose the JSON format, the CDK will print the policy validation report
to a file called `policy-validation-report.json` in the cloud assembly
directory. For the default, human-readable format, the report will be printed to
the standard output.

### For plugin authors

The communication protocol between the CDK core module and your policy tool is
defined by the `IPolicyValidationPluginBeta1` interface. To create a new plugin you must
write a class that implements this interface. There are two things you need to
implement: the plugin name (by overriding the `name` property), and the
`validate()` method.

The framework will call `validate()`, passing an `IPolicyValidationContextBeta1` object.
The location of the templates to be validated is given by `templatePaths`. The
plugin should return an instance of `PolicyValidationPluginReportBeta1`. This object
represents the report that the user wil receive at the end of the synthesis.

```python
@jsii.implements(IPolicyValidationPluginBeta1)
class MyPlugin:

    def validate(self, context):
        # First read the templates using context.templatePaths...

        # ...then perform the validation, and then compose and return the report.
        # Using hard-coded values here for better clarity:
        return PolicyValidationPluginReportBeta1(
            success=False,
            violations=[PolicyViolationBeta1(
                rule_name="CKV_AWS_117",
                description="Ensure that AWS Lambda function is configured inside a VPC",
                fix="https://docs.bridgecrew.io/docs/ensure-that-aws-lambda-function-is-configured-inside-a-vpc-1",
                violating_resources=[PolicyViolatingResourceBeta1(
                    resource_logical_id="MyFunction3BAA72D1",
                    template_path="/home/johndoe/myapp/cdk.out/MyService.template.json",
                    locations=["Properties/VpcConfig"]
                )]
            )]
        )
```

In addition to the name, plugins may optionally report their version (`version`
property ) and a list of IDs of the rules they are going to evaluate (`ruleIds`
property).

Note that plugins are not allowed to modify anything in the cloud assembly. Any
attempt to do so will result in synthesis failure.

If your plugin depends on an external tool, keep in mind that some developers may
not have that tool installed in their workstations yet. To minimize friction, we
highly recommend that you provide some installation script along with your
plugin package, to automate the whole process. Better yet, run that script as
part of the installation of your package. With `npm`, for example, you can run
add it to the `postinstall`
[script](https://docs.npmjs.com/cli/v9/using-npm/scripts) in the `package.json`
file.

## Annotations

Construct authors can add annotations to constructs to report at three different
levels: `ERROR`, `WARN`, `INFO`.

Typically warnings are added for things that are important for the user to be
aware of, but will not cause deployment errors in all cases. Some common
scenarios are (non-exhaustive list):

* Warn when the user needs to take a manual action, e.g. IAM policy should be
  added to an referenced resource.
* Warn if the user configuration might not follow best practices (but is still
  valid)
* Warn if the user is using a deprecated API

### Acknowledging Warnings

If you would like to run with `--strict` mode enabled (warnings will throw
errors) it is possible to `acknowledge` warnings to make the warning go away.

For example, if > 10 IAM managed policies are added to an IAM Group, a warning
will be created:

```text
IAM:Group:MaxPoliciesExceeded: You added 11 to IAM Group my-group. The maximum number of managed policies attached to an IAM group is 10.
```

If you have requested a [quota increase](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-quotas.html#reference_iam-quotas-entities)
you may have the ability to add > 10 managed policies which means that this
warning does not apply to you. You can acknowledge this by `acknowledging` the
warning by the `id`.

```python
Annotations.of(self).acknowledge_warning("IAM:Group:MaxPoliciesExceeded", "Account has quota increased to 20")
```

<!--END CORE DOCUMENTATION-->
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from ._jsii import *

import constructs as _constructs_77d1e7e8
from .cloud_assembly_schema import (
    AmiContextQuery as _AmiContextQuery_74bf4b1b,
    AssetManifestOptions as _AssetManifestOptions_73b93270,
    AvailabilityZonesContextQuery as _AvailabilityZonesContextQuery_715a9fea,
    BootstrapRole as _BootstrapRole_9b326056,
    CcApiContextQuery as _CcApiContextQuery_7347fbd4,
    ContextProvider as _ContextProvider_fa789bb5,
    DockerImageDestination as _DockerImageDestination_132046c7,
    DockerImageSource as _DockerImageSource_5db2cfa3,
    EndpointServiceAvailabilityZonesContextQuery as _EndpointServiceAvailabilityZonesContextQuery_ea3ca0d1,
    FileDestination as _FileDestination_7d285b38,
    FileSource as _FileSource_66254048,
    HostedZoneContextQuery as _HostedZoneContextQuery_8e6ca28f,
    KeyContextQuery as _KeyContextQuery_3ac6128d,
    LoadBalancerContextQuery as _LoadBalancerContextQuery_cb08d67c,
    LoadBalancerListenerContextQuery as _LoadBalancerListenerContextQuery_0eaf3c16,
    MissingContext as _MissingContext_0ff9e334,
    PluginContextQuery as _PluginContextQuery_31a9d073,
    SSMParameterContextQuery as _SSMParameterContextQuery_675de122,
    SecurityGroupContextQuery as _SecurityGroupContextQuery_e772f3e6,
    VpcContextQuery as _VpcContextQuery_a193c650,
)
from .cx_api import (
    CloudAssembly as _CloudAssembly_c693643e,
    CloudAssemblyBuilder as _CloudAssemblyBuilder_c90cccf3,
)


@jsii.data_type(
    jsii_type="aws-cdk-lib.AddDockerImageAssetOptions",
    jsii_struct_bases=[],
    name_mapping={"display_name": "displayName"},
)
class AddDockerImageAssetOptions:
    def __init__(self, *, display_name: typing.Optional[builtins.str] = None) -> None:
        '''Options for the addDockerImageAsset operation.

        :param display_name: A display name to associate with the asset. Default: - No display name

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            add_docker_image_asset_options = cdk.AddDockerImageAssetOptions(
                display_name="displayName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5a1ac212aad6af3e005718717aa2d00b8a797e24b74644f38d11d9f5ced18b43)
            check_type(argname="argument display_name", value=display_name, expected_type=type_hints["display_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if display_name is not None:
            self._values["display_name"] = display_name

    @builtins.property
    def display_name(self) -> typing.Optional[builtins.str]:
        '''A display name to associate with the asset.

        :default: - No display name
        '''
        result = self._values.get("display_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddDockerImageAssetOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.AddFileAssetOptions",
    jsii_struct_bases=[],
    name_mapping={"display_name": "displayName"},
)
class AddFileAssetOptions:
    def __init__(self, *, display_name: typing.Optional[builtins.str] = None) -> None:
        '''Options for the addFileAsset operation.

        :param display_name: A display name to associate with the asset. Default: - No display name

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            add_file_asset_options = cdk.AddFileAssetOptions(
                display_name="displayName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3b7bb14c4d35bab2918fcff6fafc6bf8560391fa80cf7e743756928a3ff744a2)
            check_type(argname="argument display_name", value=display_name, expected_type=type_hints["display_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if display_name is not None:
            self._values["display_name"] = display_name

    @builtins.property
    def display_name(self) -> typing.Optional[builtins.str]:
        '''A display name to associate with the asset.

        :default: - No display name
        '''
        result = self._values.get("display_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddFileAssetOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Annotations(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.Annotations"):
    '''Includes API for attaching annotations such as warning messages to constructs.

    :exampleMetadata: nofixture infused

    Example::

        import aws_cdk as cdk
        from constructs import Construct, IConstruct
        
        @jsii.implements(cdk.IAspect)
        class MyAspect:
            def visit(self, node):
                if node instanceof cdk.CfnResource && node.cfn_resource_type == "Foo::Bar":
                    self.error(node, "we do not want a Foo::Bar resource")
        
            def error(self, node, message):
                cdk.Annotations.of(node).add_error(message)
        
        class MyStack(cdk.Stack):
            def __init__(self, scope, id):
                super().__init__(scope, id)
        
                stack = cdk.Stack()
                cdk.CfnResource(stack, "Foo",
                    type="Foo::Bar",
                    properties={
                        "Fred": "Thud"
                    }
                )
                cdk.Aspects.of(stack).add(MyAspect())
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, scope: _constructs_77d1e7e8.IConstruct) -> "Annotations":
        '''Returns the annotations API for a construct scope.

        :param scope: The scope.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cfddeb4c359528028785fb7ca8a01e86bcda81d53c82cdaef6ad18dd0520ab91)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("Annotations", jsii.sinvoke(cls, "of", [scope]))

    @jsii.member(jsii_name="acknowledgeWarning")
    def acknowledge_warning(
        self,
        id: builtins.str,
        message: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Acknowledge a warning. When a warning is acknowledged for a scope all warnings that match the id will be ignored.

        The acknowledgement will apply to all child scopes

        :param id: - the id of the warning message to acknowledge.
        :param message: optional message to explain the reason for acknowledgement.

        Example::

            # my_construct: Construct
            
            Annotations.of(my_construct).acknowledge_warning("SomeWarningId", "This warning can be ignored because...")
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__58bb467dabbe29a0334a62f44fa1a8a9b59742b9dfac031624ed782055bf0ada)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument message", value=message, expected_type=type_hints["message"])
        return typing.cast(None, jsii.invoke(self, "acknowledgeWarning", [id, message]))

    @jsii.member(jsii_name="addDeprecation")
    def add_deprecation(self, api: builtins.str, message: builtins.str) -> None:
        '''Adds a deprecation warning for a specific API.

        Deprecations will be added only once per construct as a warning and will be
        deduplicated based on the ``api``.

        If the environment variable ``CDK_BLOCK_DEPRECATIONS`` is set, this method
        will throw an error instead with the deprecation message.

        :param api: The API being deprecated in the format ``module.Class.property`` (e.g. ``@aws-cdk/core.Construct.node``).
        :param message: The deprecation message to display, with information about alternatives.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b88ded54077473171d54e92828f09c821d1807418d5e202991064bb2569c24c2)
            check_type(argname="argument api", value=api, expected_type=type_hints["api"])
            check_type(argname="argument message", value=message, expected_type=type_hints["message"])
        return typing.cast(None, jsii.invoke(self, "addDeprecation", [api, message]))

    @jsii.member(jsii_name="addError")
    def add_error(self, message: builtins.str) -> None:
        '''Adds an { "error":  } metadata entry to this construct.

        The toolkit will fail deployment of any stack that has errors reported against it.

        :param message: The error message.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9858aa055da72c40f4a88e5f9f9ee8f0faebb6aa694ede90702a9558a24801e6)
            check_type(argname="argument message", value=message, expected_type=type_hints["message"])
        return typing.cast(None, jsii.invoke(self, "addError", [message]))

    @jsii.member(jsii_name="addInfo")
    def add_info(self, message: builtins.str) -> None:
        '''Adds an info metadata entry to this construct.

        The CLI will display the info message when apps are synthesized.

        :param message: The info message.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42febf8c4cbec6748c47fab1f601511d3bc57ef5bc415cb489e10605f242ffa4)
            check_type(argname="argument message", value=message, expected_type=type_hints["message"])
        return typing.cast(None, jsii.invoke(self, "addInfo", [message]))

    @jsii.member(jsii_name="addWarning")
    def add_warning(self, message: builtins.str) -> None:
        '''Adds a warning metadata entry to this construct. Prefer using ``addWarningV2``.

        The CLI will display the warning when an app is synthesized, or fail if run
        in ``--strict`` mode.

        Warnings added by this call cannot be acknowledged. This will block users from
        running in ``--strict`` mode until the deal with the warning, which makes it
        effectively not very different from ``addError``. Prefer using ``addWarningV2`` instead.

        :param message: The warning message.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e2fdcebb577d63295330a53b495fc6906cfbd98497144c53b1cb802e3e082f72)
            check_type(argname="argument message", value=message, expected_type=type_hints["message"])
        return typing.cast(None, jsii.invoke(self, "addWarning", [message]))

    @jsii.member(jsii_name="addWarningV2")
    def add_warning_v2(self, id: builtins.str, message: builtins.str) -> None:
        '''Adds an acknowledgeable warning metadata entry to this construct.

        The CLI will display the warning when an app is synthesized, or fail if run
        in ``--strict`` mode.

        If the warning is acknowledged using ``acknowledgeWarning()``, it will not be shown by
        the CLI, and will not cause ``--strict`` mode to fail synthesis.

        :param id: the unique identifier for the warning. This can be used to acknowledge the warning
        :param message: The warning message.

        Example::

            # my_construct: Construct
            
            Annotations.of(my_construct).add_warning_v2("my-library:Construct.someWarning", "Some message explaining the warning")
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__433aa46ed24ea9868fb9a2456cff447934d7579c56de53736ca48c128282d48d)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument message", value=message, expected_type=type_hints["message"])
        return typing.cast(None, jsii.invoke(self, "addWarningV2", [id, message]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.AppProps",
    jsii_struct_bases=[],
    name_mapping={
        "analytics_reporting": "analyticsReporting",
        "auto_synth": "autoSynth",
        "context": "context",
        "default_stack_synthesizer": "defaultStackSynthesizer",
        "outdir": "outdir",
        "policy_validation_beta1": "policyValidationBeta1",
        "post_cli_context": "postCliContext",
        "stack_traces": "stackTraces",
        "tree_metadata": "treeMetadata",
    },
)
class AppProps:
    def __init__(
        self,
        *,
        analytics_reporting: typing.Optional[builtins.bool] = None,
        auto_synth: typing.Optional[builtins.bool] = None,
        context: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        default_stack_synthesizer: typing.Optional["IReusableStackSynthesizer"] = None,
        outdir: typing.Optional[builtins.str] = None,
        policy_validation_beta1: typing.Optional[typing.Sequence["IPolicyValidationPluginBeta1"]] = None,
        post_cli_context: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        stack_traces: typing.Optional[builtins.bool] = None,
        tree_metadata: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Initialization props for apps.

        :param analytics_reporting: Include runtime versioning information in the Stacks of this app. Default: Value of 'aws:cdk:version-reporting' context key
        :param auto_synth: Automatically call ``synth()`` before the program exits. If you set this, you don't have to call ``synth()`` explicitly. Note that this feature is only available for certain programming languages, and calling ``synth()`` is still recommended. Default: true if running via CDK CLI (``CDK_OUTDIR`` is set), ``false`` otherwise
        :param context: Additional context values for the application. Context set by the CLI or the ``context`` key in ``cdk.json`` has precedence. Context can be read from any construct using ``node.getContext(key)``. Default: - no additional context
        :param default_stack_synthesizer: The stack synthesizer to use by default for all Stacks in the App. The Stack Synthesizer controls aspects of synthesis and deployment, like how assets are referenced and what IAM roles to use. For more information, see the README of the main CDK package. Default: - A ``DefaultStackSynthesizer`` with default settings
        :param outdir: The output directory into which to emit synthesized artifacts. You should never need to set this value. By default, the value you pass to the CLI's ``--output`` flag will be used, and if you change it to a different directory the CLI will fail to pick up the generated Cloud Assembly. This property is intended for internal and testing use. Default: - If this value is *not* set, considers the environment variable ``CDK_OUTDIR``. If ``CDK_OUTDIR`` is not defined, uses a temp directory.
        :param policy_validation_beta1: Validation plugins to run after synthesis. Default: - no validation plugins
        :param post_cli_context: Additional context values for the application. Context provided here has precedence over context set by: - The CLI via --context - The ``context`` key in ``cdk.json`` - The ``AppProps.context`` property This property is recommended over the ``AppProps.context`` property since you can make final decision over which context value to take in your app. Context can be read from any construct using ``node.getContext(key)``. Default: - no additional context
        :param stack_traces: Include construct creation stack trace in the ``aws:cdk:trace`` metadata key of all constructs. Default: true stack traces are included unless ``aws:cdk:disable-stack-trace`` is set in the context.
        :param tree_metadata: Include construct tree metadata as part of the Cloud Assembly. Default: true

        :exampleMetadata: infused

        Example::

            from aws_cdk.aws_s3 import BucketEncryption
            
            
            app = App(
                default_stack_synthesizer=AppStagingSynthesizer.default_resources(
                    app_id="my-app-id",
                    staging_bucket_encryption=BucketEncryption.S3_MANAGED,
                    deployment_identities=DeploymentIdentities.cli_credentials()
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ef2fee5e91b22ed9e2b91aa309be73835ddfb834f19eb3d41540c7e38e232930)
            check_type(argname="argument analytics_reporting", value=analytics_reporting, expected_type=type_hints["analytics_reporting"])
            check_type(argname="argument auto_synth", value=auto_synth, expected_type=type_hints["auto_synth"])
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
            check_type(argname="argument default_stack_synthesizer", value=default_stack_synthesizer, expected_type=type_hints["default_stack_synthesizer"])
            check_type(argname="argument outdir", value=outdir, expected_type=type_hints["outdir"])
            check_type(argname="argument policy_validation_beta1", value=policy_validation_beta1, expected_type=type_hints["policy_validation_beta1"])
            check_type(argname="argument post_cli_context", value=post_cli_context, expected_type=type_hints["post_cli_context"])
            check_type(argname="argument stack_traces", value=stack_traces, expected_type=type_hints["stack_traces"])
            check_type(argname="argument tree_metadata", value=tree_metadata, expected_type=type_hints["tree_metadata"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if analytics_reporting is not None:
            self._values["analytics_reporting"] = analytics_reporting
        if auto_synth is not None:
            self._values["auto_synth"] = auto_synth
        if context is not None:
            self._values["context"] = context
        if default_stack_synthesizer is not None:
            self._values["default_stack_synthesizer"] = default_stack_synthesizer
        if outdir is not None:
            self._values["outdir"] = outdir
        if policy_validation_beta1 is not None:
            self._values["policy_validation_beta1"] = policy_validation_beta1
        if post_cli_context is not None:
            self._values["post_cli_context"] = post_cli_context
        if stack_traces is not None:
            self._values["stack_traces"] = stack_traces
        if tree_metadata is not None:
            self._values["tree_metadata"] = tree_metadata

    @builtins.property
    def analytics_reporting(self) -> typing.Optional[builtins.bool]:
        '''Include runtime versioning information in the Stacks of this app.

        :default: Value of 'aws:cdk:version-reporting' context key
        '''
        result = self._values.get("analytics_reporting")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def auto_synth(self) -> typing.Optional[builtins.bool]:
        '''Automatically call ``synth()`` before the program exits.

        If you set this, you don't have to call ``synth()`` explicitly. Note that
        this feature is only available for certain programming languages, and
        calling ``synth()`` is still recommended.

        :default:

        true if running via CDK CLI (``CDK_OUTDIR`` is set), ``false``
        otherwise
        '''
        result = self._values.get("auto_synth")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def context(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Additional context values for the application.

        Context set by the CLI or the ``context`` key in ``cdk.json`` has precedence.

        Context can be read from any construct using ``node.getContext(key)``.

        :default: - no additional context
        '''
        result = self._values.get("context")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def default_stack_synthesizer(self) -> typing.Optional["IReusableStackSynthesizer"]:
        '''The stack synthesizer to use by default for all Stacks in the App.

        The Stack Synthesizer controls aspects of synthesis and deployment,
        like how assets are referenced and what IAM roles to use. For more
        information, see the README of the main CDK package.

        :default: - A ``DefaultStackSynthesizer`` with default settings
        '''
        result = self._values.get("default_stack_synthesizer")
        return typing.cast(typing.Optional["IReusableStackSynthesizer"], result)

    @builtins.property
    def outdir(self) -> typing.Optional[builtins.str]:
        '''The output directory into which to emit synthesized artifacts.

        You should never need to set this value. By default, the value you pass to
        the CLI's ``--output`` flag will be used, and if you change it to a different
        directory the CLI will fail to pick up the generated Cloud Assembly.

        This property is intended for internal and testing use.

        :default:

        - If this value is *not* set, considers the environment variable ``CDK_OUTDIR``.
        If ``CDK_OUTDIR`` is not defined, uses a temp directory.
        '''
        result = self._values.get("outdir")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def policy_validation_beta1(
        self,
    ) -> typing.Optional[typing.List["IPolicyValidationPluginBeta1"]]:
        '''Validation plugins to run after synthesis.

        :default: - no validation plugins
        '''
        result = self._values.get("policy_validation_beta1")
        return typing.cast(typing.Optional[typing.List["IPolicyValidationPluginBeta1"]], result)

    @builtins.property
    def post_cli_context(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Additional context values for the application.

        Context provided here has precedence over context set by:

        - The CLI via --context
        - The ``context`` key in ``cdk.json``
        - The ``AppProps.context`` property

        This property is recommended over the ``AppProps.context`` property since you
        can make final decision over which context value to take in your app.

        Context can be read from any construct using ``node.getContext(key)``.

        :default: - no additional context

        Example::

            # context from the CLI and from `cdk.json` are stored in the
            # CDK_CONTEXT env variable
            cli_context = JSON.parse(process.env.CDK_CONTEXT)
            
            # determine whether to take the context passed in the CLI or not
            determine_value = process.env.PROD ? cliContext.SOMEKEY : 'my-prod-value'
            App(
                post_cli_context={
                    "SOMEKEY": determine_value
                }
            )
        '''
        result = self._values.get("post_cli_context")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def stack_traces(self) -> typing.Optional[builtins.bool]:
        '''Include construct creation stack trace in the ``aws:cdk:trace`` metadata key of all constructs.

        :default: true stack traces are included unless ``aws:cdk:disable-stack-trace`` is set in the context.
        '''
        result = self._values.get("stack_traces")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def tree_metadata(self) -> typing.Optional[builtins.bool]:
        '''Include construct tree metadata as part of the Cloud Assembly.

        :default: true
        '''
        result = self._values.get("tree_metadata")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AppProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Arn(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.Arn"):
    @jsii.member(jsii_name="extractResourceName")
    @builtins.classmethod
    def extract_resource_name(
        cls,
        arn: builtins.str,
        resource_type: builtins.str,
    ) -> builtins.str:
        '''Extract the full resource name from an ARN.

        Necessary for resource names (paths) that may contain the separator, like
        ``arn:aws:iam::111111111111:role/path/to/role/name``.

        Only works if we statically know the expected ``resourceType`` beforehand, since we're going
        to use that to split the string on ':/' (and take the right-hand side).

        We can't extract the 'resourceType' from the ARN at hand, because CloudFormation Expressions
        only allow literals in the 'separator' argument to ``{ Fn::Split }``, and so it can't be
        ``{ Fn::Select: [5, { Fn::Split: [':', ARN] }}``.

        Only necessary for ARN formats for which the type-name separator is ``/``.

        :param arn: -
        :param resource_type: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8731628e38e554481b850412fb15d4515041613e4bebe842294a4341695d6ee5)
            check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
            check_type(argname="argument resource_type", value=resource_type, expected_type=type_hints["resource_type"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "extractResourceName", [arn, resource_type]))

    @jsii.member(jsii_name="format")
    @builtins.classmethod
    def format(
        cls,
        components: typing.Union["ArnComponents", typing.Dict[builtins.str, typing.Any]],
        stack: typing.Optional["Stack"] = None,
    ) -> builtins.str:
        '''Creates an ARN from components.

        If ``partition``, ``region`` or ``account`` are not specified, the stack's
        partition, region and account will be used.

        If any component is the empty string, an empty string will be inserted
        into the generated ARN at the location that component corresponds to.

        The ARN will be formatted as follows:

        arn:{partition}:{service}:{region}:{account}:{resource}{sep}{resource-name}

        The required ARN pieces that are omitted will be taken from the stack that
        the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
        can be 'undefined'.

        :param components: -
        :param stack: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__da53df6a1f50ba3bb9f39565eb12b171bd19258e304456d7b58965c67213edc7)
            check_type(argname="argument components", value=components, expected_type=type_hints["components"])
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "format", [components, stack]))

    @jsii.member(jsii_name="split")
    @builtins.classmethod
    def split(cls, arn: builtins.str, arn_format: "ArnFormat") -> "ArnComponents":
        '''Splits the provided ARN into its components.

        Works both if 'arn' is a string like 'arn:aws:s3:::bucket',
        and a Token representing a dynamic CloudFormation expression
        (in which case the returned components will also be dynamic CloudFormation expressions,
        encoded as Tokens).

        :param arn: the ARN to split into its components.
        :param arn_format: the expected format of 'arn' - depends on what format the service 'arn' represents uses.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__093b19cac4b2bc65a0e02d89aa7a33cd402d4a69d51f899836ac665722996d0d)
            check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
            check_type(argname="argument arn_format", value=arn_format, expected_type=type_hints["arn_format"])
        return typing.cast("ArnComponents", jsii.sinvoke(cls, "split", [arn, arn_format]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.ArnComponents",
    jsii_struct_bases=[],
    name_mapping={
        "resource": "resource",
        "service": "service",
        "account": "account",
        "arn_format": "arnFormat",
        "partition": "partition",
        "region": "region",
        "resource_name": "resourceName",
    },
)
class ArnComponents:
    def __init__(
        self,
        *,
        resource: builtins.str,
        service: builtins.str,
        account: typing.Optional[builtins.str] = None,
        arn_format: typing.Optional["ArnFormat"] = None,
        partition: typing.Optional[builtins.str] = None,
        region: typing.Optional[builtins.str] = None,
        resource_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param resource: Resource type (e.g. "table", "autoScalingGroup", "certificate"). For some resource types, e.g. S3 buckets, this field defines the bucket name.
        :param service: The service namespace that identifies the AWS product (for example, 's3', 'iam', 'codepipeline').
        :param account: The ID of the AWS account that owns the resource, without the hyphens. For example, 123456789012. Note that the ARNs for some resources don't require an account number, so this component might be omitted. Default: The account the stack is deployed to.
        :param arn_format: The specific ARN format to use for this ARN value. Default: - uses value of ``sep`` as the separator for formatting, ``ArnFormat.SLASH_RESOURCE_NAME`` if that property was also not provided
        :param partition: The partition that the resource is in. For standard AWS regions, the partition is aws. If you have resources in other partitions, the partition is aws-partitionname. For example, the partition for resources in the China (Beijing) region is aws-cn. Default: The AWS partition the stack is deployed to.
        :param region: The region the resource resides in. Note that the ARNs for some resources do not require a region, so this component might be omitted. Default: The region the stack is deployed to.
        :param resource_name: Resource name or path within the resource (i.e. S3 bucket object key) or a wildcard such as ``"*"``. This is service-dependent.

        :exampleMetadata: infused

        Example::

            from aws_cdk.aws_apigatewayv2_authorizers import WebSocketIamAuthorizer
            from aws_cdk.aws_apigatewayv2_integrations import WebSocketLambdaIntegration
            
            # This function handles your connect route
            # connect_handler: lambda.Function
            
            
            web_socket_api = apigwv2.WebSocketApi(self, "WebSocketApi")
            
            web_socket_api.add_route("$connect",
                integration=WebSocketLambdaIntegration("Integration", connect_handler),
                authorizer=WebSocketIamAuthorizer()
            )
            
            # Create an IAM user (identity)
            user = iam.User(self, "User")
            
            web_socket_arn = Stack.of(self).format_arn(
                service="execute-api",
                resource=web_socket_api.api_id
            )
            
            # Grant access to the IAM user
            user.attach_inline_policy(iam.Policy(self, "AllowInvoke",
                statements=[
                    iam.PolicyStatement(
                        actions=["execute-api:Invoke"],
                        effect=iam.Effect.ALLOW,
                        resources=[web_socket_arn]
                    )
                ]
            ))
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4565cb9b5469dd4d4c1d23e54f8933087065a599bfce16e56da30ffbcbeccfc0)
            check_type(argname="argument resource", value=resource, expected_type=type_hints["resource"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument account", value=account, expected_type=type_hints["account"])
            check_type(argname="argument arn_format", value=arn_format, expected_type=type_hints["arn_format"])
            check_type(argname="argument partition", value=partition, expected_type=type_hints["partition"])
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
            check_type(argname="argument resource_name", value=resource_name, expected_type=type_hints["resource_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "resource": resource,
            "service": service,
        }
        if account is not None:
            self._values["account"] = account
        if arn_format is not None:
            self._values["arn_format"] = arn_format
        if partition is not None:
            self._values["partition"] = partition
        if region is not None:
            self._values["region"] = region
        if resource_name is not None:
            self._values["resource_name"] = resource_name

    @builtins.property
    def resource(self) -> builtins.str:
        '''Resource type (e.g. "table", "autoScalingGroup", "certificate"). For some resource types, e.g. S3 buckets, this field defines the bucket name.'''
        result = self._values.get("resource")
        assert result is not None, "Required property 'resource' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''The service namespace that identifies the AWS product (for example, 's3', 'iam', 'codepipeline').'''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def account(self) -> typing.Optional[builtins.str]:
        '''The ID of the AWS account that owns the resource, without the hyphens.

        For example, 123456789012. Note that the ARNs for some resources don't
        require an account number, so this component might be omitted.

        :default: The account the stack is deployed to.
        '''
        result = self._values.get("account")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def arn_format(self) -> typing.Optional["ArnFormat"]:
        '''The specific ARN format to use for this ARN value.

        :default:

        - uses value of ``sep`` as the separator for formatting,
        ``ArnFormat.SLASH_RESOURCE_NAME`` if that property was also not provided
        '''
        result = self._values.get("arn_format")
        return typing.cast(typing.Optional["ArnFormat"], result)

    @builtins.property
    def partition(self) -> typing.Optional[builtins.str]:
        '''The partition that the resource is in.

        For standard AWS regions, the
        partition is aws. If you have resources in other partitions, the
        partition is aws-partitionname. For example, the partition for resources
        in the China (Beijing) region is aws-cn.

        :default: The AWS partition the stack is deployed to.
        '''
        result = self._values.get("partition")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def region(self) -> typing.Optional[builtins.str]:
        '''The region the resource resides in.

        Note that the ARNs for some resources
        do not require a region, so this component might be omitted.

        :default: The region the stack is deployed to.
        '''
        result = self._values.get("region")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def resource_name(self) -> typing.Optional[builtins.str]:
        '''Resource name or path within the resource (i.e. S3 bucket object key) or a wildcard such as ``"*"``. This is service-dependent.'''
        result = self._values.get("resource_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ArnComponents(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.ArnFormat")
class ArnFormat(enum.Enum):
    '''An enum representing the various ARN formats that different services use.

    :exampleMetadata: infused

    Example::

        # stack: Stack
        
        
        # Builds "arn:<PARTITION>:lambda:<REGION>:<ACCOUNT>:function:MyFunction"
        stack.format_arn(
            service="lambda",
            resource="function",
            arn_format=ArnFormat.COLON_RESOURCE_NAME,
            resource_name="MyFunction"
        )
    '''

    NO_RESOURCE_NAME = "NO_RESOURCE_NAME"
    '''This represents a format where there is no 'resourceName' part.

    This format is used for S3 resources,
    like 'arn:aws:s3:::bucket'.
    Everything after the last colon is considered the 'resource',
    even if it contains slashes,
    like in 'arn:aws:s3:::bucket/object.zip'.
    '''
    COLON_RESOURCE_NAME = "COLON_RESOURCE_NAME"
    '''This represents a format where the 'resource' and 'resourceName' parts are separated with a colon.

    Like in: 'arn:aws:service:region:account:resource:resourceName'.
    Everything after the last colon is considered the 'resourceName',
    even if it contains slashes,
    like in 'arn:aws:apigateway:region:account:resource:/test/mydemoresource/*'.
    '''
    SLASH_RESOURCE_NAME = "SLASH_RESOURCE_NAME"
    '''This represents a format where the 'resource' and 'resourceName' parts are separated with a slash.

    Like in: 'arn:aws:service:region:account:resource/resourceName'.
    Everything after the separating slash is considered the 'resourceName',
    even if it contains colons,
    like in 'arn:aws:cognito-sync:region:account:identitypool/us-east-1:1a1a1a1a-ffff-1111-9999-12345678:bla'.
    '''
    SLASH_RESOURCE_SLASH_RESOURCE_NAME = "SLASH_RESOURCE_SLASH_RESOURCE_NAME"
    '''This represents a format where the 'resource' and 'resourceName' parts are seperated with a slash, but there is also an additional slash after the colon separating 'account' from 'resource'.

    Like in: 'arn:aws:service:region:account:/resource/resourceName'.
    Note that the leading slash is *not* included in the parsed 'resource' part.
    '''


class AspectApplication(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.AspectApplication",
):
    '''Object respresenting an Aspect application.

    Stores the Aspect, the pointer to the construct it was applied
    to, and the priority value of that Aspect.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        import constructs as constructs
        
        # aspect: cdk.IAspect
        # construct: constructs.Construct
        
        aspect_application = cdk.AspectApplication(construct, aspect, 123)
    '''

    def __init__(
        self,
        construct: _constructs_77d1e7e8.IConstruct,
        aspect: "IAspect",
        priority: jsii.Number,
    ) -> None:
        '''Initializes AspectApplication object.

        :param construct: -
        :param aspect: -
        :param priority: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9e9e7eccccdbb57b560afbe6d0b0f0653f49bc2cf8739189ca2dfd2305348a44)
            check_type(argname="argument construct", value=construct, expected_type=type_hints["construct"])
            check_type(argname="argument aspect", value=aspect, expected_type=type_hints["aspect"])
            check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
        jsii.create(self.__class__, self, [construct, aspect, priority])

    @builtins.property
    @jsii.member(jsii_name="aspect")
    def aspect(self) -> "IAspect":
        '''The Aspect that was applied.'''
        return typing.cast("IAspect", jsii.get(self, "aspect"))

    @builtins.property
    @jsii.member(jsii_name="construct")
    def construct(self) -> _constructs_77d1e7e8.IConstruct:
        '''The construct that the Aspect was applied to.'''
        return typing.cast(_constructs_77d1e7e8.IConstruct, jsii.get(self, "construct"))

    @builtins.property
    @jsii.member(jsii_name="priority")
    def priority(self) -> jsii.Number:
        '''Gets the priority value.

        Sets the priority value.
        '''
        return typing.cast(jsii.Number, jsii.get(self, "priority"))

    @priority.setter
    def priority(self, value: jsii.Number) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5caf783430302b32bfab31d65627bb13dae6c0e9f42b0a5443818db8b0503d54)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "priority", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.AspectOptions",
    jsii_struct_bases=[],
    name_mapping={"priority": "priority"},
)
class AspectOptions:
    def __init__(self, *, priority: typing.Optional[jsii.Number] = None) -> None:
        '''Options when Applying an Aspect.

        :param priority: The priority value to apply on an Aspect. Priority must be a non-negative integer. Aspects that have same priority value are not guaranteed to be executed in a consistent order. Default: AspectPriority.DEFAULT

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            aspect_options = cdk.AspectOptions(
                priority=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1761263abda35b4b2f599d4ff5122c0e7ad15a95af4498d9c6e04e78bc4a4b76)
            check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if priority is not None:
            self._values["priority"] = priority

    @builtins.property
    def priority(self) -> typing.Optional[jsii.Number]:
        '''The priority value to apply on an Aspect. Priority must be a non-negative integer.

        Aspects that have same priority value are not guaranteed to be
        executed in a consistent order.

        :default: AspectPriority.DEFAULT
        '''
        result = self._values.get("priority")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AspectOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AspectPriority(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.AspectPriority"):
    '''Default Priority values for Aspects.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        aspect_priority = cdk.AspectPriority()
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.python.classproperty
    @jsii.member(jsii_name="DEFAULT")
    def DEFAULT(cls) -> jsii.Number:
        '''Default priority for Aspects that are applied without a priority.'''
        return typing.cast(jsii.Number, jsii.sget(cls, "DEFAULT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="MUTATING")
    def MUTATING(cls) -> jsii.Number:
        '''Suggested priority for Aspects that mutate the construct tree.'''
        return typing.cast(jsii.Number, jsii.sget(cls, "MUTATING"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="READONLY")
    def READONLY(cls) -> jsii.Number:
        '''Suggested priority for Aspects that only read the construct tree.'''
        return typing.cast(jsii.Number, jsii.sget(cls, "READONLY"))


class Aspects(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.Aspects"):
    '''Aspects can be applied to CDK tree scopes and can operate on the tree before synthesis.

    :exampleMetadata: nofixture infused

    Example::

        import aws_cdk as cdk
        from constructs import Construct, IConstruct
        
        @jsii.implements(cdk.IAspect)
        class MyAspect:
            def visit(self, node):
                if node instanceof cdk.CfnResource && node.cfn_resource_type == "Foo::Bar":
                    self.error(node, "we do not want a Foo::Bar resource")
        
            def error(self, node, message):
                cdk.Annotations.of(node).add_error(message)
        
        class MyStack(cdk.Stack):
            def __init__(self, scope, id):
                super().__init__(scope, id)
        
                stack = cdk.Stack()
                cdk.CfnResource(stack, "Foo",
                    type="Foo::Bar",
                    properties={
                        "Fred": "Thud"
                    }
                )
                cdk.Aspects.of(stack).add(MyAspect())
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, scope: _constructs_77d1e7e8.IConstruct) -> "Aspects":
        '''Returns the ``Aspects`` object associated with a construct scope.

        :param scope: The scope for which these aspects will apply.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c08382b7db1bf0c2efa4c2d0aa06e663a1761fa93a12226a128ebaac5b577314)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("Aspects", jsii.sinvoke(cls, "of", [scope]))

    @jsii.member(jsii_name="add")
    def add(
        self,
        aspect: "IAspect",
        *,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Adds an aspect to apply this scope before synthesis.

        :param aspect: The aspect to add.
        :param priority: The priority value to apply on an Aspect. Priority must be a non-negative integer. Aspects that have same priority value are not guaranteed to be executed in a consistent order. Default: AspectPriority.DEFAULT
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3a0ed518e4a98c831b3b8dcdf7981d87fff56769689b00526e1e9d8df1b39f84)
            check_type(argname="argument aspect", value=aspect, expected_type=type_hints["aspect"])
        options = AspectOptions(priority=priority)

        return typing.cast(None, jsii.invoke(self, "add", [aspect, options]))

    @builtins.property
    @jsii.member(jsii_name="all")
    def all(self) -> typing.List["IAspect"]:
        '''The list of aspects which were directly applied on this scope.'''
        return typing.cast(typing.List["IAspect"], jsii.get(self, "all"))

    @builtins.property
    @jsii.member(jsii_name="applied")
    def applied(self) -> typing.List[AspectApplication]:
        '''The list of aspects with priority which were directly applied on this scope.

        Also returns inherited Aspects of this node.
        '''
        return typing.cast(typing.List[AspectApplication], jsii.get(self, "applied"))


@jsii.enum(jsii_type="aws-cdk-lib.AssetHashType")
class AssetHashType(enum.Enum):
    '''The type of asset hash.

    NOTE: the hash is used in order to identify a specific revision of the asset, and
    used for optimizing and caching deployment activities related to this asset such as
    packaging, uploading to Amazon S3, etc.
    '''

    SOURCE = "SOURCE"
    '''Based on the content of the source path.

    When bundling, use ``SOURCE`` when the content of the bundling output is not
    stable across repeated bundling operations.
    '''
    OUTPUT = "OUTPUT"
    '''Based on the content of the bundling output.

    Use ``OUTPUT`` when the source of the asset is a top level folder containing
    code and/or dependencies that are not directly linked to the asset.
    '''
    CUSTOM = "CUSTOM"
    '''Use a custom hash.'''


class AssetManifestBuilder(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.AssetManifestBuilder",
):
    '''Build an asset manifest from assets added to a stack.

    This class does not need to be used by app builders; it is only necessary for building Stack Synthesizers.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        asset_manifest_builder = cdk.AssetManifestBuilder()
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="addDockerImageAsset")
    def add_docker_image_asset(
        self,
        stack: "Stack",
        source_hash: builtins.str,
        source: typing.Union[_DockerImageSource_5db2cfa3, typing.Dict[builtins.str, typing.Any]],
        dest: typing.Union[_DockerImageDestination_132046c7, typing.Dict[builtins.str, typing.Any]],
        *,
        display_name: typing.Optional[builtins.str] = None,
    ) -> _DockerImageDestination_132046c7:
        '''Add a docker asset source and destination to the manifest.

        sourceHash should be unique for every source.

        :param stack: -
        :param source_hash: -
        :param source: -
        :param dest: -
        :param display_name: A display name to associate with the asset. Default: - No display name
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e4e4609083793a8b31752be2f457b6b1f1220145a70469bafc48a142860684a6)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
            check_type(argname="argument source_hash", value=source_hash, expected_type=type_hints["source_hash"])
            check_type(argname="argument source", value=source, expected_type=type_hints["source"])
            check_type(argname="argument dest", value=dest, expected_type=type_hints["dest"])
        options = AddDockerImageAssetOptions(display_name=display_name)

        return typing.cast(_DockerImageDestination_132046c7, jsii.invoke(self, "addDockerImageAsset", [stack, source_hash, source, dest, options]))

    @jsii.member(jsii_name="addFileAsset")
    def add_file_asset(
        self,
        stack: "Stack",
        source_hash: builtins.str,
        source: typing.Union[_FileSource_66254048, typing.Dict[builtins.str, typing.Any]],
        dest: typing.Union[_FileDestination_7d285b38, typing.Dict[builtins.str, typing.Any]],
        *,
        display_name: typing.Optional[builtins.str] = None,
    ) -> _FileDestination_7d285b38:
        '''Add a file asset source and destination to the manifest.

        sourceHash should be unique for every source.

        :param stack: -
        :param source_hash: -
        :param source: -
        :param dest: -
        :param display_name: A display name to associate with the asset. Default: - No display name
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f12f649e9db582bca540895dead136ba4f8ce5c52abc11eb50e5224da26e06f5)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
            check_type(argname="argument source_hash", value=source_hash, expected_type=type_hints["source_hash"])
            check_type(argname="argument source", value=source, expected_type=type_hints["source"])
            check_type(argname="argument dest", value=dest, expected_type=type_hints["dest"])
        options = AddFileAssetOptions(display_name=display_name)

        return typing.cast(_FileDestination_7d285b38, jsii.invoke(self, "addFileAsset", [stack, source_hash, source, dest, options]))

    @jsii.member(jsii_name="defaultAddDockerImageAsset")
    def default_add_docker_image_asset(
        self,
        stack: "Stack",
        asset: typing.Union["DockerImageAssetSource", typing.Dict[builtins.str, typing.Any]],
        target: typing.Union["AssetManifestDockerImageDestination", typing.Dict[builtins.str, typing.Any]],
        *,
        display_name: typing.Optional[builtins.str] = None,
    ) -> _DockerImageDestination_132046c7:
        '''Add a docker image asset to the manifest with default settings.

        Derive the region from the stack, use the asset hash as the key, and set the prefix.

        :param stack: -
        :param asset: -
        :param target: -
        :param display_name: A display name to associate with the asset. Default: - No display name
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d8572c82a112aaa61153d2b958b8042f9e061141b483b1b0ce6429f9f7c11296)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
            check_type(argname="argument asset", value=asset, expected_type=type_hints["asset"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        options = AddDockerImageAssetOptions(display_name=display_name)

        return typing.cast(_DockerImageDestination_132046c7, jsii.invoke(self, "defaultAddDockerImageAsset", [stack, asset, target, options]))

    @jsii.member(jsii_name="defaultAddFileAsset")
    def default_add_file_asset(
        self,
        stack: "Stack",
        asset: typing.Union["FileAssetSource", typing.Dict[builtins.str, typing.Any]],
        target: typing.Union["AssetManifestFileDestination", typing.Dict[builtins.str, typing.Any]],
        *,
        display_name: typing.Optional[builtins.str] = None,
    ) -> _FileDestination_7d285b38:
        '''Add a file asset to the manifest with default settings.

        Derive the region from the stack, use the asset hash as the key, copy the
        file extension over, and set the prefix.

        :param stack: -
        :param asset: -
        :param target: -
        :param display_name: A display name to associate with the asset. Default: - No display name
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__085bf08106379af70653604775528303d429c31168f56c2439ed96f6b61197be)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
            check_type(argname="argument asset", value=asset, expected_type=type_hints["asset"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        options = AddFileAssetOptions(display_name=display_name)

        return typing.cast(_FileDestination_7d285b38, jsii.invoke(self, "defaultAddFileAsset", [stack, asset, target, options]))

    @jsii.member(jsii_name="emitManifest")
    def emit_manifest(
        self,
        stack: "Stack",
        session: "ISynthesisSession",
        options: typing.Optional[typing.Union[_AssetManifestOptions_73b93270, typing.Dict[builtins.str, typing.Any]]] = None,
        dependencies: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> builtins.str:
        '''Write the manifest to disk, and add it to the synthesis session.

        Return the artifact id, which should be added to the ``additionalDependencies``
        field of the stack artifact.

        :param stack: -
        :param session: -
        :param options: -
        :param dependencies: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d67f8740e145fcb7d2a808cf9c9594be1269f7596a2aec58692ad8b7204d7ba9)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
            check_type(argname="argument session", value=session, expected_type=type_hints["session"])
            check_type(argname="argument options", value=options, expected_type=type_hints["options"])
            check_type(argname="argument dependencies", value=dependencies, expected_type=type_hints["dependencies"])
        return typing.cast(builtins.str, jsii.invoke(self, "emitManifest", [stack, session, options, dependencies]))

    @builtins.property
    @jsii.member(jsii_name="hasAssets")
    def has_assets(self) -> builtins.bool:
        '''Whether there are any assets registered in the manifest.'''
        return typing.cast(builtins.bool, jsii.get(self, "hasAssets"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.AssetManifestDockerImageDestination",
    jsii_struct_bases=[],
    name_mapping={
        "repository_name": "repositoryName",
        "docker_tag_prefix": "dockerTagPrefix",
        "role": "role",
    },
)
class AssetManifestDockerImageDestination:
    def __init__(
        self,
        *,
        repository_name: builtins.str,
        docker_tag_prefix: typing.Optional[builtins.str] = None,
        role: typing.Optional[typing.Union["RoleOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''The destination for a docker image asset, when it is given to the AssetManifestBuilder.

        :param repository_name: Repository name where the docker image asset should be written.
        :param docker_tag_prefix: Prefix to add to the asset hash to make the Docker image tag. Default: ''
        :param role: Role to use to perform the upload. Default: - No role

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # assume_role_additional_options: Any
            
            asset_manifest_docker_image_destination = cdk.AssetManifestDockerImageDestination(
                repository_name="repositoryName",
            
                # the properties below are optional
                docker_tag_prefix="dockerTagPrefix",
                role=cdk.RoleOptions(
                    assume_role_arn="assumeRoleArn",
            
                    # the properties below are optional
                    assume_role_additional_options={
                        "assume_role_additional_options_key": assume_role_additional_options
                    },
                    assume_role_external_id="assumeRoleExternalId"
                )
            )
        '''
        if isinstance(role, dict):
            role = RoleOptions(**role)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__af3f9945f52a9416f6654c3c4619f4fd53e19865104e0818d17ae9c84bc33af4)
            check_type(argname="argument repository_name", value=repository_name, expected_type=type_hints["repository_name"])
            check_type(argname="argument docker_tag_prefix", value=docker_tag_prefix, expected_type=type_hints["docker_tag_prefix"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "repository_name": repository_name,
        }
        if docker_tag_prefix is not None:
            self._values["docker_tag_prefix"] = docker_tag_prefix
        if role is not None:
            self._values["role"] = role

    @builtins.property
    def repository_name(self) -> builtins.str:
        '''Repository name where the docker image asset should be written.'''
        result = self._values.get("repository_name")
        assert result is not None, "Required property 'repository_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def docker_tag_prefix(self) -> typing.Optional[builtins.str]:
        '''Prefix to add to the asset hash to make the Docker image tag.

        :default: ''
        '''
        result = self._values.get("docker_tag_prefix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def role(self) -> typing.Optional["RoleOptions"]:
        '''Role to use to perform the upload.

        :default: - No role
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional["RoleOptions"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssetManifestDockerImageDestination(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.AssetManifestFileDestination",
    jsii_struct_bases=[],
    name_mapping={
        "bucket_name": "bucketName",
        "bucket_prefix": "bucketPrefix",
        "role": "role",
    },
)
class AssetManifestFileDestination:
    def __init__(
        self,
        *,
        bucket_name: builtins.str,
        bucket_prefix: typing.Optional[builtins.str] = None,
        role: typing.Optional[typing.Union["RoleOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''The destination for a file asset, when it is given to the AssetManifestBuilder.

        :param bucket_name: Bucket name where the file asset should be written.
        :param bucket_prefix: Prefix to prepend to the asset hash. Default: ''
        :param role: Role to use for uploading. Default: - current role

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # assume_role_additional_options: Any
            
            asset_manifest_file_destination = cdk.AssetManifestFileDestination(
                bucket_name="bucketName",
            
                # the properties below are optional
                bucket_prefix="bucketPrefix",
                role=cdk.RoleOptions(
                    assume_role_arn="assumeRoleArn",
            
                    # the properties below are optional
                    assume_role_additional_options={
                        "assume_role_additional_options_key": assume_role_additional_options
                    },
                    assume_role_external_id="assumeRoleExternalId"
                )
            )
        '''
        if isinstance(role, dict):
            role = RoleOptions(**role)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__962f436139f591e950d1c8a45f770169d2b36054adec08e9767d43fd71c5f6e0)
            check_type(argname="argument bucket_name", value=bucket_name, expected_type=type_hints["bucket_name"])
            check_type(argname="argument bucket_prefix", value=bucket_prefix, expected_type=type_hints["bucket_prefix"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "bucket_name": bucket_name,
        }
        if bucket_prefix is not None:
            self._values["bucket_prefix"] = bucket_prefix
        if role is not None:
            self._values["role"] = role

    @builtins.property
    def bucket_name(self) -> builtins.str:
        '''Bucket name where the file asset should be written.'''
        result = self._values.get("bucket_name")
        assert result is not None, "Required property 'bucket_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def bucket_prefix(self) -> typing.Optional[builtins.str]:
        '''Prefix to prepend to the asset hash.

        :default: ''
        '''
        result = self._values.get("bucket_prefix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def role(self) -> typing.Optional["RoleOptions"]:
        '''Role to use for uploading.

        :default: - current role
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional["RoleOptions"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssetManifestFileDestination(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.AssetOptions",
    jsii_struct_bases=[],
    name_mapping={
        "asset_hash": "assetHash",
        "asset_hash_type": "assetHashType",
        "bundling": "bundling",
    },
)
class AssetOptions:
    def __init__(
        self,
        *,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[AssetHashType] = None,
        bundling: typing.Optional[typing.Union["BundlingOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Asset hash options.

        :param asset_hash: Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: Bundle the asset by executing a command in a Docker container or a custom bundling provider. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # docker_image: cdk.DockerImage
            # local_bundling: cdk.ILocalBundling
            
            asset_options = cdk.AssetOptions(
                asset_hash="assetHash",
                asset_hash_type=cdk.AssetHashType.SOURCE,
                bundling=cdk.BundlingOptions(
                    image=docker_image,
            
                    # the properties below are optional
                    bundling_file_access=cdk.BundlingFileAccess.VOLUME_COPY,
                    command=["command"],
                    entrypoint=["entrypoint"],
                    environment={
                        "environment_key": "environment"
                    },
                    local=local_bundling,
                    network="network",
                    output_type=cdk.BundlingOutput.ARCHIVED,
                    platform="platform",
                    security_opt="securityOpt",
                    user="user",
                    volumes=[cdk.DockerVolume(
                        container_path="containerPath",
                        host_path="hostPath",
            
                        # the properties below are optional
                        consistency=cdk.DockerVolumeConsistency.CONSISTENT
                    )],
                    volumes_from=["volumesFrom"],
                    working_directory="workingDirectory"
                )
            )
        '''
        if isinstance(bundling, dict):
            bundling = BundlingOptions(**bundling)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a42ea5f201cb4a825c3274d1d0377115b168fe154398d54f5681771e0d20246a)
            check_type(argname="argument asset_hash", value=asset_hash, expected_type=type_hints["asset_hash"])
            check_type(argname="argument asset_hash_type", value=asset_hash_type, expected_type=type_hints["asset_hash_type"])
            check_type(argname="argument bundling", value=bundling, expected_type=type_hints["bundling"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if asset_hash is not None:
            self._values["asset_hash"] = asset_hash
        if asset_hash_type is not None:
            self._values["asset_hash_type"] = asset_hash_type
        if bundling is not None:
            self._values["bundling"] = bundling

    @builtins.property
    def asset_hash(self) -> typing.Optional[builtins.str]:
        '''Specify a custom hash for this asset.

        If ``assetHashType`` is set it must
        be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will
        be SHA256 hashed and encoded as hex. The resulting hash will be the asset
        hash.

        NOTE: the hash is used in order to identify a specific revision of the asset, and
        used for optimizing and caching deployment activities related to this asset such as
        packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will
        need to make sure it is updated every time the asset changes, or otherwise it is
        possible that some deployments will not be invalidated.

        :default: - based on ``assetHashType``
        '''
        result = self._values.get("asset_hash")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def asset_hash_type(self) -> typing.Optional[AssetHashType]:
        '''Specifies the type of hash to calculate for this asset.

        If ``assetHash`` is configured, this option must be ``undefined`` or
        ``AssetHashType.CUSTOM``.

        :default:

        - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is
        explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        '''
        result = self._values.get("asset_hash_type")
        return typing.cast(typing.Optional[AssetHashType], result)

    @builtins.property
    def bundling(self) -> typing.Optional["BundlingOptions"]:
        '''Bundle the asset by executing a command in a Docker container or a custom bundling provider.

        The asset path will be mounted at ``/asset-input``. The Docker
        container is responsible for putting content at ``/asset-output``.
        The content at ``/asset-output`` will be zipped and used as the
        final asset.

        :default:

        - uploaded as-is to S3 if the asset is a regular file or a .zip file,
        archived into a .zip file and uploaded to S3 otherwise
        '''
        result = self._values.get("bundling")
        return typing.cast(typing.Optional["BundlingOptions"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssetOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AssetStaging(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.AssetStaging",
):
    '''Stages a file or directory from a location on the file system into a staging directory.

    This is controlled by the context key 'aws:cdk:asset-staging' and enabled
    by the CLI by default in order to ensure that when the CDK app exists, all
    assets are available for deployment. Otherwise, if an app references assets
    in temporary locations, those will not be available when it exists (see
    https://github.com/aws/aws-cdk/issues/1716).

    The ``stagedPath`` property is a stringified token that represents the location
    of the file or directory after staging. It will be resolved only during the
    "prepare" stage and may be either the original path or the staged path
    depending on the context setting.

    The file/directory are staged based on their content hash (fingerprint). This
    means that only if content was changed, copy will happen.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        # docker_image: cdk.DockerImage
        # local_bundling: cdk.ILocalBundling
        
        asset_staging = cdk.AssetStaging(self, "MyAssetStaging",
            source_path="sourcePath",
        
            # the properties below are optional
            asset_hash="assetHash",
            asset_hash_type=cdk.AssetHashType.SOURCE,
            bundling=cdk.BundlingOptions(
                image=docker_image,
        
                # the properties below are optional
                bundling_file_access=cdk.BundlingFileAccess.VOLUME_COPY,
                command=["command"],
                entrypoint=["entrypoint"],
                environment={
                    "environment_key": "environment"
                },
                local=local_bundling,
                network="network",
                output_type=cdk.BundlingOutput.ARCHIVED,
                platform="platform",
                security_opt="securityOpt",
                user="user",
                volumes=[cdk.DockerVolume(
                    container_path="containerPath",
                    host_path="hostPath",
        
                    # the properties below are optional
                    consistency=cdk.DockerVolumeConsistency.CONSISTENT
                )],
                volumes_from=["volumesFrom"],
                working_directory="workingDirectory"
            ),
            exclude=["exclude"],
            extra_hash="extraHash",
            follow=cdk.SymlinkFollowMode.NEVER,
            ignore_mode=cdk.IgnoreMode.GLOB
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        source_path: builtins.str,
        extra_hash: typing.Optional[builtins.str] = None,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[AssetHashType] = None,
        bundling: typing.Optional[typing.Union["BundlingOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional["SymlinkFollowMode"] = None,
        ignore_mode: typing.Optional["IgnoreMode"] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param source_path: The source file or directory to copy from.
        :param extra_hash: Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param asset_hash: Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: Bundle the asset by executing a command in a Docker container or a custom bundling provider. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0f053317f7c7fa8d5fea51d4f55642f8c1c852f5f6a55a582cd7d0a4bc1a77c1)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AssetStagingProps(
            source_path=source_path,
            extra_hash=extra_hash,
            asset_hash=asset_hash,
            asset_hash_type=asset_hash_type,
            bundling=bundling,
            exclude=exclude,
            follow=follow,
            ignore_mode=ignore_mode,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="clearAssetHashCache")
    @builtins.classmethod
    def clear_asset_hash_cache(cls) -> None:
        '''Clears the asset hash cache.'''
        return typing.cast(None, jsii.sinvoke(cls, "clearAssetHashCache", []))

    @jsii.member(jsii_name="relativeStagedPath")
    def relative_staged_path(self, stack: "Stack") -> builtins.str:
        '''Return the path to the staged asset, relative to the Cloud Assembly (manifest) directory of the given stack.

        Only returns a relative path if the asset was staged, returns an absolute path if
        it was not staged.

        A bundled asset might end up in the outDir and still not count as
        "staged"; if asset staging is disabled we're technically expected to
        reference source directories, but we don't have a source directory for the
        bundled outputs (as the bundle output is written to a temporary
        directory). Nevertheless, we will still return an absolute path.

        A non-obvious directory layout may look like this::

             CLOUD ASSEMBLY ROOT
               +-- asset.12345abcdef/
               +-- assembly-Stage
                     +-- MyStack.template.json
                     +-- MyStack.assets.json <- will contain { "path": "../asset.12345abcdef" }

        :param stack: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a0469992f178f5300b0ff1d8ba46d3086938c1f776c6301aa42c7eaa791b1bd2)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
        return typing.cast(builtins.str, jsii.invoke(self, "relativeStagedPath", [stack]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="BUNDLING_INPUT_DIR")
    def BUNDLING_INPUT_DIR(cls) -> builtins.str:
        '''The directory inside the bundling container into which the asset sources will be mounted.'''
        return typing.cast(builtins.str, jsii.sget(cls, "BUNDLING_INPUT_DIR"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="BUNDLING_OUTPUT_DIR")
    def BUNDLING_OUTPUT_DIR(cls) -> builtins.str:
        '''The directory inside the bundling container into which the bundled output should be written.'''
        return typing.cast(builtins.str, jsii.sget(cls, "BUNDLING_OUTPUT_DIR"))

    @builtins.property
    @jsii.member(jsii_name="absoluteStagedPath")
    def absolute_staged_path(self) -> builtins.str:
        '''Absolute path to the asset data.

        If asset staging is disabled, this will just be the source path or
        a temporary directory used for bundling.

        If asset staging is enabled it will be the staged path.

        IMPORTANT: If you are going to call ``addFileAsset()``, use
        ``relativeStagedPath()`` instead.
        '''
        return typing.cast(builtins.str, jsii.get(self, "absoluteStagedPath"))

    @builtins.property
    @jsii.member(jsii_name="assetHash")
    def asset_hash(self) -> builtins.str:
        '''A cryptographic hash of the asset.'''
        return typing.cast(builtins.str, jsii.get(self, "assetHash"))

    @builtins.property
    @jsii.member(jsii_name="isArchive")
    def is_archive(self) -> builtins.bool:
        '''Whether this asset is an archive (zip or jar).'''
        return typing.cast(builtins.bool, jsii.get(self, "isArchive"))

    @builtins.property
    @jsii.member(jsii_name="packaging")
    def packaging(self) -> "FileAssetPackaging":
        '''How this asset should be packaged.'''
        return typing.cast("FileAssetPackaging", jsii.get(self, "packaging"))

    @builtins.property
    @jsii.member(jsii_name="sourcePath")
    def source_path(self) -> builtins.str:
        '''The absolute path of the asset as it was referenced by the user.'''
        return typing.cast(builtins.str, jsii.get(self, "sourcePath"))


class Aws(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.Aws"):
    '''Accessor for pseudo parameters.

    Since pseudo parameters need to be anchored to a stack somewhere in the
    construct tree, this class takes an scope parameter; the pseudo parameter
    values can be obtained as properties from an scoped object.
    '''

    @jsii.python.classproperty
    @jsii.member(jsii_name="ACCOUNT_ID")
    def ACCOUNT_ID(cls) -> builtins.str:
        return typing.cast(builtins.str, jsii.sget(cls, "ACCOUNT_ID"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NO_VALUE")
    def NO_VALUE(cls) -> builtins.str:
        return typing.cast(builtins.str, jsii.sget(cls, "NO_VALUE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="NOTIFICATION_ARNS")
    def NOTIFICATION_ARNS(cls) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.sget(cls, "NOTIFICATION_ARNS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PARTITION")
    def PARTITION(cls) -> builtins.str:
        return typing.cast(builtins.str, jsii.sget(cls, "PARTITION"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="REGION")
    def REGION(cls) -> builtins.str:
        return typing.cast(builtins.str, jsii.sget(cls, "REGION"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="STACK_ID")
    def STACK_ID(cls) -> builtins.str:
        return typing.cast(builtins.str, jsii.sget(cls, "STACK_ID"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="STACK_NAME")
    def STACK_NAME(cls) -> builtins.str:
        return typing.cast(builtins.str, jsii.sget(cls, "STACK_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="URL_SUFFIX")
    def URL_SUFFIX(cls) -> builtins.str:
        return typing.cast(builtins.str, jsii.sget(cls, "URL_SUFFIX"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.BootstraplessSynthesizerProps",
    jsii_struct_bases=[],
    name_mapping={
        "cloud_formation_execution_role_arn": "cloudFormationExecutionRoleArn",
        "deploy_role_arn": "deployRoleArn",
    },
)
class BootstraplessSynthesizerProps:
    def __init__(
        self,
        *,
        cloud_formation_execution_role_arn: typing.Optional[builtins.str] = None,
        deploy_role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Construction properties of ``BootstraplessSynthesizer``.

        :param cloud_formation_execution_role_arn: The CFN execution Role ARN to use. Default: - No CloudFormation role (use CLI credentials)
        :param deploy_role_arn: The deploy Role ARN to use. Default: - No deploy role (use CLI credentials)

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            bootstrapless_synthesizer_props = cdk.BootstraplessSynthesizerProps(
                cloud_formation_execution_role_arn="cloudFormationExecutionRoleArn",
                deploy_role_arn="deployRoleArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3f56ec309bd76e41562731f3b986025649d4592e71caa79fed875d7020f66bfc)
            check_type(argname="argument cloud_formation_execution_role_arn", value=cloud_formation_execution_role_arn, expected_type=type_hints["cloud_formation_execution_role_arn"])
            check_type(argname="argument deploy_role_arn", value=deploy_role_arn, expected_type=type_hints["deploy_role_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cloud_formation_execution_role_arn is not None:
            self._values["cloud_formation_execution_role_arn"] = cloud_formation_execution_role_arn
        if deploy_role_arn is not None:
            self._values["deploy_role_arn"] = deploy_role_arn

    @builtins.property
    def cloud_formation_execution_role_arn(self) -> typing.Optional[builtins.str]:
        '''The CFN execution Role ARN to use.

        :default: - No CloudFormation role (use CLI credentials)
        '''
        result = self._values.get("cloud_formation_execution_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def deploy_role_arn(self) -> typing.Optional[builtins.str]:
        '''The deploy Role ARN to use.

        :default: - No deploy role (use CLI credentials)
        '''
        result = self._values.get("deploy_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BootstraplessSynthesizerProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.BundlingFileAccess")
class BundlingFileAccess(enum.Enum):
    '''The access mechanism used to make source files available to the bundling container and to return the bundling output back to the host.

    :exampleMetadata: infused

    Example::

        go.GoFunction(self, "GoFunction",
            entry="app/cmd/api",
            bundling=go.BundlingOptions(
                bundling_file_access=BundlingFileAccess.VOLUME_COPY
            )
        )
    '''

    VOLUME_COPY = "VOLUME_COPY"
    '''Creates temporary volumes and containers to copy files from the host to the bundling container and back.

    This is slower, but works also in more complex situations with remote or shared docker sockets.
    '''
    BIND_MOUNT = "BIND_MOUNT"
    '''The source and output folders will be mounted as bind mount from the host system This is faster and simpler, but less portable than ``VOLUME_COPY``.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.BundlingOptions",
    jsii_struct_bases=[],
    name_mapping={
        "image": "image",
        "bundling_file_access": "bundlingFileAccess",
        "command": "command",
        "entrypoint": "entrypoint",
        "environment": "environment",
        "local": "local",
        "network": "network",
        "output_type": "outputType",
        "platform": "platform",
        "security_opt": "securityOpt",
        "user": "user",
        "volumes": "volumes",
        "volumes_from": "volumesFrom",
        "working_directory": "workingDirectory",
    },
)
class BundlingOptions:
    def __init__(
        self,
        *,
        image: "DockerImage",
        bundling_file_access: typing.Optional[BundlingFileAccess] = None,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        local: typing.Optional["ILocalBundling"] = None,
        network: typing.Optional[builtins.str] = None,
        output_type: typing.Optional["BundlingOutput"] = None,
        platform: typing.Optional[builtins.str] = None,
        security_opt: typing.Optional[builtins.str] = None,
        user: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Sequence[typing.Union["DockerVolume", typing.Dict[builtins.str, typing.Any]]]] = None,
        volumes_from: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Bundling options.

        :param image: The Docker image where the command will run.
        :param bundling_file_access: The access mechanism used to make source files available to the bundling container and to return the bundling output back to the host. Default: - BundlingFileAccess.BIND_MOUNT
        :param command: The command to run in the Docker container. Example value: ``['npm', 'install']`` Default: - run the command defined in the image
        :param entrypoint: The entrypoint to run in the Docker container. Example value: ``['/bin/sh', '-c']`` Default: - run the entrypoint defined in the image
        :param environment: The environment variables to pass to the Docker container. Default: - no environment variables.
        :param local: Local bundling provider. The provider implements a method ``tryBundle()`` which should return ``true`` if local bundling was performed. If ``false`` is returned, docker bundling will be done. Default: - bundling will only be performed in a Docker container
        :param network: Docker `Networking options <https://docs.docker.com/engine/reference/commandline/run/#connect-a-container-to-a-network---network>`_. Default: - no networking options
        :param output_type: The type of output that this bundling operation is producing. Default: BundlingOutput.AUTO_DISCOVER
        :param platform: Platform to build for. *Requires Docker Buildx*. Specify this property to build images on a specific platform. Default: - no platform specified (the current machine architecture will be used)
        :param security_opt: `Security configuration <https://docs.docker.com/engine/reference/run/#security-configuration>`_ when running the docker container. Default: - no security options
        :param user: The user to use when running the Docker container. user | user:group | uid | uid:gid | user:gid | uid:group Default: - uid:gid of the current user or 1000:1000 on Windows
        :param volumes: Additional Docker volumes to mount. Default: - no additional volumes are mounted
        :param volumes_from: Where to mount the specified volumes from. Default: - no containers are specified to mount volumes from
        :param working_directory: Working directory inside the Docker container. Default: /asset-input

        :exampleMetadata: infused

        Example::

            import aws_cdk as cdk
            
            
            asset = Asset(self, "BundledAsset",
                path="/path/to/asset",
                bundling=cdk.BundlingOptions(
                    image=cdk.DockerImage.from_registry("alpine"),
                    command=["command-that-produces-an-archive.sh"],
                    output_type=cdk.BundlingOutput.NOT_ARCHIVED
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8093353aae305f1434cb8ad34e93d37dd68414d97a233d9a40ff7315f256e458)
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
            check_type(argname="argument bundling_file_access", value=bundling_file_access, expected_type=type_hints["bundling_file_access"])
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
            check_type(argname="argument entrypoint", value=entrypoint, expected_type=type_hints["entrypoint"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument local", value=local, expected_type=type_hints["local"])
            check_type(argname="argument network", value=network, expected_type=type_hints["network"])
            check_type(argname="argument output_type", value=output_type, expected_type=type_hints["output_type"])
            check_type(argname="argument platform", value=platform, expected_type=type_hints["platform"])
            check_type(argname="argument security_opt", value=security_opt, expected_type=type_hints["security_opt"])
            check_type(argname="argument user", value=user, expected_type=type_hints["user"])
            check_type(argname="argument volumes", value=volumes, expected_type=type_hints["volumes"])
            check_type(argname="argument volumes_from", value=volumes_from, expected_type=type_hints["volumes_from"])
            check_type(argname="argument working_directory", value=working_directory, expected_type=type_hints["working_directory"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "image": image,
        }
        if bundling_file_access is not None:
            self._values["bundling_file_access"] = bundling_file_access
        if command is not None:
            self._values["command"] = command
        if entrypoint is not None:
            self._values["entrypoint"] = entrypoint
        if environment is not None:
            self._values["environment"] = environment
        if local is not None:
            self._values["local"] = local
        if network is not None:
            self._values["network"] = network
        if output_type is not None:
            self._values["output_type"] = output_type
        if platform is not None:
            self._values["platform"] = platform
        if security_opt is not None:
            self._values["security_opt"] = security_opt
        if user is not None:
            self._values["user"] = user
        if volumes is not None:
            self._values["volumes"] = volumes
        if volumes_from is not None:
            self._values["volumes_from"] = volumes_from
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def image(self) -> "DockerImage":
        '''The Docker image where the command will run.'''
        result = self._values.get("image")
        assert result is not None, "Required property 'image' is missing"
        return typing.cast("DockerImage", result)

    @builtins.property
    def bundling_file_access(self) -> typing.Optional[BundlingFileAccess]:
        '''The access mechanism used to make source files available to the bundling container and to return the bundling output back to the host.

        :default: - BundlingFileAccess.BIND_MOUNT
        '''
        result = self._values.get("bundling_file_access")
        return typing.cast(typing.Optional[BundlingFileAccess], result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The command to run in the Docker container.

        Example value: ``['npm', 'install']``

        :default: - run the command defined in the image

        :see: https://docs.docker.com/engine/reference/run/
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entrypoint(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The entrypoint to run in the Docker container.

        Example value: ``['/bin/sh', '-c']``

        :default: - run the entrypoint defined in the image

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        '''
        result = self._values.get("entrypoint")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to pass to the Docker container.

        :default: - no environment variables.
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def local(self) -> typing.Optional["ILocalBundling"]:
        '''Local bundling provider.

        The provider implements a method ``tryBundle()`` which should return ``true``
        if local bundling was performed. If ``false`` is returned, docker bundling
        will be done.

        :default: - bundling will only be performed in a Docker container
        '''
        result = self._values.get("local")
        return typing.cast(typing.Optional["ILocalBundling"], result)

    @builtins.property
    def network(self) -> typing.Optional[builtins.str]:
        '''Docker `Networking options <https://docs.docker.com/engine/reference/commandline/run/#connect-a-container-to-a-network---network>`_.

        :default: - no networking options
        '''
        result = self._values.get("network")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_type(self) -> typing.Optional["BundlingOutput"]:
        '''The type of output that this bundling operation is producing.

        :default: BundlingOutput.AUTO_DISCOVER
        '''
        result = self._values.get("output_type")
        return typing.cast(typing.Optional["BundlingOutput"], result)

    @builtins.property
    def platform(self) -> typing.Optional[builtins.str]:
        '''Platform to build for. *Requires Docker Buildx*.

        Specify this property to build images on a specific platform.

        :default: - no platform specified (the current machine architecture will be used)
        '''
        result = self._values.get("platform")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def security_opt(self) -> typing.Optional[builtins.str]:
        '''`Security configuration <https://docs.docker.com/engine/reference/run/#security-configuration>`_ when running the docker container.

        :default: - no security options
        '''
        result = self._values.get("security_opt")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def user(self) -> typing.Optional[builtins.str]:
        '''The user to use when running the Docker container.

        user | user:group | uid | uid:gid | user:gid | uid:group

        :default: - uid:gid of the current user or 1000:1000 on Windows

        :see: https://docs.docker.com/engine/reference/run/#user
        '''
        result = self._values.get("user")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def volumes(self) -> typing.Optional[typing.List["DockerVolume"]]:
        '''Additional Docker volumes to mount.

        :default: - no additional volumes are mounted
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.List["DockerVolume"]], result)

    @builtins.property
    def volumes_from(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Where to mount the specified volumes from.

        :default: - no containers are specified to mount volumes from

        :see: https://docs.docker.com/engine/reference/commandline/run/#mount-volumes-from-container---volumes-from
        '''
        result = self._values.get("volumes_from")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''Working directory inside the Docker container.

        :default: /asset-input
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BundlingOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.BundlingOutput")
class BundlingOutput(enum.Enum):
    '''The type of output that a bundling operation is producing.

    :exampleMetadata: infused

    Example::

        import aws_cdk as cdk
        
        
        asset = Asset(self, "BundledAsset",
            path="/path/to/asset",
            bundling=cdk.BundlingOptions(
                image=cdk.DockerImage.from_registry("alpine"),
                command=["command-that-produces-an-archive.sh"],
                output_type=cdk.BundlingOutput.NOT_ARCHIVED
            )
        )
    '''

    ARCHIVED = "ARCHIVED"
    '''The bundling output directory includes a single .zip or .jar file which will be used as the final bundle. If the output directory does not include exactly a single archive, bundling will fail.'''
    NOT_ARCHIVED = "NOT_ARCHIVED"
    '''The bundling output directory contains one or more files which will be archived and uploaded as a .zip file to S3.'''
    AUTO_DISCOVER = "AUTO_DISCOVER"
    '''If the bundling output directory contains a single archive file (zip or jar) it will be used as the bundle output as-is.

    Otherwise, all the files in the bundling output directory will be zipped.
    '''
    SINGLE_FILE = "SINGLE_FILE"
    '''The bundling output directory includes a single file which will be used as the final bundle.

    If the output directory does not
    include exactly a single file, bundling will fail.

    Similar to ARCHIVED but for non-archive files
    '''


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnAutoScalingReplacingUpdate",
    jsii_struct_bases=[],
    name_mapping={"will_replace": "willReplace"},
)
class CfnAutoScalingReplacingUpdate:
    def __init__(self, *, will_replace: typing.Optional[builtins.bool] = None) -> None:
        '''Specifies whether an Auto Scaling group and the instances it contains are replaced during an update.

        During replacement,
        AWS CloudFormation retains the old group until it finishes creating the new one. If the update fails, AWS CloudFormation
        can roll back to the old Auto Scaling group and delete the new Auto Scaling group.

        While AWS CloudFormation creates the new group, it doesn't detach or attach any instances. After successfully creating
        the new Auto Scaling group, AWS CloudFormation deletes the old Auto Scaling group during the cleanup process.

        When you set the WillReplace parameter, remember to specify a matching CreationPolicy. If the minimum number of
        instances (specified by the MinSuccessfulInstancesPercent property) don't signal success within the Timeout period
        (specified in the CreationPolicy policy), the replacement update fails and AWS CloudFormation rolls back to the old
        Auto Scaling group.

        :param will_replace: 

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_auto_scaling_replacing_update = cdk.CfnAutoScalingReplacingUpdate(
                will_replace=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__64a4624c19dc262f58fbe1c1e7fe44bc86dfef800d569565b2a5f863d87def00)
            check_type(argname="argument will_replace", value=will_replace, expected_type=type_hints["will_replace"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if will_replace is not None:
            self._values["will_replace"] = will_replace

    @builtins.property
    def will_replace(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("will_replace")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAutoScalingReplacingUpdate(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnAutoScalingRollingUpdate",
    jsii_struct_bases=[],
    name_mapping={
        "max_batch_size": "maxBatchSize",
        "min_active_instances_percent": "minActiveInstancesPercent",
        "min_instances_in_service": "minInstancesInService",
        "min_successful_instances_percent": "minSuccessfulInstancesPercent",
        "pause_time": "pauseTime",
        "suspend_processes": "suspendProcesses",
        "wait_on_resource_signals": "waitOnResourceSignals",
    },
)
class CfnAutoScalingRollingUpdate:
    def __init__(
        self,
        *,
        max_batch_size: typing.Optional[jsii.Number] = None,
        min_active_instances_percent: typing.Optional[jsii.Number] = None,
        min_instances_in_service: typing.Optional[jsii.Number] = None,
        min_successful_instances_percent: typing.Optional[jsii.Number] = None,
        pause_time: typing.Optional[builtins.str] = None,
        suspend_processes: typing.Optional[typing.Sequence[builtins.str]] = None,
        wait_on_resource_signals: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''To specify how AWS CloudFormation handles rolling updates for an Auto Scaling group, use the AutoScalingRollingUpdate policy.

        Rolling updates enable you to specify whether AWS CloudFormation updates instances that are in an Auto Scaling
        group in batches or all at once.

        :param max_batch_size: Specifies the maximum number of instances that AWS CloudFormation updates.
        :param min_active_instances_percent: Specifies the percentage of instances in an Auto Scaling group that must remain in service while AWS CloudFormation updates old instances. You can specify a value from 0 to 100. AWS CloudFormation rounds to the nearest tenth of a percent. For example, if you update five instances with a minimum active percentage of 50, three instances must remain in service.
        :param min_instances_in_service: Specifies the minimum number of instances that must be in service within the Auto Scaling group while AWS CloudFormation updates old instances.
        :param min_successful_instances_percent: Specifies the percentage of instances in an Auto Scaling rolling update that must signal success for an update to succeed. You can specify a value from 0 to 100. AWS CloudFormation rounds to the nearest tenth of a percent. For example, if you update five instances with a minimum successful percentage of 50, three instances must signal success. If an instance doesn't send a signal within the time specified in the PauseTime property, AWS CloudFormation assumes that the instance wasn't updated. If you specify this property, you must also enable the WaitOnResourceSignals and PauseTime properties.
        :param pause_time: The amount of time that AWS CloudFormation pauses after making a change to a batch of instances to give those instances time to start software applications. For example, you might need to specify PauseTime when scaling up the number of instances in an Auto Scaling group. If you enable the WaitOnResourceSignals property, PauseTime is the amount of time that AWS CloudFormation should wait for the Auto Scaling group to receive the required number of valid signals from added or replaced instances. If the PauseTime is exceeded before the Auto Scaling group receives the required number of signals, the update fails. For best results, specify a time period that gives your applications sufficient time to get started. If the update needs to be rolled back, a short PauseTime can cause the rollback to fail. Specify PauseTime in the ISO8601 duration format (in the format PT#H#M#S, where each # is the number of hours, minutes, and seconds, respectively). The maximum PauseTime is one hour (PT1H).
        :param suspend_processes: Specifies the Auto Scaling processes to suspend during a stack update. Suspending processes prevents Auto Scaling from interfering with a stack update. For example, you can suspend alarming so that Auto Scaling doesn't execute scaling policies associated with an alarm. For valid values, see the ScalingProcesses.member.N parameter for the SuspendProcesses action in the Auto Scaling API Reference.
        :param wait_on_resource_signals: Specifies whether the Auto Scaling group waits on signals from new instances during an update. Use this property to ensure that instances have completed installing and configuring applications before the Auto Scaling group update proceeds. AWS CloudFormation suspends the update of an Auto Scaling group after new EC2 instances are launched into the group. AWS CloudFormation must receive a signal from each new instance within the specified PauseTime before continuing the update. To signal the Auto Scaling group, use the cfn-signal helper script or SignalResource API. To have instances wait for an Elastic Load Balancing health check before they signal success, add a health-check verification by using the cfn-init helper script. For an example, see the verify_instance_health command in the Auto Scaling rolling updates sample template.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_auto_scaling_rolling_update = cdk.CfnAutoScalingRollingUpdate(
                max_batch_size=123,
                min_active_instances_percent=123,
                min_instances_in_service=123,
                min_successful_instances_percent=123,
                pause_time="pauseTime",
                suspend_processes=["suspendProcesses"],
                wait_on_resource_signals=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__085b6d44b1c6b74e8060adbd78e13b1a188913a88389f19dce989d64cd5d0b7b)
            check_type(argname="argument max_batch_size", value=max_batch_size, expected_type=type_hints["max_batch_size"])
            check_type(argname="argument min_active_instances_percent", value=min_active_instances_percent, expected_type=type_hints["min_active_instances_percent"])
            check_type(argname="argument min_instances_in_service", value=min_instances_in_service, expected_type=type_hints["min_instances_in_service"])
            check_type(argname="argument min_successful_instances_percent", value=min_successful_instances_percent, expected_type=type_hints["min_successful_instances_percent"])
            check_type(argname="argument pause_time", value=pause_time, expected_type=type_hints["pause_time"])
            check_type(argname="argument suspend_processes", value=suspend_processes, expected_type=type_hints["suspend_processes"])
            check_type(argname="argument wait_on_resource_signals", value=wait_on_resource_signals, expected_type=type_hints["wait_on_resource_signals"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if max_batch_size is not None:
            self._values["max_batch_size"] = max_batch_size
        if min_active_instances_percent is not None:
            self._values["min_active_instances_percent"] = min_active_instances_percent
        if min_instances_in_service is not None:
            self._values["min_instances_in_service"] = min_instances_in_service
        if min_successful_instances_percent is not None:
            self._values["min_successful_instances_percent"] = min_successful_instances_percent
        if pause_time is not None:
            self._values["pause_time"] = pause_time
        if suspend_processes is not None:
            self._values["suspend_processes"] = suspend_processes
        if wait_on_resource_signals is not None:
            self._values["wait_on_resource_signals"] = wait_on_resource_signals

    @builtins.property
    def max_batch_size(self) -> typing.Optional[jsii.Number]:
        '''Specifies the maximum number of instances that AWS CloudFormation updates.'''
        result = self._values.get("max_batch_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_active_instances_percent(self) -> typing.Optional[jsii.Number]:
        '''Specifies the percentage of instances in an Auto Scaling group that must remain in service while AWS CloudFormation updates old instances.

        You can specify a value from 0 to 100. AWS CloudFormation rounds to the nearest tenth of a percent.
        For example, if you update five instances with a minimum active percentage of 50, three instances must remain in service.
        '''
        result = self._values.get("min_active_instances_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_instances_in_service(self) -> typing.Optional[jsii.Number]:
        '''Specifies the minimum number of instances that must be in service within the Auto Scaling group while AWS CloudFormation updates old instances.'''
        result = self._values.get("min_instances_in_service")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_successful_instances_percent(self) -> typing.Optional[jsii.Number]:
        '''Specifies the percentage of instances in an Auto Scaling rolling update that must signal success for an update to succeed.

        You can specify a value from 0 to 100. AWS CloudFormation rounds to the nearest tenth of a percent. For example, if you
        update five instances with a minimum successful percentage of 50, three instances must signal success.

        If an instance doesn't send a signal within the time specified in the PauseTime property, AWS CloudFormation assumes
        that the instance wasn't updated.

        If you specify this property, you must also enable the WaitOnResourceSignals and PauseTime properties.
        '''
        result = self._values.get("min_successful_instances_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def pause_time(self) -> typing.Optional[builtins.str]:
        '''The amount of time that AWS CloudFormation pauses after making a change to a batch of instances to give those instances time to start software applications.

        For example, you might need to specify PauseTime when scaling up the number of
        instances in an Auto Scaling group.

        If you enable the WaitOnResourceSignals property, PauseTime is the amount of time that AWS CloudFormation should wait
        for the Auto Scaling group to receive the required number of valid signals from added or replaced instances. If the
        PauseTime is exceeded before the Auto Scaling group receives the required number of signals, the update fails. For best
        results, specify a time period that gives your applications sufficient time to get started. If the update needs to be
        rolled back, a short PauseTime can cause the rollback to fail.

        Specify PauseTime in the ISO8601 duration format (in the format PT#H#M#S, where each # is the number of hours, minutes,
        and seconds, respectively). The maximum PauseTime is one hour (PT1H).
        '''
        result = self._values.get("pause_time")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def suspend_processes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specifies the Auto Scaling processes to suspend during a stack update.

        Suspending processes prevents Auto Scaling from
        interfering with a stack update. For example, you can suspend alarming so that Auto Scaling doesn't execute scaling
        policies associated with an alarm. For valid values, see the ScalingProcesses.member.N parameter for the SuspendProcesses
        action in the Auto Scaling API Reference.
        '''
        result = self._values.get("suspend_processes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def wait_on_resource_signals(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether the Auto Scaling group waits on signals from new instances during an update.

        Use this property to
        ensure that instances have completed installing and configuring applications before the Auto Scaling group update proceeds.
        AWS CloudFormation suspends the update of an Auto Scaling group after new EC2 instances are launched into the group.
        AWS CloudFormation must receive a signal from each new instance within the specified PauseTime before continuing the update.
        To signal the Auto Scaling group, use the cfn-signal helper script or SignalResource API.

        To have instances wait for an Elastic Load Balancing health check before they signal success, add a health-check
        verification by using the cfn-init helper script. For an example, see the verify_instance_health command in the Auto Scaling
        rolling updates sample template.
        '''
        result = self._values.get("wait_on_resource_signals")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAutoScalingRollingUpdate(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnAutoScalingScheduledAction",
    jsii_struct_bases=[],
    name_mapping={
        "ignore_unmodified_group_size_properties": "ignoreUnmodifiedGroupSizeProperties",
    },
)
class CfnAutoScalingScheduledAction:
    def __init__(
        self,
        *,
        ignore_unmodified_group_size_properties: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''With scheduled actions, the group size properties of an Auto Scaling group can change at any time.

        When you update a
        stack with an Auto Scaling group and scheduled action, AWS CloudFormation always sets the group size property values of
        your Auto Scaling group to the values that are defined in the AWS::AutoScaling::AutoScalingGroup resource of your template,
        even if a scheduled action is in effect.

        If you do not want AWS CloudFormation to change any of the group size property values when you have a scheduled action in
        effect, use the AutoScalingScheduledAction update policy to prevent AWS CloudFormation from changing the MinSize, MaxSize,
        or DesiredCapacity properties unless you have modified these values in your template.\\

        :param ignore_unmodified_group_size_properties: 

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_auto_scaling_scheduled_action = cdk.CfnAutoScalingScheduledAction(
                ignore_unmodified_group_size_properties=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fb97d190d1ae56d062fe61e7b7a1d5c6702a4ffc56969e6a4298203a32094be0)
            check_type(argname="argument ignore_unmodified_group_size_properties", value=ignore_unmodified_group_size_properties, expected_type=type_hints["ignore_unmodified_group_size_properties"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if ignore_unmodified_group_size_properties is not None:
            self._values["ignore_unmodified_group_size_properties"] = ignore_unmodified_group_size_properties

    @builtins.property
    def ignore_unmodified_group_size_properties(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("ignore_unmodified_group_size_properties")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAutoScalingScheduledAction(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.CfnCapabilities")
class CfnCapabilities(enum.Enum):
    '''Capabilities that affect whether CloudFormation is allowed to change IAM resources.'''

    NONE = "NONE"
    '''No IAM Capabilities.

    Pass this capability if you wish to block the creation IAM resources.

    :link: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-capabilities
    '''
    ANONYMOUS_IAM = "ANONYMOUS_IAM"
    '''Capability to create anonymous IAM resources.

    Pass this capability if you're only creating anonymous resources.

    :link: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-capabilities
    '''
    NAMED_IAM = "NAMED_IAM"
    '''Capability to create named IAM resources.

    Pass this capability if you're creating IAM resources that have physical
    names.

    ``CloudFormationCapabilities.NamedIAM`` implies ``CloudFormationCapabilities.IAM``; you don't have to pass both.

    :link: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-capabilities
    '''
    AUTO_EXPAND = "AUTO_EXPAND"
    '''Capability to run CloudFormation macros.

    Pass this capability if your template includes macros, for example AWS::Include or AWS::Serverless.

    :link: https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStack.html
    '''


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnCodeDeployBlueGreenAdditionalOptions",
    jsii_struct_bases=[],
    name_mapping={"termination_wait_time_in_minutes": "terminationWaitTimeInMinutes"},
)
class CfnCodeDeployBlueGreenAdditionalOptions:
    def __init__(
        self,
        *,
        termination_wait_time_in_minutes: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Additional options for the blue/green deployment.

        The type of the ``CfnCodeDeployBlueGreenHookProps.additionalOptions`` property.

        :param termination_wait_time_in_minutes: Specifies time to wait, in minutes, before terminating the blue resources. Default: - 5 minutes

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_code_deploy_blue_green_additional_options = cdk.CfnCodeDeployBlueGreenAdditionalOptions(
                termination_wait_time_in_minutes=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__79a497203131966d0d09329073a9aa1591c362b08bbd3ef60932bfaf296aee44)
            check_type(argname="argument termination_wait_time_in_minutes", value=termination_wait_time_in_minutes, expected_type=type_hints["termination_wait_time_in_minutes"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if termination_wait_time_in_minutes is not None:
            self._values["termination_wait_time_in_minutes"] = termination_wait_time_in_minutes

    @builtins.property
    def termination_wait_time_in_minutes(self) -> typing.Optional[jsii.Number]:
        '''Specifies time to wait, in minutes, before terminating the blue resources.

        :default: - 5 minutes
        '''
        result = self._values.get("termination_wait_time_in_minutes")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCodeDeployBlueGreenAdditionalOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnCodeDeployBlueGreenApplication",
    jsii_struct_bases=[],
    name_mapping={"ecs_attributes": "ecsAttributes", "target": "target"},
)
class CfnCodeDeployBlueGreenApplication:
    def __init__(
        self,
        *,
        ecs_attributes: typing.Union["CfnCodeDeployBlueGreenEcsAttributes", typing.Dict[builtins.str, typing.Any]],
        target: typing.Union["CfnCodeDeployBlueGreenApplicationTarget", typing.Dict[builtins.str, typing.Any]],
    ) -> None:
        '''The application actually being deployed.

        Type of the ``CfnCodeDeployBlueGreenHookProps.applications`` property.

        :param ecs_attributes: The detailed attributes of the deployed target.
        :param target: The target that is being deployed.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_code_deploy_blue_green_application = cdk.CfnCodeDeployBlueGreenApplication(
                ecs_attributes=cdk.CfnCodeDeployBlueGreenEcsAttributes(
                    task_definitions=["taskDefinitions"],
                    task_sets=["taskSets"],
                    traffic_routing=cdk.CfnTrafficRouting(
                        prod_traffic_route=cdk.CfnTrafficRoute(
                            logical_id="logicalId",
                            type="type"
                        ),
                        target_groups=["targetGroups"],
                        test_traffic_route=cdk.CfnTrafficRoute(
                            logical_id="logicalId",
                            type="type"
                        )
                    )
                ),
                target=cdk.CfnCodeDeployBlueGreenApplicationTarget(
                    logical_id="logicalId",
                    type="type"
                )
            )
        '''
        if isinstance(ecs_attributes, dict):
            ecs_attributes = CfnCodeDeployBlueGreenEcsAttributes(**ecs_attributes)
        if isinstance(target, dict):
            target = CfnCodeDeployBlueGreenApplicationTarget(**target)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7009fa55da2d1279f59d8dbcc140f42341267bd230a9c862c81edd737b4f6255)
            check_type(argname="argument ecs_attributes", value=ecs_attributes, expected_type=type_hints["ecs_attributes"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "ecs_attributes": ecs_attributes,
            "target": target,
        }

    @builtins.property
    def ecs_attributes(self) -> "CfnCodeDeployBlueGreenEcsAttributes":
        '''The detailed attributes of the deployed target.'''
        result = self._values.get("ecs_attributes")
        assert result is not None, "Required property 'ecs_attributes' is missing"
        return typing.cast("CfnCodeDeployBlueGreenEcsAttributes", result)

    @builtins.property
    def target(self) -> "CfnCodeDeployBlueGreenApplicationTarget":
        '''The target that is being deployed.'''
        result = self._values.get("target")
        assert result is not None, "Required property 'target' is missing"
        return typing.cast("CfnCodeDeployBlueGreenApplicationTarget", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCodeDeployBlueGreenApplication(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnCodeDeployBlueGreenApplicationTarget",
    jsii_struct_bases=[],
    name_mapping={"logical_id": "logicalId", "type": "type"},
)
class CfnCodeDeployBlueGreenApplicationTarget:
    def __init__(self, *, logical_id: builtins.str, type: builtins.str) -> None:
        '''Type of the ``CfnCodeDeployBlueGreenApplication.target`` property.

        :param logical_id: The logical id of the target resource.
        :param type: The resource type of the target being deployed. Right now, the only allowed value is 'AWS::ECS::Service'.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_code_deploy_blue_green_application_target = cdk.CfnCodeDeployBlueGreenApplicationTarget(
                logical_id="logicalId",
                type="type"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b03fca082c89add6bb633b4ce2bc31064d4450548a2843a6dcb1cebb8ed65994)
            check_type(argname="argument logical_id", value=logical_id, expected_type=type_hints["logical_id"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "logical_id": logical_id,
            "type": type,
        }

    @builtins.property
    def logical_id(self) -> builtins.str:
        '''The logical id of the target resource.'''
        result = self._values.get("logical_id")
        assert result is not None, "Required property 'logical_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type(self) -> builtins.str:
        '''The resource type of the target being deployed.

        Right now, the only allowed value is 'AWS::ECS::Service'.
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCodeDeployBlueGreenApplicationTarget(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnCodeDeployBlueGreenEcsAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "task_definitions": "taskDefinitions",
        "task_sets": "taskSets",
        "traffic_routing": "trafficRouting",
    },
)
class CfnCodeDeployBlueGreenEcsAttributes:
    def __init__(
        self,
        *,
        task_definitions: typing.Sequence[builtins.str],
        task_sets: typing.Sequence[builtins.str],
        traffic_routing: typing.Union["CfnTrafficRouting", typing.Dict[builtins.str, typing.Any]],
    ) -> None:
        '''The attributes of the ECS Service being deployed.

        Type of the ``CfnCodeDeployBlueGreenApplication.ecsAttributes`` property.

        :param task_definitions: The logical IDs of the blue and green, respectively, AWS::ECS::TaskDefinition task definitions.
        :param task_sets: The logical IDs of the blue and green, respectively, AWS::ECS::TaskSet task sets.
        :param traffic_routing: The traffic routing configuration.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_code_deploy_blue_green_ecs_attributes = cdk.CfnCodeDeployBlueGreenEcsAttributes(
                task_definitions=["taskDefinitions"],
                task_sets=["taskSets"],
                traffic_routing=cdk.CfnTrafficRouting(
                    prod_traffic_route=cdk.CfnTrafficRoute(
                        logical_id="logicalId",
                        type="type"
                    ),
                    target_groups=["targetGroups"],
                    test_traffic_route=cdk.CfnTrafficRoute(
                        logical_id="logicalId",
                        type="type"
                    )
                )
            )
        '''
        if isinstance(traffic_routing, dict):
            traffic_routing = CfnTrafficRouting(**traffic_routing)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9386f3026970a0c85d69a0af55e220d0f31455e048c93c94538d99913c52a112)
            check_type(argname="argument task_definitions", value=task_definitions, expected_type=type_hints["task_definitions"])
            check_type(argname="argument task_sets", value=task_sets, expected_type=type_hints["task_sets"])
            check_type(argname="argument traffic_routing", value=traffic_routing, expected_type=type_hints["traffic_routing"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "task_definitions": task_definitions,
            "task_sets": task_sets,
            "traffic_routing": traffic_routing,
        }

    @builtins.property
    def task_definitions(self) -> typing.List[builtins.str]:
        '''The logical IDs of the blue and green, respectively, AWS::ECS::TaskDefinition task definitions.'''
        result = self._values.get("task_definitions")
        assert result is not None, "Required property 'task_definitions' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def task_sets(self) -> typing.List[builtins.str]:
        '''The logical IDs of the blue and green, respectively, AWS::ECS::TaskSet task sets.'''
        result = self._values.get("task_sets")
        assert result is not None, "Required property 'task_sets' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def traffic_routing(self) -> "CfnTrafficRouting":
        '''The traffic routing configuration.'''
        result = self._values.get("traffic_routing")
        assert result is not None, "Required property 'traffic_routing' is missing"
        return typing.cast("CfnTrafficRouting", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCodeDeployBlueGreenEcsAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnCodeDeployBlueGreenHookProps",
    jsii_struct_bases=[],
    name_mapping={
        "applications": "applications",
        "service_role": "serviceRole",
        "additional_options": "additionalOptions",
        "lifecycle_event_hooks": "lifecycleEventHooks",
        "traffic_routing_config": "trafficRoutingConfig",
    },
)
class CfnCodeDeployBlueGreenHookProps:
    def __init__(
        self,
        *,
        applications: typing.Sequence[typing.Union[CfnCodeDeployBlueGreenApplication, typing.Dict[builtins.str, typing.Any]]],
        service_role: builtins.str,
        additional_options: typing.Optional[typing.Union[CfnCodeDeployBlueGreenAdditionalOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        lifecycle_event_hooks: typing.Optional[typing.Union["CfnCodeDeployBlueGreenLifecycleEventHooks", typing.Dict[builtins.str, typing.Any]]] = None,
        traffic_routing_config: typing.Optional[typing.Union["CfnTrafficRoutingConfig", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Construction properties of ``CfnCodeDeployBlueGreenHook``.

        :param applications: Properties of the Amazon ECS applications being deployed.
        :param service_role: The IAM Role for CloudFormation to use to perform blue-green deployments.
        :param additional_options: Additional options for the blue/green deployment. Default: - no additional options
        :param lifecycle_event_hooks: Use lifecycle event hooks to specify a Lambda function that CodeDeploy can call to validate a deployment. You can use the same function or a different one for deployment lifecycle events. Following completion of the validation tests, the Lambda ``CfnCodeDeployBlueGreenLifecycleEventHooks.afterAllowTraffic`` function calls back CodeDeploy and delivers a result of 'Succeeded' or 'Failed'. Default: - no lifecycle event hooks
        :param traffic_routing_config: Traffic routing configuration settings. Default: - time-based canary traffic shifting, with a 15% step percentage and a five minute bake time

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_code_deploy_blue_green_hook_props = cdk.CfnCodeDeployBlueGreenHookProps(
                applications=[cdk.CfnCodeDeployBlueGreenApplication(
                    ecs_attributes=cdk.CfnCodeDeployBlueGreenEcsAttributes(
                        task_definitions=["taskDefinitions"],
                        task_sets=["taskSets"],
                        traffic_routing=cdk.CfnTrafficRouting(
                            prod_traffic_route=cdk.CfnTrafficRoute(
                                logical_id="logicalId",
                                type="type"
                            ),
                            target_groups=["targetGroups"],
                            test_traffic_route=cdk.CfnTrafficRoute(
                                logical_id="logicalId",
                                type="type"
                            )
                        )
                    ),
                    target=cdk.CfnCodeDeployBlueGreenApplicationTarget(
                        logical_id="logicalId",
                        type="type"
                    )
                )],
                service_role="serviceRole",
            
                # the properties below are optional
                additional_options=cdk.CfnCodeDeployBlueGreenAdditionalOptions(
                    termination_wait_time_in_minutes=123
                ),
                lifecycle_event_hooks=cdk.CfnCodeDeployBlueGreenLifecycleEventHooks(
                    after_allow_test_traffic="afterAllowTestTraffic",
                    after_allow_traffic="afterAllowTraffic",
                    after_install="afterInstall",
                    before_allow_traffic="beforeAllowTraffic",
                    before_install="beforeInstall"
                ),
                traffic_routing_config=cdk.CfnTrafficRoutingConfig(
                    type=cdk.CfnTrafficRoutingType.ALL_AT_ONCE,
            
                    # the properties below are optional
                    time_based_canary=cdk.CfnTrafficRoutingTimeBasedCanary(
                        bake_time_mins=123,
                        step_percentage=123
                    ),
                    time_based_linear=cdk.CfnTrafficRoutingTimeBasedLinear(
                        bake_time_mins=123,
                        step_percentage=123
                    )
                )
            )
        '''
        if isinstance(additional_options, dict):
            additional_options = CfnCodeDeployBlueGreenAdditionalOptions(**additional_options)
        if isinstance(lifecycle_event_hooks, dict):
            lifecycle_event_hooks = CfnCodeDeployBlueGreenLifecycleEventHooks(**lifecycle_event_hooks)
        if isinstance(traffic_routing_config, dict):
            traffic_routing_config = CfnTrafficRoutingConfig(**traffic_routing_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__901a66ffac3933c21ba0dcae56d9d52b30713884c03a142191535c8c4035c2a2)
            check_type(argname="argument applications", value=applications, expected_type=type_hints["applications"])
            check_type(argname="argument service_role", value=service_role, expected_type=type_hints["service_role"])
            check_type(argname="argument additional_options", value=additional_options, expected_type=type_hints["additional_options"])
            check_type(argname="argument lifecycle_event_hooks", value=lifecycle_event_hooks, expected_type=type_hints["lifecycle_event_hooks"])
            check_type(argname="argument traffic_routing_config", value=traffic_routing_config, expected_type=type_hints["traffic_routing_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "applications": applications,
            "service_role": service_role,
        }
        if additional_options is not None:
            self._values["additional_options"] = additional_options
        if lifecycle_event_hooks is not None:
            self._values["lifecycle_event_hooks"] = lifecycle_event_hooks
        if traffic_routing_config is not None:
            self._values["traffic_routing_config"] = traffic_routing_config

    @builtins.property
    def applications(self) -> typing.List[CfnCodeDeployBlueGreenApplication]:
        '''Properties of the Amazon ECS applications being deployed.'''
        result = self._values.get("applications")
        assert result is not None, "Required property 'applications' is missing"
        return typing.cast(typing.List[CfnCodeDeployBlueGreenApplication], result)

    @builtins.property
    def service_role(self) -> builtins.str:
        '''The IAM Role for CloudFormation to use to perform blue-green deployments.'''
        result = self._values.get("service_role")
        assert result is not None, "Required property 'service_role' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def additional_options(
        self,
    ) -> typing.Optional[CfnCodeDeployBlueGreenAdditionalOptions]:
        '''Additional options for the blue/green deployment.

        :default: - no additional options
        '''
        result = self._values.get("additional_options")
        return typing.cast(typing.Optional[CfnCodeDeployBlueGreenAdditionalOptions], result)

    @builtins.property
    def lifecycle_event_hooks(
        self,
    ) -> typing.Optional["CfnCodeDeployBlueGreenLifecycleEventHooks"]:
        '''Use lifecycle event hooks to specify a Lambda function that CodeDeploy can call to validate a deployment.

        You can use the same function or a different one for deployment lifecycle events.
        Following completion of the validation tests,
        the Lambda ``CfnCodeDeployBlueGreenLifecycleEventHooks.afterAllowTraffic``
        function calls back CodeDeploy and delivers a result of 'Succeeded' or 'Failed'.

        :default: - no lifecycle event hooks
        '''
        result = self._values.get("lifecycle_event_hooks")
        return typing.cast(typing.Optional["CfnCodeDeployBlueGreenLifecycleEventHooks"], result)

    @builtins.property
    def traffic_routing_config(self) -> typing.Optional["CfnTrafficRoutingConfig"]:
        '''Traffic routing configuration settings.

        :default: - time-based canary traffic shifting, with a 15% step percentage and a five minute bake time
        '''
        result = self._values.get("traffic_routing_config")
        return typing.cast(typing.Optional["CfnTrafficRoutingConfig"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCodeDeployBlueGreenHookProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnCodeDeployBlueGreenLifecycleEventHooks",
    jsii_struct_bases=[],
    name_mapping={
        "after_allow_test_traffic": "afterAllowTestTraffic",
        "after_allow_traffic": "afterAllowTraffic",
        "after_install": "afterInstall",
        "before_allow_traffic": "beforeAllowTraffic",
        "before_install": "beforeInstall",
    },
)
class CfnCodeDeployBlueGreenLifecycleEventHooks:
    def __init__(
        self,
        *,
        after_allow_test_traffic: typing.Optional[builtins.str] = None,
        after_allow_traffic: typing.Optional[builtins.str] = None,
        after_install: typing.Optional[builtins.str] = None,
        before_allow_traffic: typing.Optional[builtins.str] = None,
        before_install: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Lifecycle events for blue-green deployments.

        The type of the ``CfnCodeDeployBlueGreenHookProps.lifecycleEventHooks`` property.

        :param after_allow_test_traffic: Function to use to run tasks after the test listener serves traffic to the replacement task set. Default: - none
        :param after_allow_traffic: Function to use to run tasks after the second target group serves traffic to the replacement task set. Default: - none
        :param after_install: Function to use to run tasks after the replacement task set is created and one of the target groups is associated with it. Default: - none
        :param before_allow_traffic: Function to use to run tasks after the second target group is associated with the replacement task set, but before traffic is shifted to the replacement task set. Default: - none
        :param before_install: Function to use to run tasks before the replacement task set is created. Default: - none

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_code_deploy_blue_green_lifecycle_event_hooks = cdk.CfnCodeDeployBlueGreenLifecycleEventHooks(
                after_allow_test_traffic="afterAllowTestTraffic",
                after_allow_traffic="afterAllowTraffic",
                after_install="afterInstall",
                before_allow_traffic="beforeAllowTraffic",
                before_install="beforeInstall"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__41b0072154819b77d5e53db3f653de6dd8c9860af4061e72bba0b66f0a08eb93)
            check_type(argname="argument after_allow_test_traffic", value=after_allow_test_traffic, expected_type=type_hints["after_allow_test_traffic"])
            check_type(argname="argument after_allow_traffic", value=after_allow_traffic, expected_type=type_hints["after_allow_traffic"])
            check_type(argname="argument after_install", value=after_install, expected_type=type_hints["after_install"])
            check_type(argname="argument before_allow_traffic", value=before_allow_traffic, expected_type=type_hints["before_allow_traffic"])
            check_type(argname="argument before_install", value=before_install, expected_type=type_hints["before_install"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if after_allow_test_traffic is not None:
            self._values["after_allow_test_traffic"] = after_allow_test_traffic
        if after_allow_traffic is not None:
            self._values["after_allow_traffic"] = after_allow_traffic
        if after_install is not None:
            self._values["after_install"] = after_install
        if before_allow_traffic is not None:
            self._values["before_allow_traffic"] = before_allow_traffic
        if before_install is not None:
            self._values["before_install"] = before_install

    @builtins.property
    def after_allow_test_traffic(self) -> typing.Optional[builtins.str]:
        '''Function to use to run tasks after the test listener serves traffic to the replacement task set.

        :default: - none
        '''
        result = self._values.get("after_allow_test_traffic")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def after_allow_traffic(self) -> typing.Optional[builtins.str]:
        '''Function to use to run tasks after the second target group serves traffic to the replacement task set.

        :default: - none
        '''
        result = self._values.get("after_allow_traffic")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def after_install(self) -> typing.Optional[builtins.str]:
        '''Function to use to run tasks after the replacement task set is created and one of the target groups is associated with it.

        :default: - none
        '''
        result = self._values.get("after_install")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def before_allow_traffic(self) -> typing.Optional[builtins.str]:
        '''Function to use to run tasks after the second target group is associated with the replacement task set, but before traffic is shifted to the replacement task set.

        :default: - none
        '''
        result = self._values.get("before_allow_traffic")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def before_install(self) -> typing.Optional[builtins.str]:
        '''Function to use to run tasks before the replacement task set is created.

        :default: - none
        '''
        result = self._values.get("before_install")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCodeDeployBlueGreenLifecycleEventHooks(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnCodeDeployLambdaAliasUpdate",
    jsii_struct_bases=[],
    name_mapping={
        "application_name": "applicationName",
        "deployment_group_name": "deploymentGroupName",
        "after_allow_traffic_hook": "afterAllowTrafficHook",
        "before_allow_traffic_hook": "beforeAllowTrafficHook",
    },
)
class CfnCodeDeployLambdaAliasUpdate:
    def __init__(
        self,
        *,
        application_name: builtins.str,
        deployment_group_name: builtins.str,
        after_allow_traffic_hook: typing.Optional[builtins.str] = None,
        before_allow_traffic_hook: typing.Optional[builtins.str] = None,
    ) -> None:
        '''To perform an AWS CodeDeploy deployment when the version changes on an AWS::Lambda::Alias resource, use the CodeDeployLambdaAliasUpdate update policy.

        :param application_name: The name of the AWS CodeDeploy application.
        :param deployment_group_name: The name of the AWS CodeDeploy deployment group. This is where the traffic-shifting policy is set.
        :param after_allow_traffic_hook: The name of the Lambda function to run after traffic routing completes.
        :param before_allow_traffic_hook: The name of the Lambda function to run before traffic routing starts.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_code_deploy_lambda_alias_update = cdk.CfnCodeDeployLambdaAliasUpdate(
                application_name="applicationName",
                deployment_group_name="deploymentGroupName",
            
                # the properties below are optional
                after_allow_traffic_hook="afterAllowTrafficHook",
                before_allow_traffic_hook="beforeAllowTrafficHook"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4b48f8869c46b80dc11f0e723a4d1dd66e23cd51d0ff29529a63459dd957f41d)
            check_type(argname="argument application_name", value=application_name, expected_type=type_hints["application_name"])
            check_type(argname="argument deployment_group_name", value=deployment_group_name, expected_type=type_hints["deployment_group_name"])
            check_type(argname="argument after_allow_traffic_hook", value=after_allow_traffic_hook, expected_type=type_hints["after_allow_traffic_hook"])
            check_type(argname="argument before_allow_traffic_hook", value=before_allow_traffic_hook, expected_type=type_hints["before_allow_traffic_hook"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "application_name": application_name,
            "deployment_group_name": deployment_group_name,
        }
        if after_allow_traffic_hook is not None:
            self._values["after_allow_traffic_hook"] = after_allow_traffic_hook
        if before_allow_traffic_hook is not None:
            self._values["before_allow_traffic_hook"] = before_allow_traffic_hook

    @builtins.property
    def application_name(self) -> builtins.str:
        '''The name of the AWS CodeDeploy application.'''
        result = self._values.get("application_name")
        assert result is not None, "Required property 'application_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def deployment_group_name(self) -> builtins.str:
        '''The name of the AWS CodeDeploy deployment group.

        This is where the traffic-shifting policy is set.
        '''
        result = self._values.get("deployment_group_name")
        assert result is not None, "Required property 'deployment_group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def after_allow_traffic_hook(self) -> typing.Optional[builtins.str]:
        '''The name of the Lambda function to run after traffic routing completes.'''
        result = self._values.get("after_allow_traffic_hook")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def before_allow_traffic_hook(self) -> typing.Optional[builtins.str]:
        '''The name of the Lambda function to run before traffic routing starts.'''
        result = self._values.get("before_allow_traffic_hook")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCodeDeployLambdaAliasUpdate(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnConditionProps",
    jsii_struct_bases=[],
    name_mapping={"expression": "expression"},
)
class CfnConditionProps:
    def __init__(
        self,
        *,
        expression: typing.Optional["ICfnConditionExpression"] = None,
    ) -> None:
        '''
        :param expression: The expression that the condition will evaluate. Default: - None.

        :exampleMetadata: infused

        Example::

            raw_bucket = s3.CfnBucket(self, "Bucket")
            # -or-
            raw_bucket_alt = my_bucket.node.default_child
            
            # then
            raw_bucket.cfn_options.condition = CfnCondition(self, "EnableBucket")
            raw_bucket.cfn_options.metadata = {
                "metadata_key": "MetadataValue"
            }
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e026e929ab1906fd8853f6cfbb0702ce18fcc0611e2a3155d3b01dd559b1eb89)
            check_type(argname="argument expression", value=expression, expected_type=type_hints["expression"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if expression is not None:
            self._values["expression"] = expression

    @builtins.property
    def expression(self) -> typing.Optional["ICfnConditionExpression"]:
        '''The expression that the condition will evaluate.

        :default: - None.
        '''
        result = self._values.get("expression")
        return typing.cast(typing.Optional["ICfnConditionExpression"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnConditionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnCreationPolicy",
    jsii_struct_bases=[],
    name_mapping={
        "auto_scaling_creation_policy": "autoScalingCreationPolicy",
        "resource_signal": "resourceSignal",
        "start_fleet": "startFleet",
    },
)
class CfnCreationPolicy:
    def __init__(
        self,
        *,
        auto_scaling_creation_policy: typing.Optional[typing.Union["CfnResourceAutoScalingCreationPolicy", typing.Dict[builtins.str, typing.Any]]] = None,
        resource_signal: typing.Optional[typing.Union["CfnResourceSignal", typing.Dict[builtins.str, typing.Any]]] = None,
        start_fleet: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Associate the CreationPolicy attribute with a resource to prevent its status from reaching create complete until AWS CloudFormation receives a specified number of success signals or the timeout period is exceeded.

        To signal a
        resource, you can use the cfn-signal helper script or SignalResource API. AWS CloudFormation publishes valid signals
        to the stack events so that you track the number of signals sent.

        The creation policy is invoked only when AWS CloudFormation creates the associated resource. Currently, the only
        AWS CloudFormation resources that support creation policies are AWS::AutoScaling::AutoScalingGroup, AWS::EC2::Instance,
        AWS::CloudFormation::WaitCondition and AWS::AppStream::Fleet.

        Use the CreationPolicy attribute when you want to wait on resource configuration actions before stack creation proceeds.
        For example, if you install and configure software applications on an EC2 instance, you might want those applications to
        be running before proceeding. In such cases, you can add a CreationPolicy attribute to the instance, and then send a success
        signal to the instance after the applications are installed and configured. For a detailed example, see Deploying Applications
        on Amazon EC2 with AWS CloudFormation.

        :param auto_scaling_creation_policy: For an Auto Scaling group replacement update, specifies how many instances must signal success for the update to succeed.
        :param resource_signal: When AWS CloudFormation creates the associated resource, configures the number of required success signals and the length of time that AWS CloudFormation waits for those signals.
        :param start_fleet: For an AppStream Fleet creation, specifies that the fleet is started after creation.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_creation_policy = cdk.CfnCreationPolicy(
                auto_scaling_creation_policy=cdk.CfnResourceAutoScalingCreationPolicy(
                    min_successful_instances_percent=123
                ),
                resource_signal=cdk.CfnResourceSignal(
                    count=123,
                    timeout="timeout"
                ),
                start_fleet=False
            )
        '''
        if isinstance(auto_scaling_creation_policy, dict):
            auto_scaling_creation_policy = CfnResourceAutoScalingCreationPolicy(**auto_scaling_creation_policy)
        if isinstance(resource_signal, dict):
            resource_signal = CfnResourceSignal(**resource_signal)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a7bc9de19fa72830b6b21a2c610a2f683c09b0624641711adebeb123578a5f6)
            check_type(argname="argument auto_scaling_creation_policy", value=auto_scaling_creation_policy, expected_type=type_hints["auto_scaling_creation_policy"])
            check_type(argname="argument resource_signal", value=resource_signal, expected_type=type_hints["resource_signal"])
            check_type(argname="argument start_fleet", value=start_fleet, expected_type=type_hints["start_fleet"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if auto_scaling_creation_policy is not None:
            self._values["auto_scaling_creation_policy"] = auto_scaling_creation_policy
        if resource_signal is not None:
            self._values["resource_signal"] = resource_signal
        if start_fleet is not None:
            self._values["start_fleet"] = start_fleet

    @builtins.property
    def auto_scaling_creation_policy(
        self,
    ) -> typing.Optional["CfnResourceAutoScalingCreationPolicy"]:
        '''For an Auto Scaling group replacement update, specifies how many instances must signal success for the update to succeed.'''
        result = self._values.get("auto_scaling_creation_policy")
        return typing.cast(typing.Optional["CfnResourceAutoScalingCreationPolicy"], result)

    @builtins.property
    def resource_signal(self) -> typing.Optional["CfnResourceSignal"]:
        '''When AWS CloudFormation creates the associated resource, configures the number of required success signals and the length of time that AWS CloudFormation waits for those signals.'''
        result = self._values.get("resource_signal")
        return typing.cast(typing.Optional["CfnResourceSignal"], result)

    @builtins.property
    def start_fleet(self) -> typing.Optional[builtins.bool]:
        '''For an AppStream Fleet creation, specifies that the fleet is started after creation.'''
        result = self._values.get("start_fleet")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCreationPolicy(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnCustomResourceProps",
    jsii_struct_bases=[],
    name_mapping={
        "service_token": "serviceToken",
        "service_timeout": "serviceTimeout",
    },
)
class CfnCustomResourceProps:
    def __init__(
        self,
        *,
        service_token: builtins.str,
        service_timeout: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnCustomResource``.

        :param service_token: The service token, such as an Amazon SNS topic ARN or Lambda function ARN. The service token must be from the same Region as the stack. Updates aren't supported.
        :param service_timeout: The maximum time, in seconds, that can elapse before a custom resource operation times out. The value must be an integer from 1 to 3600. The default value is 3600 seconds (1 hour).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-customresource.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_custom_resource_props = cdk.CfnCustomResourceProps(
                service_token="serviceToken",
            
                # the properties below are optional
                service_timeout=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__976a40ec6215ef243bb2eb86b5cd6657bdaac3d0ca3df51b0c6119ec77b512df)
            check_type(argname="argument service_token", value=service_token, expected_type=type_hints["service_token"])
            check_type(argname="argument service_timeout", value=service_timeout, expected_type=type_hints["service_timeout"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "service_token": service_token,
        }
        if service_timeout is not None:
            self._values["service_timeout"] = service_timeout

    @builtins.property
    def service_token(self) -> builtins.str:
        '''The service token, such as an Amazon SNS topic ARN or Lambda function ARN.

        The service token must be from the same Region as the stack.

        Updates aren't supported.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-customresource.html#cfn-cloudformation-customresource-servicetoken
        '''
        result = self._values.get("service_token")
        assert result is not None, "Required property 'service_token' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service_timeout(self) -> typing.Optional[jsii.Number]:
        '''The maximum time, in seconds, that can elapse before a custom resource operation times out.

        The value must be an integer from 1 to 3600. The default value is 3600 seconds (1 hour).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-customresource.html#cfn-cloudformation-customresource-servicetimeout
        '''
        result = self._values.get("service_timeout")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCustomResourceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.CfnDeletionPolicy")
class CfnDeletionPolicy(enum.Enum):
    '''With the DeletionPolicy attribute you can preserve or (in some cases) backup a resource when its stack is deleted.

    You specify a DeletionPolicy attribute for each resource that you want to control. If a resource has no DeletionPolicy
    attribute, AWS CloudFormation deletes the resource by default. Note that this capability also applies to update operations
    that lead to resources being removed.
    '''

    DELETE = "DELETE"
    '''AWS CloudFormation deletes the resource and all its content if applicable during stack deletion.

    You can add this
    deletion policy to any resource type. By default, if you don't specify a DeletionPolicy, AWS CloudFormation deletes
    your resources. However, be aware of the following considerations:
    '''
    RETAIN = "RETAIN"
    '''AWS CloudFormation keeps the resource without deleting the resource or its contents when its stack is deleted.

    You can add this deletion policy to any resource type. Note that when AWS CloudFormation completes the stack deletion,
    the stack will be in Delete_Complete state; however, resources that are retained continue to exist and continue to incur
    applicable charges until you delete those resources.
    '''
    RETAIN_EXCEPT_ON_CREATE = "RETAIN_EXCEPT_ON_CREATE"
    '''RetainExceptOnCreate behaves like Retain for stack operations, except for the stack operation that initially created the resource.

    If the stack operation that created the resource is rolled back, CloudFormation deletes the resource. For all other stack operations,
    such as stack deletion, CloudFormation retains the resource and its contents. The result is that new, empty, and unused resources are deleted,
    while in-use resources and their data are retained.
    '''
    SNAPSHOT = "SNAPSHOT"
    '''For resources that support snapshots (AWS::EC2::Volume, AWS::ElastiCache::CacheCluster, AWS::ElastiCache::ReplicationGroup, AWS::RDS::DBInstance, AWS::RDS::DBCluster, and AWS::Redshift::Cluster), AWS CloudFormation creates a snapshot for the resource before deleting it.

    Note that when AWS CloudFormation completes the stack deletion, the stack will be in the
    Delete_Complete state; however, the snapshots that are created with this policy continue to exist and continue to
    incur applicable charges until you delete those snapshots.
    '''


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnDynamicReferenceProps",
    jsii_struct_bases=[],
    name_mapping={"reference_key": "referenceKey", "service": "service"},
)
class CfnDynamicReferenceProps:
    def __init__(
        self,
        *,
        reference_key: builtins.str,
        service: "CfnDynamicReferenceService",
    ) -> None:
        '''Properties for a Dynamic Reference.

        :param reference_key: The reference key of the dynamic reference.
        :param service: The service to retrieve the dynamic reference from.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_dynamic_reference_props = cdk.CfnDynamicReferenceProps(
                reference_key="referenceKey",
                service=cdk.CfnDynamicReferenceService.SSM
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__12687a08c43721521799f3b9d110bf3ffab6ff24e33bf27b23f871d7023649a2)
            check_type(argname="argument reference_key", value=reference_key, expected_type=type_hints["reference_key"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "reference_key": reference_key,
            "service": service,
        }

    @builtins.property
    def reference_key(self) -> builtins.str:
        '''The reference key of the dynamic reference.'''
        result = self._values.get("reference_key")
        assert result is not None, "Required property 'reference_key' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> "CfnDynamicReferenceService":
        '''The service to retrieve the dynamic reference from.'''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast("CfnDynamicReferenceService", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDynamicReferenceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.CfnDynamicReferenceService")
class CfnDynamicReferenceService(enum.Enum):
    '''The service to retrieve the dynamic reference from.

    :exampleMetadata: infused

    Example::

        CfnDynamicReference(CfnDynamicReferenceService.SECRETS_MANAGER, "secret-id:secret-string:json-key:version-stage:version-id")
    '''

    SSM = "SSM"
    '''Plaintext value stored in AWS Systems Manager Parameter Store.'''
    SSM_SECURE = "SSM_SECURE"
    '''Secure string stored in AWS Systems Manager Parameter Store.'''
    SECRETS_MANAGER = "SECRETS_MANAGER"
    '''Secret stored in AWS Secrets Manager.'''


class CfnElement(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.CfnElement",
):
    '''An element of a CloudFormation stack.'''

    def __init__(self, scope: _constructs_77d1e7e8.Construct, id: builtins.str) -> None:
        '''Creates an entity and binds it to a tree.

        Note that the root of the tree must be a Stack object (not just any Root).

        :param scope: The parent construct.
        :param id: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__804b174e66a759a82c30b9e7c62a52c91380895b23381b35edb05e58707e3b76)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        jsii.create(self.__class__, self, [scope, id])

    @jsii.member(jsii_name="isCfnElement")
    @builtins.classmethod
    def is_cfn_element(cls, x: typing.Any) -> builtins.bool:
        '''Returns ``true`` if a construct is a stack element (i.e. part of the synthesized cloudformation template).

        Uses duck-typing instead of ``instanceof`` to allow stack elements from different
        versions of this library to be included in the same stack.

        :param x: -

        :return: The construct as a stack element or undefined if it is not a stack element.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8d66875d8623d1d223bf6084c3a8c2c175ddcee93e7a9be1026b29251696bd44)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isCfnElement", [x]))

    @jsii.member(jsii_name="overrideLogicalId")
    def override_logical_id(self, new_logical_id: builtins.str) -> None:
        '''Overrides the auto-generated logical ID with a specific ID.

        :param new_logical_id: The new logical ID to use for this stack element.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5786b9b09eedff01ff70a3bb7cd27562f344e797dfee3e18aea89baaa5c80ea4)
            check_type(argname="argument new_logical_id", value=new_logical_id, expected_type=type_hints["new_logical_id"])
        return typing.cast(None, jsii.invoke(self, "overrideLogicalId", [new_logical_id]))

    @builtins.property
    @jsii.member(jsii_name="creationStack")
    def creation_stack(self) -> typing.List[builtins.str]:
        '''
        :return:

        the stack trace of the point where this Resource was created from, sourced
        from the +metadata+ entry typed +aws:cdk:logicalId+, and with the bottom-most
        node +internal+ entries filtered.
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "creationStack"))

    @builtins.property
    @jsii.member(jsii_name="logicalId")
    def logical_id(self) -> builtins.str:
        '''The logical ID for this CloudFormation stack element.

        The logical ID of the element
        is calculated from the path of the resource node in the construct tree.

        To override this value, use ``overrideLogicalId(newLogicalId)``.

        :return:

        the logical ID as a stringified token. This value will only get
        resolved during synthesis.
        '''
        return typing.cast(builtins.str, jsii.get(self, "logicalId"))

    @builtins.property
    @jsii.member(jsii_name="stack")
    def stack(self) -> "Stack":
        '''The stack in which this element is defined.

        CfnElements must be defined within a stack scope (directly or indirectly).
        '''
        return typing.cast("Stack", jsii.get(self, "stack"))


class _CfnElementProxy(CfnElement):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, CfnElement).__jsii_proxy_class__ = lambda : _CfnElementProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnGuardHookProps",
    jsii_struct_bases=[],
    name_mapping={
        "alias": "alias",
        "execution_role": "executionRole",
        "failure_mode": "failureMode",
        "hook_status": "hookStatus",
        "rule_location": "ruleLocation",
        "target_operations": "targetOperations",
        "log_bucket": "logBucket",
        "options": "options",
        "stack_filters": "stackFilters",
        "target_filters": "targetFilters",
    },
)
class CfnGuardHookProps:
    def __init__(
        self,
        *,
        alias: builtins.str,
        execution_role: builtins.str,
        failure_mode: builtins.str,
        hook_status: builtins.str,
        rule_location: typing.Union["IResolvable", typing.Union["CfnGuardHook.S3LocationProperty", typing.Dict[builtins.str, typing.Any]]],
        target_operations: typing.Sequence[builtins.str],
        log_bucket: typing.Optional[builtins.str] = None,
        options: typing.Optional[typing.Union["IResolvable", typing.Union["CfnGuardHook.OptionsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        stack_filters: typing.Optional[typing.Union["IResolvable", typing.Union["CfnGuardHook.StackFiltersProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        target_filters: typing.Optional[typing.Union["IResolvable", typing.Union["CfnGuardHook.TargetFiltersProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnGuardHook``.

        :param alias: The type name alias for the Hook. This alias must be unique per account and Region. The alias must be in the form ``Name1::Name2::Name3`` and must not begin with ``AWS`` . For example, ``Private::Guard::MyTestHook`` .
        :param execution_role: The IAM role that the Hook assumes to retrieve your Guard rules from S3 and optionally write a detailed Guard output report back.
        :param failure_mode: Specifies how the Hook responds when rules fail their evaluation. - ``FAIL`` : Prevents the action from proceeding. This is helpful for enforcing strict compliance or security policies. - ``WARN`` : Issues warnings to users but allows actions to continue. This is useful for non-critical validations or informational checks. Default: - "WARN"
        :param hook_status: Specifies if the Hook is ``ENABLED`` or ``DISABLED`` . Default: - "DISABLED"
        :param rule_location: Specifies the S3 location of your Guard rules.
        :param target_operations: Specifies the list of operations the Hook is run against. For more information, see `Hook targets <https://docs.aws.amazon.com/cloudformation-cli/latest/hooks-userguide/hooks-concepts.html#hook-terms-hook-target>`_ in the *AWS CloudFormation Hooks User Guide* . Valid values: ``STACK`` | ``RESOURCE`` | ``CHANGE_SET`` | ``CLOUD_CONTROL``
        :param log_bucket: Specifies the name of an S3 bucket to store the Guard output report. This report contains the results of your Guard rule validations.
        :param options: Specifies the S3 location of your input parameters.
        :param stack_filters: Specifies the stack level filters for the Hook. Example stack level filter in JSON: ``"StackFilters": {"FilteringCriteria": "ALL", "StackNames": {"Exclude": [ "stack-1", "stack-2"]}}`` Example stack level filter in YAML: ``StackFilters: FilteringCriteria: ALL StackNames: Exclude: - stack-1 - stack-2``
        :param target_filters: Specifies the target filters for the Hook. Example target filter in JSON: ``"TargetFilters": {"Actions": [ "Create", "Update", "Delete" ]}`` Example target filter in YAML: ``TargetFilters: Actions: - CREATE - UPDATE - DELETE``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-guardhook.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_guard_hook_props = cdk.CfnGuardHookProps(
                alias="alias",
                execution_role="executionRole",
                failure_mode="failureMode",
                hook_status="hookStatus",
                rule_location=cdk.CfnGuardHook.S3LocationProperty(
                    uri="uri",
            
                    # the properties below are optional
                    version_id="versionId"
                ),
                target_operations=["targetOperations"],
            
                # the properties below are optional
                log_bucket="logBucket",
                options=cdk.CfnGuardHook.OptionsProperty(
                    input_params=cdk.CfnGuardHook.S3LocationProperty(
                        uri="uri",
            
                        # the properties below are optional
                        version_id="versionId"
                    )
                ),
                stack_filters=cdk.CfnGuardHook.StackFiltersProperty(
                    filtering_criteria="filteringCriteria",
            
                    # the properties below are optional
                    stack_names=cdk.CfnGuardHook.StackNamesProperty(
                        exclude=["exclude"],
                        include=["include"]
                    ),
                    stack_roles=cdk.CfnGuardHook.StackRolesProperty(
                        exclude=["exclude"],
                        include=["include"]
                    )
                ),
                target_filters=cdk.CfnGuardHook.TargetFiltersProperty(
                    targets=[cdk.CfnGuardHook.HookTargetProperty(
                        action="action",
                        invocation_point="invocationPoint",
                        target_name="targetName"
                    )],
            
                    # the properties below are optional
                    actions=["actions"],
                    invocation_points=["invocationPoints"],
                    target_names=["targetNames"]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0af504bd749e0a62686c10b562deecc015abe54143ca61ed442f20ab5b08fee9)
            check_type(argname="argument alias", value=alias, expected_type=type_hints["alias"])
            check_type(argname="argument execution_role", value=execution_role, expected_type=type_hints["execution_role"])
            check_type(argname="argument failure_mode", value=failure_mode, expected_type=type_hints["failure_mode"])
            check_type(argname="argument hook_status", value=hook_status, expected_type=type_hints["hook_status"])
            check_type(argname="argument rule_location", value=rule_location, expected_type=type_hints["rule_location"])
            check_type(argname="argument target_operations", value=target_operations, expected_type=type_hints["target_operations"])
            check_type(argname="argument log_bucket", value=log_bucket, expected_type=type_hints["log_bucket"])
            check_type(argname="argument options", value=options, expected_type=type_hints["options"])
            check_type(argname="argument stack_filters", value=stack_filters, expected_type=type_hints["stack_filters"])
            check_type(argname="argument target_filters", value=target_filters, expected_type=type_hints["target_filters"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "alias": alias,
            "execution_role": execution_role,
            "failure_mode": failure_mode,
            "hook_status": hook_status,
            "rule_location": rule_location,
            "target_operations": target_operations,
        }
        if log_bucket is not None:
            self._values["log_bucket"] = log_bucket
        if options is not None:
            self._values["options"] = options
        if stack_filters is not None:
            self._values["stack_filters"] = stack_filters
        if target_filters is not None:
            self._values["target_filters"] = target_filters

    @builtins.property
    def alias(self) -> builtins.str:
        '''The type name alias for the Hook. This alias must be unique per account and Region.

        The alias must be in the form ``Name1::Name2::Name3`` and must not begin with ``AWS`` . For example, ``Private::Guard::MyTestHook`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-guardhook.html#cfn-cloudformation-guardhook-alias
        '''
        result = self._values.get("alias")
        assert result is not None, "Required property 'alias' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def execution_role(self) -> builtins.str:
        '''The IAM role that the Hook assumes to retrieve your Guard rules from S3 and optionally write a detailed Guard output report back.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-guardhook.html#cfn-cloudformation-guardhook-executionrole
        '''
        result = self._values.get("execution_role")
        assert result is not None, "Required property 'execution_role' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def failure_mode(self) -> builtins.str:
        '''Specifies how the Hook responds when rules fail their evaluation.

        - ``FAIL`` : Prevents the action from proceeding. This is helpful for enforcing strict compliance or security policies.
        - ``WARN`` : Issues warnings to users but allows actions to continue. This is useful for non-critical validations or informational checks.

        :default: - "WARN"

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-guardhook.html#cfn-cloudformation-guardhook-failuremode
        '''
        result = self._values.get("failure_mode")
        assert result is not None, "Required property 'failure_mode' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def hook_status(self) -> builtins.str:
        '''Specifies if the Hook is ``ENABLED`` or ``DISABLED`` .

        :default: - "DISABLED"

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-guardhook.html#cfn-cloudformation-guardhook-hookstatus
        '''
        result = self._values.get("hook_status")
        assert result is not None, "Required property 'hook_status' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rule_location(
        self,
    ) -> typing.Union["IResolvable", "CfnGuardHook.S3LocationProperty"]:
        '''Specifies the S3 location of your Guard rules.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-guardhook.html#cfn-cloudformation-guardhook-rulelocation
        '''
        result = self._values.get("rule_location")
        assert result is not None, "Required property 'rule_location' is missing"
        return typing.cast(typing.Union["IResolvable", "CfnGuardHook.S3LocationProperty"], result)

    @builtins.property
    def target_operations(self) -> typing.List[builtins.str]:
        '''Specifies the list of operations the Hook is run against.

        For more information, see `Hook targets <https://docs.aws.amazon.com/cloudformation-cli/latest/hooks-userguide/hooks-concepts.html#hook-terms-hook-target>`_ in the *AWS CloudFormation Hooks User Guide* .

        Valid values: ``STACK`` | ``RESOURCE`` | ``CHANGE_SET`` | ``CLOUD_CONTROL``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-guardhook.html#cfn-cloudformation-guardhook-targetoperations
        '''
        result = self._values.get("target_operations")
        assert result is not None, "Required property 'target_operations' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def log_bucket(self) -> typing.Optional[builtins.str]:
        '''Specifies the name of an S3 bucket to store the Guard output report.

        This report contains the results of your Guard rule validations.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-guardhook.html#cfn-cloudformation-guardhook-logbucket
        '''
        result = self._values.get("log_bucket")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def options(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", "CfnGuardHook.OptionsProperty"]]:
        '''Specifies the S3 location of your input parameters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-guardhook.html#cfn-cloudformation-guardhook-options
        '''
        result = self._values.get("options")
        return typing.cast(typing.Optional[typing.Union["IResolvable", "CfnGuardHook.OptionsProperty"]], result)

    @builtins.property
    def stack_filters(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", "CfnGuardHook.StackFiltersProperty"]]:
        '''Specifies the stack level filters for the Hook.

        Example stack level filter in JSON:

        ``"StackFilters": {"FilteringCriteria": "ALL", "StackNames": {"Exclude": [ "stack-1", "stack-2"]}}``

        Example stack level filter in YAML:

        ``StackFilters: FilteringCriteria: ALL StackNames: Exclude: - stack-1 - stack-2``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-guardhook.html#cfn-cloudformation-guardhook-stackfilters
        '''
        result = self._values.get("stack_filters")
        return typing.cast(typing.Optional[typing.Union["IResolvable", "CfnGuardHook.StackFiltersProperty"]], result)

    @builtins.property
    def target_filters(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", "CfnGuardHook.TargetFiltersProperty"]]:
        '''Specifies the target filters for the Hook.

        Example target filter in JSON:

        ``"TargetFilters": {"Actions": [ "Create", "Update", "Delete" ]}``

        Example target filter in YAML:

        ``TargetFilters: Actions: - CREATE - UPDATE - DELETE``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-guardhook.html#cfn-cloudformation-guardhook-targetfilters
        '''
        result = self._values.get("target_filters")
        return typing.cast(typing.Optional[typing.Union["IResolvable", "CfnGuardHook.TargetFiltersProperty"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGuardHookProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CfnHook(CfnElement, metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.CfnHook"):
    '''Represents a CloudFormation resource.

    :exampleMetadata: infused

    Example::

        # cfn_template: cfn_inc.CfnInclude
        
        # mutating the hook
        # my_role: iam.Role
        
        hook = cfn_template.get_hook("MyOutput")
        code_deploy_hook = hook
        code_deploy_hook.service_role = my_role.role_arn
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        type: builtins.str,
        properties: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''Creates a new Hook object.

        :param scope: -
        :param id: -
        :param type: The type of the hook (for example, "AWS::CodeDeploy::BlueGreen").
        :param properties: The properties of the hook. Default: - no properties
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__34e5d88f44cad60f887c90c156a334d23c105a3fb5b0b86b06864bcfb01af902)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnHookProps(type=type, properties=properties)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5eca06810c909a846c0c688dd775e5c3976c4e682bff494498e73ace0f6f3e4e)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], jsii.invoke(self, "renderProperties", [props]))

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''The type of the hook (for example, "AWS::CodeDeploy::BlueGreen").'''
        return typing.cast(builtins.str, jsii.get(self, "type"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnHookDefaultVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "type_name": "typeName",
        "type_version_arn": "typeVersionArn",
        "version_id": "versionId",
    },
)
class CfnHookDefaultVersionProps:
    def __init__(
        self,
        *,
        type_name: typing.Optional[builtins.str] = None,
        type_version_arn: typing.Optional[builtins.str] = None,
        version_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnHookDefaultVersion``.

        :param type_name: The name of the Hook. You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .
        :param type_version_arn: The version ID of the type configuration. You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .
        :param version_id: The version ID of the type specified. You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hookdefaultversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_hook_default_version_props = cdk.CfnHookDefaultVersionProps(
                type_name="typeName",
                type_version_arn="typeVersionArn",
                version_id="versionId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d4d2c1075e69e97d9331f4b9a20c1ded57360b265253a7ce845015b76ef2c37)
            check_type(argname="argument type_name", value=type_name, expected_type=type_hints["type_name"])
            check_type(argname="argument type_version_arn", value=type_version_arn, expected_type=type_hints["type_version_arn"])
            check_type(argname="argument version_id", value=version_id, expected_type=type_hints["version_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if type_name is not None:
            self._values["type_name"] = type_name
        if type_version_arn is not None:
            self._values["type_version_arn"] = type_version_arn
        if version_id is not None:
            self._values["version_id"] = version_id

    @builtins.property
    def type_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Hook.

        You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hookdefaultversion.html#cfn-cloudformation-hookdefaultversion-typename
        '''
        result = self._values.get("type_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type_version_arn(self) -> typing.Optional[builtins.str]:
        '''The version ID of the type configuration.

        You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hookdefaultversion.html#cfn-cloudformation-hookdefaultversion-typeversionarn
        '''
        result = self._values.get("type_version_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def version_id(self) -> typing.Optional[builtins.str]:
        '''The version ID of the type specified.

        You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hookdefaultversion.html#cfn-cloudformation-hookdefaultversion-versionid
        '''
        result = self._values.get("version_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnHookDefaultVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnHookProps",
    jsii_struct_bases=[],
    name_mapping={"type": "type", "properties": "properties"},
)
class CfnHookProps:
    def __init__(
        self,
        *,
        type: builtins.str,
        properties: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''Construction properties of ``CfnHook``.

        :param type: The type of the hook (for example, "AWS::CodeDeploy::BlueGreen").
        :param properties: The properties of the hook. Default: - no properties

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # properties: Any
            
            cfn_hook_props = cdk.CfnHookProps(
                type="type",
            
                # the properties below are optional
                properties={
                    "properties_key": properties
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fa84c2bfdeaacb225d7d9642f6a0632f904419b60f805f60b2cf30f235119b13)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument properties", value=properties, expected_type=type_hints["properties"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }
        if properties is not None:
            self._values["properties"] = properties

    @builtins.property
    def type(self) -> builtins.str:
        '''The type of the hook (for example, "AWS::CodeDeploy::BlueGreen").'''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def properties(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The properties of the hook.

        :default: - no properties
        '''
        result = self._values.get("properties")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnHookProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnHookTypeConfigProps",
    jsii_struct_bases=[],
    name_mapping={
        "configuration": "configuration",
        "configuration_alias": "configurationAlias",
        "type_arn": "typeArn",
        "type_name": "typeName",
    },
)
class CfnHookTypeConfigProps:
    def __init__(
        self,
        *,
        configuration: builtins.str,
        configuration_alias: typing.Optional[builtins.str] = None,
        type_arn: typing.Optional[builtins.str] = None,
        type_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnHookTypeConfig``.

        :param configuration: Specifies the activated Hook type configuration, in this AWS account and AWS Region . You must specify either ``TypeName`` and ``Configuration`` or ``TypeArn`` and ``Configuration`` .
        :param configuration_alias: An alias by which to refer to this configuration data. Defaults to ``default`` alias. Hook types currently support default configuration alias. Default: - "default"
        :param type_arn: The Amazon Resource Number (ARN) for the Hook to set ``Configuration`` for. You must specify either ``TypeName`` and ``Configuration`` or ``TypeArn`` and ``Configuration`` .
        :param type_name: The unique name for your Hook. Specifies a three-part namespace for your Hook, with a recommended pattern of ``Organization::Service::Hook`` . You must specify either ``TypeName`` and ``Configuration`` or ``TypeArn`` and ``Configuration`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hooktypeconfig.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_hook_type_config_props = cdk.CfnHookTypeConfigProps(
                configuration="configuration",
            
                # the properties below are optional
                configuration_alias="configurationAlias",
                type_arn="typeArn",
                type_name="typeName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5a7ce592465f8cab9579ea1453e8eeccc9a2ff221395549f342f20f69d762a72)
            check_type(argname="argument configuration", value=configuration, expected_type=type_hints["configuration"])
            check_type(argname="argument configuration_alias", value=configuration_alias, expected_type=type_hints["configuration_alias"])
            check_type(argname="argument type_arn", value=type_arn, expected_type=type_hints["type_arn"])
            check_type(argname="argument type_name", value=type_name, expected_type=type_hints["type_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "configuration": configuration,
        }
        if configuration_alias is not None:
            self._values["configuration_alias"] = configuration_alias
        if type_arn is not None:
            self._values["type_arn"] = type_arn
        if type_name is not None:
            self._values["type_name"] = type_name

    @builtins.property
    def configuration(self) -> builtins.str:
        '''Specifies the activated Hook type configuration, in this AWS account and AWS Region .

        You must specify either ``TypeName`` and ``Configuration`` or ``TypeArn`` and ``Configuration`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hooktypeconfig.html#cfn-cloudformation-hooktypeconfig-configuration
        '''
        result = self._values.get("configuration")
        assert result is not None, "Required property 'configuration' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def configuration_alias(self) -> typing.Optional[builtins.str]:
        '''An alias by which to refer to this configuration data.

        Defaults to ``default`` alias. Hook types currently support default configuration alias.

        :default: - "default"

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hooktypeconfig.html#cfn-cloudformation-hooktypeconfig-configurationalias
        '''
        result = self._values.get("configuration_alias")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Number (ARN) for the Hook to set ``Configuration`` for.

        You must specify either ``TypeName`` and ``Configuration`` or ``TypeArn`` and ``Configuration`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hooktypeconfig.html#cfn-cloudformation-hooktypeconfig-typearn
        '''
        result = self._values.get("type_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type_name(self) -> typing.Optional[builtins.str]:
        '''The unique name for your Hook.

        Specifies a three-part namespace for your Hook, with a recommended pattern of ``Organization::Service::Hook`` .

        You must specify either ``TypeName`` and ``Configuration`` or ``TypeArn`` and ``Configuration`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hooktypeconfig.html#cfn-cloudformation-hooktypeconfig-typename
        '''
        result = self._values.get("type_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnHookTypeConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnHookVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "schema_handler_package": "schemaHandlerPackage",
        "type_name": "typeName",
        "execution_role_arn": "executionRoleArn",
        "logging_config": "loggingConfig",
    },
)
class CfnHookVersionProps:
    def __init__(
        self,
        *,
        schema_handler_package: builtins.str,
        type_name: builtins.str,
        execution_role_arn: typing.Optional[builtins.str] = None,
        logging_config: typing.Optional[typing.Union["IResolvable", typing.Union["CfnHookVersion.LoggingConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnHookVersion``.

        :param schema_handler_package: A URL to the Amazon S3 bucket containing the Hook project package that contains the necessary files for the Hook you want to register. For information on generating a schema handler package, see `Modeling custom CloudFormation Hooks <https://docs.aws.amazon.com/cloudformation-cli/latest/hooks-userguide/hooks-model.html>`_ in the *AWS CloudFormation Hooks User Guide* . .. epigraph:: To register the Hook, you must have ``s3:GetObject`` permissions to access the S3 objects.
        :param type_name: The unique name for your hook. Specifies a three-part namespace for your hook, with a recommended pattern of ``Organization::Service::Hook`` . .. epigraph:: The following organization namespaces are reserved and can't be used in your hook type names: - ``Alexa`` - ``AMZN`` - ``Amazon`` - ``ASK`` - ``AWS`` - ``Custom`` - ``Dev``
        :param execution_role_arn: The Amazon Resource Name (ARN) of the task execution role that grants the Hook permission.
        :param logging_config: Contains logging configuration information for an extension.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hookversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_hook_version_props = cdk.CfnHookVersionProps(
                schema_handler_package="schemaHandlerPackage",
                type_name="typeName",
            
                # the properties below are optional
                execution_role_arn="executionRoleArn",
                logging_config=cdk.CfnHookVersion.LoggingConfigProperty(
                    log_group_name="logGroupName",
                    log_role_arn="logRoleArn"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1bf57ffb1ed0d81059c9db4f1de2ad76bfbe14ae3d69e26fc60ae87eda289375)
            check_type(argname="argument schema_handler_package", value=schema_handler_package, expected_type=type_hints["schema_handler_package"])
            check_type(argname="argument type_name", value=type_name, expected_type=type_hints["type_name"])
            check_type(argname="argument execution_role_arn", value=execution_role_arn, expected_type=type_hints["execution_role_arn"])
            check_type(argname="argument logging_config", value=logging_config, expected_type=type_hints["logging_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "schema_handler_package": schema_handler_package,
            "type_name": type_name,
        }
        if execution_role_arn is not None:
            self._values["execution_role_arn"] = execution_role_arn
        if logging_config is not None:
            self._values["logging_config"] = logging_config

    @builtins.property
    def schema_handler_package(self) -> builtins.str:
        '''A URL to the Amazon S3 bucket containing the Hook project package that contains the necessary files for the Hook you want to register.

        For information on generating a schema handler package, see `Modeling custom CloudFormation Hooks <https://docs.aws.amazon.com/cloudformation-cli/latest/hooks-userguide/hooks-model.html>`_ in the *AWS CloudFormation Hooks User Guide* .
        .. epigraph::

           To register the Hook, you must have ``s3:GetObject`` permissions to access the S3 objects.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hookversion.html#cfn-cloudformation-hookversion-schemahandlerpackage
        '''
        result = self._values.get("schema_handler_package")
        assert result is not None, "Required property 'schema_handler_package' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type_name(self) -> builtins.str:
        '''The unique name for your hook.

        Specifies a three-part namespace for your hook, with a recommended pattern of ``Organization::Service::Hook`` .
        .. epigraph::

           The following organization namespaces are reserved and can't be used in your hook type names:

           - ``Alexa``
           - ``AMZN``
           - ``Amazon``
           - ``ASK``
           - ``AWS``
           - ``Custom``
           - ``Dev``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hookversion.html#cfn-cloudformation-hookversion-typename
        '''
        result = self._values.get("type_name")
        assert result is not None, "Required property 'type_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def execution_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the task execution role that grants the Hook permission.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hookversion.html#cfn-cloudformation-hookversion-executionrolearn
        '''
        result = self._values.get("execution_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def logging_config(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", "CfnHookVersion.LoggingConfigProperty"]]:
        '''Contains logging configuration information for an extension.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-hookversion.html#cfn-cloudformation-hookversion-loggingconfig
        '''
        result = self._values.get("logging_config")
        return typing.cast(typing.Optional[typing.Union["IResolvable", "CfnHookVersion.LoggingConfigProperty"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnHookVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnJsonProps",
    jsii_struct_bases=[],
    name_mapping={"value": "value"},
)
class CfnJsonProps:
    def __init__(self, *, value: typing.Any) -> None:
        '''
        :param value: The value to resolve. Can be any JavaScript object, including tokens and references in keys or values.

        :exampleMetadata: infused

        Example::

            tag_param = CfnParameter(self, "TagName")
            
            string_equals = CfnJson(self, "ConditionJson",
                value={
                    "f"aws:PrincipalTag/{tagParam.valueAsString}"": True
                }
            )
            
            principal = iam.AccountRootPrincipal().with_conditions({
                "StringEquals": string_equals
            })
            
            iam.Role(self, "MyRole", assumed_by=principal)
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__87eed2422211d2fd267f6174466957d5a425714d5c1428b9824d42eae88c3f27)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "value": value,
        }

    @builtins.property
    def value(self) -> typing.Any:
        '''The value to resolve.

        Can be any JavaScript object, including tokens and
        references in keys or values.
        '''
        result = self._values.get("value")
        assert result is not None, "Required property 'value' is missing"
        return typing.cast(typing.Any, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnJsonProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnLambdaHookProps",
    jsii_struct_bases=[],
    name_mapping={
        "alias": "alias",
        "execution_role": "executionRole",
        "failure_mode": "failureMode",
        "hook_status": "hookStatus",
        "lambda_function": "lambdaFunction",
        "target_operations": "targetOperations",
        "stack_filters": "stackFilters",
        "target_filters": "targetFilters",
    },
)
class CfnLambdaHookProps:
    def __init__(
        self,
        *,
        alias: builtins.str,
        execution_role: builtins.str,
        failure_mode: builtins.str,
        hook_status: builtins.str,
        lambda_function: builtins.str,
        target_operations: typing.Sequence[builtins.str],
        stack_filters: typing.Optional[typing.Union["IResolvable", typing.Union["CfnLambdaHook.StackFiltersProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        target_filters: typing.Optional[typing.Union["IResolvable", typing.Union["CfnLambdaHook.TargetFiltersProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnLambdaHook``.

        :param alias: The type name alias for the Hook. This alias must be unique per account and Region. The alias must be in the form ``Name1::Name2::Name3`` and must not begin with ``AWS`` . For example, ``Private::Lambda::MyTestHook`` .
        :param execution_role: The IAM role that the Hook assumes to invoke your Lambda function.
        :param failure_mode: Specifies how the Hook responds when the Lambda function invoked by the Hook returns a ``FAILED`` response. - ``FAIL`` : Prevents the action from proceeding. This is helpful for enforcing strict compliance or security policies. - ``WARN`` : Issues warnings to users but allows actions to continue. This is useful for non-critical validations or informational checks.
        :param hook_status: Specifies if the Hook is ``ENABLED`` or ``DISABLED`` . Default: - "ENABLED"
        :param lambda_function: Specifies the Lambda function for the Hook. You can use:. - The full Amazon Resource Name (ARN) without a suffix. - A qualified ARN with a version or alias suffix.
        :param target_operations: Specifies the list of operations the Hook is run against. For more information, see `Hook targets <https://docs.aws.amazon.com/cloudformation-cli/latest/hooks-userguide/hooks-concepts.html#hook-terms-hook-target>`_ in the *AWS CloudFormation Hooks User Guide* . Valid values: ``STACK`` | ``RESOURCE`` | ``CHANGE_SET`` | ``CLOUD_CONTROL``
        :param stack_filters: Specifies the stack level filters for the Hook. Example stack level filter in JSON: ``"StackFilters": {"FilteringCriteria": "ALL", "StackNames": {"Exclude": [ "stack-1", "stack-2"]}}`` Example stack level filter in YAML: ``StackFilters: FilteringCriteria: ALL StackNames: Exclude: - stack-1 - stack-2``
        :param target_filters: Specifies the target filters for the Hook. Example target filter in JSON: ``"TargetFilters": {"Actions": [ "Create", "Update", "Delete" ]}`` Example target filter in YAML: ``TargetFilters: Actions: - CREATE - UPDATE - DELETE``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-lambdahook.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_lambda_hook_props = cdk.CfnLambdaHookProps(
                alias="alias",
                execution_role="executionRole",
                failure_mode="failureMode",
                hook_status="hookStatus",
                lambda_function="lambdaFunction",
                target_operations=["targetOperations"],
            
                # the properties below are optional
                stack_filters=cdk.CfnLambdaHook.StackFiltersProperty(
                    filtering_criteria="filteringCriteria",
            
                    # the properties below are optional
                    stack_names=cdk.CfnLambdaHook.StackNamesProperty(
                        exclude=["exclude"],
                        include=["include"]
                    ),
                    stack_roles=cdk.CfnLambdaHook.StackRolesProperty(
                        exclude=["exclude"],
                        include=["include"]
                    )
                ),
                target_filters=cdk.CfnLambdaHook.TargetFiltersProperty(
                    targets=[cdk.CfnLambdaHook.HookTargetProperty(
                        action="action",
                        invocation_point="invocationPoint",
                        target_name="targetName"
                    )],
            
                    # the properties below are optional
                    actions=["actions"],
                    invocation_points=["invocationPoints"],
                    target_names=["targetNames"]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4a131cdc86d452e41ccdf75ad468f0b377a8ec03cc7f13c95235a7c90e881d29)
            check_type(argname="argument alias", value=alias, expected_type=type_hints["alias"])
            check_type(argname="argument execution_role", value=execution_role, expected_type=type_hints["execution_role"])
            check_type(argname="argument failure_mode", value=failure_mode, expected_type=type_hints["failure_mode"])
            check_type(argname="argument hook_status", value=hook_status, expected_type=type_hints["hook_status"])
            check_type(argname="argument lambda_function", value=lambda_function, expected_type=type_hints["lambda_function"])
            check_type(argname="argument target_operations", value=target_operations, expected_type=type_hints["target_operations"])
            check_type(argname="argument stack_filters", value=stack_filters, expected_type=type_hints["stack_filters"])
            check_type(argname="argument target_filters", value=target_filters, expected_type=type_hints["target_filters"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "alias": alias,
            "execution_role": execution_role,
            "failure_mode": failure_mode,
            "hook_status": hook_status,
            "lambda_function": lambda_function,
            "target_operations": target_operations,
        }
        if stack_filters is not None:
            self._values["stack_filters"] = stack_filters
        if target_filters is not None:
            self._values["target_filters"] = target_filters

    @builtins.property
    def alias(self) -> builtins.str:
        '''The type name alias for the Hook. This alias must be unique per account and Region.

        The alias must be in the form ``Name1::Name2::Name3`` and must not begin with ``AWS`` . For example, ``Private::Lambda::MyTestHook`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-lambdahook.html#cfn-cloudformation-lambdahook-alias
        '''
        result = self._values.get("alias")
        assert result is not None, "Required property 'alias' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def execution_role(self) -> builtins.str:
        '''The IAM role that the Hook assumes to invoke your Lambda function.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-lambdahook.html#cfn-cloudformation-lambdahook-executionrole
        '''
        result = self._values.get("execution_role")
        assert result is not None, "Required property 'execution_role' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def failure_mode(self) -> builtins.str:
        '''Specifies how the Hook responds when the Lambda function invoked by the Hook returns a ``FAILED`` response.

        - ``FAIL`` : Prevents the action from proceeding. This is helpful for enforcing strict compliance or security policies.
        - ``WARN`` : Issues warnings to users but allows actions to continue. This is useful for non-critical validations or informational checks.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-lambdahook.html#cfn-cloudformation-lambdahook-failuremode
        '''
        result = self._values.get("failure_mode")
        assert result is not None, "Required property 'failure_mode' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def hook_status(self) -> builtins.str:
        '''Specifies if the Hook is ``ENABLED`` or ``DISABLED`` .

        :default: - "ENABLED"

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-lambdahook.html#cfn-cloudformation-lambdahook-hookstatus
        '''
        result = self._values.get("hook_status")
        assert result is not None, "Required property 'hook_status' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def lambda_function(self) -> builtins.str:
        '''Specifies the Lambda function for the Hook. You can use:.

        - The full Amazon Resource Name (ARN) without a suffix.
        - A qualified ARN with a version or alias suffix.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-lambdahook.html#cfn-cloudformation-lambdahook-lambdafunction
        '''
        result = self._values.get("lambda_function")
        assert result is not None, "Required property 'lambda_function' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_operations(self) -> typing.List[builtins.str]:
        '''Specifies the list of operations the Hook is run against.

        For more information, see `Hook targets <https://docs.aws.amazon.com/cloudformation-cli/latest/hooks-userguide/hooks-concepts.html#hook-terms-hook-target>`_ in the *AWS CloudFormation Hooks User Guide* .

        Valid values: ``STACK`` | ``RESOURCE`` | ``CHANGE_SET`` | ``CLOUD_CONTROL``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-lambdahook.html#cfn-cloudformation-lambdahook-targetoperations
        '''
        result = self._values.get("target_operations")
        assert result is not None, "Required property 'target_operations' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def stack_filters(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", "CfnLambdaHook.StackFiltersProperty"]]:
        '''Specifies the stack level filters for the Hook.

        Example stack level filter in JSON:

        ``"StackFilters": {"FilteringCriteria": "ALL", "StackNames": {"Exclude": [ "stack-1", "stack-2"]}}``

        Example stack level filter in YAML:

        ``StackFilters: FilteringCriteria: ALL StackNames: Exclude: - stack-1 - stack-2``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-lambdahook.html#cfn-cloudformation-lambdahook-stackfilters
        '''
        result = self._values.get("stack_filters")
        return typing.cast(typing.Optional[typing.Union["IResolvable", "CfnLambdaHook.StackFiltersProperty"]], result)

    @builtins.property
    def target_filters(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", "CfnLambdaHook.TargetFiltersProperty"]]:
        '''Specifies the target filters for the Hook.

        Example target filter in JSON:

        ``"TargetFilters": {"Actions": [ "Create", "Update", "Delete" ]}``

        Example target filter in YAML:

        ``TargetFilters: Actions: - CREATE - UPDATE - DELETE``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-lambdahook.html#cfn-cloudformation-lambdahook-targetfilters
        '''
        result = self._values.get("target_filters")
        return typing.cast(typing.Optional[typing.Union["IResolvable", "CfnLambdaHook.TargetFiltersProperty"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnLambdaHookProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnMacroProps",
    jsii_struct_bases=[],
    name_mapping={
        "function_name": "functionName",
        "name": "name",
        "description": "description",
        "log_group_name": "logGroupName",
        "log_role_arn": "logRoleArn",
    },
)
class CfnMacroProps:
    def __init__(
        self,
        *,
        function_name: builtins.str,
        name: builtins.str,
        description: typing.Optional[builtins.str] = None,
        log_group_name: typing.Optional[builtins.str] = None,
        log_role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnMacro``.

        :param function_name: The Amazon Resource Name (ARN) of the underlying Lambda function that you want CloudFormation to invoke when the macro is run.
        :param name: The name of the macro. The name of the macro must be unique across all macros in the account.
        :param description: A description of the macro.
        :param log_group_name: The CloudWatch Logs group to which CloudFormation sends error logging information when invoking the macro's underlying Lambda function. This will be an existing CloudWatch Logs LogGroup. Neither CloudFormation or Lambda will create the group.
        :param log_role_arn: The ARN of the role CloudFormation should assume when sending log entries to CloudWatch Logs .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-macro.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_macro_props = cdk.CfnMacroProps(
                function_name="functionName",
                name="name",
            
                # the properties below are optional
                description="description",
                log_group_name="logGroupName",
                log_role_arn="logRoleArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__37ca317726d1a81abd91af8b3515fe38803c55d61fd20d81e54fc711ec81f086)
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument log_group_name", value=log_group_name, expected_type=type_hints["log_group_name"])
            check_type(argname="argument log_role_arn", value=log_role_arn, expected_type=type_hints["log_role_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_name": function_name,
            "name": name,
        }
        if description is not None:
            self._values["description"] = description
        if log_group_name is not None:
            self._values["log_group_name"] = log_group_name
        if log_role_arn is not None:
            self._values["log_role_arn"] = log_role_arn

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the underlying Lambda function that you want CloudFormation to invoke when the macro is run.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-macro.html#cfn-cloudformation-macro-functionname
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of the macro.

        The name of the macro must be unique across all macros in the account.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-macro.html#cfn-cloudformation-macro-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the macro.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-macro.html#cfn-cloudformation-macro-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_group_name(self) -> typing.Optional[builtins.str]:
        '''The CloudWatch Logs group to which CloudFormation sends error logging information when invoking the macro's underlying Lambda function.

        This will be an existing CloudWatch Logs LogGroup. Neither CloudFormation or Lambda will create the group.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-macro.html#cfn-cloudformation-macro-loggroupname
        '''
        result = self._values.get("log_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_role_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the role CloudFormation should assume when sending log entries to CloudWatch Logs .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-macro.html#cfn-cloudformation-macro-logrolearn
        '''
        result = self._values.get("log_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnMacroProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnMappingProps",
    jsii_struct_bases=[],
    name_mapping={"lazy": "lazy", "mapping": "mapping"},
)
class CfnMappingProps:
    def __init__(
        self,
        *,
        lazy: typing.Optional[builtins.bool] = None,
        mapping: typing.Optional[typing.Mapping[builtins.str, typing.Mapping[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''
        :param lazy: 
        :param mapping: Mapping of key to a set of corresponding set of named values. The key identifies a map of name-value pairs and must be unique within the mapping. For example, if you want to set values based on a region, you can create a mapping that uses the region name as a key and contains the values you want to specify for each specific region. Default: - No mapping.

        :exampleMetadata: infused

        Example::

            region_table = CfnMapping(self, "RegionTable",
                mapping={
                    "us-east-1": {
                        "region_name": "US East (N. Virginia)"
                    },
                    "us-east-2": {
                        "region_name": "US East (Ohio)"
                    }
                },
                lazy=True
            )
            
            region_table.find_in_map("us-east-2", "regionName")
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1661eec8c4dedc5a4b0f85332d5dfada759b3dcd7d56d62c059c454e7cbfc838)
            check_type(argname="argument lazy", value=lazy, expected_type=type_hints["lazy"])
            check_type(argname="argument mapping", value=mapping, expected_type=type_hints["mapping"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if lazy is not None:
            self._values["lazy"] = lazy
        if mapping is not None:
            self._values["mapping"] = mapping

    @builtins.property
    def lazy(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("lazy")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mapping(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Mapping[builtins.str, typing.Any]]]:
        '''Mapping of key to a set of corresponding set of named values.

        The key identifies a map of name-value pairs and must be unique within the mapping.

        For example, if you want to set values based on a region, you can create a mapping
        that uses the region name as a key and contains the values you want to specify for
        each specific region.

        :default: - No mapping.
        '''
        result = self._values.get("mapping")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Mapping[builtins.str, typing.Any]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnMappingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnModuleDefaultVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "arn": "arn",
        "module_name": "moduleName",
        "version_id": "versionId",
    },
)
class CfnModuleDefaultVersionProps:
    def __init__(
        self,
        *,
        arn: typing.Optional[builtins.str] = None,
        module_name: typing.Optional[builtins.str] = None,
        version_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnModuleDefaultVersion``.

        :param arn: The Amazon Resource Name (ARN) of the module version to set as the default version. Conditional: You must specify either ``Arn`` , or ``ModuleName`` and ``VersionId`` .
        :param module_name: The name of the module. Conditional: You must specify either ``Arn`` , or ``ModuleName`` and ``VersionId`` .
        :param version_id: The ID for the specific version of the module. Conditional: You must specify either ``Arn`` , or ``ModuleName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-moduledefaultversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_module_default_version_props = cdk.CfnModuleDefaultVersionProps(
                arn="arn",
                module_name="moduleName",
                version_id="versionId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9cc56e2d0e6a70cdbb8dcc60b487f94b24b2369c1b8d4849abf2f6e4ed165b40)
            check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
            check_type(argname="argument module_name", value=module_name, expected_type=type_hints["module_name"])
            check_type(argname="argument version_id", value=version_id, expected_type=type_hints["version_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if arn is not None:
            self._values["arn"] = arn
        if module_name is not None:
            self._values["module_name"] = module_name
        if version_id is not None:
            self._values["version_id"] = version_id

    @builtins.property
    def arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the module version to set as the default version.

        Conditional: You must specify either ``Arn`` , or ``ModuleName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-moduledefaultversion.html#cfn-cloudformation-moduledefaultversion-arn
        '''
        result = self._values.get("arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def module_name(self) -> typing.Optional[builtins.str]:
        '''The name of the module.

        Conditional: You must specify either ``Arn`` , or ``ModuleName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-moduledefaultversion.html#cfn-cloudformation-moduledefaultversion-modulename
        '''
        result = self._values.get("module_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def version_id(self) -> typing.Optional[builtins.str]:
        '''The ID for the specific version of the module.

        Conditional: You must specify either ``Arn`` , or ``ModuleName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-moduledefaultversion.html#cfn-cloudformation-moduledefaultversion-versionid
        '''
        result = self._values.get("version_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnModuleDefaultVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnModuleVersionProps",
    jsii_struct_bases=[],
    name_mapping={"module_name": "moduleName", "module_package": "modulePackage"},
)
class CfnModuleVersionProps:
    def __init__(
        self,
        *,
        module_name: builtins.str,
        module_package: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnModuleVersion``.

        :param module_name: The name of the module being registered.
        :param module_package: A URL to the S3 bucket containing the package that contains the template fragment and schema files for the module version to register. For more information, see `Module structure and requirements <https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/modules-structure.html>`_ in the *AWS CloudFormation Command Line Interface (CLI) User Guide* . .. epigraph:: To register the module version, you must have ``s3:GetObject`` permissions to access the S3 objects.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-moduleversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_module_version_props = cdk.CfnModuleVersionProps(
                module_name="moduleName",
                module_package="modulePackage"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0236c4123691aa817c64c404366ded37cea57bfc4cbbc1ab802b2d110f89e859)
            check_type(argname="argument module_name", value=module_name, expected_type=type_hints["module_name"])
            check_type(argname="argument module_package", value=module_package, expected_type=type_hints["module_package"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "module_name": module_name,
            "module_package": module_package,
        }

    @builtins.property
    def module_name(self) -> builtins.str:
        '''The name of the module being registered.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-moduleversion.html#cfn-cloudformation-moduleversion-modulename
        '''
        result = self._values.get("module_name")
        assert result is not None, "Required property 'module_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def module_package(self) -> builtins.str:
        '''A URL to the S3 bucket containing the package that contains the template fragment and schema files for the module version to register.

        For more information, see `Module structure and requirements <https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/modules-structure.html>`_ in the *AWS CloudFormation Command Line Interface (CLI) User Guide* .
        .. epigraph::

           To register the module version, you must have ``s3:GetObject`` permissions to access the S3 objects.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-moduleversion.html#cfn-cloudformation-moduleversion-modulepackage
        '''
        result = self._values.get("module_package")
        assert result is not None, "Required property 'module_package' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnModuleVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CfnOutput(CfnElement, metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.CfnOutput"):
    '''
    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        # add service account
        service_account = cluster.add_service_account("MyServiceAccount")
        
        bucket = s3.Bucket(self, "Bucket")
        bucket.grant_read_write(service_account)
        
        mypod = cluster.add_manifest("mypod", {
            "api_version": "v1",
            "kind": "Pod",
            "metadata": {"name": "mypod"},
            "spec": {
                "service_account_name": service_account.service_account_name,
                "containers": [{
                    "name": "hello",
                    "image": "paulbouwer/hello-kubernetes:1.5",
                    "ports": [{"container_port": 8080}]
                }
                ]
            }
        })
        
        # create the resource after the service account.
        mypod.node.add_dependency(service_account)
        
        # print the IAM role arn for this service account
        CfnOutput(self, "ServiceAccountIamRole", value=service_account.role.role_arn)
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        value: builtins.str,
        condition: typing.Optional["CfnCondition"] = None,
        description: typing.Optional[builtins.str] = None,
        export_name: typing.Optional[builtins.str] = None,
        key: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Creates an CfnOutput value for this stack.

        :param scope: The parent construct.
        :param id: -
        :param value: The value of the property returned by the aws cloudformation describe-stacks command. The value of an output can include literals, parameter references, pseudo-parameters, a mapping value, or intrinsic functions.
        :param condition: A condition to associate with this output value. If the condition evaluates to ``false``, this output value will not be included in the stack. Default: - No condition is associated with the output.
        :param description: A String type that describes the output value. The description can be a maximum of 4 K in length. Default: - No description.
        :param export_name: The name used to export the value of this output across stacks. To import the value from another stack, use ``Fn.importValue(exportName)``. Default: - the output is not exported
        :param key: The key of the property returned by aws cloudformation describe-stacks command.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c35d48c38799aea9675a664131517d899601afec8cd5325c458a0328f96d1d10)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnOutputProps(
            value=value,
            condition=condition,
            description=description,
            export_name=export_name,
            key=key,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="importValue")
    def import_value(self) -> builtins.str:
        '''Return the ``Fn.importValue`` expression to import this value into another stack.

        The returned value should not be used in the same stack, but in a
        different one. It must be deployed to the same environment, as
        CloudFormation exports can only be imported in the same Region and
        account.

        The is no automatic registration of dependencies between stacks when using
        this mechanism, so you should make sure to deploy them in the right order
        yourself.

        You can use this mechanism to share values across Stacks in different
        Stages. If you intend to share the value to another Stack inside the same
        Stage, the automatic cross-stack referencing mechanism is more convenient.
        '''
        return typing.cast(builtins.str, jsii.get(self, "importValue"))

    @builtins.property
    @jsii.member(jsii_name="value")
    def value(self) -> typing.Any:
        '''The value of the property returned by the aws cloudformation describe-stacks command.

        The value of an output can include literals, parameter references, pseudo-parameters,
        a mapping value, or intrinsic functions.
        '''
        return typing.cast(typing.Any, jsii.get(self, "value"))

    @value.setter
    def value(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ae04907c14401d9f8daddd7f2ed7732084e0b1937706e17d55614ea6380a3816)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "value", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="condition")
    def condition(self) -> typing.Optional["CfnCondition"]:
        '''A condition to associate with this output value.

        If the condition evaluates
        to ``false``, this output value will not be included in the stack.

        :default: - No condition is associated with the output.
        '''
        return typing.cast(typing.Optional["CfnCondition"], jsii.get(self, "condition"))

    @condition.setter
    def condition(self, value: typing.Optional["CfnCondition"]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8704d4bb35aeed5dde58d118c74dd1b1dc0b548e02f85e734228b0c76dd257d6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "condition", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A String type that describes the output value.

        The description can be a maximum of 4 K in length.

        :default: - No description.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fee62e15bb63daed4d2af15024b68eda73936d1f22f424c9162a603bbf238303)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="exportName")
    def export_name(self) -> typing.Optional[builtins.str]:
        '''The name used to export the value of this output across stacks.

        To use the value in another stack, pass the value of
        ``output.importValue`` to it.

        :default: - the output is not exported
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "exportName"))

    @export_name.setter
    def export_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a6f99116a425880b0b5f808fc659a8fe40a778123bb6026c47ec6d6324bbbd47)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "exportName", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnOutputProps",
    jsii_struct_bases=[],
    name_mapping={
        "value": "value",
        "condition": "condition",
        "description": "description",
        "export_name": "exportName",
        "key": "key",
    },
)
class CfnOutputProps:
    def __init__(
        self,
        *,
        value: builtins.str,
        condition: typing.Optional["CfnCondition"] = None,
        description: typing.Optional[builtins.str] = None,
        export_name: typing.Optional[builtins.str] = None,
        key: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param value: The value of the property returned by the aws cloudformation describe-stacks command. The value of an output can include literals, parameter references, pseudo-parameters, a mapping value, or intrinsic functions.
        :param condition: A condition to associate with this output value. If the condition evaluates to ``false``, this output value will not be included in the stack. Default: - No condition is associated with the output.
        :param description: A String type that describes the output value. The description can be a maximum of 4 K in length. Default: - No description.
        :param export_name: The name used to export the value of this output across stacks. To import the value from another stack, use ``Fn.importValue(exportName)``. Default: - the output is not exported
        :param key: The key of the property returned by aws cloudformation describe-stacks command.

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            # add service account
            service_account = cluster.add_service_account("MyServiceAccount")
            
            bucket = s3.Bucket(self, "Bucket")
            bucket.grant_read_write(service_account)
            
            mypod = cluster.add_manifest("mypod", {
                "api_version": "v1",
                "kind": "Pod",
                "metadata": {"name": "mypod"},
                "spec": {
                    "service_account_name": service_account.service_account_name,
                    "containers": [{
                        "name": "hello",
                        "image": "paulbouwer/hello-kubernetes:1.5",
                        "ports": [{"container_port": 8080}]
                    }
                    ]
                }
            })
            
            # create the resource after the service account.
            mypod.node.add_dependency(service_account)
            
            # print the IAM role arn for this service account
            CfnOutput(self, "ServiceAccountIamRole", value=service_account.role.role_arn)
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6af828294354071e80ffd088d31250ef749d4b932a346aa42aa54bd94579c610)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            check_type(argname="argument condition", value=condition, expected_type=type_hints["condition"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument export_name", value=export_name, expected_type=type_hints["export_name"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "value": value,
        }
        if condition is not None:
            self._values["condition"] = condition
        if description is not None:
            self._values["description"] = description
        if export_name is not None:
            self._values["export_name"] = export_name
        if key is not None:
            self._values["key"] = key

    @builtins.property
    def value(self) -> builtins.str:
        '''The value of the property returned by the aws cloudformation describe-stacks command.

        The value of an output can include literals, parameter references, pseudo-parameters,
        a mapping value, or intrinsic functions.
        '''
        result = self._values.get("value")
        assert result is not None, "Required property 'value' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def condition(self) -> typing.Optional["CfnCondition"]:
        '''A condition to associate with this output value.

        If the condition evaluates
        to ``false``, this output value will not be included in the stack.

        :default: - No condition is associated with the output.
        '''
        result = self._values.get("condition")
        return typing.cast(typing.Optional["CfnCondition"], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A String type that describes the output value.

        The description can be a maximum of 4 K in length.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def export_name(self) -> typing.Optional[builtins.str]:
        '''The name used to export the value of this output across stacks.

        To import the value from another stack, use ``Fn.importValue(exportName)``.

        :default: - the output is not exported
        '''
        result = self._values.get("export_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def key(self) -> typing.Optional[builtins.str]:
        '''The key of the property returned by aws cloudformation describe-stacks command.'''
        result = self._values.get("key")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnOutputProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CfnParameter(
    CfnElement,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.CfnParameter",
):
    '''A CloudFormation parameter.

    Use the optional Parameters section to customize your templates.
    Parameters enable you to input custom values to your template each time you create or
    update a stack.

    :exampleMetadata: infused

    Example::

        my_topic = sns.Topic(self, "MyTopic")
        url = CfnParameter(self, "url-param")
        
        my_topic.add_subscription(subscriptions.UrlSubscription(url.value_as_string))
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        allowed_pattern: typing.Optional[builtins.str] = None,
        allowed_values: typing.Optional[typing.Sequence[builtins.str]] = None,
        constraint_description: typing.Optional[builtins.str] = None,
        default: typing.Any = None,
        description: typing.Optional[builtins.str] = None,
        max_length: typing.Optional[jsii.Number] = None,
        max_value: typing.Optional[jsii.Number] = None,
        min_length: typing.Optional[jsii.Number] = None,
        min_value: typing.Optional[jsii.Number] = None,
        no_echo: typing.Optional[builtins.bool] = None,
        type: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Creates a parameter construct.

        Note that the name (logical ID) of the parameter will derive from it's ``coname`` and location
        within the stack. Therefore, it is recommended that parameters are defined at the stack level.

        :param scope: The parent construct.
        :param id: -
        :param allowed_pattern: A regular expression that represents the patterns to allow for String types. Default: - No constraints on patterns allowed for parameter.
        :param allowed_values: An array containing the list of values allowed for the parameter. Default: - No constraints on values allowed for parameter.
        :param constraint_description: A string that explains a constraint when the constraint is violated. For example, without a constraint description, a parameter that has an allowed pattern of [A-Za-z0-9]+ displays the following error message when the user specifies an invalid value: Default: - No description with customized error message when user specifies invalid values.
        :param default: A value of the appropriate type for the template to use if no value is specified when a stack is created. If you define constraints for the parameter, you must specify a value that adheres to those constraints. Default: - No default value for parameter.
        :param description: A string of up to 4000 characters that describes the parameter. Default: - No description for the parameter.
        :param max_length: An integer value that determines the largest number of characters you want to allow for String types. Default: - None.
        :param max_value: A numeric value that determines the largest numeric value you want to allow for Number types. Default: - None.
        :param min_length: An integer value that determines the smallest number of characters you want to allow for String types. Default: - None.
        :param min_value: A numeric value that determines the smallest numeric value you want to allow for Number types. Default: - None.
        :param no_echo: Whether to mask the parameter value when anyone makes a call that describes the stack. If you set the value to ``true``, the parameter value is masked with asterisks (``*****``). Default: - Parameter values are not masked.
        :param type: The data type for the parameter (DataType). Default: String
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cc555548af6d6231010b36b8e59a6893b7de17b7441d093d1cbf57f7f49aca44)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnParameterProps(
            allowed_pattern=allowed_pattern,
            allowed_values=allowed_values,
            constraint_description=constraint_description,
            default=default,
            description=description,
            max_length=max_length,
            max_value=max_value,
            min_length=min_length,
            min_value=min_value,
            no_echo=no_echo,
            type=type,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="resolve")
    def resolve(self, _context: "IResolveContext") -> typing.Any:
        '''
        :param _context: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__13c68dd59f6890fa9ec1f34216f0bbdad4d86fbebba45e297f1647ad8960b8a5)
            check_type(argname="argument _context", value=_context, expected_type=type_hints["_context"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [_context]))

    @builtins.property
    @jsii.member(jsii_name="value")
    def value(self) -> "IResolvable":
        '''The parameter value as a Token.'''
        return typing.cast("IResolvable", jsii.get(self, "value"))

    @builtins.property
    @jsii.member(jsii_name="valueAsList")
    def value_as_list(self) -> typing.List[builtins.str]:
        '''The parameter value, if it represents a string list.'''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "valueAsList"))

    @builtins.property
    @jsii.member(jsii_name="valueAsNumber")
    def value_as_number(self) -> jsii.Number:
        '''The parameter value, if it represents a number.'''
        return typing.cast(jsii.Number, jsii.get(self, "valueAsNumber"))

    @builtins.property
    @jsii.member(jsii_name="valueAsString")
    def value_as_string(self) -> builtins.str:
        '''The parameter value, if it represents a string.'''
        return typing.cast(builtins.str, jsii.get(self, "valueAsString"))

    @builtins.property
    @jsii.member(jsii_name="default")
    def default(self) -> typing.Any:
        '''A value of the appropriate type for the template to use if no value is specified when a stack is created.

        If you define constraints for the parameter, you must specify
        a value that adheres to those constraints.

        :default: - No default value for parameter.
        '''
        return typing.cast(typing.Any, jsii.get(self, "default"))

    @default.setter
    def default(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__890e37ff3466d7066d047df24be9cc48dec6995687b94045bf1a867a84bfcb00)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "default", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="noEcho")
    def no_echo(self) -> builtins.bool:
        '''Indicates if this parameter is configured with "NoEcho" enabled.'''
        return typing.cast(builtins.bool, jsii.get(self, "noEcho"))

    @no_echo.setter
    def no_echo(self, value: builtins.bool) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b9bc125d000f64952fca0539cfebab47b021d967867b6fe369026609c704ce51)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "noEcho", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''The data type for the parameter (DataType).

        :default: String
        '''
        return typing.cast(builtins.str, jsii.get(self, "type"))

    @type.setter
    def type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a7e2790a72711c495dfbc2fbf18f125f5cafa092b80a53f60e4a83ca213fbacc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "type", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="allowedPattern")
    def allowed_pattern(self) -> typing.Optional[builtins.str]:
        '''A regular expression that represents the patterns to allow for String types.

        :default: - No constraints on patterns allowed for parameter.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "allowedPattern"))

    @allowed_pattern.setter
    def allowed_pattern(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bccc804e9bde09df07b9c43eca0d49a9a986e8d6463373c956d71c78852b10ea)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allowedPattern", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="allowedValues")
    def allowed_values(self) -> typing.Optional[typing.List[builtins.str]]:
        '''An array containing the list of values allowed for the parameter.

        :default: - No constraints on values allowed for parameter.
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "allowedValues"))

    @allowed_values.setter
    def allowed_values(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__244e39c60d4038554921b86623b26479586dc80a8377d492f3e3687126aff5c8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allowedValues", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="constraintDescription")
    def constraint_description(self) -> typing.Optional[builtins.str]:
        '''A string that explains a constraint when the constraint is violated.

        For example, without a constraint description, a parameter that has an allowed
        pattern of [A-Za-z0-9]+ displays the following error message when the user specifies
        an invalid value:

        :default: - No description with customized error message when user specifies invalid values.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "constraintDescription"))

    @constraint_description.setter
    def constraint_description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bc159a89368624d20193ad929b2b370445b7ebd9f1ea6879030e1454ca997298)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "constraintDescription", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A string of up to 4000 characters that describes the parameter.

        :default: - No description for the parameter.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f02c11a5a6084ab7285739b645ff4db52fa4ab9d537140e0040eb928df488a0a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="maxLength")
    def max_length(self) -> typing.Optional[jsii.Number]:
        '''An integer value that determines the largest number of characters you want to allow for String types.

        :default: - None.
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxLength"))

    @max_length.setter
    def max_length(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b8c7f7988f60b65db86ed4c303c4c30645856005b14ba31e44cbb0370593ba38)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maxLength", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="maxValue")
    def max_value(self) -> typing.Optional[jsii.Number]:
        '''A numeric value that determines the largest numeric value you want to allow for Number types.

        :default: - None.
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxValue"))

    @max_value.setter
    def max_value(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f637122dfdd53cfdf3313b8dc0055a13240a2701e688fdd73b3b873bf5585c19)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maxValue", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="minLength")
    def min_length(self) -> typing.Optional[jsii.Number]:
        '''An integer value that determines the smallest number of characters you want to allow for String types.

        :default: - None.
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "minLength"))

    @min_length.setter
    def min_length(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ece8fbd2ab6623d4345ceb33f2bc952eae9ceea3fd07df99365a2e77df0255cb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "minLength", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="minValue")
    def min_value(self) -> typing.Optional[jsii.Number]:
        '''A numeric value that determines the smallest numeric value you want to allow for Number types.

        :default: - None.
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "minValue"))

    @min_value.setter
    def min_value(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__520687c32986ebe96570a7b651014e5422b522582468d929a83c9bdcbbd914c1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "minValue", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnParameterProps",
    jsii_struct_bases=[],
    name_mapping={
        "allowed_pattern": "allowedPattern",
        "allowed_values": "allowedValues",
        "constraint_description": "constraintDescription",
        "default": "default",
        "description": "description",
        "max_length": "maxLength",
        "max_value": "maxValue",
        "min_length": "minLength",
        "min_value": "minValue",
        "no_echo": "noEcho",
        "type": "type",
    },
)
class CfnParameterProps:
    def __init__(
        self,
        *,
        allowed_pattern: typing.Optional[builtins.str] = None,
        allowed_values: typing.Optional[typing.Sequence[builtins.str]] = None,
        constraint_description: typing.Optional[builtins.str] = None,
        default: typing.Any = None,
        description: typing.Optional[builtins.str] = None,
        max_length: typing.Optional[jsii.Number] = None,
        max_value: typing.Optional[jsii.Number] = None,
        min_length: typing.Optional[jsii.Number] = None,
        min_value: typing.Optional[jsii.Number] = None,
        no_echo: typing.Optional[builtins.bool] = None,
        type: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param allowed_pattern: A regular expression that represents the patterns to allow for String types. Default: - No constraints on patterns allowed for parameter.
        :param allowed_values: An array containing the list of values allowed for the parameter. Default: - No constraints on values allowed for parameter.
        :param constraint_description: A string that explains a constraint when the constraint is violated. For example, without a constraint description, a parameter that has an allowed pattern of [A-Za-z0-9]+ displays the following error message when the user specifies an invalid value: Default: - No description with customized error message when user specifies invalid values.
        :param default: A value of the appropriate type for the template to use if no value is specified when a stack is created. If you define constraints for the parameter, you must specify a value that adheres to those constraints. Default: - No default value for parameter.
        :param description: A string of up to 4000 characters that describes the parameter. Default: - No description for the parameter.
        :param max_length: An integer value that determines the largest number of characters you want to allow for String types. Default: - None.
        :param max_value: A numeric value that determines the largest numeric value you want to allow for Number types. Default: - None.
        :param min_length: An integer value that determines the smallest number of characters you want to allow for String types. Default: - None.
        :param min_value: A numeric value that determines the smallest numeric value you want to allow for Number types. Default: - None.
        :param no_echo: Whether to mask the parameter value when anyone makes a call that describes the stack. If you set the value to ``true``, the parameter value is masked with asterisks (``*****``). Default: - Parameter values are not masked.
        :param type: The data type for the parameter (DataType). Default: String

        :exampleMetadata: infused

        Example::

            param = CfnParameter(self, "ParameterName")
            
            # If the parameter is a String
            param.value_as_string
            
            # If the parameter is a Number
            param.value_as_number
            
            # If the parameter is a List
            param.value_as_list
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6d425fa390023c3d5a204065fa1d0abca878458a00e8f9bd0859be42e4cc9e66)
            check_type(argname="argument allowed_pattern", value=allowed_pattern, expected_type=type_hints["allowed_pattern"])
            check_type(argname="argument allowed_values", value=allowed_values, expected_type=type_hints["allowed_values"])
            check_type(argname="argument constraint_description", value=constraint_description, expected_type=type_hints["constraint_description"])
            check_type(argname="argument default", value=default, expected_type=type_hints["default"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument max_length", value=max_length, expected_type=type_hints["max_length"])
            check_type(argname="argument max_value", value=max_value, expected_type=type_hints["max_value"])
            check_type(argname="argument min_length", value=min_length, expected_type=type_hints["min_length"])
            check_type(argname="argument min_value", value=min_value, expected_type=type_hints["min_value"])
            check_type(argname="argument no_echo", value=no_echo, expected_type=type_hints["no_echo"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allowed_pattern is not None:
            self._values["allowed_pattern"] = allowed_pattern
        if allowed_values is not None:
            self._values["allowed_values"] = allowed_values
        if constraint_description is not None:
            self._values["constraint_description"] = constraint_description
        if default is not None:
            self._values["default"] = default
        if description is not None:
            self._values["description"] = description
        if max_length is not None:
            self._values["max_length"] = max_length
        if max_value is not None:
            self._values["max_value"] = max_value
        if min_length is not None:
            self._values["min_length"] = min_length
        if min_value is not None:
            self._values["min_value"] = min_value
        if no_echo is not None:
            self._values["no_echo"] = no_echo
        if type is not None:
            self._values["type"] = type

    @builtins.property
    def allowed_pattern(self) -> typing.Optional[builtins.str]:
        '''A regular expression that represents the patterns to allow for String types.

        :default: - No constraints on patterns allowed for parameter.
        '''
        result = self._values.get("allowed_pattern")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def allowed_values(self) -> typing.Optional[typing.List[builtins.str]]:
        '''An array containing the list of values allowed for the parameter.

        :default: - No constraints on values allowed for parameter.
        '''
        result = self._values.get("allowed_values")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def constraint_description(self) -> typing.Optional[builtins.str]:
        '''A string that explains a constraint when the constraint is violated.

        For example, without a constraint description, a parameter that has an allowed
        pattern of [A-Za-z0-9]+ displays the following error message when the user specifies
        an invalid value:

        :default: - No description with customized error message when user specifies invalid values.
        '''
        result = self._values.get("constraint_description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def default(self) -> typing.Any:
        '''A value of the appropriate type for the template to use if no value is specified when a stack is created.

        If you define constraints for the parameter, you must specify
        a value that adheres to those constraints.

        :default: - No default value for parameter.
        '''
        result = self._values.get("default")
        return typing.cast(typing.Any, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A string of up to 4000 characters that describes the parameter.

        :default: - No description for the parameter.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_length(self) -> typing.Optional[jsii.Number]:
        '''An integer value that determines the largest number of characters you want to allow for String types.

        :default: - None.
        '''
        result = self._values.get("max_length")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_value(self) -> typing.Optional[jsii.Number]:
        '''A numeric value that determines the largest numeric value you want to allow for Number types.

        :default: - None.
        '''
        result = self._values.get("max_value")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_length(self) -> typing.Optional[jsii.Number]:
        '''An integer value that determines the smallest number of characters you want to allow for String types.

        :default: - None.
        '''
        result = self._values.get("min_length")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_value(self) -> typing.Optional[jsii.Number]:
        '''A numeric value that determines the smallest numeric value you want to allow for Number types.

        :default: - None.
        '''
        result = self._values.get("min_value")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def no_echo(self) -> typing.Optional[builtins.bool]:
        '''Whether to mask the parameter value when anyone makes a call that describes the stack.

        If you set the value to ``true``, the parameter value is masked with asterisks (``*****``).

        :default: - Parameter values are not masked.
        '''
        result = self._values.get("no_echo")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def type(self) -> typing.Optional[builtins.str]:
        '''The data type for the parameter (DataType).

        :default: String
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnParameterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnPublicTypeVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "arn": "arn",
        "log_delivery_bucket": "logDeliveryBucket",
        "public_version_number": "publicVersionNumber",
        "type": "type",
        "type_name": "typeName",
    },
)
class CfnPublicTypeVersionProps:
    def __init__(
        self,
        *,
        arn: typing.Optional[builtins.str] = None,
        log_delivery_bucket: typing.Optional[builtins.str] = None,
        public_version_number: typing.Optional[builtins.str] = None,
        type: typing.Optional[builtins.str] = None,
        type_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnPublicTypeVersion``.

        :param arn: The Amazon Resource Number (ARN) of the extension. Conditional: You must specify ``Arn`` , or ``TypeName`` and ``Type`` .
        :param log_delivery_bucket: The S3 bucket to which CloudFormation delivers the contract test execution logs. CloudFormation delivers the logs by the time contract testing has completed and the extension has been assigned a test type status of ``PASSED`` or ``FAILED`` . The user initiating the stack operation must be able to access items in the specified S3 bucket. Specifically, the user needs the following permissions: - s3:GetObject - s3:PutObject
        :param public_version_number: The version number to assign to this version of the extension. Use the following format, and adhere to semantic versioning when assigning a version number to your extension: ``MAJOR.MINOR.PATCH`` For more information, see `Semantic Versioning 2.0.0 <https://docs.aws.amazon.com/https://semver.org/>`_ . If you don't specify a version number, CloudFormation increments the version number by one minor version release. You cannot specify a version number the first time you publish a type. CloudFormation automatically sets the first version number to be ``1.0.0`` .
        :param type: The type of the extension to test. Conditional: You must specify ``Arn`` , or ``TypeName`` and ``Type`` .
        :param type_name: The name of the extension to test. Conditional: You must specify ``Arn`` , or ``TypeName`` and ``Type`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-publictypeversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_public_type_version_props = cdk.CfnPublicTypeVersionProps(
                arn="arn",
                log_delivery_bucket="logDeliveryBucket",
                public_version_number="publicVersionNumber",
                type="type",
                type_name="typeName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bdc5db969e191a5bff30053fc7d54639715ea15209b6396ee3a6bb316d1f37bc)
            check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
            check_type(argname="argument log_delivery_bucket", value=log_delivery_bucket, expected_type=type_hints["log_delivery_bucket"])
            check_type(argname="argument public_version_number", value=public_version_number, expected_type=type_hints["public_version_number"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument type_name", value=type_name, expected_type=type_hints["type_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if arn is not None:
            self._values["arn"] = arn
        if log_delivery_bucket is not None:
            self._values["log_delivery_bucket"] = log_delivery_bucket
        if public_version_number is not None:
            self._values["public_version_number"] = public_version_number
        if type is not None:
            self._values["type"] = type
        if type_name is not None:
            self._values["type_name"] = type_name

    @builtins.property
    def arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Number (ARN) of the extension.

        Conditional: You must specify ``Arn`` , or ``TypeName`` and ``Type`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-publictypeversion.html#cfn-cloudformation-publictypeversion-arn
        '''
        result = self._values.get("arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_delivery_bucket(self) -> typing.Optional[builtins.str]:
        '''The S3 bucket to which CloudFormation delivers the contract test execution logs.

        CloudFormation delivers the logs by the time contract testing has completed and the extension has been assigned a test type status of ``PASSED`` or ``FAILED`` .

        The user initiating the stack operation must be able to access items in the specified S3 bucket. Specifically, the user needs the following permissions:

        - s3:GetObject
        - s3:PutObject

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-publictypeversion.html#cfn-cloudformation-publictypeversion-logdeliverybucket
        '''
        result = self._values.get("log_delivery_bucket")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def public_version_number(self) -> typing.Optional[builtins.str]:
        '''The version number to assign to this version of the extension.

        Use the following format, and adhere to semantic versioning when assigning a version number to your extension:

        ``MAJOR.MINOR.PATCH``

        For more information, see `Semantic Versioning 2.0.0 <https://docs.aws.amazon.com/https://semver.org/>`_ .

        If you don't specify a version number, CloudFormation increments the version number by one minor version release.

        You cannot specify a version number the first time you publish a type. CloudFormation automatically sets the first version number to be ``1.0.0`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-publictypeversion.html#cfn-cloudformation-publictypeversion-publicversionnumber
        '''
        result = self._values.get("public_version_number")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type(self) -> typing.Optional[builtins.str]:
        '''The type of the extension to test.

        Conditional: You must specify ``Arn`` , or ``TypeName`` and ``Type`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-publictypeversion.html#cfn-cloudformation-publictypeversion-type
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type_name(self) -> typing.Optional[builtins.str]:
        '''The name of the extension to test.

        Conditional: You must specify ``Arn`` , or ``TypeName`` and ``Type`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-publictypeversion.html#cfn-cloudformation-publictypeversion-typename
        '''
        result = self._values.get("type_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnPublicTypeVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnPublisherProps",
    jsii_struct_bases=[],
    name_mapping={
        "accept_terms_and_conditions": "acceptTermsAndConditions",
        "connection_arn": "connectionArn",
    },
)
class CfnPublisherProps:
    def __init__(
        self,
        *,
        accept_terms_and_conditions: typing.Union[builtins.bool, "IResolvable"],
        connection_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnPublisher``.

        :param accept_terms_and_conditions: Whether you accept the `Terms and Conditions <https://docs.aws.amazon.com/https://cloudformation-registry-documents.s3.amazonaws.com/Terms_and_Conditions_for_AWS_CloudFormation_Registry_Publishers.pdf>`_ for publishing extensions in the CloudFormation registry. You must accept the terms and conditions in order to register to publish public extensions to the CloudFormation registry. The default is ``false`` .
        :param connection_arn: If you are using a Bitbucket or GitHub account for identity verification, the Amazon Resource Name (ARN) for your connection to that account. For more information, see `Prerequisite: Registering your account to publish CloudFormation extensions <https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/publish-extension.html#publish-extension-prereqs>`_ in the *AWS CloudFormation Command Line Interface (CLI) User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-publisher.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_publisher_props = cdk.CfnPublisherProps(
                accept_terms_and_conditions=False,
            
                # the properties below are optional
                connection_arn="connectionArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f03e133e4177dc4ea11f9dd025e802bf66991f843afdd43544239ad9ef44f390)
            check_type(argname="argument accept_terms_and_conditions", value=accept_terms_and_conditions, expected_type=type_hints["accept_terms_and_conditions"])
            check_type(argname="argument connection_arn", value=connection_arn, expected_type=type_hints["connection_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "accept_terms_and_conditions": accept_terms_and_conditions,
        }
        if connection_arn is not None:
            self._values["connection_arn"] = connection_arn

    @builtins.property
    def accept_terms_and_conditions(self) -> typing.Union[builtins.bool, "IResolvable"]:
        '''Whether you accept the `Terms and Conditions <https://docs.aws.amazon.com/https://cloudformation-registry-documents.s3.amazonaws.com/Terms_and_Conditions_for_AWS_CloudFormation_Registry_Publishers.pdf>`_ for publishing extensions in the CloudFormation registry. You must accept the terms and conditions in order to register to publish public extensions to the CloudFormation registry.

        The default is ``false`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-publisher.html#cfn-cloudformation-publisher-accepttermsandconditions
        '''
        result = self._values.get("accept_terms_and_conditions")
        assert result is not None, "Required property 'accept_terms_and_conditions' is missing"
        return typing.cast(typing.Union[builtins.bool, "IResolvable"], result)

    @builtins.property
    def connection_arn(self) -> typing.Optional[builtins.str]:
        '''If you are using a Bitbucket or GitHub account for identity verification, the Amazon Resource Name (ARN) for your connection to that account.

        For more information, see `Prerequisite: Registering your account to publish CloudFormation extensions <https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/publish-extension.html#publish-extension-prereqs>`_ in the *AWS CloudFormation Command Line Interface (CLI) User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-publisher.html#cfn-cloudformation-publisher-connectionarn
        '''
        result = self._values.get("connection_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnPublisherProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CfnRefElement(
    CfnElement,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.CfnRefElement",
):
    '''Base class for referencable CloudFormation constructs which are not Resources.

    These constructs are things like Conditions and Parameters, can be
    referenced by taking the ``.ref`` attribute.

    Resource constructs do not inherit from CfnRefElement because they have their
    own, more specific types returned from the .ref attribute. Also, some
    resources aren't referencable at all (such as BucketPolicies or GatewayAttachments).
    '''

    def __init__(self, scope: _constructs_77d1e7e8.Construct, id: builtins.str) -> None:
        '''Creates an entity and binds it to a tree.

        Note that the root of the tree must be a Stack object (not just any Root).

        :param scope: The parent construct.
        :param id: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f9abc715bfd6e5d6183be29ffb2ffc20d3c34d09eab6455193440ec7215e9a59)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        jsii.create(self.__class__, self, [scope, id])

    @builtins.property
    @jsii.member(jsii_name="ref")
    def ref(self) -> builtins.str:
        '''Return a string that will be resolved to a CloudFormation ``{ Ref }`` for this element.

        If, by any chance, the intrinsic reference of a resource is not a string, you could
        coerce it to an IResolvable through ``Lazy.any({ produce: resource.ref })``.
        '''
        return typing.cast(builtins.str, jsii.get(self, "ref"))


class _CfnRefElementProxy(
    CfnRefElement,
    jsii.proxy_for(CfnElement), # type: ignore[misc]
):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, CfnRefElement).__jsii_proxy_class__ = lambda : _CfnRefElementProxy


class CfnResource(
    CfnRefElement,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.CfnResource",
):
    '''Represents a CloudFormation resource.

    :exampleMetadata: infused

    Example::

        @jsii.implements(ITaggable)
        class MyConstruct(Resource):
        
            def __init__(self, scope, id):
                super().__init__(scope, id)
        
                CfnResource(self, "Resource",
                    type="Whatever::The::Type",
                    properties={
                        # ...
                        "Tags": self.tags.rendered_tags
                    }
                )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        type: builtins.str,
        properties: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''Creates a resource construct.

        :param scope: -
        :param id: -
        :param type: CloudFormation resource type (e.g. ``AWS::S3::Bucket``).
        :param properties: Resource properties. Default: - No resource properties.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4afb094eec247e91f482fc7589690d86cf04ae10fd6531df97175e4925198ea1)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnResourceProps(type=type, properties=properties)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="isCfnResource")
    @builtins.classmethod
    def is_cfn_resource(cls, x: typing.Any) -> builtins.bool:
        '''Check whether the given object is a CfnResource.

        :param x: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42efefba6abf0aa8b28673336bf91cf7ae70877da6014c54788b61960dde5e49)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isCfnResource", [x]))

    @jsii.member(jsii_name="addDeletionOverride")
    def add_deletion_override(self, path: builtins.str) -> None:
        '''Syntactic sugar for ``addOverride(path, undefined)``.

        :param path: The path of the value to delete.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d61ae6a11d09a1a26f0f017f5b0013de9a5a3ea6bc9598ac2626a5dafee771cf)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        return typing.cast(None, jsii.invoke(self, "addDeletionOverride", [path]))

    @jsii.member(jsii_name="addDependency")
    def add_dependency(self, target: "CfnResource") -> None:
        '''Indicates that this resource depends on another resource and cannot be provisioned unless the other resource has been successfully provisioned.

        This can be used for resources across stacks (or nested stack) boundaries
        and the dependency will automatically be transferred to the relevant scope.

        :param target: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0da09f5241780cd5f46998f6bfaeddabf1e5cbbe59f9160f5880e9c15e66e0e7)
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        return typing.cast(None, jsii.invoke(self, "addDependency", [target]))

    @jsii.member(jsii_name="addDependsOn")
    def add_depends_on(self, target: "CfnResource") -> None:
        '''(deprecated) Indicates that this resource depends on another resource and cannot be provisioned unless the other resource has been successfully provisioned.

        :param target: -

        :deprecated: use addDependency

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__633a933e050129722586a882acec3ec7f135dbd7138050527b6e7678314c7b78)
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        return typing.cast(None, jsii.invoke(self, "addDependsOn", [target]))

    @jsii.member(jsii_name="addMetadata")
    def add_metadata(self, key: builtins.str, value: typing.Any) -> None:
        '''Add a value to the CloudFormation Resource Metadata.

        :param key: -
        :param value: -

        :see:

        https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/metadata-section-structure.html

        Note that this is a different set of metadata from CDK node metadata; this
        metadata ends up in the stack template under the resource, whereas CDK
        node metadata ends up in the Cloud Assembly.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0f7315710a6a0583f83431b2d7e8a7d656caf8761c847b8e848df9cdf3b031e3)
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast(None, jsii.invoke(self, "addMetadata", [key, value]))

    @jsii.member(jsii_name="addOverride")
    def add_override(self, path: builtins.str, value: typing.Any) -> None:
        '''Adds an override to the synthesized CloudFormation resource.

        To add a
        property override, either use ``addPropertyOverride`` or prefix ``path`` with
        "Properties." (i.e. ``Properties.TopicName``).

        If the override is nested, separate each nested level using a dot (.) in the path parameter.
        If there is an array as part of the nesting, specify the index in the path.

        To include a literal ``.`` in the property name, prefix with a ``\\``. In most
        programming languages you will need to write this as ``"\\\\."`` because the
        ``\\`` itself will need to be escaped.

        For example::

           cfn_resource.add_override("Properties.GlobalSecondaryIndexes.0.Projection.NonKeyAttributes", ["myattribute"])
           cfn_resource.add_override("Properties.GlobalSecondaryIndexes.1.ProjectionType", "INCLUDE")

        would add the overrides Example::

           "Properties": {
             "GlobalSecondaryIndexes": [
               {
                 "Projection": {
                   "NonKeyAttributes": [ "myattribute" ]
                   ...
                 }
                 ...
               },
               {
                 "ProjectionType": "INCLUDE"
                 ...
               },
             ]
             ...
           }

        The ``value`` argument to ``addOverride`` will not be processed or translated
        in any way. Pass raw JSON values in here with the correct capitalization
        for CloudFormation. If you pass CDK classes or structs, they will be
        rendered with lowercased key names, and CloudFormation will reject the
        template.

        :param path: - The path of the property, you can use dot notation to override values in complex types. Any intermediate keys will be created as needed.
        :param value: - The value. Could be primitive or complex.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__052babe539d9a9cdaadc8ff62b1e226402627d26ce5a33dd81aed6063f918d4a)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast(None, jsii.invoke(self, "addOverride", [path, value]))

    @jsii.member(jsii_name="addPropertyDeletionOverride")
    def add_property_deletion_override(self, property_path: builtins.str) -> None:
        '''Adds an override that deletes the value of a property from the resource definition.

        :param property_path: The path to the property.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ea1ac642ce0831440629dcf5fd31b0aa5397afbdbf7f761a3a0fa3d76166925)
            check_type(argname="argument property_path", value=property_path, expected_type=type_hints["property_path"])
        return typing.cast(None, jsii.invoke(self, "addPropertyDeletionOverride", [property_path]))

    @jsii.member(jsii_name="addPropertyOverride")
    def add_property_override(
        self,
        property_path: builtins.str,
        value: typing.Any,
    ) -> None:
        '''Adds an override to a resource property.

        Syntactic sugar for ``addOverride("Properties.<...>", value)``.

        :param property_path: The path of the property.
        :param value: The value.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e340b61f684215fcb94d252c520e9bcc509c45926e9f2e5afc7d390043b3c6fe)
            check_type(argname="argument property_path", value=property_path, expected_type=type_hints["property_path"])
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast(None, jsii.invoke(self, "addPropertyOverride", [property_path, value]))

    @jsii.member(jsii_name="applyRemovalPolicy")
    def apply_removal_policy(
        self,
        policy: typing.Optional["RemovalPolicy"] = None,
        *,
        apply_to_update_replace_policy: typing.Optional[builtins.bool] = None,
        default: typing.Optional["RemovalPolicy"] = None,
    ) -> None:
        '''Sets the deletion policy of the resource based on the removal policy specified.

        The Removal Policy controls what happens to this resource when it stops
        being managed by CloudFormation, either because you've removed it from the
        CDK application or because you've made a change that requires the resource
        to be replaced.

        The resource can be deleted (``RemovalPolicy.DESTROY``), or left in your AWS
        account for data recovery and cleanup later (``RemovalPolicy.RETAIN``). In some
        cases, a snapshot can be taken of the resource prior to deletion
        (``RemovalPolicy.SNAPSHOT``). A list of resources that support this policy
        can be found in the following link:

        :param policy: -
        :param apply_to_update_replace_policy: Apply the same deletion policy to the resource's "UpdateReplacePolicy". Default: true
        :param default: The default policy to apply in case the removal policy is not defined. Default: - Default value is resource specific. To determine the default value for a resource, please consult that specific resource's documentation.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html#aws-attribute-deletionpolicy-options
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8570110d1ab37276230f4f064e5cb13f761914d97258121c4873f4b0f565b155)
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
        options = RemovalPolicyOptions(
            apply_to_update_replace_policy=apply_to_update_replace_policy,
            default=default,
        )

        return typing.cast(None, jsii.invoke(self, "applyRemovalPolicy", [policy, options]))

    @jsii.member(jsii_name="getAtt")
    def get_att(
        self,
        attribute_name: builtins.str,
        type_hint: typing.Optional["ResolutionTypeHint"] = None,
    ) -> "Reference":
        '''Returns a token for an runtime attribute of this resource.

        Ideally, use generated attribute accessors (e.g. ``resource.arn``), but this can be used for future compatibility
        in case there is no generated attribute.

        :param attribute_name: The name of the attribute.
        :param type_hint: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a344cd489f092bca856db0edd466fe7f9ad59c6c42ea468109c576321c97018)
            check_type(argname="argument attribute_name", value=attribute_name, expected_type=type_hints["attribute_name"])
            check_type(argname="argument type_hint", value=type_hint, expected_type=type_hints["type_hint"])
        return typing.cast("Reference", jsii.invoke(self, "getAtt", [attribute_name, type_hint]))

    @jsii.member(jsii_name="getMetadata")
    def get_metadata(self, key: builtins.str) -> typing.Any:
        '''Retrieve a value value from the CloudFormation Resource Metadata.

        :param key: -

        :see:

        https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/metadata-section-structure.html

        Note that this is a different set of metadata from CDK node metadata; this
        metadata ends up in the stack template under the resource, whereas CDK
        node metadata ends up in the Cloud Assembly.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a786d5d2fd06dbd13085cfc35e4b9e4c2e20482af7e283453701f97310718591)
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
        return typing.cast(typing.Any, jsii.invoke(self, "getMetadata", [key]))

    @jsii.member(jsii_name="obtainDependencies")
    def obtain_dependencies(self) -> typing.List[typing.Union["Stack", "CfnResource"]]:
        '''Retrieves an array of resources this resource depends on.

        This assembles dependencies on resources across stacks (including nested stacks)
        automatically.
        '''
        return typing.cast(typing.List[typing.Union["Stack", "CfnResource"]], jsii.invoke(self, "obtainDependencies", []))

    @jsii.member(jsii_name="obtainResourceDependencies")
    def obtain_resource_dependencies(self) -> typing.List["CfnResource"]:
        '''Get a shallow copy of dependencies between this resource and other resources in the same stack.'''
        return typing.cast(typing.List["CfnResource"], jsii.invoke(self, "obtainResourceDependencies", []))

    @jsii.member(jsii_name="removeDependency")
    def remove_dependency(self, target: "CfnResource") -> None:
        '''Indicates that this resource no longer depends on another resource.

        This can be used for resources across stacks (including nested stacks)
        and the dependency will automatically be removed from the relevant scope.

        :param target: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8d3c93668fadffc0a4bcf670183e1a56913f35ccfc896b1244a755cebde69857)
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        return typing.cast(None, jsii.invoke(self, "removeDependency", [target]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30756c631b4bc150b58c85f68c4380245795007bb88e644c940a0be0818b6fb8)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.member(jsii_name="replaceDependency")
    def replace_dependency(
        self,
        target: "CfnResource",
        new_target: "CfnResource",
    ) -> None:
        '''Replaces one dependency with another.

        :param target: The dependency to replace.
        :param new_target: The new dependency to add.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__747b5cda36f3b9af07fcb86f87b648dc5bcb137281ba393c777416060ee21b1e)
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
            check_type(argname="argument new_target", value=new_target, expected_type=type_hints["new_target"])
        return typing.cast(None, jsii.invoke(self, "replaceDependency", [target, new_target]))

    @jsii.member(jsii_name="shouldSynthesize")
    def _should_synthesize(self) -> builtins.bool:
        '''Can be overridden by subclasses to determine if this resource will be rendered into the cloudformation template.

        :return:

        ``true`` if the resource should be included or ``false`` is the resource
        should be omitted.
        '''
        return typing.cast(builtins.bool, jsii.invoke(self, "shouldSynthesize", []))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Returns a string representation of this construct.

        :return: a string representation of this resource
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.member(jsii_name="validateProperties")
    def _validate_properties(self, _properties: typing.Any) -> None:
        '''
        :param _properties: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__69184ab365a6785998fae341892d7d63a931fb2898e050ae2a7a5726ee7d49ce)
            check_type(argname="argument _properties", value=_properties, expected_type=type_hints["_properties"])
        return typing.cast(None, jsii.invoke(self, "validateProperties", [_properties]))

    @builtins.property
    @jsii.member(jsii_name="cfnOptions")
    def cfn_options(self) -> "ICfnResourceOptions":
        '''Options for this resource, such as condition, update policy etc.'''
        return typing.cast("ICfnResourceOptions", jsii.get(self, "cfnOptions"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="cfnResourceType")
    def cfn_resource_type(self) -> builtins.str:
        '''AWS resource type.'''
        return typing.cast(builtins.str, jsii.get(self, "cfnResourceType"))

    @builtins.property
    @jsii.member(jsii_name="updatedProperites")
    def _updated_properites(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''(deprecated) Deprecated.

        :deprecated:

        use ``updatedProperties``

        Return properties modified after initiation

        Resources that expose mutable properties should override this function to
        collect and return the properties object for this resource.

        :stability: deprecated
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "updatedProperites"))

    @builtins.property
    @jsii.member(jsii_name="updatedProperties")
    def _updated_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Return properties modified after initiation.

        Resources that expose mutable properties should override this function to
        collect and return the properties object for this resource.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "updatedProperties"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnResourceAutoScalingCreationPolicy",
    jsii_struct_bases=[],
    name_mapping={"min_successful_instances_percent": "minSuccessfulInstancesPercent"},
)
class CfnResourceAutoScalingCreationPolicy:
    def __init__(
        self,
        *,
        min_successful_instances_percent: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''For an Auto Scaling group replacement update, specifies how many instances must signal success for the update to succeed.

        :param min_successful_instances_percent: Specifies the percentage of instances in an Auto Scaling replacement update that must signal success for the update to succeed. You can specify a value from 0 to 100. AWS CloudFormation rounds to the nearest tenth of a percent. For example, if you update five instances with a minimum successful percentage of 50, three instances must signal success. If an instance doesn't send a signal within the time specified by the Timeout property, AWS CloudFormation assumes that the instance wasn't created.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_resource_auto_scaling_creation_policy = cdk.CfnResourceAutoScalingCreationPolicy(
                min_successful_instances_percent=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__25fd3d530bb114204b4dcd6ba84f95a9d57ba9552b9b68fadabc485576d13b59)
            check_type(argname="argument min_successful_instances_percent", value=min_successful_instances_percent, expected_type=type_hints["min_successful_instances_percent"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if min_successful_instances_percent is not None:
            self._values["min_successful_instances_percent"] = min_successful_instances_percent

    @builtins.property
    def min_successful_instances_percent(self) -> typing.Optional[jsii.Number]:
        '''Specifies the percentage of instances in an Auto Scaling replacement update that must signal success for the update to succeed.

        You can specify a value from 0 to 100. AWS CloudFormation rounds to the nearest tenth of a percent.
        For example, if you update five instances with a minimum successful percentage of 50, three instances must signal success.
        If an instance doesn't send a signal within the time specified by the Timeout property, AWS CloudFormation assumes that the
        instance wasn't created.
        '''
        result = self._values.get("min_successful_instances_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnResourceAutoScalingCreationPolicy(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnResourceDefaultVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "type_name": "typeName",
        "type_version_arn": "typeVersionArn",
        "version_id": "versionId",
    },
)
class CfnResourceDefaultVersionProps:
    def __init__(
        self,
        *,
        type_name: typing.Optional[builtins.str] = None,
        type_version_arn: typing.Optional[builtins.str] = None,
        version_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnResourceDefaultVersion``.

        :param type_name: The name of the resource. Conditional: You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .
        :param type_version_arn: The Amazon Resource Name (ARN) of the resource version. Conditional: You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .
        :param version_id: The ID of a specific version of the resource. The version ID is the value at the end of the Amazon Resource Name (ARN) assigned to the resource version when it's registered. Conditional: You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-resourcedefaultversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_resource_default_version_props = cdk.CfnResourceDefaultVersionProps(
                type_name="typeName",
                type_version_arn="typeVersionArn",
                version_id="versionId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3f1523608cf6fe45952d0f5d1f32008dd9f0562a1c719af62d043d6a71ac6e7b)
            check_type(argname="argument type_name", value=type_name, expected_type=type_hints["type_name"])
            check_type(argname="argument type_version_arn", value=type_version_arn, expected_type=type_hints["type_version_arn"])
            check_type(argname="argument version_id", value=version_id, expected_type=type_hints["version_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if type_name is not None:
            self._values["type_name"] = type_name
        if type_version_arn is not None:
            self._values["type_version_arn"] = type_version_arn
        if version_id is not None:
            self._values["version_id"] = version_id

    @builtins.property
    def type_name(self) -> typing.Optional[builtins.str]:
        '''The name of the resource.

        Conditional: You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-resourcedefaultversion.html#cfn-cloudformation-resourcedefaultversion-typename
        '''
        result = self._values.get("type_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type_version_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the resource version.

        Conditional: You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-resourcedefaultversion.html#cfn-cloudformation-resourcedefaultversion-typeversionarn
        '''
        result = self._values.get("type_version_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def version_id(self) -> typing.Optional[builtins.str]:
        '''The ID of a specific version of the resource.

        The version ID is the value at the end of the Amazon Resource Name (ARN) assigned to the resource version when it's registered.

        Conditional: You must specify either ``TypeVersionArn`` , or ``TypeName`` and ``VersionId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-resourcedefaultversion.html#cfn-cloudformation-resourcedefaultversion-versionid
        '''
        result = self._values.get("version_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnResourceDefaultVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnResourceProps",
    jsii_struct_bases=[],
    name_mapping={"type": "type", "properties": "properties"},
)
class CfnResourceProps:
    def __init__(
        self,
        *,
        type: builtins.str,
        properties: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''
        :param type: CloudFormation resource type (e.g. ``AWS::S3::Bucket``).
        :param properties: Resource properties. Default: - No resource properties.

        :exampleMetadata: infused

        Example::

            @jsii.implements(ITaggable)
            class MyConstruct(Resource):
            
                def __init__(self, scope, id):
                    super().__init__(scope, id)
            
                    CfnResource(self, "Resource",
                        type="Whatever::The::Type",
                        properties={
                            # ...
                            "Tags": self.tags.rendered_tags
                        }
                    )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d1b62b09ffbab6ff0f59d664e89ba6054339a477f55c26bf979a6779e45c976)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument properties", value=properties, expected_type=type_hints["properties"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }
        if properties is not None:
            self._values["properties"] = properties

    @builtins.property
    def type(self) -> builtins.str:
        '''CloudFormation resource type (e.g. ``AWS::S3::Bucket``).'''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def properties(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Resource properties.

        :default: - No resource properties.
        '''
        result = self._values.get("properties")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnResourceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnResourceSignal",
    jsii_struct_bases=[],
    name_mapping={"count": "count", "timeout": "timeout"},
)
class CfnResourceSignal:
    def __init__(
        self,
        *,
        count: typing.Optional[jsii.Number] = None,
        timeout: typing.Optional[builtins.str] = None,
    ) -> None:
        '''When AWS CloudFormation creates the associated resource, configures the number of required success signals and the length of time that AWS CloudFormation waits for those signals.

        :param count: The number of success signals AWS CloudFormation must receive before it sets the resource status as CREATE_COMPLETE. If the resource receives a failure signal or doesn't receive the specified number of signals before the timeout period expires, the resource creation fails and AWS CloudFormation rolls the stack back.
        :param timeout: The length of time that AWS CloudFormation waits for the number of signals that was specified in the Count property. The timeout period starts after AWS CloudFormation starts creating the resource, and the timeout expires no sooner than the time you specify but can occur shortly thereafter. The maximum time that you can specify is 12 hours.

        :exampleMetadata: infused

        Example::

            # resource: CfnResource
            
            
            resource.cfn_options.creation_policy = CfnCreationPolicy(
                resource_signal=CfnResourceSignal(
                    count=3,
                    timeout="PR15M"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ec2fe3952f235e3aa253ca9c0c8adb63d8a53064e0a0f7cd52639ce2b7c6ee7)
            check_type(argname="argument count", value=count, expected_type=type_hints["count"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if count is not None:
            self._values["count"] = count
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def count(self) -> typing.Optional[jsii.Number]:
        '''The number of success signals AWS CloudFormation must receive before it sets the resource status as CREATE_COMPLETE.

        If the resource receives a failure signal or doesn't receive the specified number of signals before the timeout period
        expires, the resource creation fails and AWS CloudFormation rolls the stack back.
        '''
        result = self._values.get("count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def timeout(self) -> typing.Optional[builtins.str]:
        '''The length of time that AWS CloudFormation waits for the number of signals that was specified in the Count property.

        The timeout period starts after AWS CloudFormation starts creating the resource, and the timeout expires no sooner
        than the time you specify but can occur shortly thereafter. The maximum time that you can specify is 12 hours.
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnResourceSignal(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnResourceVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "schema_handler_package": "schemaHandlerPackage",
        "type_name": "typeName",
        "execution_role_arn": "executionRoleArn",
        "logging_config": "loggingConfig",
    },
)
class CfnResourceVersionProps:
    def __init__(
        self,
        *,
        schema_handler_package: builtins.str,
        type_name: builtins.str,
        execution_role_arn: typing.Optional[builtins.str] = None,
        logging_config: typing.Optional[typing.Union["IResolvable", typing.Union["CfnResourceVersion.LoggingConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnResourceVersion``.

        :param schema_handler_package: A URL to the S3 bucket containing the resource project package that contains the necessary files for the resource you want to register. For information on generating a schema handler package, see `Modeling resource types to use with AWS CloudFormation <https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-model.html>`_ in the *AWS CloudFormation Command Line Interface (CLI) User Guide* . .. epigraph:: To register the resource version, you must have ``s3:GetObject`` permissions to access the S3 objects.
        :param type_name: The name of the resource being registered. We recommend that resource names adhere to the following pattern: *company_or_organization* :: *service* :: *type* . .. epigraph:: The following organization namespaces are reserved and can't be used in your resource names: - ``Alexa`` - ``AMZN`` - ``Amazon`` - ``AWS`` - ``Custom`` - ``Dev``
        :param execution_role_arn: The Amazon Resource Name (ARN) of the IAM role for CloudFormation to assume when invoking the resource. If your resource calls AWS APIs in any of its handlers, you must create an IAM execution role that includes the necessary permissions to call those AWS APIs, and provision that execution role in your account. When CloudFormation needs to invoke the resource type handler, CloudFormation assumes this execution role to create a temporary session token, which it then passes to the resource type handler, thereby supplying your resource type with the appropriate credentials.
        :param logging_config: Logging configuration information for a resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-resourceversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_resource_version_props = cdk.CfnResourceVersionProps(
                schema_handler_package="schemaHandlerPackage",
                type_name="typeName",
            
                # the properties below are optional
                execution_role_arn="executionRoleArn",
                logging_config=cdk.CfnResourceVersion.LoggingConfigProperty(
                    log_group_name="logGroupName",
                    log_role_arn="logRoleArn"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9e4f7bf87543cdb186b133d3f86faf2b73a94e38243a2b4a2c84aeb302725007)
            check_type(argname="argument schema_handler_package", value=schema_handler_package, expected_type=type_hints["schema_handler_package"])
            check_type(argname="argument type_name", value=type_name, expected_type=type_hints["type_name"])
            check_type(argname="argument execution_role_arn", value=execution_role_arn, expected_type=type_hints["execution_role_arn"])
            check_type(argname="argument logging_config", value=logging_config, expected_type=type_hints["logging_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "schema_handler_package": schema_handler_package,
            "type_name": type_name,
        }
        if execution_role_arn is not None:
            self._values["execution_role_arn"] = execution_role_arn
        if logging_config is not None:
            self._values["logging_config"] = logging_config

    @builtins.property
    def schema_handler_package(self) -> builtins.str:
        '''A URL to the S3 bucket containing the resource project package that contains the necessary files for the resource you want to register.

        For information on generating a schema handler package, see `Modeling resource types to use with AWS CloudFormation <https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-model.html>`_ in the *AWS CloudFormation Command Line Interface (CLI) User Guide* .
        .. epigraph::

           To register the resource version, you must have ``s3:GetObject`` permissions to access the S3 objects.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-resourceversion.html#cfn-cloudformation-resourceversion-schemahandlerpackage
        '''
        result = self._values.get("schema_handler_package")
        assert result is not None, "Required property 'schema_handler_package' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type_name(self) -> builtins.str:
        '''The name of the resource being registered.

        We recommend that resource names adhere to the following pattern: *company_or_organization* :: *service* :: *type* .
        .. epigraph::

           The following organization namespaces are reserved and can't be used in your resource names:

           - ``Alexa``
           - ``AMZN``
           - ``Amazon``
           - ``AWS``
           - ``Custom``
           - ``Dev``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-resourceversion.html#cfn-cloudformation-resourceversion-typename
        '''
        result = self._values.get("type_name")
        assert result is not None, "Required property 'type_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def execution_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the IAM role for CloudFormation to assume when invoking the resource.

        If your resource calls AWS APIs in any of its handlers, you must create an IAM execution role that includes the necessary permissions to call those AWS APIs, and provision that execution role in your account. When CloudFormation needs to invoke the resource type handler, CloudFormation assumes this execution role to create a temporary session token, which it then passes to the resource type handler, thereby supplying your resource type with the appropriate credentials.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-resourceversion.html#cfn-cloudformation-resourceversion-executionrolearn
        '''
        result = self._values.get("execution_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def logging_config(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", "CfnResourceVersion.LoggingConfigProperty"]]:
        '''Logging configuration information for a resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-resourceversion.html#cfn-cloudformation-resourceversion-loggingconfig
        '''
        result = self._values.get("logging_config")
        return typing.cast(typing.Optional[typing.Union["IResolvable", "CfnResourceVersion.LoggingConfigProperty"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnResourceVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CfnRule(CfnRefElement, metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.CfnRule"):
    '''The Rules that define template constraints in an AWS Service Catalog portfolio describe when end users can use the template and which values they can specify for parameters that are declared in the AWS CloudFormation template used to create the product they are attempting to use.

    Rules
    are useful for preventing end users from inadvertently specifying an incorrect value.
    For example, you can add a rule to verify whether end users specified a valid subnet in a
    given VPC or used m1.small instance types for test environments. AWS CloudFormation uses
    rules to validate parameter values before it creates the resources for the product.

    A rule can include a RuleCondition property and must include an Assertions property.
    For each rule, you can define only one rule condition; you can define one or more asserts within the Assertions property.
    You define a rule condition and assertions by using rule-specific intrinsic functions.

    :link: https://docs.aws.amazon.com/servicecatalog/latest/adminguide/reference-template_constraint_rules.html
    :exampleMetadata: infused

    Example::

        # cfn_template: cfn_inc.CfnInclude
        
        # mutating the rule
        # my_parameter: core.CfnParameter
        
        rule = cfn_template.get_rule("MyRule")
        rule.add_assertion(core.Fn.condition_contains(["m1.small"], my_parameter.value_as_string), "MyParameter has to be m1.small")
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        assertions: typing.Optional[typing.Sequence[typing.Union["CfnRuleAssertion", typing.Dict[builtins.str, typing.Any]]]] = None,
        rule_condition: typing.Optional["ICfnConditionExpression"] = None,
    ) -> None:
        '''Creates and adds a rule.

        :param scope: The parent construct.
        :param id: -
        :param assertions: Assertions which define the rule. Default: - No assertions for the rule.
        :param rule_condition: If the rule condition evaluates to false, the rule doesn't take effect. If the function in the rule condition evaluates to true, expressions in each assert are evaluated and applied. Default: - Rule's assertions will always take effect.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d455dbd13c2ca15e4654b058261bb797a2b8efaf4f8db7b0f07cbe89c05c8527)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnRuleProps(assertions=assertions, rule_condition=rule_condition)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addAssertion")
    def add_assertion(
        self,
        condition: "ICfnConditionExpression",
        description: builtins.str,
    ) -> None:
        '''Adds an assertion to the rule.

        :param condition: The expression to evaluation.
        :param description: The description of the assertion.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5dc815799188b2f1db51243f3a7be6e8952d98b380ad198621c6f8a16d08a58a)
            check_type(argname="argument condition", value=condition, expected_type=type_hints["condition"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        return typing.cast(None, jsii.invoke(self, "addAssertion", [condition, description]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnRuleAssertion",
    jsii_struct_bases=[],
    name_mapping={"assert_": "assert", "assert_description": "assertDescription"},
)
class CfnRuleAssertion:
    def __init__(
        self,
        *,
        assert_: "ICfnConditionExpression",
        assert_description: builtins.str,
    ) -> None:
        '''A rule assertion.

        :param assert_: The assertion.
        :param assert_description: The assertion description.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # cfn_condition_expression: cdk.ICfnConditionExpression
            
            cfn_rule_assertion = cdk.CfnRuleAssertion(
                assert=cfn_condition_expression,
                assert_description="assertDescription"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8934aa228e7286677d27268135368d752ef82baad27417cf4b0858e2a70c7281)
            check_type(argname="argument assert_", value=assert_, expected_type=type_hints["assert_"])
            check_type(argname="argument assert_description", value=assert_description, expected_type=type_hints["assert_description"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "assert_": assert_,
            "assert_description": assert_description,
        }

    @builtins.property
    def assert_(self) -> "ICfnConditionExpression":
        '''The assertion.'''
        result = self._values.get("assert_")
        assert result is not None, "Required property 'assert_' is missing"
        return typing.cast("ICfnConditionExpression", result)

    @builtins.property
    def assert_description(self) -> builtins.str:
        '''The assertion description.'''
        result = self._values.get("assert_description")
        assert result is not None, "Required property 'assert_description' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRuleAssertion(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnRuleProps",
    jsii_struct_bases=[],
    name_mapping={"assertions": "assertions", "rule_condition": "ruleCondition"},
)
class CfnRuleProps:
    def __init__(
        self,
        *,
        assertions: typing.Optional[typing.Sequence[typing.Union[CfnRuleAssertion, typing.Dict[builtins.str, typing.Any]]]] = None,
        rule_condition: typing.Optional["ICfnConditionExpression"] = None,
    ) -> None:
        '''A rule can include a RuleCondition property and must include an Assertions property.

        For each rule, you can define only one rule condition; you can define one or more asserts within the Assertions property.
        You define a rule condition and assertions by using rule-specific intrinsic functions.

        You can use the following rule-specific intrinsic functions to define rule conditions and assertions:

        Fn::And
        Fn::Contains
        Fn::EachMemberEquals
        Fn::EachMemberIn
        Fn::Equals
        Fn::If
        Fn::Not
        Fn::Or
        Fn::RefAll
        Fn::ValueOf
        Fn::ValueOfAll

        https://docs.aws.amazon.com/servicecatalog/latest/adminguide/reference-template_constraint_rules.html

        :param assertions: Assertions which define the rule. Default: - No assertions for the rule.
        :param rule_condition: If the rule condition evaluates to false, the rule doesn't take effect. If the function in the rule condition evaluates to true, expressions in each assert are evaluated and applied. Default: - Rule's assertions will always take effect.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # cfn_condition_expression: cdk.ICfnConditionExpression
            
            cfn_rule_props = cdk.CfnRuleProps(
                assertions=[cdk.CfnRuleAssertion(
                    assert=cfn_condition_expression,
                    assert_description="assertDescription"
                )],
                rule_condition=cfn_condition_expression
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b4a89ebbdf831c87a631869cbefb7a1b6d6a2f1ce5b8030f61d58f4736db7754)
            check_type(argname="argument assertions", value=assertions, expected_type=type_hints["assertions"])
            check_type(argname="argument rule_condition", value=rule_condition, expected_type=type_hints["rule_condition"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if assertions is not None:
            self._values["assertions"] = assertions
        if rule_condition is not None:
            self._values["rule_condition"] = rule_condition

    @builtins.property
    def assertions(self) -> typing.Optional[typing.List[CfnRuleAssertion]]:
        '''Assertions which define the rule.

        :default: - No assertions for the rule.
        '''
        result = self._values.get("assertions")
        return typing.cast(typing.Optional[typing.List[CfnRuleAssertion]], result)

    @builtins.property
    def rule_condition(self) -> typing.Optional["ICfnConditionExpression"]:
        '''If the rule condition evaluates to false, the rule doesn't take effect.

        If the function in the rule condition evaluates to true, expressions in each assert are evaluated and applied.

        :default: - Rule's assertions will always take effect.
        '''
        result = self._values.get("rule_condition")
        return typing.cast(typing.Optional["ICfnConditionExpression"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRuleProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnStackProps",
    jsii_struct_bases=[],
    name_mapping={
        "notification_arns": "notificationArns",
        "parameters": "parameters",
        "tags": "tags",
        "template_url": "templateUrl",
        "timeout_in_minutes": "timeoutInMinutes",
    },
)
class CfnStackProps:
    def __init__(
        self,
        *,
        notification_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        parameters: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], "IResolvable"]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union["CfnTag", typing.Dict[builtins.str, typing.Any]]]] = None,
        template_url: typing.Optional[builtins.str] = None,
        timeout_in_minutes: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnStack``.

        :param notification_arns: The Amazon SNS topic ARNs to publish stack related events. You can find your Amazon SNS topic ARNs using the Amazon SNS console or your Command Line Interface (CLI).
        :param parameters: The set value pairs that represent the parameters passed to CloudFormation when this nested stack is created. Each parameter has a name corresponding to a parameter defined in the embedded template and a value representing the value that you want to set for the parameter. .. epigraph:: If you use the ``Ref`` function to pass a parameter value to a nested stack, comma-delimited list parameters must be of type ``String`` . In other words, you can't pass values that are of type ``CommaDelimitedList`` to nested stacks. Required if the nested stack requires input parameters. Whether an update causes interruptions depends on the resources that are being updated. An update never causes a nested stack to be replaced.
        :param tags: Key-value pairs to associate with this stack. CloudFormation also propagates these tags to the resources created in the stack. A maximum number of 50 tags can be specified.
        :param template_url: The URL of a file containing the template body. The URL must point to a template (max size: 1 MB) that's located in an Amazon S3 bucket. The location for an Amazon S3 bucket must start with ``https://`` . Whether an update causes interruptions depends on the resources that are being updated. An update never causes a nested stack to be replaced.
        :param timeout_in_minutes: The length of time, in minutes, that CloudFormation waits for the nested stack to reach the ``CREATE_COMPLETE`` state. The default is no timeout. When CloudFormation detects that the nested stack has reached the ``CREATE_COMPLETE`` state, it marks the nested stack resource as ``CREATE_COMPLETE`` in the parent stack and resumes creating the parent stack. If the timeout period expires before the nested stack reaches ``CREATE_COMPLETE`` , CloudFormation marks the nested stack as failed and rolls back both the nested stack and parent stack. Updates aren't supported.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stack.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_stack_props = cdk.CfnStackProps(
                notification_arns=["notificationArns"],
                parameters={
                    "parameters_key": "parameters"
                },
                tags=[cdk.CfnTag(
                    key="key",
                    value="value"
                )],
                template_url="templateUrl",
                timeout_in_minutes=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9fb527d4bac73dc363319bbe1cf2be973d3eb9db93eb4ec913a8151682c3f223)
            check_type(argname="argument notification_arns", value=notification_arns, expected_type=type_hints["notification_arns"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument template_url", value=template_url, expected_type=type_hints["template_url"])
            check_type(argname="argument timeout_in_minutes", value=timeout_in_minutes, expected_type=type_hints["timeout_in_minutes"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if notification_arns is not None:
            self._values["notification_arns"] = notification_arns
        if parameters is not None:
            self._values["parameters"] = parameters
        if tags is not None:
            self._values["tags"] = tags
        if template_url is not None:
            self._values["template_url"] = template_url
        if timeout_in_minutes is not None:
            self._values["timeout_in_minutes"] = timeout_in_minutes

    @builtins.property
    def notification_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The Amazon SNS topic ARNs to publish stack related events.

        You can find your Amazon SNS topic ARNs using the Amazon SNS console or your Command Line Interface (CLI).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stack.html#cfn-cloudformation-stack-notificationarns
        '''
        result = self._values.get("notification_arns")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def parameters(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], "IResolvable"]]:
        '''The set value pairs that represent the parameters passed to CloudFormation when this nested stack is created.

        Each parameter has a name corresponding to a parameter defined in the embedded template and a value representing the value that you want to set for the parameter.
        .. epigraph::

           If you use the ``Ref`` function to pass a parameter value to a nested stack, comma-delimited list parameters must be of type ``String`` . In other words, you can't pass values that are of type ``CommaDelimitedList`` to nested stacks.

        Required if the nested stack requires input parameters.

        Whether an update causes interruptions depends on the resources that are being updated. An update never causes a nested stack to be replaced.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stack.html#cfn-cloudformation-stack-parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], "IResolvable"]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List["CfnTag"]]:
        '''Key-value pairs to associate with this stack.

        CloudFormation also propagates these tags to the resources created in the stack. A maximum number of 50 tags can be specified.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stack.html#cfn-cloudformation-stack-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List["CfnTag"]], result)

    @builtins.property
    def template_url(self) -> typing.Optional[builtins.str]:
        '''The URL of a file containing the template body.

        The URL must point to a template (max size: 1 MB) that's located in an Amazon S3 bucket. The location for an Amazon S3 bucket must start with ``https://`` .

        Whether an update causes interruptions depends on the resources that are being updated. An update never causes a nested stack to be replaced.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stack.html#cfn-cloudformation-stack-templateurl
        '''
        result = self._values.get("template_url")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def timeout_in_minutes(self) -> typing.Optional[jsii.Number]:
        '''The length of time, in minutes, that CloudFormation waits for the nested stack to reach the ``CREATE_COMPLETE`` state.

        The default is no timeout. When CloudFormation detects that the nested stack has reached the ``CREATE_COMPLETE`` state, it marks the nested stack resource as ``CREATE_COMPLETE`` in the parent stack and resumes creating the parent stack. If the timeout period expires before the nested stack reaches ``CREATE_COMPLETE`` , CloudFormation marks the nested stack as failed and rolls back both the nested stack and parent stack.

        Updates aren't supported.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stack.html#cfn-cloudformation-stack-timeoutinminutes
        '''
        result = self._values.get("timeout_in_minutes")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnStackProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnStackSetProps",
    jsii_struct_bases=[],
    name_mapping={
        "permission_model": "permissionModel",
        "stack_set_name": "stackSetName",
        "administration_role_arn": "administrationRoleArn",
        "auto_deployment": "autoDeployment",
        "call_as": "callAs",
        "capabilities": "capabilities",
        "description": "description",
        "execution_role_name": "executionRoleName",
        "managed_execution": "managedExecution",
        "operation_preferences": "operationPreferences",
        "parameters": "parameters",
        "stack_instances_group": "stackInstancesGroup",
        "tags": "tags",
        "template_body": "templateBody",
        "template_url": "templateUrl",
    },
)
class CfnStackSetProps:
    def __init__(
        self,
        *,
        permission_model: builtins.str,
        stack_set_name: builtins.str,
        administration_role_arn: typing.Optional[builtins.str] = None,
        auto_deployment: typing.Optional[typing.Union["IResolvable", typing.Union["CfnStackSet.AutoDeploymentProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        call_as: typing.Optional[builtins.str] = None,
        capabilities: typing.Optional[typing.Sequence[builtins.str]] = None,
        description: typing.Optional[builtins.str] = None,
        execution_role_name: typing.Optional[builtins.str] = None,
        managed_execution: typing.Any = None,
        operation_preferences: typing.Optional[typing.Union["IResolvable", typing.Union["CfnStackSet.OperationPreferencesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        parameters: typing.Optional[typing.Union["IResolvable", typing.Sequence[typing.Union["IResolvable", typing.Union["CfnStackSet.ParameterProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        stack_instances_group: typing.Optional[typing.Union["IResolvable", typing.Sequence[typing.Union["IResolvable", typing.Union["CfnStackSet.StackInstancesProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union["CfnTag", typing.Dict[builtins.str, typing.Any]]]] = None,
        template_body: typing.Optional[builtins.str] = None,
        template_url: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnStackSet``.

        :param permission_model: Describes how the IAM roles required for stack set operations are created. - With ``SELF_MANAGED`` permissions, you must create the administrator and execution roles required to deploy to target accounts. For more information, see `Grant self-managed permissions <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html>`_ in the *AWS CloudFormation User Guide* . - With ``SERVICE_MANAGED`` permissions, StackSets automatically creates the IAM roles required to deploy to accounts managed by AWS Organizations . For more information, see `Activate trusted access for stack sets with AWS Organizations <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-activate-trusted-access.html>`_ in the *AWS CloudFormation User Guide* .
        :param stack_set_name: The name to associate with the stack set. The name must be unique in the Region where you create your stack set.
        :param administration_role_arn: The Amazon Resource Number (ARN) of the IAM role to use to create this stack set. Specify an IAM role only if you are using customized administrator roles to control which users or groups can manage specific stack sets within the same administrator account. Use customized administrator roles to control which users or groups can manage specific stack sets within the same administrator account. For more information, see `Grant self-managed permissions <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html>`_ in the *AWS CloudFormation User Guide* . Valid only if the permissions model is ``SELF_MANAGED`` .
        :param auto_deployment: Describes whether StackSets automatically deploys to AWS Organizations accounts that are added to a target organization or organizational unit (OU). For more information, see `Manage automatic deployments for CloudFormation StackSets that use service-managed permissions <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-manage-auto-deployment.html>`_ in the *AWS CloudFormation User Guide* . Required if the permissions model is ``SERVICE_MANAGED`` . (Not used with self-managed permissions.)
        :param call_as: Specifies whether you are acting as an account administrator in the organization's management account or as a delegated administrator in a member account. By default, ``SELF`` is specified. Use ``SELF`` for stack sets with self-managed permissions. - To create a stack set with service-managed permissions while signed in to the management account, specify ``SELF`` . - To create a stack set with service-managed permissions while signed in to a delegated administrator account, specify ``DELEGATED_ADMIN`` . Your AWS account must be registered as a delegated admin in the management account. For more information, see `Register a delegated administrator <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-delegated-admin.html>`_ in the *AWS CloudFormation User Guide* . Stack sets with service-managed permissions are created in the management account, including stack sets that are created by delegated administrators. Valid only if the permissions model is ``SERVICE_MANAGED`` .
        :param capabilities: The capabilities that are allowed in the stack set. Some stack set templates might include resources that can affect permissions in your AWS account —for example, by creating new IAM users. For more information, see `Acknowledging IAM resources in CloudFormation templates <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/control-access-with-iam.html#using-iam-capabilities>`_ in the *AWS CloudFormation User Guide* .
        :param description: A description of the stack set.
        :param execution_role_name: The name of the IAM execution role to use to create the stack set. If you don't specify an execution role, CloudFormation uses the ``AWSCloudFormationStackSetExecutionRole`` role for the stack set operation. Valid only if the permissions model is ``SELF_MANAGED`` . *Pattern* : ``[a-zA-Z_0-9+=,.@-]+``
        :param managed_execution: Describes whether StackSets performs non-conflicting operations concurrently and queues conflicting operations. When active, StackSets performs non-conflicting operations concurrently and queues conflicting operations. After conflicting operations finish, StackSets starts queued operations in request order. .. epigraph:: If there are already running or queued operations, StackSets queues all incoming operations even if they are non-conflicting. You can't modify your stack set's execution configuration while there are running or queued operations for that stack set. When inactive (default), StackSets performs one operation at a time in request order.
        :param operation_preferences: The user-specified preferences for how CloudFormation performs a stack set operation.
        :param parameters: The input parameters for the stack set template.
        :param stack_instances_group: A group of stack instances with parameters in some specific accounts and Regions.
        :param tags: Key-value pairs to associate with this stack. CloudFormation also propagates these tags to supported resources in the stack. You can specify a maximum number of 50 tags. If you don't specify this parameter, CloudFormation doesn't modify the stack's tags. If you specify an empty value, CloudFormation removes all associated tags.
        :param template_body: The structure that contains the template body, with a minimum length of 1 byte and a maximum length of 51,200 bytes. You must include either ``TemplateURL`` or ``TemplateBody`` in a StackSet, but you can't use both. Dynamic references in the ``TemplateBody`` may not work correctly in all cases. It's recommended to pass templates containing dynamic references through ``TemplateUrl`` instead.
        :param template_url: The URL of a file containing the template body. The URL must point to a template (max size: 1 MB) that's located in an Amazon S3 bucket or a Systems Manager document. The location for an Amazon S3 bucket must start with ``https://`` . Conditional: You must specify only one of the following parameters: ``TemplateBody`` , ``TemplateURL`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # managed_execution: Any
            
            cfn_stack_set_props = cdk.CfnStackSetProps(
                permission_model="permissionModel",
                stack_set_name="stackSetName",
            
                # the properties below are optional
                administration_role_arn="administrationRoleArn",
                auto_deployment=cdk.CfnStackSet.AutoDeploymentProperty(
                    enabled=False,
                    retain_stacks_on_account_removal=False
                ),
                call_as="callAs",
                capabilities=["capabilities"],
                description="description",
                execution_role_name="executionRoleName",
                managed_execution=managed_execution,
                operation_preferences=cdk.CfnStackSet.OperationPreferencesProperty(
                    concurrency_mode="concurrencyMode",
                    failure_tolerance_count=123,
                    failure_tolerance_percentage=123,
                    max_concurrent_count=123,
                    max_concurrent_percentage=123,
                    region_concurrency_type="regionConcurrencyType",
                    region_order=["regionOrder"]
                ),
                parameters=[cdk.CfnStackSet.ParameterProperty(
                    parameter_key="parameterKey",
                    parameter_value="parameterValue"
                )],
                stack_instances_group=[cdk.CfnStackSet.StackInstancesProperty(
                    deployment_targets=cdk.CfnStackSet.DeploymentTargetsProperty(
                        account_filter_type="accountFilterType",
                        accounts=["accounts"],
                        accounts_url="accountsUrl",
                        organizational_unit_ids=["organizationalUnitIds"]
                    ),
                    regions=["regions"],
            
                    # the properties below are optional
                    parameter_overrides=[cdk.CfnStackSet.ParameterProperty(
                        parameter_key="parameterKey",
                        parameter_value="parameterValue"
                    )]
                )],
                tags=[cdk.CfnTag(
                    key="key",
                    value="value"
                )],
                template_body="templateBody",
                template_url="templateUrl"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e15110ff9adbcf3562be747350b3e0243d50e812fb8981e69fee1a1e0eab4c60)
            check_type(argname="argument permission_model", value=permission_model, expected_type=type_hints["permission_model"])
            check_type(argname="argument stack_set_name", value=stack_set_name, expected_type=type_hints["stack_set_name"])
            check_type(argname="argument administration_role_arn", value=administration_role_arn, expected_type=type_hints["administration_role_arn"])
            check_type(argname="argument auto_deployment", value=auto_deployment, expected_type=type_hints["auto_deployment"])
            check_type(argname="argument call_as", value=call_as, expected_type=type_hints["call_as"])
            check_type(argname="argument capabilities", value=capabilities, expected_type=type_hints["capabilities"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument execution_role_name", value=execution_role_name, expected_type=type_hints["execution_role_name"])
            check_type(argname="argument managed_execution", value=managed_execution, expected_type=type_hints["managed_execution"])
            check_type(argname="argument operation_preferences", value=operation_preferences, expected_type=type_hints["operation_preferences"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument stack_instances_group", value=stack_instances_group, expected_type=type_hints["stack_instances_group"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument template_body", value=template_body, expected_type=type_hints["template_body"])
            check_type(argname="argument template_url", value=template_url, expected_type=type_hints["template_url"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "permission_model": permission_model,
            "stack_set_name": stack_set_name,
        }
        if administration_role_arn is not None:
            self._values["administration_role_arn"] = administration_role_arn
        if auto_deployment is not None:
            self._values["auto_deployment"] = auto_deployment
        if call_as is not None:
            self._values["call_as"] = call_as
        if capabilities is not None:
            self._values["capabilities"] = capabilities
        if description is not None:
            self._values["description"] = description
        if execution_role_name is not None:
            self._values["execution_role_name"] = execution_role_name
        if managed_execution is not None:
            self._values["managed_execution"] = managed_execution
        if operation_preferences is not None:
            self._values["operation_preferences"] = operation_preferences
        if parameters is not None:
            self._values["parameters"] = parameters
        if stack_instances_group is not None:
            self._values["stack_instances_group"] = stack_instances_group
        if tags is not None:
            self._values["tags"] = tags
        if template_body is not None:
            self._values["template_body"] = template_body
        if template_url is not None:
            self._values["template_url"] = template_url

    @builtins.property
    def permission_model(self) -> builtins.str:
        '''Describes how the IAM roles required for stack set operations are created.

        - With ``SELF_MANAGED`` permissions, you must create the administrator and execution roles required to deploy to target accounts. For more information, see `Grant self-managed permissions <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html>`_ in the *AWS CloudFormation User Guide* .
        - With ``SERVICE_MANAGED`` permissions, StackSets automatically creates the IAM roles required to deploy to accounts managed by AWS Organizations . For more information, see `Activate trusted access for stack sets with AWS Organizations <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-activate-trusted-access.html>`_ in the *AWS CloudFormation User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-permissionmodel
        '''
        result = self._values.get("permission_model")
        assert result is not None, "Required property 'permission_model' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def stack_set_name(self) -> builtins.str:
        '''The name to associate with the stack set.

        The name must be unique in the Region where you create your stack set.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-stacksetname
        '''
        result = self._values.get("stack_set_name")
        assert result is not None, "Required property 'stack_set_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def administration_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Number (ARN) of the IAM role to use to create this stack set.

        Specify an IAM role only if you are using customized administrator roles to control which users or groups can manage specific stack sets within the same administrator account.

        Use customized administrator roles to control which users or groups can manage specific stack sets within the same administrator account. For more information, see `Grant self-managed permissions <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html>`_ in the *AWS CloudFormation User Guide* .

        Valid only if the permissions model is ``SELF_MANAGED`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-administrationrolearn
        '''
        result = self._values.get("administration_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auto_deployment(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", "CfnStackSet.AutoDeploymentProperty"]]:
        '''Describes whether StackSets automatically deploys to AWS Organizations accounts that are added to a target organization or organizational unit (OU).

        For more information, see `Manage automatic deployments for CloudFormation StackSets that use service-managed permissions <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-manage-auto-deployment.html>`_ in the *AWS CloudFormation User Guide* .

        Required if the permissions model is ``SERVICE_MANAGED`` . (Not used with self-managed permissions.)

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-autodeployment
        '''
        result = self._values.get("auto_deployment")
        return typing.cast(typing.Optional[typing.Union["IResolvable", "CfnStackSet.AutoDeploymentProperty"]], result)

    @builtins.property
    def call_as(self) -> typing.Optional[builtins.str]:
        '''Specifies whether you are acting as an account administrator in the organization's management account or as a delegated administrator in a member account.

        By default, ``SELF`` is specified. Use ``SELF`` for stack sets with self-managed permissions.

        - To create a stack set with service-managed permissions while signed in to the management account, specify ``SELF`` .
        - To create a stack set with service-managed permissions while signed in to a delegated administrator account, specify ``DELEGATED_ADMIN`` .

        Your AWS account must be registered as a delegated admin in the management account. For more information, see `Register a delegated administrator <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-orgs-delegated-admin.html>`_ in the *AWS CloudFormation User Guide* .

        Stack sets with service-managed permissions are created in the management account, including stack sets that are created by delegated administrators.

        Valid only if the permissions model is ``SERVICE_MANAGED`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-callas
        '''
        result = self._values.get("call_as")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def capabilities(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The capabilities that are allowed in the stack set.

        Some stack set templates might include resources that can affect permissions in your AWS account —for example, by creating new IAM users. For more information, see `Acknowledging IAM resources in CloudFormation templates <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/control-access-with-iam.html#using-iam-capabilities>`_ in the *AWS CloudFormation User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-capabilities
        '''
        result = self._values.get("capabilities")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the stack set.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def execution_role_name(self) -> typing.Optional[builtins.str]:
        '''The name of the IAM execution role to use to create the stack set.

        If you don't specify an execution role, CloudFormation uses the ``AWSCloudFormationStackSetExecutionRole`` role for the stack set operation.

        Valid only if the permissions model is ``SELF_MANAGED`` .

        *Pattern* : ``[a-zA-Z_0-9+=,.@-]+``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-executionrolename
        '''
        result = self._values.get("execution_role_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def managed_execution(self) -> typing.Any:
        '''Describes whether StackSets performs non-conflicting operations concurrently and queues conflicting operations.

        When active, StackSets performs non-conflicting operations concurrently and queues conflicting operations. After conflicting operations finish, StackSets starts queued operations in request order.
        .. epigraph::

           If there are already running or queued operations, StackSets queues all incoming operations even if they are non-conflicting.

           You can't modify your stack set's execution configuration while there are running or queued operations for that stack set.

        When inactive (default), StackSets performs one operation at a time in request order.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-managedexecution
        '''
        result = self._values.get("managed_execution")
        return typing.cast(typing.Any, result)

    @builtins.property
    def operation_preferences(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", "CfnStackSet.OperationPreferencesProperty"]]:
        '''The user-specified preferences for how CloudFormation performs a stack set operation.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-operationpreferences
        '''
        result = self._values.get("operation_preferences")
        return typing.cast(typing.Optional[typing.Union["IResolvable", "CfnStackSet.OperationPreferencesProperty"]], result)

    @builtins.property
    def parameters(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", typing.List[typing.Union["IResolvable", "CfnStackSet.ParameterProperty"]]]]:
        '''The input parameters for the stack set template.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Union["IResolvable", typing.List[typing.Union["IResolvable", "CfnStackSet.ParameterProperty"]]]], result)

    @builtins.property
    def stack_instances_group(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", typing.List[typing.Union["IResolvable", "CfnStackSet.StackInstancesProperty"]]]]:
        '''A group of stack instances with parameters in some specific accounts and Regions.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-stackinstancesgroup
        '''
        result = self._values.get("stack_instances_group")
        return typing.cast(typing.Optional[typing.Union["IResolvable", typing.List[typing.Union["IResolvable", "CfnStackSet.StackInstancesProperty"]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List["CfnTag"]]:
        '''Key-value pairs to associate with this stack.

        CloudFormation also propagates these tags to supported resources in the stack. You can specify a maximum number of 50 tags.

        If you don't specify this parameter, CloudFormation doesn't modify the stack's tags. If you specify an empty value, CloudFormation removes all associated tags.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List["CfnTag"]], result)

    @builtins.property
    def template_body(self) -> typing.Optional[builtins.str]:
        '''The structure that contains the template body, with a minimum length of 1 byte and a maximum length of 51,200 bytes.

        You must include either ``TemplateURL`` or ``TemplateBody`` in a StackSet, but you can't use both. Dynamic references in the ``TemplateBody`` may not work correctly in all cases. It's recommended to pass templates containing dynamic references through ``TemplateUrl`` instead.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-templatebody
        '''
        result = self._values.get("template_body")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def template_url(self) -> typing.Optional[builtins.str]:
        '''The URL of a file containing the template body.

        The URL must point to a template (max size: 1 MB) that's located in an Amazon S3 bucket or a Systems Manager document. The location for an Amazon S3 bucket must start with ``https://`` .

        Conditional: You must specify only one of the following parameters: ``TemplateBody`` , ``TemplateURL`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-stackset.html#cfn-cloudformation-stackset-templateurl
        '''
        result = self._values.get("template_url")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnStackSetProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnTag",
    jsii_struct_bases=[],
    name_mapping={"key": "key", "value": "value"},
)
class CfnTag:
    def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
        '''
        :param key: 
        :param value: 

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-resource-tags.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_tag = cdk.CfnTag(
                key="key",
                value="value"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b6c1d7ff52297a62b95240cca6767683e442ad58822715479502fcdabe3de5db)
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "key": key,
            "value": value,
        }

    @builtins.property
    def key(self) -> builtins.str:
        '''
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-resource-tags.html#cfn-resource-tags-key
        '''
        result = self._values.get("key")
        assert result is not None, "Required property 'key' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def value(self) -> builtins.str:
        '''
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-resource-tags.html#cfn-resource-tags-value
        '''
        result = self._values.get("value")
        assert result is not None, "Required property 'value' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnTag(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnTrafficRoute",
    jsii_struct_bases=[],
    name_mapping={"logical_id": "logicalId", "type": "type"},
)
class CfnTrafficRoute:
    def __init__(self, *, logical_id: builtins.str, type: builtins.str) -> None:
        '''A traffic route, representing where the traffic is being directed to.

        :param logical_id: The logical id of the target resource.
        :param type: The resource type of the route. Today, the only allowed value is 'AWS::ElasticLoadBalancingV2::Listener'.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_traffic_route = cdk.CfnTrafficRoute(
                logical_id="logicalId",
                type="type"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7815e2892b74599171d75efec7545ad1f01fcd2f149448a5cdfe4bc6890a080f)
            check_type(argname="argument logical_id", value=logical_id, expected_type=type_hints["logical_id"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "logical_id": logical_id,
            "type": type,
        }

    @builtins.property
    def logical_id(self) -> builtins.str:
        '''The logical id of the target resource.'''
        result = self._values.get("logical_id")
        assert result is not None, "Required property 'logical_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type(self) -> builtins.str:
        '''The resource type of the route.

        Today, the only allowed value is 'AWS::ElasticLoadBalancingV2::Listener'.
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnTrafficRoute(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnTrafficRouting",
    jsii_struct_bases=[],
    name_mapping={
        "prod_traffic_route": "prodTrafficRoute",
        "target_groups": "targetGroups",
        "test_traffic_route": "testTrafficRoute",
    },
)
class CfnTrafficRouting:
    def __init__(
        self,
        *,
        prod_traffic_route: typing.Union[CfnTrafficRoute, typing.Dict[builtins.str, typing.Any]],
        target_groups: typing.Sequence[builtins.str],
        test_traffic_route: typing.Union[CfnTrafficRoute, typing.Dict[builtins.str, typing.Any]],
    ) -> None:
        '''Type of the ``CfnCodeDeployBlueGreenEcsAttributes.trafficRouting`` property.

        :param prod_traffic_route: The listener to be used by your load balancer to direct traffic to your target groups.
        :param target_groups: The logical IDs of the blue and green, respectively, AWS::ElasticLoadBalancingV2::TargetGroup target groups.
        :param test_traffic_route: The listener to be used by your load balancer to direct traffic to your target groups.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_traffic_routing = cdk.CfnTrafficRouting(
                prod_traffic_route=cdk.CfnTrafficRoute(
                    logical_id="logicalId",
                    type="type"
                ),
                target_groups=["targetGroups"],
                test_traffic_route=cdk.CfnTrafficRoute(
                    logical_id="logicalId",
                    type="type"
                )
            )
        '''
        if isinstance(prod_traffic_route, dict):
            prod_traffic_route = CfnTrafficRoute(**prod_traffic_route)
        if isinstance(test_traffic_route, dict):
            test_traffic_route = CfnTrafficRoute(**test_traffic_route)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c531a698004654beb12e486c72d7fbcbc2b0f1f3e370fc1b4c5546e24feef8a8)
            check_type(argname="argument prod_traffic_route", value=prod_traffic_route, expected_type=type_hints["prod_traffic_route"])
            check_type(argname="argument target_groups", value=target_groups, expected_type=type_hints["target_groups"])
            check_type(argname="argument test_traffic_route", value=test_traffic_route, expected_type=type_hints["test_traffic_route"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "prod_traffic_route": prod_traffic_route,
            "target_groups": target_groups,
            "test_traffic_route": test_traffic_route,
        }

    @builtins.property
    def prod_traffic_route(self) -> CfnTrafficRoute:
        '''The listener to be used by your load balancer to direct traffic to your target groups.'''
        result = self._values.get("prod_traffic_route")
        assert result is not None, "Required property 'prod_traffic_route' is missing"
        return typing.cast(CfnTrafficRoute, result)

    @builtins.property
    def target_groups(self) -> typing.List[builtins.str]:
        '''The logical IDs of the blue and green, respectively, AWS::ElasticLoadBalancingV2::TargetGroup target groups.'''
        result = self._values.get("target_groups")
        assert result is not None, "Required property 'target_groups' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def test_traffic_route(self) -> CfnTrafficRoute:
        '''The listener to be used by your load balancer to direct traffic to your target groups.'''
        result = self._values.get("test_traffic_route")
        assert result is not None, "Required property 'test_traffic_route' is missing"
        return typing.cast(CfnTrafficRoute, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnTrafficRouting(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnTrafficRoutingConfig",
    jsii_struct_bases=[],
    name_mapping={
        "type": "type",
        "time_based_canary": "timeBasedCanary",
        "time_based_linear": "timeBasedLinear",
    },
)
class CfnTrafficRoutingConfig:
    def __init__(
        self,
        *,
        type: "CfnTrafficRoutingType",
        time_based_canary: typing.Optional[typing.Union["CfnTrafficRoutingTimeBasedCanary", typing.Dict[builtins.str, typing.Any]]] = None,
        time_based_linear: typing.Optional[typing.Union["CfnTrafficRoutingTimeBasedLinear", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Traffic routing configuration settings.

        The type of the ``CfnCodeDeployBlueGreenHookProps.trafficRoutingConfig`` property.

        :param type: The type of traffic shifting used by the blue-green deployment configuration.
        :param time_based_canary: The configuration for traffic routing when ``type`` is ``CfnTrafficRoutingType.TIME_BASED_CANARY``. Default: - none
        :param time_based_linear: The configuration for traffic routing when ``type`` is ``CfnTrafficRoutingType.TIME_BASED_LINEAR``. Default: - none

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_traffic_routing_config = cdk.CfnTrafficRoutingConfig(
                type=cdk.CfnTrafficRoutingType.ALL_AT_ONCE,
            
                # the properties below are optional
                time_based_canary=cdk.CfnTrafficRoutingTimeBasedCanary(
                    bake_time_mins=123,
                    step_percentage=123
                ),
                time_based_linear=cdk.CfnTrafficRoutingTimeBasedLinear(
                    bake_time_mins=123,
                    step_percentage=123
                )
            )
        '''
        if isinstance(time_based_canary, dict):
            time_based_canary = CfnTrafficRoutingTimeBasedCanary(**time_based_canary)
        if isinstance(time_based_linear, dict):
            time_based_linear = CfnTrafficRoutingTimeBasedLinear(**time_based_linear)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4adc4b40f5a5e0f413f1441482e77ecc912a1e21e82a96417469454f48f09193)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument time_based_canary", value=time_based_canary, expected_type=type_hints["time_based_canary"])
            check_type(argname="argument time_based_linear", value=time_based_linear, expected_type=type_hints["time_based_linear"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }
        if time_based_canary is not None:
            self._values["time_based_canary"] = time_based_canary
        if time_based_linear is not None:
            self._values["time_based_linear"] = time_based_linear

    @builtins.property
    def type(self) -> "CfnTrafficRoutingType":
        '''The type of traffic shifting used by the blue-green deployment configuration.'''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("CfnTrafficRoutingType", result)

    @builtins.property
    def time_based_canary(self) -> typing.Optional["CfnTrafficRoutingTimeBasedCanary"]:
        '''The configuration for traffic routing when ``type`` is ``CfnTrafficRoutingType.TIME_BASED_CANARY``.

        :default: - none
        '''
        result = self._values.get("time_based_canary")
        return typing.cast(typing.Optional["CfnTrafficRoutingTimeBasedCanary"], result)

    @builtins.property
    def time_based_linear(self) -> typing.Optional["CfnTrafficRoutingTimeBasedLinear"]:
        '''The configuration for traffic routing when ``type`` is ``CfnTrafficRoutingType.TIME_BASED_LINEAR``.

        :default: - none
        '''
        result = self._values.get("time_based_linear")
        return typing.cast(typing.Optional["CfnTrafficRoutingTimeBasedLinear"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnTrafficRoutingConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnTrafficRoutingTimeBasedCanary",
    jsii_struct_bases=[],
    name_mapping={
        "bake_time_mins": "bakeTimeMins",
        "step_percentage": "stepPercentage",
    },
)
class CfnTrafficRoutingTimeBasedCanary:
    def __init__(
        self,
        *,
        bake_time_mins: typing.Optional[jsii.Number] = None,
        step_percentage: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''The traffic routing configuration if ``CfnTrafficRoutingConfig.type`` is ``CfnTrafficRoutingType.TIME_BASED_CANARY``.

        :param bake_time_mins: The number of minutes between the first and second traffic shifts of a time-based canary deployment. Default: 5
        :param step_percentage: The percentage of traffic to shift in the first increment of a time-based canary deployment. The step percentage must be 14% or greater. Default: 15

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_traffic_routing_time_based_canary = cdk.CfnTrafficRoutingTimeBasedCanary(
                bake_time_mins=123,
                step_percentage=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30d087f849cfe834e3a855b35755eb114a2bbe0d53b477ec503f319d80742cfc)
            check_type(argname="argument bake_time_mins", value=bake_time_mins, expected_type=type_hints["bake_time_mins"])
            check_type(argname="argument step_percentage", value=step_percentage, expected_type=type_hints["step_percentage"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if bake_time_mins is not None:
            self._values["bake_time_mins"] = bake_time_mins
        if step_percentage is not None:
            self._values["step_percentage"] = step_percentage

    @builtins.property
    def bake_time_mins(self) -> typing.Optional[jsii.Number]:
        '''The number of minutes between the first and second traffic shifts of a time-based canary deployment.

        :default: 5
        '''
        result = self._values.get("bake_time_mins")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def step_percentage(self) -> typing.Optional[jsii.Number]:
        '''The percentage of traffic to shift in the first increment of a time-based canary deployment.

        The step percentage must be 14% or greater.

        :default: 15
        '''
        result = self._values.get("step_percentage")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnTrafficRoutingTimeBasedCanary(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnTrafficRoutingTimeBasedLinear",
    jsii_struct_bases=[],
    name_mapping={
        "bake_time_mins": "bakeTimeMins",
        "step_percentage": "stepPercentage",
    },
)
class CfnTrafficRoutingTimeBasedLinear:
    def __init__(
        self,
        *,
        bake_time_mins: typing.Optional[jsii.Number] = None,
        step_percentage: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''The traffic routing configuration if ``CfnTrafficRoutingConfig.type`` is ``CfnTrafficRoutingType.TIME_BASED_LINEAR``.

        :param bake_time_mins: The number of minutes between the first and second traffic shifts of a time-based linear deployment. Default: 5
        :param step_percentage: The percentage of traffic that is shifted at the start of each increment of a time-based linear deployment. The step percentage must be 14% or greater. Default: 15

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_traffic_routing_time_based_linear = cdk.CfnTrafficRoutingTimeBasedLinear(
                bake_time_mins=123,
                step_percentage=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1f89baecbe790ff6e386ade23f6d38c3ba2fc1da33154750afd39879f4188e72)
            check_type(argname="argument bake_time_mins", value=bake_time_mins, expected_type=type_hints["bake_time_mins"])
            check_type(argname="argument step_percentage", value=step_percentage, expected_type=type_hints["step_percentage"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if bake_time_mins is not None:
            self._values["bake_time_mins"] = bake_time_mins
        if step_percentage is not None:
            self._values["step_percentage"] = step_percentage

    @builtins.property
    def bake_time_mins(self) -> typing.Optional[jsii.Number]:
        '''The number of minutes between the first and second traffic shifts of a time-based linear deployment.

        :default: 5
        '''
        result = self._values.get("bake_time_mins")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def step_percentage(self) -> typing.Optional[jsii.Number]:
        '''The percentage of traffic that is shifted at the start of each increment of a time-based linear deployment.

        The step percentage must be 14% or greater.

        :default: 15
        '''
        result = self._values.get("step_percentage")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnTrafficRoutingTimeBasedLinear(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.CfnTrafficRoutingType")
class CfnTrafficRoutingType(enum.Enum):
    '''The possible types of traffic shifting for the blue-green deployment configuration.

    The type of the ``CfnTrafficRoutingConfig.type`` property.
    '''

    ALL_AT_ONCE = "ALL_AT_ONCE"
    '''Switch from blue to green at once.'''
    TIME_BASED_CANARY = "TIME_BASED_CANARY"
    '''Specifies a configuration that shifts traffic from blue to green in two increments.'''
    TIME_BASED_LINEAR = "TIME_BASED_LINEAR"
    '''Specifies a configuration that shifts traffic from blue to green in equal increments, with an equal number of minutes between each increment.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnTypeActivationProps",
    jsii_struct_bases=[],
    name_mapping={
        "auto_update": "autoUpdate",
        "execution_role_arn": "executionRoleArn",
        "logging_config": "loggingConfig",
        "major_version": "majorVersion",
        "public_type_arn": "publicTypeArn",
        "publisher_id": "publisherId",
        "type": "type",
        "type_name": "typeName",
        "type_name_alias": "typeNameAlias",
        "version_bump": "versionBump",
    },
)
class CfnTypeActivationProps:
    def __init__(
        self,
        *,
        auto_update: typing.Optional[typing.Union[builtins.bool, "IResolvable"]] = None,
        execution_role_arn: typing.Optional[builtins.str] = None,
        logging_config: typing.Optional[typing.Union["IResolvable", typing.Union["CfnTypeActivation.LoggingConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        major_version: typing.Optional[builtins.str] = None,
        public_type_arn: typing.Optional[builtins.str] = None,
        publisher_id: typing.Optional[builtins.str] = None,
        type: typing.Optional[builtins.str] = None,
        type_name: typing.Optional[builtins.str] = None,
        type_name_alias: typing.Optional[builtins.str] = None,
        version_bump: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnTypeActivation``.

        :param auto_update: Whether to automatically update the extension in this account and Region when a new *minor* version is published by the extension publisher. Major versions released by the publisher must be manually updated. The default is ``true`` .
        :param execution_role_arn: The name of the IAM execution role to use to activate the extension.
        :param logging_config: Specifies logging configuration information for an extension.
        :param major_version: The major version of this extension you want to activate, if multiple major versions are available. The default is the latest major version. CloudFormation uses the latest available *minor* version of the major version selected. You can specify ``MajorVersion`` or ``VersionBump`` , but not both.
        :param public_type_arn: The Amazon Resource Number (ARN) of the public extension. Conditional: You must specify ``PublicTypeArn`` , or ``TypeName`` , ``Type`` , and ``PublisherId`` .
        :param publisher_id: The ID of the extension publisher. Conditional: You must specify ``PublicTypeArn`` , or ``TypeName`` , ``Type`` , and ``PublisherId`` .
        :param type: The extension type. Conditional: You must specify ``PublicTypeArn`` , or ``TypeName`` , ``Type`` , and ``PublisherId`` .
        :param type_name: The name of the extension. Conditional: You must specify ``PublicTypeArn`` , or ``TypeName`` , ``Type`` , and ``PublisherId`` .
        :param type_name_alias: An alias to assign to the public extension, in this account and Region. If you specify an alias for the extension, CloudFormation treats the alias as the extension type name within this account and Region. You must use the alias to refer to the extension in your templates, API calls, and CloudFormation console. An extension alias must be unique within a given account and Region. You can activate the same public resource multiple times in the same account and Region, using different type name aliases.
        :param version_bump: Manually updates a previously-activated type to a new major or minor version, if available. You can also use this parameter to update the value of ``AutoUpdate`` . - ``MAJOR`` : CloudFormation updates the extension to the newest major version, if one is available. - ``MINOR`` : CloudFormation updates the extension to the newest minor version, if one is available.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-typeactivation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_type_activation_props = cdk.CfnTypeActivationProps(
                auto_update=False,
                execution_role_arn="executionRoleArn",
                logging_config=cdk.CfnTypeActivation.LoggingConfigProperty(
                    log_group_name="logGroupName",
                    log_role_arn="logRoleArn"
                ),
                major_version="majorVersion",
                public_type_arn="publicTypeArn",
                publisher_id="publisherId",
                type="type",
                type_name="typeName",
                type_name_alias="typeNameAlias",
                version_bump="versionBump"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a433522c98b0bfa51cbfe5f74bac91e1f2e68987f761fa799c6050eff0ff77c)
            check_type(argname="argument auto_update", value=auto_update, expected_type=type_hints["auto_update"])
            check_type(argname="argument execution_role_arn", value=execution_role_arn, expected_type=type_hints["execution_role_arn"])
            check_type(argname="argument logging_config", value=logging_config, expected_type=type_hints["logging_config"])
            check_type(argname="argument major_version", value=major_version, expected_type=type_hints["major_version"])
            check_type(argname="argument public_type_arn", value=public_type_arn, expected_type=type_hints["public_type_arn"])
            check_type(argname="argument publisher_id", value=publisher_id, expected_type=type_hints["publisher_id"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument type_name", value=type_name, expected_type=type_hints["type_name"])
            check_type(argname="argument type_name_alias", value=type_name_alias, expected_type=type_hints["type_name_alias"])
            check_type(argname="argument version_bump", value=version_bump, expected_type=type_hints["version_bump"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if auto_update is not None:
            self._values["auto_update"] = auto_update
        if execution_role_arn is not None:
            self._values["execution_role_arn"] = execution_role_arn
        if logging_config is not None:
            self._values["logging_config"] = logging_config
        if major_version is not None:
            self._values["major_version"] = major_version
        if public_type_arn is not None:
            self._values["public_type_arn"] = public_type_arn
        if publisher_id is not None:
            self._values["publisher_id"] = publisher_id
        if type is not None:
            self._values["type"] = type
        if type_name is not None:
            self._values["type_name"] = type_name
        if type_name_alias is not None:
            self._values["type_name_alias"] = type_name_alias
        if version_bump is not None:
            self._values["version_bump"] = version_bump

    @builtins.property
    def auto_update(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, "IResolvable"]]:
        '''Whether to automatically update the extension in this account and Region when a new *minor* version is published by the extension publisher.

        Major versions released by the publisher must be manually updated.

        The default is ``true`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-typeactivation.html#cfn-cloudformation-typeactivation-autoupdate
        '''
        result = self._values.get("auto_update")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, "IResolvable"]], result)

    @builtins.property
    def execution_role_arn(self) -> typing.Optional[builtins.str]:
        '''The name of the IAM execution role to use to activate the extension.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-typeactivation.html#cfn-cloudformation-typeactivation-executionrolearn
        '''
        result = self._values.get("execution_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def logging_config(
        self,
    ) -> typing.Optional[typing.Union["IResolvable", "CfnTypeActivation.LoggingConfigProperty"]]:
        '''Specifies logging configuration information for an extension.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-typeactivation.html#cfn-cloudformation-typeactivation-loggingconfig
        '''
        result = self._values.get("logging_config")
        return typing.cast(typing.Optional[typing.Union["IResolvable", "CfnTypeActivation.LoggingConfigProperty"]], result)

    @builtins.property
    def major_version(self) -> typing.Optional[builtins.str]:
        '''The major version of this extension you want to activate, if multiple major versions are available.

        The default is the latest major version. CloudFormation uses the latest available *minor* version of the major version selected.

        You can specify ``MajorVersion`` or ``VersionBump`` , but not both.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-typeactivation.html#cfn-cloudformation-typeactivation-majorversion
        '''
        result = self._values.get("major_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def public_type_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Number (ARN) of the public extension.

        Conditional: You must specify ``PublicTypeArn`` , or ``TypeName`` , ``Type`` , and ``PublisherId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-typeactivation.html#cfn-cloudformation-typeactivation-publictypearn
        '''
        result = self._values.get("public_type_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def publisher_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the extension publisher.

        Conditional: You must specify ``PublicTypeArn`` , or ``TypeName`` , ``Type`` , and ``PublisherId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-typeactivation.html#cfn-cloudformation-typeactivation-publisherid
        '''
        result = self._values.get("publisher_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type(self) -> typing.Optional[builtins.str]:
        '''The extension type.

        Conditional: You must specify ``PublicTypeArn`` , or ``TypeName`` , ``Type`` , and ``PublisherId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-typeactivation.html#cfn-cloudformation-typeactivation-type
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type_name(self) -> typing.Optional[builtins.str]:
        '''The name of the extension.

        Conditional: You must specify ``PublicTypeArn`` , or ``TypeName`` , ``Type`` , and ``PublisherId`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-typeactivation.html#cfn-cloudformation-typeactivation-typename
        '''
        result = self._values.get("type_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type_name_alias(self) -> typing.Optional[builtins.str]:
        '''An alias to assign to the public extension, in this account and Region.

        If you specify an alias for the extension, CloudFormation treats the alias as the extension type name within this account and Region. You must use the alias to refer to the extension in your templates, API calls, and CloudFormation console.

        An extension alias must be unique within a given account and Region. You can activate the same public resource multiple times in the same account and Region, using different type name aliases.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-typeactivation.html#cfn-cloudformation-typeactivation-typenamealias
        '''
        result = self._values.get("type_name_alias")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def version_bump(self) -> typing.Optional[builtins.str]:
        '''Manually updates a previously-activated type to a new major or minor version, if available.

        You can also use this parameter to update the value of ``AutoUpdate`` .

        - ``MAJOR`` : CloudFormation updates the extension to the newest major version, if one is available.
        - ``MINOR`` : CloudFormation updates the extension to the newest minor version, if one is available.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-typeactivation.html#cfn-cloudformation-typeactivation-versionbump
        '''
        result = self._values.get("version_bump")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnTypeActivationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnUpdatePolicy",
    jsii_struct_bases=[],
    name_mapping={
        "auto_scaling_replacing_update": "autoScalingReplacingUpdate",
        "auto_scaling_rolling_update": "autoScalingRollingUpdate",
        "auto_scaling_scheduled_action": "autoScalingScheduledAction",
        "code_deploy_lambda_alias_update": "codeDeployLambdaAliasUpdate",
        "enable_version_upgrade": "enableVersionUpgrade",
        "use_online_resharding": "useOnlineResharding",
    },
)
class CfnUpdatePolicy:
    def __init__(
        self,
        *,
        auto_scaling_replacing_update: typing.Optional[typing.Union[CfnAutoScalingReplacingUpdate, typing.Dict[builtins.str, typing.Any]]] = None,
        auto_scaling_rolling_update: typing.Optional[typing.Union[CfnAutoScalingRollingUpdate, typing.Dict[builtins.str, typing.Any]]] = None,
        auto_scaling_scheduled_action: typing.Optional[typing.Union[CfnAutoScalingScheduledAction, typing.Dict[builtins.str, typing.Any]]] = None,
        code_deploy_lambda_alias_update: typing.Optional[typing.Union[CfnCodeDeployLambdaAliasUpdate, typing.Dict[builtins.str, typing.Any]]] = None,
        enable_version_upgrade: typing.Optional[builtins.bool] = None,
        use_online_resharding: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Use the UpdatePolicy attribute to specify how AWS CloudFormation handles updates to the AWS::AutoScaling::AutoScalingGroup resource.

        AWS CloudFormation invokes one of three update policies depending on the type of change you make or whether a
        scheduled action is associated with the Auto Scaling group.

        :param auto_scaling_replacing_update: Specifies whether an Auto Scaling group and the instances it contains are replaced during an update. During replacement, AWS CloudFormation retains the old group until it finishes creating the new one. If the update fails, AWS CloudFormation can roll back to the old Auto Scaling group and delete the new Auto Scaling group.
        :param auto_scaling_rolling_update: To specify how AWS CloudFormation handles rolling updates for an Auto Scaling group, use the AutoScalingRollingUpdate policy. Rolling updates enable you to specify whether AWS CloudFormation updates instances that are in an Auto Scaling group in batches or all at once.
        :param auto_scaling_scheduled_action: To specify how AWS CloudFormation handles updates for the MinSize, MaxSize, and DesiredCapacity properties when the AWS::AutoScaling::AutoScalingGroup resource has an associated scheduled action, use the AutoScalingScheduledAction policy.
        :param code_deploy_lambda_alias_update: To perform an AWS CodeDeploy deployment when the version changes on an AWS::Lambda::Alias resource, use the CodeDeployLambdaAliasUpdate update policy.
        :param enable_version_upgrade: To upgrade an Amazon ES domain to a new version of Elasticsearch rather than replacing the entire AWS::Elasticsearch::Domain resource, use the EnableVersionUpgrade update policy.
        :param use_online_resharding: To modify a replication group's shards by adding or removing shards, rather than replacing the entire AWS::ElastiCache::ReplicationGroup resource, use the UseOnlineResharding update policy.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_update_policy = cdk.CfnUpdatePolicy(
                auto_scaling_replacing_update=cdk.CfnAutoScalingReplacingUpdate(
                    will_replace=False
                ),
                auto_scaling_rolling_update=cdk.CfnAutoScalingRollingUpdate(
                    max_batch_size=123,
                    min_active_instances_percent=123,
                    min_instances_in_service=123,
                    min_successful_instances_percent=123,
                    pause_time="pauseTime",
                    suspend_processes=["suspendProcesses"],
                    wait_on_resource_signals=False
                ),
                auto_scaling_scheduled_action=cdk.CfnAutoScalingScheduledAction(
                    ignore_unmodified_group_size_properties=False
                ),
                code_deploy_lambda_alias_update=cdk.CfnCodeDeployLambdaAliasUpdate(
                    application_name="applicationName",
                    deployment_group_name="deploymentGroupName",
            
                    # the properties below are optional
                    after_allow_traffic_hook="afterAllowTrafficHook",
                    before_allow_traffic_hook="beforeAllowTrafficHook"
                ),
                enable_version_upgrade=False,
                use_online_resharding=False
            )
        '''
        if isinstance(auto_scaling_replacing_update, dict):
            auto_scaling_replacing_update = CfnAutoScalingReplacingUpdate(**auto_scaling_replacing_update)
        if isinstance(auto_scaling_rolling_update, dict):
            auto_scaling_rolling_update = CfnAutoScalingRollingUpdate(**auto_scaling_rolling_update)
        if isinstance(auto_scaling_scheduled_action, dict):
            auto_scaling_scheduled_action = CfnAutoScalingScheduledAction(**auto_scaling_scheduled_action)
        if isinstance(code_deploy_lambda_alias_update, dict):
            code_deploy_lambda_alias_update = CfnCodeDeployLambdaAliasUpdate(**code_deploy_lambda_alias_update)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6add1df70ff4410a8895aba5994eb8a4312fc5ba98cfb1bf45516d19d0d4f411)
            check_type(argname="argument auto_scaling_replacing_update", value=auto_scaling_replacing_update, expected_type=type_hints["auto_scaling_replacing_update"])
            check_type(argname="argument auto_scaling_rolling_update", value=auto_scaling_rolling_update, expected_type=type_hints["auto_scaling_rolling_update"])
            check_type(argname="argument auto_scaling_scheduled_action", value=auto_scaling_scheduled_action, expected_type=type_hints["auto_scaling_scheduled_action"])
            check_type(argname="argument code_deploy_lambda_alias_update", value=code_deploy_lambda_alias_update, expected_type=type_hints["code_deploy_lambda_alias_update"])
            check_type(argname="argument enable_version_upgrade", value=enable_version_upgrade, expected_type=type_hints["enable_version_upgrade"])
            check_type(argname="argument use_online_resharding", value=use_online_resharding, expected_type=type_hints["use_online_resharding"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if auto_scaling_replacing_update is not None:
            self._values["auto_scaling_replacing_update"] = auto_scaling_replacing_update
        if auto_scaling_rolling_update is not None:
            self._values["auto_scaling_rolling_update"] = auto_scaling_rolling_update
        if auto_scaling_scheduled_action is not None:
            self._values["auto_scaling_scheduled_action"] = auto_scaling_scheduled_action
        if code_deploy_lambda_alias_update is not None:
            self._values["code_deploy_lambda_alias_update"] = code_deploy_lambda_alias_update
        if enable_version_upgrade is not None:
            self._values["enable_version_upgrade"] = enable_version_upgrade
        if use_online_resharding is not None:
            self._values["use_online_resharding"] = use_online_resharding

    @builtins.property
    def auto_scaling_replacing_update(
        self,
    ) -> typing.Optional[CfnAutoScalingReplacingUpdate]:
        '''Specifies whether an Auto Scaling group and the instances it contains are replaced during an update.

        During replacement,
        AWS CloudFormation retains the old group until it finishes creating the new one. If the update fails, AWS CloudFormation
        can roll back to the old Auto Scaling group and delete the new Auto Scaling group.
        '''
        result = self._values.get("auto_scaling_replacing_update")
        return typing.cast(typing.Optional[CfnAutoScalingReplacingUpdate], result)

    @builtins.property
    def auto_scaling_rolling_update(
        self,
    ) -> typing.Optional[CfnAutoScalingRollingUpdate]:
        '''To specify how AWS CloudFormation handles rolling updates for an Auto Scaling group, use the AutoScalingRollingUpdate policy.

        Rolling updates enable you to specify whether AWS CloudFormation updates instances that are in an Auto Scaling
        group in batches or all at once.
        '''
        result = self._values.get("auto_scaling_rolling_update")
        return typing.cast(typing.Optional[CfnAutoScalingRollingUpdate], result)

    @builtins.property
    def auto_scaling_scheduled_action(
        self,
    ) -> typing.Optional[CfnAutoScalingScheduledAction]:
        '''To specify how AWS CloudFormation handles updates for the MinSize, MaxSize, and DesiredCapacity properties when the AWS::AutoScaling::AutoScalingGroup resource has an associated scheduled action, use the AutoScalingScheduledAction policy.'''
        result = self._values.get("auto_scaling_scheduled_action")
        return typing.cast(typing.Optional[CfnAutoScalingScheduledAction], result)

    @builtins.property
    def code_deploy_lambda_alias_update(
        self,
    ) -> typing.Optional[CfnCodeDeployLambdaAliasUpdate]:
        '''To perform an AWS CodeDeploy deployment when the version changes on an AWS::Lambda::Alias resource, use the CodeDeployLambdaAliasUpdate update policy.'''
        result = self._values.get("code_deploy_lambda_alias_update")
        return typing.cast(typing.Optional[CfnCodeDeployLambdaAliasUpdate], result)

    @builtins.property
    def enable_version_upgrade(self) -> typing.Optional[builtins.bool]:
        '''To upgrade an Amazon ES domain to a new version of Elasticsearch rather than replacing the entire AWS::Elasticsearch::Domain resource, use the EnableVersionUpgrade update policy.'''
        result = self._values.get("enable_version_upgrade")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def use_online_resharding(self) -> typing.Optional[builtins.bool]:
        '''To modify a replication group's shards by adding or removing shards, rather than replacing the entire AWS::ElastiCache::ReplicationGroup resource, use the UseOnlineResharding update policy.'''
        result = self._values.get("use_online_resharding")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUpdatePolicy(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnWaitConditionHandleProps",
    jsii_struct_bases=[],
    name_mapping={},
)
class CfnWaitConditionHandleProps:
    def __init__(self) -> None:
        '''Properties for defining a ``CfnWaitConditionHandle``.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-waitconditionhandle.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_wait_condition_handle_props = cdk.CfnWaitConditionHandleProps()
        '''
        self._values: typing.Dict[builtins.str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnWaitConditionHandleProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CfnWaitConditionProps",
    jsii_struct_bases=[],
    name_mapping={"count": "count", "handle": "handle", "timeout": "timeout"},
)
class CfnWaitConditionProps:
    def __init__(
        self,
        *,
        count: typing.Optional[jsii.Number] = None,
        handle: typing.Optional[builtins.str] = None,
        timeout: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnWaitCondition``.

        :param count: The number of success signals that CloudFormation must receive before it continues the stack creation process. When the wait condition receives the requisite number of success signals, CloudFormation resumes the creation of the stack. If the wait condition doesn't receive the specified number of success signals before the Timeout period expires, CloudFormation assumes that the wait condition has failed and rolls the stack back. Updates aren't supported.
        :param handle: A reference to the wait condition handle used to signal this wait condition. Use the ``Ref`` intrinsic function to specify an `AWS::CloudFormation::WaitConditionHandle <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-waitconditionhandle.html>`_ resource. Anytime you add a ``WaitCondition`` resource during a stack update, you must associate the wait condition with a new WaitConditionHandle resource. Don't reuse an old wait condition handle that has already been defined in the template. If you reuse a wait condition handle, the wait condition might evaluate old signals from a previous create or update stack command. Updates aren't supported.
        :param timeout: The length of time (in seconds) to wait for the number of signals that the ``Count`` property specifies. ``Timeout`` is a minimum-bound property, meaning the timeout occurs no sooner than the time you specify, but can occur shortly thereafter. The maximum time that can be specified for this property is 12 hours (43200 seconds). Updates aren't supported.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-waitcondition.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cfn_wait_condition_props = cdk.CfnWaitConditionProps(
                count=123,
                handle="handle",
                timeout="timeout"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__90d0d725667e58fc0126ff3258a3bf8a24b61dbe45104919280018f7283dac97)
            check_type(argname="argument count", value=count, expected_type=type_hints["count"])
            check_type(argname="argument handle", value=handle, expected_type=type_hints["handle"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if count is not None:
            self._values["count"] = count
        if handle is not None:
            self._values["handle"] = handle
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def count(self) -> typing.Optional[jsii.Number]:
        '''The number of success signals that CloudFormation must receive before it continues the stack creation process.

        When the wait condition receives the requisite number of success signals, CloudFormation resumes the creation of the stack. If the wait condition doesn't receive the specified number of success signals before the Timeout period expires, CloudFormation assumes that the wait condition has failed and rolls the stack back.

        Updates aren't supported.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-waitcondition.html#cfn-cloudformation-waitcondition-count
        '''
        result = self._values.get("count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def handle(self) -> typing.Optional[builtins.str]:
        '''A reference to the wait condition handle used to signal this wait condition.

        Use the ``Ref`` intrinsic function to specify an `AWS::CloudFormation::WaitConditionHandle <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-waitconditionhandle.html>`_ resource.

        Anytime you add a ``WaitCondition`` resource during a stack update, you must associate the wait condition with a new WaitConditionHandle resource. Don't reuse an old wait condition handle that has already been defined in the template. If you reuse a wait condition handle, the wait condition might evaluate old signals from a previous create or update stack command.

        Updates aren't supported.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-waitcondition.html#cfn-cloudformation-waitcondition-handle
        '''
        result = self._values.get("handle")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def timeout(self) -> typing.Optional[builtins.str]:
        '''The length of time (in seconds) to wait for the number of signals that the ``Count`` property specifies.

        ``Timeout`` is a minimum-bound property, meaning the timeout occurs no sooner than the time you specify, but can occur shortly thereafter. The maximum time that can be specified for this property is 12 hours (43200 seconds).

        Updates aren't supported.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-waitcondition.html#cfn-cloudformation-waitcondition-timeout
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnWaitConditionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CliCredentialsStackSynthesizerProps",
    jsii_struct_bases=[],
    name_mapping={
        "bucket_prefix": "bucketPrefix",
        "docker_tag_prefix": "dockerTagPrefix",
        "file_assets_bucket_name": "fileAssetsBucketName",
        "image_assets_repository_name": "imageAssetsRepositoryName",
        "qualifier": "qualifier",
    },
)
class CliCredentialsStackSynthesizerProps:
    def __init__(
        self,
        *,
        bucket_prefix: typing.Optional[builtins.str] = None,
        docker_tag_prefix: typing.Optional[builtins.str] = None,
        file_assets_bucket_name: typing.Optional[builtins.str] = None,
        image_assets_repository_name: typing.Optional[builtins.str] = None,
        qualifier: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for the CliCredentialsStackSynthesizer.

        :param bucket_prefix: bucketPrefix to use while storing S3 Assets. Default: - DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PREFIX
        :param docker_tag_prefix: A prefix to use while tagging and uploading Docker images to ECR. This does not add any separators - the source hash will be appended to this string directly. Default: - DefaultStackSynthesizer.DEFAULT_DOCKER_ASSET_PREFIX
        :param file_assets_bucket_name: Name of the S3 bucket to hold file assets. You must supply this if you have given a non-standard name to the staging bucket. The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will be replaced with the values of qualifier and the stack's account and region, respectively. Default: DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME
        :param image_assets_repository_name: Name of the ECR repository to hold Docker Image assets. You must supply this if you have given a non-standard name to the ECR repository. The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will be replaced with the values of qualifier and the stack's account and region, respectively. Default: DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME
        :param qualifier: Qualifier to disambiguate multiple environments in the same account. You can use this and leave the other naming properties empty if you have deployed the bootstrap environment with standard names but only different qualifiers. Default: - Value of context key '@aws-cdk/core:bootstrapQualifier' if set, otherwise ``DefaultStackSynthesizer.DEFAULT_QUALIFIER``

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            cli_credentials_stack_synthesizer_props = cdk.CliCredentialsStackSynthesizerProps(
                bucket_prefix="bucketPrefix",
                docker_tag_prefix="dockerTagPrefix",
                file_assets_bucket_name="fileAssetsBucketName",
                image_assets_repository_name="imageAssetsRepositoryName",
                qualifier="qualifier"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__582012b1da715680697bba8cad465d232673167a367d4087116d4afdd0c9f6d3)
            check_type(argname="argument bucket_prefix", value=bucket_prefix, expected_type=type_hints["bucket_prefix"])
            check_type(argname="argument docker_tag_prefix", value=docker_tag_prefix, expected_type=type_hints["docker_tag_prefix"])
            check_type(argname="argument file_assets_bucket_name", value=file_assets_bucket_name, expected_type=type_hints["file_assets_bucket_name"])
            check_type(argname="argument image_assets_repository_name", value=image_assets_repository_name, expected_type=type_hints["image_assets_repository_name"])
            check_type(argname="argument qualifier", value=qualifier, expected_type=type_hints["qualifier"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if bucket_prefix is not None:
            self._values["bucket_prefix"] = bucket_prefix
        if docker_tag_prefix is not None:
            self._values["docker_tag_prefix"] = docker_tag_prefix
        if file_assets_bucket_name is not None:
            self._values["file_assets_bucket_name"] = file_assets_bucket_name
        if image_assets_repository_name is not None:
            self._values["image_assets_repository_name"] = image_assets_repository_name
        if qualifier is not None:
            self._values["qualifier"] = qualifier

    @builtins.property
    def bucket_prefix(self) -> typing.Optional[builtins.str]:
        '''bucketPrefix to use while storing S3 Assets.

        :default: - DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PREFIX
        '''
        result = self._values.get("bucket_prefix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def docker_tag_prefix(self) -> typing.Optional[builtins.str]:
        '''A prefix to use while tagging and uploading Docker images to ECR.

        This does not add any separators - the source hash will be appended to
        this string directly.

        :default: - DefaultStackSynthesizer.DEFAULT_DOCKER_ASSET_PREFIX
        '''
        result = self._values.get("docker_tag_prefix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def file_assets_bucket_name(self) -> typing.Optional[builtins.str]:
        '''Name of the S3 bucket to hold file assets.

        You must supply this if you have given a non-standard name to the staging bucket.

        The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will
        be replaced with the values of qualifier and the stack's account and region,
        respectively.

        :default: DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME
        '''
        result = self._values.get("file_assets_bucket_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def image_assets_repository_name(self) -> typing.Optional[builtins.str]:
        '''Name of the ECR repository to hold Docker Image assets.

        You must supply this if you have given a non-standard name to the ECR repository.

        The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will
        be replaced with the values of qualifier and the stack's account and region,
        respectively.

        :default: DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME
        '''
        result = self._values.get("image_assets_repository_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def qualifier(self) -> typing.Optional[builtins.str]:
        '''Qualifier to disambiguate multiple environments in the same account.

        You can use this and leave the other naming properties empty if you have deployed
        the bootstrap environment with standard names but only different qualifiers.

        :default: - Value of context key '@aws-cdk/core:bootstrapQualifier' if set, otherwise ``DefaultStackSynthesizer.DEFAULT_QUALIFIER``
        '''
        result = self._values.get("qualifier")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CliCredentialsStackSynthesizerProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ContextProvider(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.ContextProvider"):
    '''Base class for the model side of context providers.

    Instances of this class communicate with context provider plugins in the 'cdk
    toolkit' via context variables (input), outputting specialized queries for
    more context variables (output).

    ContextProvider needs access to a Construct to hook into the context mechanism.
    '''

    @jsii.member(jsii_name="getKey")
    @builtins.classmethod
    def get_key(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        *,
        provider: builtins.str,
        include_environment: typing.Optional[builtins.bool] = None,
        props: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> "GetContextKeyResult":
        '''
        :param scope: -
        :param provider: The context provider to query.
        :param include_environment: Whether to include the stack's account and region automatically. Default: true
        :param props: Provider-specific properties.

        :return: the context key or undefined if a key cannot be rendered (due to tokens used in any of the props)
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__43574b9eb91c9a4b6203c6c4dae8ace9123c2746c25560e39a6b8a8db6ca67ff)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        options = GetContextKeyOptions(
            provider=provider, include_environment=include_environment, props=props
        )

        return typing.cast("GetContextKeyResult", jsii.sinvoke(cls, "getKey", [scope, options]))

    @jsii.member(jsii_name="getValue")
    @builtins.classmethod
    def get_value(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        *,
        dummy_value: typing.Any,
        ignore_error_on_missing_context: typing.Optional[builtins.bool] = None,
        must_exist: typing.Optional[builtins.bool] = None,
        provider: builtins.str,
        include_environment: typing.Optional[builtins.bool] = None,
        props: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> "GetContextValueResult":
        '''
        :param scope: -
        :param dummy_value: The value to return if the lookup has not yet been performed. Upon first synthesis, the lookups has not yet been performed. The ``getValue()`` operation returns this value instead, so that synthesis can proceed. After synthesis completes the first time, the actual lookup will be performed and synthesis will run again with the *real* value. Dummy values should preferably have valid shapes so that downstream consumers of lookup values don't throw validation exceptions if they encounter a dummy value (or all possible downstream consumers need to effectively check for the well-known shape of the dummy value); throwing an exception would error out the synthesis operation and prevent the lookup and the second, real, synthesis from happening. Connection to mustExist ``dummyValue`` is also used as the official value to return if the lookup has failed and ``mustExist == false``.
        :param ignore_error_on_missing_context: (deprecated) Ignore a lookup failure and return the ``dummyValue`` instead. ``mustExist`` is the recommended alias for this deprecated property (note that its value is reversed). Default: false
        :param must_exist: Whether the resource must exist. If this is set (the default), the query fails if the value or resource we tried to look up doesn't exist. If this is ``false`` and the value we tried to look up could not be found, the failure is suppressed and ``dummyValue`` is officially returned instead. When this happens, ``dummyValue`` is encoded into cached context and it will never be refreshed anymore until the user runs ``cdk context --reset <key>``. Note that it is not possible for the CDK app code to make a distinction between "the lookup has not been performed yet" and "the lookup didn't find anything and we returned a default value instead". Context providers This feature must explicitly be supported by context providers. It is currently supported by: - KMS key provider - SSM parameter provider Note to implementors The dummy value should not be returned for all SDK lookup failures. For example, "no network" or "no credentials" or "malformed query" should not lead to the dummy value being returned. Only the case of "no such resource" should. Default: true
        :param provider: The context provider to query.
        :param include_environment: Whether to include the stack's account and region automatically. Default: true
        :param props: Provider-specific properties.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cbcd83febe5237c988f26712b8a838ca5d2e34ea47490d008757c21a6870a77b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        options = GetContextValueOptions(
            dummy_value=dummy_value,
            ignore_error_on_missing_context=ignore_error_on_missing_context,
            must_exist=must_exist,
            provider=provider,
            include_environment=include_environment,
            props=props,
        )

        return typing.cast("GetContextValueResult", jsii.sinvoke(cls, "getValue", [scope, options]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.CopyOptions",
    jsii_struct_bases=[],
    name_mapping={
        "exclude": "exclude",
        "follow": "follow",
        "ignore_mode": "ignoreMode",
    },
)
class CopyOptions:
    def __init__(
        self,
        *,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional["SymlinkFollowMode"] = None,
        ignore_mode: typing.Optional["IgnoreMode"] = None,
    ) -> None:
        '''Options applied when copying directories.

        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            copy_options = cdk.CopyOptions(
                exclude=["exclude"],
                follow=cdk.SymlinkFollowMode.NEVER,
                ignore_mode=cdk.IgnoreMode.GLOB
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__80d5759670f4f104ff92870b1bbd513c39e43f1023a912e9dbdce77f980f218d)
            check_type(argname="argument exclude", value=exclude, expected_type=type_hints["exclude"])
            check_type(argname="argument follow", value=follow, expected_type=type_hints["follow"])
            check_type(argname="argument ignore_mode", value=ignore_mode, expected_type=type_hints["ignore_mode"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if exclude is not None:
            self._values["exclude"] = exclude
        if follow is not None:
            self._values["follow"] = follow
        if ignore_mode is not None:
            self._values["ignore_mode"] = ignore_mode

    @builtins.property
    def exclude(self) -> typing.Optional[typing.List[builtins.str]]:
        '''File paths matching the patterns will be excluded.

        See ``ignoreMode`` to set the matching behavior.
        Has no effect on Assets bundled using the ``bundling`` property.

        :default: - nothing is excluded
        '''
        result = self._values.get("exclude")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def follow(self) -> typing.Optional["SymlinkFollowMode"]:
        '''A strategy for how to handle symlinks.

        :default: SymlinkFollowMode.NEVER
        '''
        result = self._values.get("follow")
        return typing.cast(typing.Optional["SymlinkFollowMode"], result)

    @builtins.property
    def ignore_mode(self) -> typing.Optional["IgnoreMode"]:
        '''The ignore behavior to use for ``exclude`` patterns.

        :default: IgnoreMode.GLOB
        '''
        result = self._values.get("ignore_mode")
        return typing.cast(typing.Optional["IgnoreMode"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CopyOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CustomResourceProps",
    jsii_struct_bases=[],
    name_mapping={
        "service_token": "serviceToken",
        "pascal_case_properties": "pascalCaseProperties",
        "properties": "properties",
        "removal_policy": "removalPolicy",
        "resource_type": "resourceType",
        "service_timeout": "serviceTimeout",
    },
)
class CustomResourceProps:
    def __init__(
        self,
        *,
        service_token: builtins.str,
        pascal_case_properties: typing.Optional[builtins.bool] = None,
        properties: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        removal_policy: typing.Optional["RemovalPolicy"] = None,
        resource_type: typing.Optional[builtins.str] = None,
        service_timeout: typing.Optional["Duration"] = None,
    ) -> None:
        '''Properties to provide a Lambda-backed custom resource.

        :param service_token: The ARN of the provider which implements this custom resource type. You can implement a provider by listening to raw AWS CloudFormation events and specify the ARN of an SNS topic (``topic.topicArn``) or the ARN of an AWS Lambda function (``lambda.functionArn``) or use the CDK's custom `resource provider framework <https://docs.aws.amazon.com/cdk/api/latest/docs/custom-resources-readme.html>`_ which makes it easier to implement robust providers. Provider framework:: // use the provider framework from aws-cdk/custom-resources: const provider = new customresources.Provider(this, 'ResourceProvider', { onEventHandler, isCompleteHandler, // optional }); new CustomResource(this, 'MyResource', { serviceToken: provider.serviceToken, }); AWS Lambda function (not recommended to use AWS Lambda Functions directly, see the module README):: // invoke an AWS Lambda function when a lifecycle event occurs: new CustomResource(this, 'MyResource', { serviceToken: myFunction.functionArn, }); SNS topic (not recommended to use AWS Lambda Functions directly, see the module README):: // publish lifecycle events to an SNS topic: new CustomResource(this, 'MyResource', { serviceToken: myTopic.topicArn, }); Maps to `ServiceToken <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-customresource.html#cfn-cloudformation-customresource-servicetoken>`_ property for the ``AWS::CloudFormation::CustomResource`` resource
        :param pascal_case_properties: Convert all property keys to pascal case. Default: false
        :param properties: Properties to pass to the Lambda. Values in this ``properties`` dictionary can possibly overwrite other values in ``CustomResourceProps`` E.g. ``ServiceToken`` and ``ServiceTimeout`` It is recommended to avoid using same keys that exist in ``CustomResourceProps`` Default: - No properties.
        :param removal_policy: The policy to apply when this resource is removed from the application. Default: cdk.RemovalPolicy.Destroy
        :param resource_type: For custom resources, you can specify AWS::CloudFormation::CustomResource (the default) as the resource type, or you can specify your own resource type name. For example, you can use "Custom::MyCustomResourceTypeName". Custom resource type names must begin with "Custom::" and can include alphanumeric characters and the following characters: _@-. You can specify a custom resource type name up to a maximum length of 60 characters. You cannot change the type during an update. Using your own resource type names helps you quickly differentiate the types of custom resources in your stack. For example, if you had two custom resources that conduct two different ping tests, you could name their type as Custom::PingTester to make them easily identifiable as ping testers (instead of using AWS::CloudFormation::CustomResource). Default: - AWS::CloudFormation::CustomResource
        :param service_timeout: The maximum time that can elapse before a custom resource operation times out. The value must be between 1 second and 3600 seconds. Maps to `ServiceTimeout <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-customresource.html#cfn-cloudformation-customresource-servicetimeout>`_ property for the ``AWS::CloudFormation::CustomResource`` resource A token can be specified for this property, but it must be specified with ``Duration.seconds()``. Default: Duration.seconds(3600)

        :exampleMetadata: infused

        Example::

            stack = Stack()
            dur_token = CfnParameter(stack, "MyParameter",
                type="Number",
                default=60
            )
            CustomResource(stack, "MyCustomResource",
                service_token="MyServiceToken",
                service_timeout=Duration.seconds(dur_token.value_as_number)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9dc07c260551c6e828a1b20303b5a44e226e85166f6249c94e31d02717edd2f7)
            check_type(argname="argument service_token", value=service_token, expected_type=type_hints["service_token"])
            check_type(argname="argument pascal_case_properties", value=pascal_case_properties, expected_type=type_hints["pascal_case_properties"])
            check_type(argname="argument properties", value=properties, expected_type=type_hints["properties"])
            check_type(argname="argument removal_policy", value=removal_policy, expected_type=type_hints["removal_policy"])
            check_type(argname="argument resource_type", value=resource_type, expected_type=type_hints["resource_type"])
            check_type(argname="argument service_timeout", value=service_timeout, expected_type=type_hints["service_timeout"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "service_token": service_token,
        }
        if pascal_case_properties is not None:
            self._values["pascal_case_properties"] = pascal_case_properties
        if properties is not None:
            self._values["properties"] = properties
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy
        if resource_type is not None:
            self._values["resource_type"] = resource_type
        if service_timeout is not None:
            self._values["service_timeout"] = service_timeout

    @builtins.property
    def service_token(self) -> builtins.str:
        '''The ARN of the provider which implements this custom resource type.

        You can implement a provider by listening to raw AWS CloudFormation events
        and specify the ARN of an SNS topic (``topic.topicArn``) or the ARN of an AWS
        Lambda function (``lambda.functionArn``) or use the CDK's custom `resource
        provider framework <https://docs.aws.amazon.com/cdk/api/latest/docs/custom-resources-readme.html>`_ which makes it easier to implement robust providers.

        Provider framework::

           # use the provider framework from aws-cdk/custom-resources:
           provider = customresources.Provider(self, "ResourceProvider",
               on_event_handler=on_event_handler,
               is_complete_handler=is_complete_handler
           )

           CustomResource(self, "MyResource",
               service_token=provider.service_token
           )

        AWS Lambda function (not recommended to use AWS Lambda Functions directly,
        see the module README)::

           # invoke an AWS Lambda function when a lifecycle event occurs:
           CustomResource(self, "MyResource",
               service_token=my_function.function_arn
           )

        SNS topic (not recommended to use AWS Lambda Functions directly, see the
        module README)::

           # publish lifecycle events to an SNS topic:
           CustomResource(self, "MyResource",
               service_token=my_topic.topic_arn
           )

        Maps to `ServiceToken <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-customresource.html#cfn-cloudformation-customresource-servicetoken>`_ property for the ``AWS::CloudFormation::CustomResource`` resource
        '''
        result = self._values.get("service_token")
        assert result is not None, "Required property 'service_token' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def pascal_case_properties(self) -> typing.Optional[builtins.bool]:
        '''Convert all property keys to pascal case.

        :default: false
        '''
        result = self._values.get("pascal_case_properties")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def properties(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Properties to pass to the Lambda.

        Values in this ``properties`` dictionary can possibly overwrite other values in ``CustomResourceProps``
        E.g. ``ServiceToken`` and ``ServiceTimeout``
        It is recommended to avoid using same keys that exist in ``CustomResourceProps``

        :default: - No properties.
        '''
        result = self._values.get("properties")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def removal_policy(self) -> typing.Optional["RemovalPolicy"]:
        '''The policy to apply when this resource is removed from the application.

        :default: cdk.RemovalPolicy.Destroy
        '''
        result = self._values.get("removal_policy")
        return typing.cast(typing.Optional["RemovalPolicy"], result)

    @builtins.property
    def resource_type(self) -> typing.Optional[builtins.str]:
        '''For custom resources, you can specify AWS::CloudFormation::CustomResource (the default) as the resource type, or you can specify your own resource type name.

        For example, you can use "Custom::MyCustomResourceTypeName".

        Custom resource type names must begin with "Custom::" and can include
        alphanumeric characters and the following characters: _@-. You can specify
        a custom resource type name up to a maximum length of 60 characters. You
        cannot change the type during an update.

        Using your own resource type names helps you quickly differentiate the
        types of custom resources in your stack. For example, if you had two custom
        resources that conduct two different ping tests, you could name their type
        as Custom::PingTester to make them easily identifiable as ping testers
        (instead of using AWS::CloudFormation::CustomResource).

        :default: - AWS::CloudFormation::CustomResource

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html#aws-cfn-resource-type-name
        '''
        result = self._values.get("resource_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_timeout(self) -> typing.Optional["Duration"]:
        '''The maximum time that can elapse before a custom resource operation times out.

        The value must be between 1 second and 3600 seconds.

        Maps to `ServiceTimeout <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-customresource.html#cfn-cloudformation-customresource-servicetimeout>`_ property for the ``AWS::CloudFormation::CustomResource`` resource

        A token can be specified for this property, but it must be specified with ``Duration.seconds()``.

        :default: Duration.seconds(3600)

        Example::

            stack = Stack()
            dur_token = CfnParameter(stack, "MyParameter",
                type="Number",
                default=60
            )
            CustomResource(stack, "MyCustomResource",
                service_token="MyServiceToken",
                service_timeout=Duration.seconds(dur_token.value_as_number)
            )
        '''
        result = self._values.get("service_timeout")
        return typing.cast(typing.Optional["Duration"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CustomResourceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CustomResourceProviderBase(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.CustomResourceProviderBase",
):
    '''Base class for creating a custom resource provider.'''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        code_directory: builtins.str,
        runtime_name: builtins.str,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        memory_size: typing.Optional["Size"] = None,
        policy_statements: typing.Optional[typing.Sequence[typing.Any]] = None,
        timeout: typing.Optional["Duration"] = None,
        use_cfn_response_wrapper: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param code_directory: A local file system directory with the provider's code. The code will be bundled into a zip asset and wired to the provider's AWS Lambda function.
        :param runtime_name: The AWS Lambda runtime and version name to use for the provider.
        :param description: A description of the function. Default: - No description.
        :param environment: Key-value pairs that are passed to Lambda as Environment. Default: - No environment variables.
        :param memory_size: The amount of memory that your function has access to. Increasing the function's memory also increases its CPU allocation. Default: Size.mebibytes(128)
        :param policy_statements: A set of IAM policy statements to include in the inline policy of the provider's lambda function. **Please note**: these are direct IAM JSON policy blobs, *not* ``iam.PolicyStatement`` objects like you will see in the rest of the CDK. Default: - no additional inline policy
        :param timeout: AWS Lambda timeout for the provider. Default: Duration.minutes(15)
        :param use_cfn_response_wrapper: Whether or not the cloudformation response wrapper (``nodejs-entrypoint.ts``) is used. If set to ``true``, ``nodejs-entrypoint.js`` is bundled in the same asset as the custom resource and set as the entrypoint. If set to ``false``, the custom resource provided is the entrypoint. Default: - ``true`` if ``inlineCode: false`` and ``false`` otherwise.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b472990efe22be52456a5e812dc6396350b7010d1113310d62ef01f74c1769e1)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CustomResourceProviderBaseProps(
            code_directory=code_directory,
            runtime_name=runtime_name,
            description=description,
            environment=environment,
            memory_size=memory_size,
            policy_statements=policy_statements,
            timeout=timeout,
            use_cfn_response_wrapper=use_cfn_response_wrapper,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addToRolePolicy")
    def add_to_role_policy(self, statement: typing.Any) -> None:
        '''Add an IAM policy statement to the inline policy of the provider's lambda function's role.

        **Please note**: this is a direct IAM JSON policy blob, *not* a ``iam.PolicyStatement``
        object like you will see in the rest of the CDK.

        :param statement: -

        Example::

            # my_provider: CustomResourceProvider
            
            
            my_provider.add_to_role_policy({
                "Effect": "Allow",
                "Action": "s3:GetObject",
                "Resource": "*"
            })
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d7b4da0f6df59a6071a8ead45156197cdf0751659059c019b964df5f5dd58b9a)
            check_type(argname="argument statement", value=statement, expected_type=type_hints["statement"])
        return typing.cast(None, jsii.invoke(self, "addToRolePolicy", [statement]))

    @builtins.property
    @jsii.member(jsii_name="codeHash")
    def code_hash(self) -> builtins.str:
        '''The hash of the lambda code backing this provider.

        Can be used to trigger updates
        on code changes, even when the properties of a custom resource remain unchanged.
        '''
        return typing.cast(builtins.str, jsii.get(self, "codeHash"))

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        '''The ARN of the provider's AWS Lambda function role.'''
        return typing.cast(builtins.str, jsii.get(self, "roleArn"))

    @builtins.property
    @jsii.member(jsii_name="serviceToken")
    def service_token(self) -> builtins.str:
        '''The ARN of the provider's AWS Lambda function which should be used as the ``serviceToken`` when defining a custom resource.'''
        return typing.cast(builtins.str, jsii.get(self, "serviceToken"))


class _CustomResourceProviderBaseProxy(CustomResourceProviderBase):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, CustomResourceProviderBase).__jsii_proxy_class__ = lambda : _CustomResourceProviderBaseProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.CustomResourceProviderOptions",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "environment": "environment",
        "memory_size": "memorySize",
        "policy_statements": "policyStatements",
        "timeout": "timeout",
        "use_cfn_response_wrapper": "useCfnResponseWrapper",
    },
)
class CustomResourceProviderOptions:
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        memory_size: typing.Optional["Size"] = None,
        policy_statements: typing.Optional[typing.Sequence[typing.Any]] = None,
        timeout: typing.Optional["Duration"] = None,
        use_cfn_response_wrapper: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Initialization options for custom resource providers.

        :param description: A description of the function. Default: - No description.
        :param environment: Key-value pairs that are passed to Lambda as Environment. Default: - No environment variables.
        :param memory_size: The amount of memory that your function has access to. Increasing the function's memory also increases its CPU allocation. Default: Size.mebibytes(128)
        :param policy_statements: A set of IAM policy statements to include in the inline policy of the provider's lambda function. **Please note**: these are direct IAM JSON policy blobs, *not* ``iam.PolicyStatement`` objects like you will see in the rest of the CDK. Default: - no additional inline policy
        :param timeout: AWS Lambda timeout for the provider. Default: Duration.minutes(15)
        :param use_cfn_response_wrapper: Whether or not the cloudformation response wrapper (``nodejs-entrypoint.ts``) is used. If set to ``true``, ``nodejs-entrypoint.js`` is bundled in the same asset as the custom resource and set as the entrypoint. If set to ``false``, the custom resource provided is the entrypoint. Default: - ``true`` if ``inlineCode: false`` and ``false`` otherwise.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # policy_statements: Any
            # size: cdk.Size
            
            custom_resource_provider_options = cdk.CustomResourceProviderOptions(
                description="description",
                environment={
                    "environment_key": "environment"
                },
                memory_size=size,
                policy_statements=[policy_statements],
                timeout=cdk.Duration.minutes(30),
                use_cfn_response_wrapper=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7473e2978820a8ffdbda9b6e4ccae286256e80d7f830b3838e774e0dd9a8605f)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument memory_size", value=memory_size, expected_type=type_hints["memory_size"])
            check_type(argname="argument policy_statements", value=policy_statements, expected_type=type_hints["policy_statements"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument use_cfn_response_wrapper", value=use_cfn_response_wrapper, expected_type=type_hints["use_cfn_response_wrapper"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if environment is not None:
            self._values["environment"] = environment
        if memory_size is not None:
            self._values["memory_size"] = memory_size
        if policy_statements is not None:
            self._values["policy_statements"] = policy_statements
        if timeout is not None:
            self._values["timeout"] = timeout
        if use_cfn_response_wrapper is not None:
            self._values["use_cfn_response_wrapper"] = use_cfn_response_wrapper

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the function.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Key-value pairs that are passed to Lambda as Environment.

        :default: - No environment variables.
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def memory_size(self) -> typing.Optional["Size"]:
        '''The amount of memory that your function has access to.

        Increasing the
        function's memory also increases its CPU allocation.

        :default: Size.mebibytes(128)
        '''
        result = self._values.get("memory_size")
        return typing.cast(typing.Optional["Size"], result)

    @builtins.property
    def policy_statements(self) -> typing.Optional[typing.List[typing.Any]]:
        '''A set of IAM policy statements to include in the inline policy of the provider's lambda function.

        **Please note**: these are direct IAM JSON policy blobs, *not* ``iam.PolicyStatement``
        objects like you will see in the rest of the CDK.

        :default: - no additional inline policy

        Example::

            provider = CustomResourceProvider.get_or_create_provider(self, "Custom::MyCustomResourceType",
                code_directory=f"{__dirname}/my-handler",
                runtime=CustomResourceProviderRuntime.NODEJS_18_X,
                policy_statements=[{
                    "Effect": "Allow",
                    "Action": "s3:PutObject*",
                    "Resource": "*"
                }
                ]
            )
        '''
        result = self._values.get("policy_statements")
        return typing.cast(typing.Optional[typing.List[typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional["Duration"]:
        '''AWS Lambda timeout for the provider.

        :default: Duration.minutes(15)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional["Duration"], result)

    @builtins.property
    def use_cfn_response_wrapper(self) -> typing.Optional[builtins.bool]:
        '''Whether or not the cloudformation response wrapper (``nodejs-entrypoint.ts``) is used. If set to ``true``, ``nodejs-entrypoint.js`` is bundled in the same asset as the custom resource and set as the entrypoint. If set to ``false``, the custom resource provided is the entrypoint.

        :default: - ``true`` if ``inlineCode: false`` and ``false`` otherwise.
        '''
        result = self._values.get("use_cfn_response_wrapper")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CustomResourceProviderOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.CustomResourceProviderProps",
    jsii_struct_bases=[CustomResourceProviderOptions],
    name_mapping={
        "description": "description",
        "environment": "environment",
        "memory_size": "memorySize",
        "policy_statements": "policyStatements",
        "timeout": "timeout",
        "use_cfn_response_wrapper": "useCfnResponseWrapper",
        "code_directory": "codeDirectory",
        "runtime": "runtime",
    },
)
class CustomResourceProviderProps(CustomResourceProviderOptions):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        memory_size: typing.Optional["Size"] = None,
        policy_statements: typing.Optional[typing.Sequence[typing.Any]] = None,
        timeout: typing.Optional["Duration"] = None,
        use_cfn_response_wrapper: typing.Optional[builtins.bool] = None,
        code_directory: builtins.str,
        runtime: "CustomResourceProviderRuntime",
    ) -> None:
        '''Initialization properties for ``CustomResourceProvider``.

        :param description: A description of the function. Default: - No description.
        :param environment: Key-value pairs that are passed to Lambda as Environment. Default: - No environment variables.
        :param memory_size: The amount of memory that your function has access to. Increasing the function's memory also increases its CPU allocation. Default: Size.mebibytes(128)
        :param policy_statements: A set of IAM policy statements to include in the inline policy of the provider's lambda function. **Please note**: these are direct IAM JSON policy blobs, *not* ``iam.PolicyStatement`` objects like you will see in the rest of the CDK. Default: - no additional inline policy
        :param timeout: AWS Lambda timeout for the provider. Default: Duration.minutes(15)
        :param use_cfn_response_wrapper: Whether or not the cloudformation response wrapper (``nodejs-entrypoint.ts``) is used. If set to ``true``, ``nodejs-entrypoint.js`` is bundled in the same asset as the custom resource and set as the entrypoint. If set to ``false``, the custom resource provided is the entrypoint. Default: - ``true`` if ``inlineCode: false`` and ``false`` otherwise.
        :param code_directory: A local file system directory with the provider's code. The code will be bundled into a zip asset and wired to the provider's AWS Lambda function.
        :param runtime: The AWS Lambda runtime and version to use for the provider.

        :exampleMetadata: infused

        Example::

            provider = CustomResourceProvider.get_or_create_provider(self, "Custom::MyCustomResourceType",
                code_directory=f"{__dirname}/my-handler",
                runtime=CustomResourceProviderRuntime.NODEJS_18_X
            )
            provider.add_to_role_policy({
                "Effect": "Allow",
                "Action": "s3:GetObject",
                "Resource": "*"
            })
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__67b3f7332c978ef4e7c7aecd62d50434c044f1fb228ab5cf8711c9ad036743c1)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument memory_size", value=memory_size, expected_type=type_hints["memory_size"])
            check_type(argname="argument policy_statements", value=policy_statements, expected_type=type_hints["policy_statements"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument use_cfn_response_wrapper", value=use_cfn_response_wrapper, expected_type=type_hints["use_cfn_response_wrapper"])
            check_type(argname="argument code_directory", value=code_directory, expected_type=type_hints["code_directory"])
            check_type(argname="argument runtime", value=runtime, expected_type=type_hints["runtime"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "code_directory": code_directory,
            "runtime": runtime,
        }
        if description is not None:
            self._values["description"] = description
        if environment is not None:
            self._values["environment"] = environment
        if memory_size is not None:
            self._values["memory_size"] = memory_size
        if policy_statements is not None:
            self._values["policy_statements"] = policy_statements
        if timeout is not None:
            self._values["timeout"] = timeout
        if use_cfn_response_wrapper is not None:
            self._values["use_cfn_response_wrapper"] = use_cfn_response_wrapper

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the function.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Key-value pairs that are passed to Lambda as Environment.

        :default: - No environment variables.
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def memory_size(self) -> typing.Optional["Size"]:
        '''The amount of memory that your function has access to.

        Increasing the
        function's memory also increases its CPU allocation.

        :default: Size.mebibytes(128)
        '''
        result = self._values.get("memory_size")
        return typing.cast(typing.Optional["Size"], result)

    @builtins.property
    def policy_statements(self) -> typing.Optional[typing.List[typing.Any]]:
        '''A set of IAM policy statements to include in the inline policy of the provider's lambda function.

        **Please note**: these are direct IAM JSON policy blobs, *not* ``iam.PolicyStatement``
        objects like you will see in the rest of the CDK.

        :default: - no additional inline policy

        Example::

            provider = CustomResourceProvider.get_or_create_provider(self, "Custom::MyCustomResourceType",
                code_directory=f"{__dirname}/my-handler",
                runtime=CustomResourceProviderRuntime.NODEJS_18_X,
                policy_statements=[{
                    "Effect": "Allow",
                    "Action": "s3:PutObject*",
                    "Resource": "*"
                }
                ]
            )
        '''
        result = self._values.get("policy_statements")
        return typing.cast(typing.Optional[typing.List[typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional["Duration"]:
        '''AWS Lambda timeout for the provider.

        :default: Duration.minutes(15)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional["Duration"], result)

    @builtins.property
    def use_cfn_response_wrapper(self) -> typing.Optional[builtins.bool]:
        '''Whether or not the cloudformation response wrapper (``nodejs-entrypoint.ts``) is used. If set to ``true``, ``nodejs-entrypoint.js`` is bundled in the same asset as the custom resource and set as the entrypoint. If set to ``false``, the custom resource provided is the entrypoint.

        :default: - ``true`` if ``inlineCode: false`` and ``false`` otherwise.
        '''
        result = self._values.get("use_cfn_response_wrapper")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def code_directory(self) -> builtins.str:
        '''A local file system directory with the provider's code.

        The code will be
        bundled into a zip asset and wired to the provider's AWS Lambda function.
        '''
        result = self._values.get("code_directory")
        assert result is not None, "Required property 'code_directory' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def runtime(self) -> "CustomResourceProviderRuntime":
        '''The AWS Lambda runtime and version to use for the provider.'''
        result = self._values.get("runtime")
        assert result is not None, "Required property 'runtime' is missing"
        return typing.cast("CustomResourceProviderRuntime", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CustomResourceProviderProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.CustomResourceProviderRuntime")
class CustomResourceProviderRuntime(enum.Enum):
    '''The lambda runtime to use for the resource provider.

    This also indicates
    which language is used for the handler.

    :exampleMetadata: infused

    Example::

        provider = CustomResourceProvider.get_or_create_provider(self, "Custom::MyCustomResourceType",
            code_directory=f"{__dirname}/my-handler",
            runtime=CustomResourceProviderRuntime.NODEJS_18_X
        )
        provider.add_to_role_policy({
            "Effect": "Allow",
            "Action": "s3:GetObject",
            "Resource": "*"
        })
    '''

    NODEJS_12_X = "NODEJS_12_X"
    '''(deprecated) Node.js 12.x.

    :deprecated: Use latest version

    :stability: deprecated
    '''
    NODEJS_14_X = "NODEJS_14_X"
    '''(deprecated) Node.js 14.x.

    :deprecated: Use latest version

    :stability: deprecated
    '''
    NODEJS_16_X = "NODEJS_16_X"
    '''Node.js 16.x.'''
    NODEJS_18_X = "NODEJS_18_X"
    '''Node.js 18.x.'''
    NODEJS_20_X = "NODEJS_20_X"
    '''Node.js 20.x.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.DefaultStackSynthesizerProps",
    jsii_struct_bases=[],
    name_mapping={
        "bootstrap_stack_version_ssm_parameter": "bootstrapStackVersionSsmParameter",
        "bucket_prefix": "bucketPrefix",
        "cloud_formation_execution_role": "cloudFormationExecutionRole",
        "deploy_role_additional_options": "deployRoleAdditionalOptions",
        "deploy_role_arn": "deployRoleArn",
        "deploy_role_external_id": "deployRoleExternalId",
        "docker_tag_prefix": "dockerTagPrefix",
        "file_asset_publishing_external_id": "fileAssetPublishingExternalId",
        "file_asset_publishing_role_additional_options": "fileAssetPublishingRoleAdditionalOptions",
        "file_asset_publishing_role_arn": "fileAssetPublishingRoleArn",
        "file_assets_bucket_name": "fileAssetsBucketName",
        "generate_bootstrap_version_rule": "generateBootstrapVersionRule",
        "image_asset_publishing_external_id": "imageAssetPublishingExternalId",
        "image_asset_publishing_role_additional_options": "imageAssetPublishingRoleAdditionalOptions",
        "image_asset_publishing_role_arn": "imageAssetPublishingRoleArn",
        "image_assets_repository_name": "imageAssetsRepositoryName",
        "lookup_role_additional_options": "lookupRoleAdditionalOptions",
        "lookup_role_arn": "lookupRoleArn",
        "lookup_role_external_id": "lookupRoleExternalId",
        "qualifier": "qualifier",
        "use_lookup_role_for_stack_operations": "useLookupRoleForStackOperations",
    },
)
class DefaultStackSynthesizerProps:
    def __init__(
        self,
        *,
        bootstrap_stack_version_ssm_parameter: typing.Optional[builtins.str] = None,
        bucket_prefix: typing.Optional[builtins.str] = None,
        cloud_formation_execution_role: typing.Optional[builtins.str] = None,
        deploy_role_additional_options: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        deploy_role_arn: typing.Optional[builtins.str] = None,
        deploy_role_external_id: typing.Optional[builtins.str] = None,
        docker_tag_prefix: typing.Optional[builtins.str] = None,
        file_asset_publishing_external_id: typing.Optional[builtins.str] = None,
        file_asset_publishing_role_additional_options: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        file_asset_publishing_role_arn: typing.Optional[builtins.str] = None,
        file_assets_bucket_name: typing.Optional[builtins.str] = None,
        generate_bootstrap_version_rule: typing.Optional[builtins.bool] = None,
        image_asset_publishing_external_id: typing.Optional[builtins.str] = None,
        image_asset_publishing_role_additional_options: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        image_asset_publishing_role_arn: typing.Optional[builtins.str] = None,
        image_assets_repository_name: typing.Optional[builtins.str] = None,
        lookup_role_additional_options: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        lookup_role_arn: typing.Optional[builtins.str] = None,
        lookup_role_external_id: typing.Optional[builtins.str] = None,
        qualifier: typing.Optional[builtins.str] = None,
        use_lookup_role_for_stack_operations: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Configuration properties for DefaultStackSynthesizer.

        :param bootstrap_stack_version_ssm_parameter: Bootstrap stack version SSM parameter. The placeholder ``${Qualifier}`` will be replaced with the value of qualifier. Default: DefaultStackSynthesizer.DEFAULT_BOOTSTRAP_STACK_VERSION_SSM_PARAMETER
        :param bucket_prefix: bucketPrefix to use while storing S3 Assets. Default: - DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PREFIX
        :param cloud_formation_execution_role: The role CloudFormation will assume when deploying the Stack. You must supply this if you have given a non-standard name to the execution role. The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will be replaced with the values of qualifier and the stack's account and region, respectively. Default: DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN
        :param deploy_role_additional_options: Additional options to pass to STS when assuming the deploy role. - ``RoleArn`` should not be used. Use the dedicated ``deployRoleArn`` property instead. - ``ExternalId`` should not be used. Use the dedicated ``deployRoleExternalId`` instead. - ``TransitiveTagKeys`` defaults to use all keys (if any) specified in ``Tags``. E.g, all tags are transitive by default. Default: - No additional options.
        :param deploy_role_arn: The role to assume to initiate a deployment in this environment. You must supply this if you have given a non-standard name to the publishing role. The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will be replaced with the values of qualifier and the stack's account and region, respectively. Default: DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN
        :param deploy_role_external_id: External ID to use when assuming role for cloudformation deployments. Default: - No external ID
        :param docker_tag_prefix: A prefix to use while tagging and uploading Docker images to ECR. This does not add any separators - the source hash will be appended to this string directly. Default: - DefaultStackSynthesizer.DEFAULT_DOCKER_ASSET_PREFIX
        :param file_asset_publishing_external_id: External ID to use when assuming role for file asset publishing. Default: - No external ID
        :param file_asset_publishing_role_additional_options: Additional options to pass to STS when assuming the file asset publishing. - ``RoleArn`` should not be used. Use the dedicated ``fileAssetPublishingRoleArn`` property instead. - ``ExternalId`` should not be used. Use the dedicated ``fileAssetPublishingExternalId`` instead. - ``TransitiveTagKeys`` defaults to use all keys (if any) specified in ``Tags``. E.g, all tags are transitive by default. Default: - No additional options.
        :param file_asset_publishing_role_arn: The role to use to publish file assets to the S3 bucket in this environment. You must supply this if you have given a non-standard name to the publishing role. The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will be replaced with the values of qualifier and the stack's account and region, respectively. Default: DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PUBLISHING_ROLE_ARN
        :param file_assets_bucket_name: Name of the S3 bucket to hold file assets. You must supply this if you have given a non-standard name to the staging bucket. The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will be replaced with the values of qualifier and the stack's account and region, respectively. Default: DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME
        :param generate_bootstrap_version_rule: Whether to add a Rule to the stack template verifying the bootstrap stack version. This generally should be left set to ``true``, unless you explicitly want to be able to deploy to an unbootstrapped environment. Default: true
        :param image_asset_publishing_external_id: External ID to use when assuming role for image asset publishing. Default: - No external ID
        :param image_asset_publishing_role_additional_options: Additional options to pass to STS when assuming the image asset publishing. - ``RoleArn`` should not be used. Use the dedicated ``imageAssetPublishingRoleArn`` property instead. - ``ExternalId`` should not be used. Use the dedicated ``imageAssetPublishingExternalId`` instead. - ``TransitiveTagKeys`` defaults to use all keys (if any) specified in ``Tags``. E.g, all tags are transitive by default. Default: - No additional options.
        :param image_asset_publishing_role_arn: The role to use to publish image assets to the ECR repository in this environment. You must supply this if you have given a non-standard name to the publishing role. The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will be replaced with the values of qualifier and the stack's account and region, respectively. Default: DefaultStackSynthesizer.DEFAULT_IMAGE_ASSET_PUBLISHING_ROLE_ARN
        :param image_assets_repository_name: Name of the ECR repository to hold Docker Image assets. You must supply this if you have given a non-standard name to the ECR repository. The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will be replaced with the values of qualifier and the stack's account and region, respectively. Default: DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME
        :param lookup_role_additional_options: Additional options to pass to STS when assuming the lookup role. - ``RoleArn`` should not be used. Use the dedicated ``lookupRoleArn`` property instead. - ``ExternalId`` should not be used. Use the dedicated ``lookupRoleExternalId`` instead. - ``TransitiveTagKeys`` defaults to use all keys (if any) specified in ``Tags``. E.g, all tags are transitive by default. Default: - No additional options.
        :param lookup_role_arn: The role to use to look up values from the target AWS account during synthesis. Default: - None
        :param lookup_role_external_id: External ID to use when assuming lookup role. Default: - No external ID
        :param qualifier: Qualifier to disambiguate multiple environments in the same account. You can use this and leave the other naming properties empty if you have deployed the bootstrap environment with standard names but only different qualifiers. Default: - Value of context key '@aws-cdk/core:bootstrapQualifier' if set, otherwise ``DefaultStackSynthesizer.DEFAULT_QUALIFIER``
        :param use_lookup_role_for_stack_operations: Use the bootstrapped lookup role for (read-only) stack operations. Use the lookup role when performing a ``cdk diff``. If set to ``false``, the ``deploy role`` credentials will be used to perform a ``cdk diff``. Requires bootstrap stack version 8. Default: true

        :exampleMetadata: infused

        Example::

            MyStack(app, "MyStack",
                synthesizer=DefaultStackSynthesizer(
                    file_assets_bucket_name="amzn-s3-demo-bucket"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__06784f4f0c2d983d957b0f4fbf46c4e4e85e842ac87bb6c7b582ce30bcd481ad)
            check_type(argname="argument bootstrap_stack_version_ssm_parameter", value=bootstrap_stack_version_ssm_parameter, expected_type=type_hints["bootstrap_stack_version_ssm_parameter"])
            check_type(argname="argument bucket_prefix", value=bucket_prefix, expected_type=type_hints["bucket_prefix"])
            check_type(argname="argument cloud_formation_execution_role", value=cloud_formation_execution_role, expected_type=type_hints["cloud_formation_execution_role"])
            check_type(argname="argument deploy_role_additional_options", value=deploy_role_additional_options, expected_type=type_hints["deploy_role_additional_options"])
            check_type(argname="argument deploy_role_arn", value=deploy_role_arn, expected_type=type_hints["deploy_role_arn"])
            check_type(argname="argument deploy_role_external_id", value=deploy_role_external_id, expected_type=type_hints["deploy_role_external_id"])
            check_type(argname="argument docker_tag_prefix", value=docker_tag_prefix, expected_type=type_hints["docker_tag_prefix"])
            check_type(argname="argument file_asset_publishing_external_id", value=file_asset_publishing_external_id, expected_type=type_hints["file_asset_publishing_external_id"])
            check_type(argname="argument file_asset_publishing_role_additional_options", value=file_asset_publishing_role_additional_options, expected_type=type_hints["file_asset_publishing_role_additional_options"])
            check_type(argname="argument file_asset_publishing_role_arn", value=file_asset_publishing_role_arn, expected_type=type_hints["file_asset_publishing_role_arn"])
            check_type(argname="argument file_assets_bucket_name", value=file_assets_bucket_name, expected_type=type_hints["file_assets_bucket_name"])
            check_type(argname="argument generate_bootstrap_version_rule", value=generate_bootstrap_version_rule, expected_type=type_hints["generate_bootstrap_version_rule"])
            check_type(argname="argument image_asset_publishing_external_id", value=image_asset_publishing_external_id, expected_type=type_hints["image_asset_publishing_external_id"])
            check_type(argname="argument image_asset_publishing_role_additional_options", value=image_asset_publishing_role_additional_options, expected_type=type_hints["image_asset_publishing_role_additional_options"])
            check_type(argname="argument image_asset_publishing_role_arn", value=image_asset_publishing_role_arn, expected_type=type_hints["image_asset_publishing_role_arn"])
            check_type(argname="argument image_assets_repository_name", value=image_assets_repository_name, expected_type=type_hints["image_assets_repository_name"])
            check_type(argname="argument lookup_role_additional_options", value=lookup_role_additional_options, expected_type=type_hints["lookup_role_additional_options"])
            check_type(argname="argument lookup_role_arn", value=lookup_role_arn, expected_type=type_hints["lookup_role_arn"])
            check_type(argname="argument lookup_role_external_id", value=lookup_role_external_id, expected_type=type_hints["lookup_role_external_id"])
            check_type(argname="argument qualifier", value=qualifier, expected_type=type_hints["qualifier"])
            check_type(argname="argument use_lookup_role_for_stack_operations", value=use_lookup_role_for_stack_operations, expected_type=type_hints["use_lookup_role_for_stack_operations"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if bootstrap_stack_version_ssm_parameter is not None:
            self._values["bootstrap_stack_version_ssm_parameter"] = bootstrap_stack_version_ssm_parameter
        if bucket_prefix is not None:
            self._values["bucket_prefix"] = bucket_prefix
        if cloud_formation_execution_role is not None:
            self._values["cloud_formation_execution_role"] = cloud_formation_execution_role
        if deploy_role_additional_options is not None:
            self._values["deploy_role_additional_options"] = deploy_role_additional_options
        if deploy_role_arn is not None:
            self._values["deploy_role_arn"] = deploy_role_arn
        if deploy_role_external_id is not None:
            self._values["deploy_role_external_id"] = deploy_role_external_id
        if docker_tag_prefix is not None:
            self._values["docker_tag_prefix"] = docker_tag_prefix
        if file_asset_publishing_external_id is not None:
            self._values["file_asset_publishing_external_id"] = file_asset_publishing_external_id
        if file_asset_publishing_role_additional_options is not None:
            self._values["file_asset_publishing_role_additional_options"] = file_asset_publishing_role_additional_options
        if file_asset_publishing_role_arn is not None:
            self._values["file_asset_publishing_role_arn"] = file_asset_publishing_role_arn
        if file_assets_bucket_name is not None:
            self._values["file_assets_bucket_name"] = file_assets_bucket_name
        if generate_bootstrap_version_rule is not None:
            self._values["generate_bootstrap_version_rule"] = generate_bootstrap_version_rule
        if image_asset_publishing_external_id is not None:
            self._values["image_asset_publishing_external_id"] = image_asset_publishing_external_id
        if image_asset_publishing_role_additional_options is not None:
            self._values["image_asset_publishing_role_additional_options"] = image_asset_publishing_role_additional_options
        if image_asset_publishing_role_arn is not None:
            self._values["image_asset_publishing_role_arn"] = image_asset_publishing_role_arn
        if image_assets_repository_name is not None:
            self._values["image_assets_repository_name"] = image_assets_repository_name
        if lookup_role_additional_options is not None:
            self._values["lookup_role_additional_options"] = lookup_role_additional_options
        if lookup_role_arn is not None:
            self._values["lookup_role_arn"] = lookup_role_arn
        if lookup_role_external_id is not None:
            self._values["lookup_role_external_id"] = lookup_role_external_id
        if qualifier is not None:
            self._values["qualifier"] = qualifier
        if use_lookup_role_for_stack_operations is not None:
            self._values["use_lookup_role_for_stack_operations"] = use_lookup_role_for_stack_operations

    @builtins.property
    def bootstrap_stack_version_ssm_parameter(self) -> typing.Optional[builtins.str]:
        '''Bootstrap stack version SSM parameter.

        The placeholder ``${Qualifier}`` will be replaced with the value of qualifier.

        :default: DefaultStackSynthesizer.DEFAULT_BOOTSTRAP_STACK_VERSION_SSM_PARAMETER
        '''
        result = self._values.get("bootstrap_stack_version_ssm_parameter")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def bucket_prefix(self) -> typing.Optional[builtins.str]:
        '''bucketPrefix to use while storing S3 Assets.

        :default: - DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PREFIX
        '''
        result = self._values.get("bucket_prefix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cloud_formation_execution_role(self) -> typing.Optional[builtins.str]:
        '''The role CloudFormation will assume when deploying the Stack.

        You must supply this if you have given a non-standard name to the execution role.

        The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will
        be replaced with the values of qualifier and the stack's account and region,
        respectively.

        :default: DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN
        '''
        result = self._values.get("cloud_formation_execution_role")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def deploy_role_additional_options(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Additional options to pass to STS when assuming the deploy role.

        - ``RoleArn`` should not be used. Use the dedicated ``deployRoleArn`` property instead.
        - ``ExternalId`` should not be used. Use the dedicated ``deployRoleExternalId`` instead.
        - ``TransitiveTagKeys`` defaults to use all keys (if any) specified in ``Tags``. E.g, all tags are transitive by default.

        :default: - No additional options.

        :see: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/STS.html#assumeRole-property
        '''
        result = self._values.get("deploy_role_additional_options")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def deploy_role_arn(self) -> typing.Optional[builtins.str]:
        '''The role to assume to initiate a deployment in this environment.

        You must supply this if you have given a non-standard name to the publishing role.

        The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will
        be replaced with the values of qualifier and the stack's account and region,
        respectively.

        :default: DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN
        '''
        result = self._values.get("deploy_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def deploy_role_external_id(self) -> typing.Optional[builtins.str]:
        '''External ID to use when assuming role for cloudformation deployments.

        :default: - No external ID
        '''
        result = self._values.get("deploy_role_external_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def docker_tag_prefix(self) -> typing.Optional[builtins.str]:
        '''A prefix to use while tagging and uploading Docker images to ECR.

        This does not add any separators - the source hash will be appended to
        this string directly.

        :default: - DefaultStackSynthesizer.DEFAULT_DOCKER_ASSET_PREFIX
        '''
        result = self._values.get("docker_tag_prefix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def file_asset_publishing_external_id(self) -> typing.Optional[builtins.str]:
        '''External ID to use when assuming role for file asset publishing.

        :default: - No external ID
        '''
        result = self._values.get("file_asset_publishing_external_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def file_asset_publishing_role_additional_options(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Additional options to pass to STS when assuming the file asset publishing.

        - ``RoleArn`` should not be used. Use the dedicated ``fileAssetPublishingRoleArn`` property instead.
        - ``ExternalId`` should not be used. Use the dedicated ``fileAssetPublishingExternalId`` instead.
        - ``TransitiveTagKeys`` defaults to use all keys (if any) specified in ``Tags``. E.g, all tags are transitive by default.

        :default: - No additional options.

        :see: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/STS.html#assumeRole-property
        '''
        result = self._values.get("file_asset_publishing_role_additional_options")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def file_asset_publishing_role_arn(self) -> typing.Optional[builtins.str]:
        '''The role to use to publish file assets to the S3 bucket in this environment.

        You must supply this if you have given a non-standard name to the publishing role.

        The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will
        be replaced with the values of qualifier and the stack's account and region,
        respectively.

        :default: DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PUBLISHING_ROLE_ARN
        '''
        result = self._values.get("file_asset_publishing_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def file_assets_bucket_name(self) -> typing.Optional[builtins.str]:
        '''Name of the S3 bucket to hold file assets.

        You must supply this if you have given a non-standard name to the staging bucket.

        The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will
        be replaced with the values of qualifier and the stack's account and region,
        respectively.

        :default: DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME
        '''
        result = self._values.get("file_assets_bucket_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def generate_bootstrap_version_rule(self) -> typing.Optional[builtins.bool]:
        '''Whether to add a Rule to the stack template verifying the bootstrap stack version.

        This generally should be left set to ``true``, unless you explicitly
        want to be able to deploy to an unbootstrapped environment.

        :default: true
        '''
        result = self._values.get("generate_bootstrap_version_rule")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def image_asset_publishing_external_id(self) -> typing.Optional[builtins.str]:
        '''External ID to use when assuming role for image asset publishing.

        :default: - No external ID
        '''
        result = self._values.get("image_asset_publishing_external_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def image_asset_publishing_role_additional_options(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Additional options to pass to STS when assuming the image asset publishing.

        - ``RoleArn`` should not be used. Use the dedicated ``imageAssetPublishingRoleArn`` property instead.
        - ``ExternalId`` should not be used. Use the dedicated ``imageAssetPublishingExternalId`` instead.
        - ``TransitiveTagKeys`` defaults to use all keys (if any) specified in ``Tags``. E.g, all tags are transitive by default.

        :default: - No additional options.

        :see: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/STS.html#assumeRole-property
        '''
        result = self._values.get("image_asset_publishing_role_additional_options")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def image_asset_publishing_role_arn(self) -> typing.Optional[builtins.str]:
        '''The role to use to publish image assets to the ECR repository in this environment.

        You must supply this if you have given a non-standard name to the publishing role.

        The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will
        be replaced with the values of qualifier and the stack's account and region,
        respectively.

        :default: DefaultStackSynthesizer.DEFAULT_IMAGE_ASSET_PUBLISHING_ROLE_ARN
        '''
        result = self._values.get("image_asset_publishing_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def image_assets_repository_name(self) -> typing.Optional[builtins.str]:
        '''Name of the ECR repository to hold Docker Image assets.

        You must supply this if you have given a non-standard name to the ECR repository.

        The placeholders ``${Qualifier}``, ``${AWS::AccountId}`` and ``${AWS::Region}`` will
        be replaced with the values of qualifier and the stack's account and region,
        respectively.

        :default: DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME
        '''
        result = self._values.get("image_assets_repository_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def lookup_role_additional_options(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Additional options to pass to STS when assuming the lookup role.

        - ``RoleArn`` should not be used. Use the dedicated ``lookupRoleArn`` property instead.
        - ``ExternalId`` should not be used. Use the dedicated ``lookupRoleExternalId`` instead.
        - ``TransitiveTagKeys`` defaults to use all keys (if any) specified in ``Tags``. E.g, all tags are transitive by default.

        :default: - No additional options.

        :see: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/STS.html#assumeRole-property
        '''
        result = self._values.get("lookup_role_additional_options")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def lookup_role_arn(self) -> typing.Optional[builtins.str]:
        '''The role to use to look up values from the target AWS account during synthesis.

        :default: - None
        '''
        result = self._values.get("lookup_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def lookup_role_external_id(self) -> typing.Optional[builtins.str]:
        '''External ID to use when assuming lookup role.

        :default: - No external ID
        '''
        result = self._values.get("lookup_role_external_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def qualifier(self) -> typing.Optional[builtins.str]:
        '''Qualifier to disambiguate multiple environments in the same account.

        You can use this and leave the other naming properties empty if you have deployed
        the bootstrap environment with standard names but only different qualifiers.

        :default: - Value of context key '@aws-cdk/core:bootstrapQualifier' if set, otherwise ``DefaultStackSynthesizer.DEFAULT_QUALIFIER``
        '''
        result = self._values.get("qualifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def use_lookup_role_for_stack_operations(self) -> typing.Optional[builtins.bool]:
        '''Use the bootstrapped lookup role for (read-only) stack operations.

        Use the lookup role when performing a ``cdk diff``. If set to ``false``, the
        ``deploy role`` credentials will be used to perform a ``cdk diff``.

        Requires bootstrap stack version 8.

        :default: true
        '''
        result = self._values.get("use_lookup_role_for_stack_operations")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DefaultStackSynthesizerProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.DockerBuildOptions",
    jsii_struct_bases=[],
    name_mapping={
        "build_args": "buildArgs",
        "cache_disabled": "cacheDisabled",
        "cache_from": "cacheFrom",
        "cache_to": "cacheTo",
        "file": "file",
        "platform": "platform",
        "target_stage": "targetStage",
    },
)
class DockerBuildOptions:
    def __init__(
        self,
        *,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        cache_disabled: typing.Optional[builtins.bool] = None,
        cache_from: typing.Optional[typing.Sequence[typing.Union["DockerCacheOption", typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_to: typing.Optional[typing.Union["DockerCacheOption", typing.Dict[builtins.str, typing.Any]]] = None,
        file: typing.Optional[builtins.str] = None,
        platform: typing.Optional[builtins.str] = None,
        target_stage: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Docker build options.

        :param build_args: Build args. Default: - no build args
        :param cache_disabled: Disable the cache and pass ``--no-cache`` to the ``docker build`` command. Default: - cache is used
        :param cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from args are passed
        :param cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to args are passed
        :param file: Name of the Dockerfile, must relative to the docker build path. Default: ``Dockerfile``
        :param platform: Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*. Example value: ``linux/amd64`` Default: - no platform specified
        :param target_stage: Set build target for multi-stage container builds. Any stage defined afterwards will be ignored. Example value: ``build-env`` Default: - Build all stages defined in the Dockerfile

        :exampleMetadata: infused

        Example::

            lambda_.Function(self, "Function",
                code=lambda_.Code.from_asset("/path/to/handler",
                    bundling=BundlingOptions(
                        image=DockerImage.from_build("/path/to/dir/with/DockerFile",
                            build_args={
                                "ARG1": "value1"
                            }
                        ),
                        command=["my", "cool", "command"]
                    )
                ),
                runtime=lambda_.Runtime.PYTHON_3_9,
                handler="index.handler"
            )
        '''
        if isinstance(cache_to, dict):
            cache_to = DockerCacheOption(**cache_to)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e285633ad5a50949fcc0277b0b38d3679a006cdac41feaef60665b393932370c)
            check_type(argname="argument build_args", value=build_args, expected_type=type_hints["build_args"])
            check_type(argname="argument cache_disabled", value=cache_disabled, expected_type=type_hints["cache_disabled"])
            check_type(argname="argument cache_from", value=cache_from, expected_type=type_hints["cache_from"])
            check_type(argname="argument cache_to", value=cache_to, expected_type=type_hints["cache_to"])
            check_type(argname="argument file", value=file, expected_type=type_hints["file"])
            check_type(argname="argument platform", value=platform, expected_type=type_hints["platform"])
            check_type(argname="argument target_stage", value=target_stage, expected_type=type_hints["target_stage"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if build_args is not None:
            self._values["build_args"] = build_args
        if cache_disabled is not None:
            self._values["cache_disabled"] = cache_disabled
        if cache_from is not None:
            self._values["cache_from"] = cache_from
        if cache_to is not None:
            self._values["cache_to"] = cache_to
        if file is not None:
            self._values["file"] = file
        if platform is not None:
            self._values["platform"] = platform
        if target_stage is not None:
            self._values["target_stage"] = target_stage

    @builtins.property
    def build_args(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Build args.

        :default: - no build args
        '''
        result = self._values.get("build_args")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def cache_disabled(self) -> typing.Optional[builtins.bool]:
        '''Disable the cache and pass ``--no-cache`` to the ``docker build`` command.

        :default: - cache is used
        '''
        result = self._values.get("cache_disabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cache_from(self) -> typing.Optional[typing.List["DockerCacheOption"]]:
        '''Cache from options to pass to the ``docker build`` command.

        :default: - no cache from args are passed
        '''
        result = self._values.get("cache_from")
        return typing.cast(typing.Optional[typing.List["DockerCacheOption"]], result)

    @builtins.property
    def cache_to(self) -> typing.Optional["DockerCacheOption"]:
        '''Cache to options to pass to the ``docker build`` command.

        :default: - no cache to args are passed
        '''
        result = self._values.get("cache_to")
        return typing.cast(typing.Optional["DockerCacheOption"], result)

    @builtins.property
    def file(self) -> typing.Optional[builtins.str]:
        '''Name of the Dockerfile, must relative to the docker build path.

        :default: ``Dockerfile``
        '''
        result = self._values.get("file")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def platform(self) -> typing.Optional[builtins.str]:
        '''Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*.

        Example value: ``linux/amd64``

        :default: - no platform specified
        '''
        result = self._values.get("platform")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def target_stage(self) -> typing.Optional[builtins.str]:
        '''Set build target for multi-stage container builds. Any stage defined afterwards will be ignored.

        Example value: ``build-env``

        :default: - Build all stages defined in the Dockerfile
        '''
        result = self._values.get("target_stage")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerBuildOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DockerBuildSecret(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.DockerBuildSecret",
):
    '''Methods to build Docker CLI arguments for builds using secrets.

    Docker BuildKit must be enabled to use build secrets.

    :see: https://docs.docker.com/build/buildkit/
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        docker_build_secret = cdk.DockerBuildSecret()
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromSrc")
    @builtins.classmethod
    def from_src(cls, src: builtins.str) -> builtins.str:
        '''A Docker build secret from a file source.

        :param src: The path to the source file, relative to the build directory.

        :return: The latter half required for ``--secret``
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7d1e7678741a1e416d6b7fb3002e643dfa382c25976bfcfcbcf283b7c67f4970)
            check_type(argname="argument src", value=src, expected_type=type_hints["src"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "fromSrc", [src]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.DockerCacheOption",
    jsii_struct_bases=[],
    name_mapping={"type": "type", "params": "params"},
)
class DockerCacheOption:
    def __init__(
        self,
        *,
        type: builtins.str,
        params: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Options for configuring the Docker cache backend.

        :param type: The type of cache to use. Refer to https://docs.docker.com/build/cache/backends/ for full list of backends. Default: - unspecified
        :param params: Any parameters to pass into the docker cache backend configuration. Refer to https://docs.docker.com/build/cache/backends/ for cache backend configuration. Default: {} No options provided

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            docker_cache_option = cdk.DockerCacheOption(
                type="type",
            
                # the properties below are optional
                params={
                    "params_key": "params"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__766712fdaaa81e1448ebeec7ea9ce7cba7253ecb6623db4aed769612ac92000b)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument params", value=params, expected_type=type_hints["params"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }
        if params is not None:
            self._values["params"] = params

    @builtins.property
    def type(self) -> builtins.str:
        '''The type of cache to use.

        Refer to https://docs.docker.com/build/cache/backends/ for full list of backends.

        :default: - unspecified

        Example::

            "registry"
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def params(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Any parameters to pass into the docker cache backend configuration.

        Refer to https://docs.docker.com/build/cache/backends/ for cache backend configuration.

        :default: {} No options provided

        Example::

            # branch: str
            
            
            params = {
                "ref": f"12345678.dkr.ecr.us-west-2.amazonaws.com/cache:{branch}",
                "mode": "max"
            }
        '''
        result = self._values.get("params")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerCacheOption(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DockerImage(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.DockerImage"):
    '''A Docker image.

    :exampleMetadata: infused

    Example::

        entry = "/path/to/function"
        image = DockerImage.from_build(entry)
        
        python.PythonFunction(self, "function",
            entry=entry,
            runtime=Runtime.PYTHON_3_8,
            bundling=python.BundlingOptions(
                build_args={"PIP_INDEX_URL": "https://your.index.url/simple/", "PIP_EXTRA_INDEX_URL": "https://your.extra-index.url/simple/"}
            )
        )
    '''

    def __init__(
        self,
        image: builtins.str,
        _image_hash: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param image: The Docker image.
        :param _image_hash: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__afcabd4c5219ebfac1973d723059ba93a731b03246e398629716d62624820799)
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
            check_type(argname="argument _image_hash", value=_image_hash, expected_type=type_hints["_image_hash"])
        jsii.create(self.__class__, self, [image, _image_hash])

    @jsii.member(jsii_name="fromBuild")
    @builtins.classmethod
    def from_build(
        cls,
        path: builtins.str,
        *,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        cache_disabled: typing.Optional[builtins.bool] = None,
        cache_from: typing.Optional[typing.Sequence[typing.Union[DockerCacheOption, typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_to: typing.Optional[typing.Union[DockerCacheOption, typing.Dict[builtins.str, typing.Any]]] = None,
        file: typing.Optional[builtins.str] = None,
        platform: typing.Optional[builtins.str] = None,
        target_stage: typing.Optional[builtins.str] = None,
    ) -> "DockerImage":
        '''Builds a Docker image.

        :param path: The path to the directory containing the Docker file.
        :param build_args: Build args. Default: - no build args
        :param cache_disabled: Disable the cache and pass ``--no-cache`` to the ``docker build`` command. Default: - cache is used
        :param cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from args are passed
        :param cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to args are passed
        :param file: Name of the Dockerfile, must relative to the docker build path. Default: ``Dockerfile``
        :param platform: Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*. Example value: ``linux/amd64`` Default: - no platform specified
        :param target_stage: Set build target for multi-stage container builds. Any stage defined afterwards will be ignored. Example value: ``build-env`` Default: - Build all stages defined in the Dockerfile
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ee160dc0667b6e8f45df2c2f176a77d9d6a3ece4b83cdf7804625adf8be35a98)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        options = DockerBuildOptions(
            build_args=build_args,
            cache_disabled=cache_disabled,
            cache_from=cache_from,
            cache_to=cache_to,
            file=file,
            platform=platform,
            target_stage=target_stage,
        )

        return typing.cast("DockerImage", jsii.sinvoke(cls, "fromBuild", [path, options]))

    @jsii.member(jsii_name="fromRegistry")
    @builtins.classmethod
    def from_registry(cls, image: builtins.str) -> "DockerImage":
        '''Reference an image on DockerHub or another online registry.

        :param image: the image name.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__834262a369d8f018a0ee1c55f041b796b1abf0ba5ed44798aa1dec1ec75d2a4c)
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
        return typing.cast("DockerImage", jsii.sinvoke(cls, "fromRegistry", [image]))

    @jsii.member(jsii_name="cp")
    def cp(
        self,
        image_path: builtins.str,
        output_path: typing.Optional[builtins.str] = None,
    ) -> builtins.str:
        '''Copies a file or directory out of the Docker image to the local filesystem.

        If ``outputPath`` is omitted the destination path is a temporary directory.

        :param image_path: the path in the Docker image.
        :param output_path: the destination path for the copy operation.

        :return: the destination path
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cb61c06ba40ead83f1b09c17bb3a2b26ed82691d6069e845da6d1fc0cb20dfb3)
            check_type(argname="argument image_path", value=image_path, expected_type=type_hints["image_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
        return typing.cast(builtins.str, jsii.invoke(self, "cp", [image_path, output_path]))

    @jsii.member(jsii_name="run")
    def run(
        self,
        *,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        network: typing.Optional[builtins.str] = None,
        platform: typing.Optional[builtins.str] = None,
        security_opt: typing.Optional[builtins.str] = None,
        user: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Sequence[typing.Union["DockerVolume", typing.Dict[builtins.str, typing.Any]]]] = None,
        volumes_from: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Runs a Docker image.

        :param command: The command to run in the container. Default: - run the command defined in the image
        :param entrypoint: The entrypoint to run in the container. Default: - run the entrypoint defined in the image
        :param environment: The environment variables to pass to the container. Default: - no environment variables.
        :param network: Docker `Networking options <https://docs.docker.com/engine/reference/commandline/run/#connect-a-container-to-a-network---network>`_. Default: - no networking options
        :param platform: Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*. Example value: ``linux/amd64`` Default: - no platform specified
        :param security_opt: `Security configuration <https://docs.docker.com/engine/reference/run/#security-configuration>`_ when running the docker container. Default: - no security options
        :param user: The user to use when running the container. Default: - root or image default
        :param volumes: Docker volumes to mount. Default: - no volumes are mounted
        :param volumes_from: Where to mount the specified volumes from. Default: - no containers are specified to mount volumes from
        :param working_directory: Working directory inside the container. Default: - image default
        '''
        options = DockerRunOptions(
            command=command,
            entrypoint=entrypoint,
            environment=environment,
            network=network,
            platform=platform,
            security_opt=security_opt,
            user=user,
            volumes=volumes,
            volumes_from=volumes_from,
            working_directory=working_directory,
        )

        return typing.cast(None, jsii.invoke(self, "run", [options]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> builtins.str:
        '''Provides a stable representation of this image for JSON serialization.

        :return: The overridden image name if set or image hash name in that order
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toJSON", []))

    @builtins.property
    @jsii.member(jsii_name="image")
    def image(self) -> builtins.str:
        '''The Docker image.'''
        return typing.cast(builtins.str, jsii.get(self, "image"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.DockerImageAssetLocation",
    jsii_struct_bases=[],
    name_mapping={
        "image_uri": "imageUri",
        "repository_name": "repositoryName",
        "image_tag": "imageTag",
    },
)
class DockerImageAssetLocation:
    def __init__(
        self,
        *,
        image_uri: builtins.str,
        repository_name: builtins.str,
        image_tag: typing.Optional[builtins.str] = None,
    ) -> None:
        '''The location of the published docker image.

        This is where the image can be
        consumed at runtime.

        :param image_uri: The URI of the image in Amazon ECR (including a tag).
        :param repository_name: The name of the ECR repository.
        :param image_tag: The tag of the image in Amazon ECR. Default: - the hash of the asset, or the ``dockerTagPrefix`` concatenated with the asset hash if a ``dockerTagPrefix`` is specified in the stack synthesizer

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            docker_image_asset_location = cdk.DockerImageAssetLocation(
                image_uri="imageUri",
                repository_name="repositoryName",
            
                # the properties below are optional
                image_tag="imageTag"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__51b92762f4a556fe422660fbe3d17a86cdb642aa57d38d933aaaadde092119a4)
            check_type(argname="argument image_uri", value=image_uri, expected_type=type_hints["image_uri"])
            check_type(argname="argument repository_name", value=repository_name, expected_type=type_hints["repository_name"])
            check_type(argname="argument image_tag", value=image_tag, expected_type=type_hints["image_tag"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "image_uri": image_uri,
            "repository_name": repository_name,
        }
        if image_tag is not None:
            self._values["image_tag"] = image_tag

    @builtins.property
    def image_uri(self) -> builtins.str:
        '''The URI of the image in Amazon ECR (including a tag).'''
        result = self._values.get("image_uri")
        assert result is not None, "Required property 'image_uri' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def repository_name(self) -> builtins.str:
        '''The name of the ECR repository.'''
        result = self._values.get("repository_name")
        assert result is not None, "Required property 'repository_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def image_tag(self) -> typing.Optional[builtins.str]:
        '''The tag of the image in Amazon ECR.

        :default: - the hash of the asset, or the ``dockerTagPrefix`` concatenated with the asset hash if a ``dockerTagPrefix`` is specified in the stack synthesizer
        '''
        result = self._values.get("image_tag")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerImageAssetLocation(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.DockerImageAssetSource",
    jsii_struct_bases=[],
    name_mapping={
        "source_hash": "sourceHash",
        "asset_name": "assetName",
        "directory_name": "directoryName",
        "display_name": "displayName",
        "docker_build_args": "dockerBuildArgs",
        "docker_build_secrets": "dockerBuildSecrets",
        "docker_build_ssh": "dockerBuildSsh",
        "docker_build_target": "dockerBuildTarget",
        "docker_cache_disabled": "dockerCacheDisabled",
        "docker_cache_from": "dockerCacheFrom",
        "docker_cache_to": "dockerCacheTo",
        "docker_file": "dockerFile",
        "docker_outputs": "dockerOutputs",
        "executable": "executable",
        "network_mode": "networkMode",
        "platform": "platform",
    },
)
class DockerImageAssetSource:
    def __init__(
        self,
        *,
        source_hash: builtins.str,
        asset_name: typing.Optional[builtins.str] = None,
        directory_name: typing.Optional[builtins.str] = None,
        display_name: typing.Optional[builtins.str] = None,
        docker_build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_build_secrets: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_build_ssh: typing.Optional[builtins.str] = None,
        docker_build_target: typing.Optional[builtins.str] = None,
        docker_cache_disabled: typing.Optional[builtins.bool] = None,
        docker_cache_from: typing.Optional[typing.Sequence[typing.Union[DockerCacheOption, typing.Dict[builtins.str, typing.Any]]]] = None,
        docker_cache_to: typing.Optional[typing.Union[DockerCacheOption, typing.Dict[builtins.str, typing.Any]]] = None,
        docker_file: typing.Optional[builtins.str] = None,
        docker_outputs: typing.Optional[typing.Sequence[builtins.str]] = None,
        executable: typing.Optional[typing.Sequence[builtins.str]] = None,
        network_mode: typing.Optional[builtins.str] = None,
        platform: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param source_hash: The hash of the contents of the docker build context. This hash is used throughout the system to identify this image and avoid duplicate work in case the source did not change. NOTE: this means that if you wish to update your docker image, you must make a modification to the source (e.g. add some metadata to your Dockerfile).
        :param asset_name: Unique identifier of the docker image asset and its potential revisions. Required if using AppScopedStagingSynthesizer. Default: - no asset name
        :param directory_name: The directory where the Dockerfile is stored, must be relative to the cloud assembly root. Default: - Exactly one of ``directoryName`` and ``executable`` is required
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. Default: - The asset hash is used to display the asset
        :param docker_build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Only allowed when ``directoryName`` is specified. Default: - no build args are passed
        :param docker_build_secrets: Build secrets to pass to the ``docker build`` command. Since Docker build secrets are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Only allowed when ``directoryName`` is specified. Default: - no build secrets are passed
        :param docker_build_ssh: SSH agent socket or keys to pass to the ``docker buildx`` command. Default: - no ssh arg is passed
        :param docker_build_target: Docker target to build to. Only allowed when ``directoryName`` is specified. Default: - no target
        :param docker_cache_disabled: Disable the cache and pass ``--no-cache`` to the ``docker build`` command. Default: - cache is used
        :param docker_cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from args are passed
        :param docker_cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to args are passed
        :param docker_file: Path to the Dockerfile (relative to the directory). Only allowed when ``directoryName`` is specified. Default: - no file
        :param docker_outputs: Outputs to pass to the ``docker build`` command. Default: - no build args are passed
        :param executable: An external command that will produce the packaged asset. The command should produce the name of a local Docker image on ``stdout``. Default: - Exactly one of ``directoryName`` and ``executable`` is required
        :param network_mode: Networking mode for the RUN commands during build. *Requires Docker Engine API v1.25+*. Specify this property to build images on a specific networking mode. Default: - no networking mode specified
        :param platform: Platform to build for. *Requires Docker Buildx*. Specify this property to build images on a specific platform. Default: - no platform specified (the current machine architecture will be used)

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            docker_image_asset_source = cdk.DockerImageAssetSource(
                source_hash="sourceHash",
            
                # the properties below are optional
                asset_name="assetName",
                directory_name="directoryName",
                display_name="displayName",
                docker_build_args={
                    "docker_build_args_key": "dockerBuildArgs"
                },
                docker_build_secrets={
                    "docker_build_secrets_key": "dockerBuildSecrets"
                },
                docker_build_ssh="dockerBuildSsh",
                docker_build_target="dockerBuildTarget",
                docker_cache_disabled=False,
                docker_cache_from=[cdk.DockerCacheOption(
                    type="type",
            
                    # the properties below are optional
                    params={
                        "params_key": "params"
                    }
                )],
                docker_cache_to=cdk.DockerCacheOption(
                    type="type",
            
                    # the properties below are optional
                    params={
                        "params_key": "params"
                    }
                ),
                docker_file="dockerFile",
                docker_outputs=["dockerOutputs"],
                executable=["executable"],
                network_mode="networkMode",
                platform="platform"
            )
        '''
        if isinstance(docker_cache_to, dict):
            docker_cache_to = DockerCacheOption(**docker_cache_to)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9dc551978a96bea78751c4b4d32e87775d67d2a68f351884cf05c35d5c042160)
            check_type(argname="argument source_hash", value=source_hash, expected_type=type_hints["source_hash"])
            check_type(argname="argument asset_name", value=asset_name, expected_type=type_hints["asset_name"])
            check_type(argname="argument directory_name", value=directory_name, expected_type=type_hints["directory_name"])
            check_type(argname="argument display_name", value=display_name, expected_type=type_hints["display_name"])
            check_type(argname="argument docker_build_args", value=docker_build_args, expected_type=type_hints["docker_build_args"])
            check_type(argname="argument docker_build_secrets", value=docker_build_secrets, expected_type=type_hints["docker_build_secrets"])
            check_type(argname="argument docker_build_ssh", value=docker_build_ssh, expected_type=type_hints["docker_build_ssh"])
            check_type(argname="argument docker_build_target", value=docker_build_target, expected_type=type_hints["docker_build_target"])
            check_type(argname="argument docker_cache_disabled", value=docker_cache_disabled, expected_type=type_hints["docker_cache_disabled"])
            check_type(argname="argument docker_cache_from", value=docker_cache_from, expected_type=type_hints["docker_cache_from"])
            check_type(argname="argument docker_cache_to", value=docker_cache_to, expected_type=type_hints["docker_cache_to"])
            check_type(argname="argument docker_file", value=docker_file, expected_type=type_hints["docker_file"])
            check_type(argname="argument docker_outputs", value=docker_outputs, expected_type=type_hints["docker_outputs"])
            check_type(argname="argument executable", value=executable, expected_type=type_hints["executable"])
            check_type(argname="argument network_mode", value=network_mode, expected_type=type_hints["network_mode"])
            check_type(argname="argument platform", value=platform, expected_type=type_hints["platform"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "source_hash": source_hash,
        }
        if asset_name is not None:
            self._values["asset_name"] = asset_name
        if directory_name is not None:
            self._values["directory_name"] = directory_name
        if display_name is not None:
            self._values["display_name"] = display_name
        if docker_build_args is not None:
            self._values["docker_build_args"] = docker_build_args
        if docker_build_secrets is not None:
            self._values["docker_build_secrets"] = docker_build_secrets
        if docker_build_ssh is not None:
            self._values["docker_build_ssh"] = docker_build_ssh
        if docker_build_target is not None:
            self._values["docker_build_target"] = docker_build_target
        if docker_cache_disabled is not None:
            self._values["docker_cache_disabled"] = docker_cache_disabled
        if docker_cache_from is not None:
            self._values["docker_cache_from"] = docker_cache_from
        if docker_cache_to is not None:
            self._values["docker_cache_to"] = docker_cache_to
        if docker_file is not None:
            self._values["docker_file"] = docker_file
        if docker_outputs is not None:
            self._values["docker_outputs"] = docker_outputs
        if executable is not None:
            self._values["executable"] = executable
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if platform is not None:
            self._values["platform"] = platform

    @builtins.property
    def source_hash(self) -> builtins.str:
        '''The hash of the contents of the docker build context.

        This hash is used
        throughout the system to identify this image and avoid duplicate work
        in case the source did not change.

        NOTE: this means that if you wish to update your docker image, you
        must make a modification to the source (e.g. add some metadata to your Dockerfile).
        '''
        result = self._values.get("source_hash")
        assert result is not None, "Required property 'source_hash' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def asset_name(self) -> typing.Optional[builtins.str]:
        '''Unique identifier of the docker image asset and its potential revisions.

        Required if using AppScopedStagingSynthesizer.

        :default: - no asset name
        '''
        result = self._values.get("asset_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def directory_name(self) -> typing.Optional[builtins.str]:
        '''The directory where the Dockerfile is stored, must be relative to the cloud assembly root.

        :default: - Exactly one of ``directoryName`` and ``executable`` is required
        '''
        result = self._values.get("directory_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def display_name(self) -> typing.Optional[builtins.str]:
        '''A display name for this asset.

        If supplied, the display name will be used in locations where the asset
        identifier is printed, like in the CLI progress information.

        :default: - The asset hash is used to display the asset
        '''
        result = self._values.get("display_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def docker_build_args(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Build args to pass to the ``docker build`` command.

        Since Docker build arguments are resolved before deployment, keys and
        values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or
        ``queue.queueUrl``).

        Only allowed when ``directoryName`` is specified.

        :default: - no build args are passed
        '''
        result = self._values.get("docker_build_args")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def docker_build_secrets(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Build secrets to pass to the ``docker build`` command.

        Since Docker build secrets are resolved before deployment, keys and
        values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or
        ``queue.queueUrl``).

        Only allowed when ``directoryName`` is specified.

        :default: - no build secrets are passed
        '''
        result = self._values.get("docker_build_secrets")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def docker_build_ssh(self) -> typing.Optional[builtins.str]:
        '''SSH agent socket or keys to pass to the ``docker buildx`` command.

        :default: - no ssh arg is passed
        '''
        result = self._values.get("docker_build_ssh")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def docker_build_target(self) -> typing.Optional[builtins.str]:
        '''Docker target to build to.

        Only allowed when ``directoryName`` is specified.

        :default: - no target
        '''
        result = self._values.get("docker_build_target")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def docker_cache_disabled(self) -> typing.Optional[builtins.bool]:
        '''Disable the cache and pass ``--no-cache`` to the ``docker build`` command.

        :default: - cache is used
        '''
        result = self._values.get("docker_cache_disabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def docker_cache_from(self) -> typing.Optional[typing.List[DockerCacheOption]]:
        '''Cache from options to pass to the ``docker build`` command.

        :default: - no cache from args are passed
        '''
        result = self._values.get("docker_cache_from")
        return typing.cast(typing.Optional[typing.List[DockerCacheOption]], result)

    @builtins.property
    def docker_cache_to(self) -> typing.Optional[DockerCacheOption]:
        '''Cache to options to pass to the ``docker build`` command.

        :default: - no cache to args are passed
        '''
        result = self._values.get("docker_cache_to")
        return typing.cast(typing.Optional[DockerCacheOption], result)

    @builtins.property
    def docker_file(self) -> typing.Optional[builtins.str]:
        '''Path to the Dockerfile (relative to the directory).

        Only allowed when ``directoryName`` is specified.

        :default: - no file
        '''
        result = self._values.get("docker_file")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def docker_outputs(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Outputs to pass to the ``docker build`` command.

        :default: - no build args are passed
        '''
        result = self._values.get("docker_outputs")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def executable(self) -> typing.Optional[typing.List[builtins.str]]:
        '''An external command that will produce the packaged asset.

        The command should produce the name of a local Docker image on ``stdout``.

        :default: - Exactly one of ``directoryName`` and ``executable`` is required
        '''
        result = self._values.get("executable")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def network_mode(self) -> typing.Optional[builtins.str]:
        '''Networking mode for the RUN commands during build. *Requires Docker Engine API v1.25+*.

        Specify this property to build images on a specific networking mode.

        :default: - no networking mode specified
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def platform(self) -> typing.Optional[builtins.str]:
        '''Platform to build for. *Requires Docker Buildx*.

        Specify this property to build images on a specific platform.

        :default: - no platform specified (the current machine architecture will be used)
        '''
        result = self._values.get("platform")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerImageAssetSource(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.DockerRunOptions",
    jsii_struct_bases=[],
    name_mapping={
        "command": "command",
        "entrypoint": "entrypoint",
        "environment": "environment",
        "network": "network",
        "platform": "platform",
        "security_opt": "securityOpt",
        "user": "user",
        "volumes": "volumes",
        "volumes_from": "volumesFrom",
        "working_directory": "workingDirectory",
    },
)
class DockerRunOptions:
    def __init__(
        self,
        *,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        network: typing.Optional[builtins.str] = None,
        platform: typing.Optional[builtins.str] = None,
        security_opt: typing.Optional[builtins.str] = None,
        user: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Sequence[typing.Union["DockerVolume", typing.Dict[builtins.str, typing.Any]]]] = None,
        volumes_from: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Docker run options.

        :param command: The command to run in the container. Default: - run the command defined in the image
        :param entrypoint: The entrypoint to run in the container. Default: - run the entrypoint defined in the image
        :param environment: The environment variables to pass to the container. Default: - no environment variables.
        :param network: Docker `Networking options <https://docs.docker.com/engine/reference/commandline/run/#connect-a-container-to-a-network---network>`_. Default: - no networking options
        :param platform: Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*. Example value: ``linux/amd64`` Default: - no platform specified
        :param security_opt: `Security configuration <https://docs.docker.com/engine/reference/run/#security-configuration>`_ when running the docker container. Default: - no security options
        :param user: The user to use when running the container. Default: - root or image default
        :param volumes: Docker volumes to mount. Default: - no volumes are mounted
        :param volumes_from: Where to mount the specified volumes from. Default: - no containers are specified to mount volumes from
        :param working_directory: Working directory inside the container. Default: - image default

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            docker_run_options = cdk.DockerRunOptions(
                command=["command"],
                entrypoint=["entrypoint"],
                environment={
                    "environment_key": "environment"
                },
                network="network",
                platform="platform",
                security_opt="securityOpt",
                user="user",
                volumes=[cdk.DockerVolume(
                    container_path="containerPath",
                    host_path="hostPath",
            
                    # the properties below are optional
                    consistency=cdk.DockerVolumeConsistency.CONSISTENT
                )],
                volumes_from=["volumesFrom"],
                working_directory="workingDirectory"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e06b5fa3fffdf538ee232f49e80ef834239758fe0bf1634f5ac7a528e98b45b)
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
            check_type(argname="argument entrypoint", value=entrypoint, expected_type=type_hints["entrypoint"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument network", value=network, expected_type=type_hints["network"])
            check_type(argname="argument platform", value=platform, expected_type=type_hints["platform"])
            check_type(argname="argument security_opt", value=security_opt, expected_type=type_hints["security_opt"])
            check_type(argname="argument user", value=user, expected_type=type_hints["user"])
            check_type(argname="argument volumes", value=volumes, expected_type=type_hints["volumes"])
            check_type(argname="argument volumes_from", value=volumes_from, expected_type=type_hints["volumes_from"])
            check_type(argname="argument working_directory", value=working_directory, expected_type=type_hints["working_directory"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if command is not None:
            self._values["command"] = command
        if entrypoint is not None:
            self._values["entrypoint"] = entrypoint
        if environment is not None:
            self._values["environment"] = environment
        if network is not None:
            self._values["network"] = network
        if platform is not None:
            self._values["platform"] = platform
        if security_opt is not None:
            self._values["security_opt"] = security_opt
        if user is not None:
            self._values["user"] = user
        if volumes is not None:
            self._values["volumes"] = volumes
        if volumes_from is not None:
            self._values["volumes_from"] = volumes_from
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The command to run in the container.

        :default: - run the command defined in the image
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entrypoint(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The entrypoint to run in the container.

        :default: - run the entrypoint defined in the image
        '''
        result = self._values.get("entrypoint")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to pass to the container.

        :default: - no environment variables.
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def network(self) -> typing.Optional[builtins.str]:
        '''Docker `Networking options <https://docs.docker.com/engine/reference/commandline/run/#connect-a-container-to-a-network---network>`_.

        :default: - no networking options
        '''
        result = self._values.get("network")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def platform(self) -> typing.Optional[builtins.str]:
        '''Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*.

        Example value: ``linux/amd64``

        :default: - no platform specified
        '''
        result = self._values.get("platform")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def security_opt(self) -> typing.Optional[builtins.str]:
        '''`Security configuration <https://docs.docker.com/engine/reference/run/#security-configuration>`_ when running the docker container.

        :default: - no security options
        '''
        result = self._values.get("security_opt")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def user(self) -> typing.Optional[builtins.str]:
        '''The user to use when running the container.

        :default: - root or image default
        '''
        result = self._values.get("user")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def volumes(self) -> typing.Optional[typing.List["DockerVolume"]]:
        '''Docker volumes to mount.

        :default: - no volumes are mounted
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.List["DockerVolume"]], result)

    @builtins.property
    def volumes_from(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Where to mount the specified volumes from.

        :default: - no containers are specified to mount volumes from

        :see: https://docs.docker.com/engine/reference/commandline/run/#mount-volumes-from-container---volumes-from
        '''
        result = self._values.get("volumes_from")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''Working directory inside the container.

        :default: - image default
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerRunOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.DockerVolume",
    jsii_struct_bases=[],
    name_mapping={
        "container_path": "containerPath",
        "host_path": "hostPath",
        "consistency": "consistency",
    },
)
class DockerVolume:
    def __init__(
        self,
        *,
        container_path: builtins.str,
        host_path: builtins.str,
        consistency: typing.Optional["DockerVolumeConsistency"] = None,
    ) -> None:
        '''A Docker volume.

        :param container_path: The path where the file or directory is mounted in the container.
        :param host_path: The path to the file or directory on the host machine.
        :param consistency: Mount consistency. Only applicable for macOS Default: DockerConsistency.DELEGATED

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            docker_volume = cdk.DockerVolume(
                container_path="containerPath",
                host_path="hostPath",
            
                # the properties below are optional
                consistency=cdk.DockerVolumeConsistency.CONSISTENT
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d274148e728fd69c5453232f9e0da6fcf792a946fc0b1fc1eba997ab152010a7)
            check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
            check_type(argname="argument host_path", value=host_path, expected_type=type_hints["host_path"])
            check_type(argname="argument consistency", value=consistency, expected_type=type_hints["consistency"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "container_path": container_path,
            "host_path": host_path,
        }
        if consistency is not None:
            self._values["consistency"] = consistency

    @builtins.property
    def container_path(self) -> builtins.str:
        '''The path where the file or directory is mounted in the container.'''
        result = self._values.get("container_path")
        assert result is not None, "Required property 'container_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def host_path(self) -> builtins.str:
        '''The path to the file or directory on the host machine.'''
        result = self._values.get("host_path")
        assert result is not None, "Required property 'host_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def consistency(self) -> typing.Optional["DockerVolumeConsistency"]:
        '''Mount consistency.

        Only applicable for macOS

        :default: DockerConsistency.DELEGATED

        :see: https://docs.docker.com/storage/bind-mounts/#configure-mount-consistency-for-macos
        '''
        result = self._values.get("consistency")
        return typing.cast(typing.Optional["DockerVolumeConsistency"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerVolume(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.DockerVolumeConsistency")
class DockerVolumeConsistency(enum.Enum):
    '''Supported Docker volume consistency types.

    Only valid on macOS due to the way file storage works on Mac
    '''

    CONSISTENT = "CONSISTENT"
    '''Read/write operations inside the Docker container are applied immediately on the mounted host machine volumes.'''
    DELEGATED = "DELEGATED"
    '''Read/write operations on mounted Docker volumes are first written inside the container and then synchronized to the host machine.'''
    CACHED = "CACHED"
    '''Read/write operations on mounted Docker volumes are first applied on the host machine and then synchronized to the container.'''


class Duration(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.Duration"):
    '''Represents a length of time.

    The amount can be specified either as a literal value (e.g: ``10``) which
    cannot be negative, or as an unresolved number token.

    When the amount is passed as a token, unit conversion is not possible.

    :exampleMetadata: infused

    Example::

        # my_role: iam.Role
        
        cr.AwsCustomResource(self, "Customized",
            role=my_role,  # must be assumable by the `lambda.amazonaws.com` service principal
            timeout=Duration.minutes(10),  # defaults to 2 minutes
            memory_size=1025,  # defaults to 512 if installLatestAwsSdk is true
            log_group=logs.LogGroup(self, "AwsCustomResourceLogs",
                retention=logs.RetentionDays.ONE_DAY
            ),
            function_name="my-custom-name",  # defaults to a CloudFormation generated name
            removal_policy=RemovalPolicy.RETAIN,  # defaults to `RemovalPolicy.DESTROY`
            policy=cr.AwsCustomResourcePolicy.from_sdk_calls(
                resources=cr.AwsCustomResourcePolicy.ANY_RESOURCE
            )
        )
    '''

    @jsii.member(jsii_name="days")
    @builtins.classmethod
    def days(cls, amount: jsii.Number) -> "Duration":
        '''Create a Duration representing an amount of days.

        :param amount: the amount of Days the ``Duration`` will represent.

        :return: a new ``Duration`` representing ``amount`` Days.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6d35b52ff87a2ed50c3b59d28ab5eae57e7c5e956544b6b955e397e12d9ed376)
            check_type(argname="argument amount", value=amount, expected_type=type_hints["amount"])
        return typing.cast("Duration", jsii.sinvoke(cls, "days", [amount]))

    @jsii.member(jsii_name="hours")
    @builtins.classmethod
    def hours(cls, amount: jsii.Number) -> "Duration":
        '''Create a Duration representing an amount of hours.

        :param amount: the amount of Hours the ``Duration`` will represent.

        :return: a new ``Duration`` representing ``amount`` Hours.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4971cd3466a4bda25651dc63c9c3ac5831595be36ca40b3f673003d1904cf222)
            check_type(argname="argument amount", value=amount, expected_type=type_hints["amount"])
        return typing.cast("Duration", jsii.sinvoke(cls, "hours", [amount]))

    @jsii.member(jsii_name="millis")
    @builtins.classmethod
    def millis(cls, amount: jsii.Number) -> "Duration":
        '''Create a Duration representing an amount of milliseconds.

        :param amount: the amount of Milliseconds the ``Duration`` will represent.

        :return: a new ``Duration`` representing ``amount`` ms.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__007deebc2aa960ff9a2820096711cbc9245edecd93b75496bab54f508ba71b82)
            check_type(argname="argument amount", value=amount, expected_type=type_hints["amount"])
        return typing.cast("Duration", jsii.sinvoke(cls, "millis", [amount]))

    @jsii.member(jsii_name="minutes")
    @builtins.classmethod
    def minutes(cls, amount: jsii.Number) -> "Duration":
        '''Create a Duration representing an amount of minutes.

        :param amount: the amount of Minutes the ``Duration`` will represent.

        :return: a new ``Duration`` representing ``amount`` Minutes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9f70a9d736969af4fb67f92aeb4042a361b80edda93a6c43fe44b68de83c7f69)
            check_type(argname="argument amount", value=amount, expected_type=type_hints["amount"])
        return typing.cast("Duration", jsii.sinvoke(cls, "minutes", [amount]))

    @jsii.member(jsii_name="parse")
    @builtins.classmethod
    def parse(cls, duration: builtins.str) -> "Duration":
        '''Parse a period formatted according to the ISO 8601 standard.

        Days are the largest ISO duration supported, i.e.,
        weeks, months, and years are not supported.

        :param duration: an ISO-formatted duration to be parsed.

        :return: the parsed ``Duration``.

        :see: https://www.iso.org/standard/70907.html

        Example::

            # This represents 1 day, 2 hours, 3 minutes, 4 seconds, and 567 milliseconds.
            "P1DT2H3M4.567S"
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c524b27dfe746532718570f907f5cd9142c9c49d701f766c3c586fa9a766c4d)
            check_type(argname="argument duration", value=duration, expected_type=type_hints["duration"])
        return typing.cast("Duration", jsii.sinvoke(cls, "parse", [duration]))

    @jsii.member(jsii_name="seconds")
    @builtins.classmethod
    def seconds(cls, amount: jsii.Number) -> "Duration":
        '''Create a Duration representing an amount of seconds.

        :param amount: the amount of Seconds the ``Duration`` will represent.

        :return: a new ``Duration`` representing ``amount`` Seconds.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0b3e911777f25164a2e8f5c4b2244a3c01a3b10cf7293e74fc6231343fa015b9)
            check_type(argname="argument amount", value=amount, expected_type=type_hints["amount"])
        return typing.cast("Duration", jsii.sinvoke(cls, "seconds", [amount]))

    @jsii.member(jsii_name="formatTokenToNumber")
    def format_token_to_number(self) -> builtins.str:
        '''Returns stringified number of duration.'''
        return typing.cast(builtins.str, jsii.invoke(self, "formatTokenToNumber", []))

    @jsii.member(jsii_name="isUnresolved")
    def is_unresolved(self) -> builtins.bool:
        '''Checks if duration is a token or a resolvable object.'''
        return typing.cast(builtins.bool, jsii.invoke(self, "isUnresolved", []))

    @jsii.member(jsii_name="minus")
    def minus(self, rhs: "Duration") -> "Duration":
        '''Substract two Durations together.

        :param rhs: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b53234d382f75e3ee8969215583a45bb1ffe95f1988426b3905df37377686706)
            check_type(argname="argument rhs", value=rhs, expected_type=type_hints["rhs"])
        return typing.cast("Duration", jsii.invoke(self, "minus", [rhs]))

    @jsii.member(jsii_name="plus")
    def plus(self, rhs: "Duration") -> "Duration":
        '''Add two Durations together.

        :param rhs: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__61726a74091f076f8854079d8bbffd3760a58132ff9a1b0a0074e4f29448e488)
            check_type(argname="argument rhs", value=rhs, expected_type=type_hints["rhs"])
        return typing.cast("Duration", jsii.invoke(self, "plus", [rhs]))

    @jsii.member(jsii_name="toDays")
    def to_days(
        self,
        *,
        integral: typing.Optional[builtins.bool] = None,
    ) -> jsii.Number:
        '''Return the total number of days in this Duration.

        :param integral: If ``true``, conversions into a larger time unit (e.g. ``Seconds`` to ``Minutes``) will fail if the result is not an integer. Default: true

        :return: the value of this ``Duration`` expressed in Days.
        '''
        opts = TimeConversionOptions(integral=integral)

        return typing.cast(jsii.Number, jsii.invoke(self, "toDays", [opts]))

    @jsii.member(jsii_name="toHours")
    def to_hours(
        self,
        *,
        integral: typing.Optional[builtins.bool] = None,
    ) -> jsii.Number:
        '''Return the total number of hours in this Duration.

        :param integral: If ``true``, conversions into a larger time unit (e.g. ``Seconds`` to ``Minutes``) will fail if the result is not an integer. Default: true

        :return: the value of this ``Duration`` expressed in Hours.
        '''
        opts = TimeConversionOptions(integral=integral)

        return typing.cast(jsii.Number, jsii.invoke(self, "toHours", [opts]))

    @jsii.member(jsii_name="toHumanString")
    def to_human_string(self) -> builtins.str:
        '''Turn this duration into a human-readable string.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toHumanString", []))

    @jsii.member(jsii_name="toIsoString")
    def to_iso_string(self) -> builtins.str:
        '''Return an ISO 8601 representation of this period.

        :return: a string starting with 'P' describing the period

        :see: https://www.iso.org/standard/70907.html
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toIsoString", []))

    @jsii.member(jsii_name="toMilliseconds")
    def to_milliseconds(
        self,
        *,
        integral: typing.Optional[builtins.bool] = None,
    ) -> jsii.Number:
        '''Return the total number of milliseconds in this Duration.

        :param integral: If ``true``, conversions into a larger time unit (e.g. ``Seconds`` to ``Minutes``) will fail if the result is not an integer. Default: true

        :return: the value of this ``Duration`` expressed in Milliseconds.
        '''
        opts = TimeConversionOptions(integral=integral)

        return typing.cast(jsii.Number, jsii.invoke(self, "toMilliseconds", [opts]))

    @jsii.member(jsii_name="toMinutes")
    def to_minutes(
        self,
        *,
        integral: typing.Optional[builtins.bool] = None,
    ) -> jsii.Number:
        '''Return the total number of minutes in this Duration.

        :param integral: If ``true``, conversions into a larger time unit (e.g. ``Seconds`` to ``Minutes``) will fail if the result is not an integer. Default: true

        :return: the value of this ``Duration`` expressed in Minutes.
        '''
        opts = TimeConversionOptions(integral=integral)

        return typing.cast(jsii.Number, jsii.invoke(self, "toMinutes", [opts]))

    @jsii.member(jsii_name="toSeconds")
    def to_seconds(
        self,
        *,
        integral: typing.Optional[builtins.bool] = None,
    ) -> jsii.Number:
        '''Return the total number of seconds in this Duration.

        :param integral: If ``true``, conversions into a larger time unit (e.g. ``Seconds`` to ``Minutes``) will fail if the result is not an integer. Default: true

        :return: the value of this ``Duration`` expressed in Seconds.
        '''
        opts = TimeConversionOptions(integral=integral)

        return typing.cast(jsii.Number, jsii.invoke(self, "toSeconds", [opts]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Returns a string representation of this ``Duration``.

        This is is never the right function to use when you want to use the ``Duration``
        object in a template. Use ``toSeconds()``, ``toMinutes()``, ``toDays()``, etc. instead.
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.member(jsii_name="unitLabel")
    def unit_label(self) -> builtins.str:
        '''Returns unit of the duration.'''
        return typing.cast(builtins.str, jsii.invoke(self, "unitLabel", []))


@jsii.data_type(
    jsii_type="aws-cdk-lib.EncodingOptions",
    jsii_struct_bases=[],
    name_mapping={"display_hint": "displayHint"},
)
class EncodingOptions:
    def __init__(self, *, display_hint: typing.Optional[builtins.str] = None) -> None:
        '''Properties to string encodings.

        :param display_hint: A hint for the Token's purpose when stringifying it.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            encoding_options = cdk.EncodingOptions(
                display_hint="displayHint"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6df869dd9914b4e4179a939e6f293b7e954c45075b45db7793ac8114e7a465ce)
            check_type(argname="argument display_hint", value=display_hint, expected_type=type_hints["display_hint"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if display_hint is not None:
            self._values["display_hint"] = display_hint

    @builtins.property
    def display_hint(self) -> typing.Optional[builtins.str]:
        '''A hint for the Token's purpose when stringifying it.'''
        result = self._values.get("display_hint")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EncodingOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.Environment",
    jsii_struct_bases=[],
    name_mapping={"account": "account", "region": "region"},
)
class Environment:
    def __init__(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        region: typing.Optional[builtins.str] = None,
    ) -> None:
        '''The deployment environment for a stack.

        :param account: The AWS account ID for this environment. This can be either a concrete value such as ``585191031104`` or ``Aws.ACCOUNT_ID`` which indicates that account ID will only be determined during deployment (it will resolve to the CloudFormation intrinsic ``{"Ref":"AWS::AccountId"}``). Note that certain features, such as cross-stack references and environmental context providers require concrete region information and will cause this stack to emit synthesis errors. Default: Aws.ACCOUNT_ID which means that the stack will be account-agnostic.
        :param region: The AWS region for this environment. This can be either a concrete value such as ``eu-west-2`` or ``Aws.REGION`` which indicates that account ID will only be determined during deployment (it will resolve to the CloudFormation intrinsic ``{"Ref":"AWS::Region"}``). Note that certain features, such as cross-stack references and environmental context providers require concrete region information and will cause this stack to emit synthesis errors. Default: Aws.REGION which means that the stack will be region-agnostic.

        :exampleMetadata: infused

        Example::

            import aws_cdk as cdk
            import aws_cdk.aws_cloudwatch as cloudwatch
            
            
            app = cdk.App()
            stack = cdk.Stack(app, "Stack", env=cdk.Environment(region="us-west-2"))
            
            global_table = dynamodb.TableV2(stack, "GlobalTable",
                partition_key=dynamodb.Attribute(name="pk", type=dynamodb.AttributeType.STRING),
                replicas=[dynamodb.ReplicaTableProps(region="us-east-1"), dynamodb.ReplicaTableProps(region="us-east-2")
                ]
            )
            
            # metric is only for the table in us-west-2
            metric = global_table.metric_consumed_read_capacity_units()
            
            cloudwatch.Alarm(self, "Alarm",
                metric=metric,
                evaluation_periods=1,
                threshold=1
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__779551ef0a4b144070fd2c3e88ff076e32ad12d30facdc65a940b7a8791f27c6)
            check_type(argname="argument account", value=account, expected_type=type_hints["account"])
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if account is not None:
            self._values["account"] = account
        if region is not None:
            self._values["region"] = region

    @builtins.property
    def account(self) -> typing.Optional[builtins.str]:
        '''The AWS account ID for this environment.

        This can be either a concrete value such as ``585191031104`` or ``Aws.ACCOUNT_ID`` which
        indicates that account ID will only be determined during deployment (it
        will resolve to the CloudFormation intrinsic ``{"Ref":"AWS::AccountId"}``).
        Note that certain features, such as cross-stack references and
        environmental context providers require concrete region information and
        will cause this stack to emit synthesis errors.

        :default: Aws.ACCOUNT_ID which means that the stack will be account-agnostic.
        '''
        result = self._values.get("account")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def region(self) -> typing.Optional[builtins.str]:
        '''The AWS region for this environment.

        This can be either a concrete value such as ``eu-west-2`` or ``Aws.REGION``
        which indicates that account ID will only be determined during deployment
        (it will resolve to the CloudFormation intrinsic ``{"Ref":"AWS::Region"}``).
        Note that certain features, such as cross-stack references and
        environmental context providers require concrete region information and
        will cause this stack to emit synthesis errors.

        :default: Aws.REGION which means that the stack will be region-agnostic.
        '''
        result = self._values.get("region")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Environment(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Errors(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.Errors"):
    '''Helper to check if an error is of a certain type.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        errors = cdk.Errors()
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="isAssertionError")
    @builtins.classmethod
    def is_assertion_error(cls, x: typing.Any) -> builtins.bool:
        '''Test whether the given error is a AssertionError.

        An AssertionError is thrown when an assertion fails.

        :param x: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4b06766267738419297aaf93a2368e0c38c274adc14053ce29fd653d61c4cef4)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isAssertionError", [x]))

    @jsii.member(jsii_name="isCloudAssemblyError")
    @builtins.classmethod
    def is_cloud_assembly_error(cls, x: typing.Any) -> builtins.bool:
        '''Test whether the given error is a CloudAssemblyError.

        A CloudAssemblyError is thrown for unexpected problems with the synthesized assembly.

        :param x: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1cadecf7c75d3f55ff2fcddf8a0d000d9f8e54b48218f1336d97358901d38c0b)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isCloudAssemblyError", [x]))

    @jsii.member(jsii_name="isConstructError")
    @builtins.classmethod
    def is_construct_error(cls, x: typing.Any) -> builtins.bool:
        '''Test whether the given errors is a ConstructionError.

        A ConstructionError is a generic error that will be thrown during the App construction or synthesis.
        To check for more specific errors, use the respective methods.

        :param x: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c62bc5a6fe032556ae3c32e105ddaad3f4ca89286fd26884af6347d3364c5ae)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isConstructError", [x]))

    @jsii.member(jsii_name="isValidationError")
    @builtins.classmethod
    def is_validation_error(cls, x: typing.Any) -> builtins.bool:
        '''Test whether the given error is a ValidationError.

        A ValidationError is thrown when input props are failing to pass the rules of the construct.
        It usually means the underlying CloudFormation resource(s) would not deploy with a given configuration.

        :param x: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c9afe20487c547932befb7a2f6567c38793e8d7df55f47046e1cfb408408d733)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isValidationError", [x]))


class Expiration(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.Expiration"):
    '''Represents a date of expiration.

    The amount can be specified either as a Date object, timestamp, Duration or string.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        expiration = cdk.Expiration.after(cdk.Duration.minutes(30))
    '''

    @jsii.member(jsii_name="after")
    @builtins.classmethod
    def after(cls, t: Duration) -> "Expiration":
        '''Expire once the specified duration has passed since deployment time.

        :param t: the duration to wait before expiring.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__086ea26442655ff69dcf03ab98e8412412bd4dc9ce30d181e3a559323dcede99)
            check_type(argname="argument t", value=t, expected_type=type_hints["t"])
        return typing.cast("Expiration", jsii.sinvoke(cls, "after", [t]))

    @jsii.member(jsii_name="atDate")
    @builtins.classmethod
    def at_date(cls, d: datetime.datetime) -> "Expiration":
        '''Expire at the specified date.

        :param d: date to expire at.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__60d2d587f79bb343f8bbbcb98d52686c89c7d8f870797316e20aa1999b5b45dc)
            check_type(argname="argument d", value=d, expected_type=type_hints["d"])
        return typing.cast("Expiration", jsii.sinvoke(cls, "atDate", [d]))

    @jsii.member(jsii_name="atTimestamp")
    @builtins.classmethod
    def at_timestamp(cls, t: jsii.Number) -> "Expiration":
        '''Expire at the specified timestamp.

        :param t: timestamp in unix milliseconds.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e41b659a7e5fb0d4b671745db5a1b7f56e8c8fbe0476e736b514c5d94dc7ea05)
            check_type(argname="argument t", value=t, expected_type=type_hints["t"])
        return typing.cast("Expiration", jsii.sinvoke(cls, "atTimestamp", [t]))

    @jsii.member(jsii_name="fromString")
    @builtins.classmethod
    def from_string(cls, s: builtins.str) -> "Expiration":
        '''Expire at specified date, represented as a string.

        :param s: the string that represents date to expire at.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__10eb28faeee7e95390800c5e4d339f10c402a59a71e74cfdf390d458d7c6f148)
            check_type(argname="argument s", value=s, expected_type=type_hints["s"])
        return typing.cast("Expiration", jsii.sinvoke(cls, "fromString", [s]))

    @jsii.member(jsii_name="isAfter")
    def is_after(self, t: Duration) -> builtins.bool:
        '''Check if Expiration expires after input.

        :param t: the duration to check against.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e35ded8f0cea6e064fdf08f1b09e709340c46045a992438e9091ae73f7c07e9f)
            check_type(argname="argument t", value=t, expected_type=type_hints["t"])
        return typing.cast(builtins.bool, jsii.invoke(self, "isAfter", [t]))

    @jsii.member(jsii_name="isBefore")
    def is_before(self, t: Duration) -> builtins.bool:
        '''Check if Expiration expires before input.

        :param t: the duration to check against.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dc6818df505088eacca22c0f0266f3c1617959e13b5241f901c71411d0042bdc)
            check_type(argname="argument t", value=t, expected_type=type_hints["t"])
        return typing.cast(builtins.bool, jsii.invoke(self, "isBefore", [t]))

    @jsii.member(jsii_name="toEpoch")
    def to_epoch(self) -> jsii.Number:
        '''Expiration Value in a formatted Unix Epoch Time in seconds.'''
        return typing.cast(jsii.Number, jsii.invoke(self, "toEpoch", []))

    @builtins.property
    @jsii.member(jsii_name="date")
    def date(self) -> datetime.datetime:
        '''Expiration value as a Date object.'''
        return typing.cast(datetime.datetime, jsii.get(self, "date"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.ExportValueOptions",
    jsii_struct_bases=[],
    name_mapping={"description": "description", "name": "name"},
)
class ExportValueOptions:
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options for the ``stack.exportValue()`` method.

        :param description: The description of the outputs. Default: - No description
        :param name: The name of the export to create. Default: - A name is automatically chosen

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            export_value_options = cdk.ExportValueOptions(
                description="description",
                name="name"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__03cd5df1b4b793501aa18abc1ba0ae9423e6661d00a8437be18434d086a1ba09)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if name is not None:
            self._values["name"] = name

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the outputs.

        :default: - No description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of the export to create.

        :default: - A name is automatically chosen
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExportValueOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class FeatureFlags(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.FeatureFlags"):
    '''Features that are implemented behind a flag in order to preserve backwards compatibility for existing apps.

    The list of flags are available in the
    ``aws-cdk-lib/cx-api`` module.

    The state of the flag for this application is stored as a CDK context variable.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        feature_flags = cdk.FeatureFlags.of(self)
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, scope: _constructs_77d1e7e8.IConstruct) -> "FeatureFlags":
        '''Inspect feature flags on the construct node's context.

        :param scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__732604d3e1707214238ffd36606fc4b1e8d8b3e7c572b68abef7ecd85e3cea54)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("FeatureFlags", jsii.sinvoke(cls, "of", [scope]))

    @jsii.member(jsii_name="isEnabled")
    def is_enabled(self, feature_flag: builtins.str) -> typing.Optional[builtins.bool]:
        '''Check whether a feature flag is enabled.

        If configured, the flag is present in
        the construct node context. Falls back to the defaults defined in the ``cx-api``
        module.

        :param feature_flag: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5f9a58ffd4a4395222a034ab55211a0310f41607dd92bdff98df5adf5c2e2e5a)
            check_type(argname="argument feature_flag", value=feature_flag, expected_type=type_hints["feature_flag"])
        return typing.cast(typing.Optional[builtins.bool], jsii.invoke(self, "isEnabled", [feature_flag]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.FileAssetLocation",
    jsii_struct_bases=[],
    name_mapping={
        "bucket_name": "bucketName",
        "http_url": "httpUrl",
        "object_key": "objectKey",
        "s3_object_url": "s3ObjectUrl",
        "kms_key_arn": "kmsKeyArn",
        "s3_object_url_with_placeholders": "s3ObjectUrlWithPlaceholders",
    },
)
class FileAssetLocation:
    def __init__(
        self,
        *,
        bucket_name: builtins.str,
        http_url: builtins.str,
        object_key: builtins.str,
        s3_object_url: builtins.str,
        kms_key_arn: typing.Optional[builtins.str] = None,
        s3_object_url_with_placeholders: typing.Optional[builtins.str] = None,
    ) -> None:
        '''The location of the published file asset.

        This is where the asset
        can be consumed at runtime.

        :param bucket_name: The name of the Amazon S3 bucket.
        :param http_url: The HTTP URL of this asset on Amazon S3. This value suitable for inclusion in a CloudFormation template, and may be an encoded token. Example value: ``https://s3-us-east-1.amazonaws.com/mybucket/myobject``
        :param object_key: The Amazon S3 object key.
        :param s3_object_url: The S3 URL of this asset on Amazon S3. This value suitable for inclusion in a CloudFormation template, and may be an encoded token. Example value: ``s3://mybucket/myobject``
        :param kms_key_arn: The ARN of the KMS key used to encrypt the file asset bucket, if any. The CDK bootstrap stack comes with a key policy that does not require setting this property, so you only need to set this property if you have customized the bootstrap stack to require it. Default: - Asset bucket is not encrypted, or decryption permissions are defined by a Key Policy.
        :param s3_object_url_with_placeholders: Like ``s3ObjectUrl``, but not suitable for CloudFormation consumption. If there are placeholders in the S3 URL, they will be returned un-replaced and un-evaluated. Default: - This feature cannot be used

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            file_asset_location = cdk.FileAssetLocation(
                bucket_name="bucketName",
                http_url="httpUrl",
                object_key="objectKey",
                s3_object_url="s3ObjectUrl",
            
                # the properties below are optional
                kms_key_arn="kmsKeyArn",
                s3_object_url_with_placeholders="s3ObjectUrlWithPlaceholders"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c6040a289d602953238a210c76c205ec629f4246f10403569f27eaf5b194f958)
            check_type(argname="argument bucket_name", value=bucket_name, expected_type=type_hints["bucket_name"])
            check_type(argname="argument http_url", value=http_url, expected_type=type_hints["http_url"])
            check_type(argname="argument object_key", value=object_key, expected_type=type_hints["object_key"])
            check_type(argname="argument s3_object_url", value=s3_object_url, expected_type=type_hints["s3_object_url"])
            check_type(argname="argument kms_key_arn", value=kms_key_arn, expected_type=type_hints["kms_key_arn"])
            check_type(argname="argument s3_object_url_with_placeholders", value=s3_object_url_with_placeholders, expected_type=type_hints["s3_object_url_with_placeholders"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "bucket_name": bucket_name,
            "http_url": http_url,
            "object_key": object_key,
            "s3_object_url": s3_object_url,
        }
        if kms_key_arn is not None:
            self._values["kms_key_arn"] = kms_key_arn
        if s3_object_url_with_placeholders is not None:
            self._values["s3_object_url_with_placeholders"] = s3_object_url_with_placeholders

    @builtins.property
    def bucket_name(self) -> builtins.str:
        '''The name of the Amazon S3 bucket.'''
        result = self._values.get("bucket_name")
        assert result is not None, "Required property 'bucket_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def http_url(self) -> builtins.str:
        '''The HTTP URL of this asset on Amazon S3.

        This value suitable for inclusion in a CloudFormation template, and
        may be an encoded token.

        Example value: ``https://s3-us-east-1.amazonaws.com/mybucket/myobject``
        '''
        result = self._values.get("http_url")
        assert result is not None, "Required property 'http_url' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def object_key(self) -> builtins.str:
        '''The Amazon S3 object key.'''
        result = self._values.get("object_key")
        assert result is not None, "Required property 'object_key' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def s3_object_url(self) -> builtins.str:
        '''The S3 URL of this asset on Amazon S3.

        This value suitable for inclusion in a CloudFormation template, and
        may be an encoded token.

        Example value: ``s3://mybucket/myobject``
        '''
        result = self._values.get("s3_object_url")
        assert result is not None, "Required property 's3_object_url' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def kms_key_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the KMS key used to encrypt the file asset bucket, if any.

        The CDK bootstrap stack comes with a key policy that does not require
        setting this property, so you only need to set this property if you
        have customized the bootstrap stack to require it.

        :default:

        - Asset bucket is not encrypted, or decryption permissions are
        defined by a Key Policy.
        '''
        result = self._values.get("kms_key_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def s3_object_url_with_placeholders(self) -> typing.Optional[builtins.str]:
        '''Like ``s3ObjectUrl``, but not suitable for CloudFormation consumption.

        If there are placeholders in the S3 URL, they will be returned un-replaced
        and un-evaluated.

        :default: - This feature cannot be used
        '''
        result = self._values.get("s3_object_url_with_placeholders")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FileAssetLocation(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.FileAssetPackaging")
class FileAssetPackaging(enum.Enum):
    '''Packaging modes for file assets.'''

    ZIP_DIRECTORY = "ZIP_DIRECTORY"
    '''The asset source path points to a directory, which should be archived using zip and and then uploaded to Amazon S3.'''
    FILE = "FILE"
    '''The asset source path points to a single file, which should be uploaded to Amazon S3.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.FileAssetSource",
    jsii_struct_bases=[],
    name_mapping={
        "source_hash": "sourceHash",
        "deploy_time": "deployTime",
        "display_name": "displayName",
        "executable": "executable",
        "file_name": "fileName",
        "packaging": "packaging",
    },
)
class FileAssetSource:
    def __init__(
        self,
        *,
        source_hash: builtins.str,
        deploy_time: typing.Optional[builtins.bool] = None,
        display_name: typing.Optional[builtins.str] = None,
        executable: typing.Optional[typing.Sequence[builtins.str]] = None,
        file_name: typing.Optional[builtins.str] = None,
        packaging: typing.Optional[FileAssetPackaging] = None,
    ) -> None:
        '''Represents the source for a file asset.

        :param source_hash: A hash on the content source. This hash is used to uniquely identify this asset throughout the system. If this value doesn't change, the asset will not be rebuilt or republished.
        :param deploy_time: Whether or not the asset needs to exist beyond deployment time; i.e. are copied over to a different location and not needed afterwards. Setting this property to true has an impact on the lifecycle of the asset, because we will assume that it is safe to delete after the CloudFormation deployment succeeds. For example, Lambda Function assets are copied over to Lambda during deployment. Therefore, it is not necessary to store the asset in S3, so we consider those deployTime assets. Default: false
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. Default: - The asset hash is used to display the asset
        :param executable: An external command that will produce the packaged asset. The command should produce the location of a ZIP file on ``stdout``. Default: - Exactly one of ``fileName`` and ``executable`` is required
        :param file_name: The path, relative to the root of the cloud assembly, in which this asset source resides. This can be a path to a file or a directory, depending on the packaging type. Default: - Exactly one of ``fileName`` and ``executable`` is required
        :param packaging: Which type of packaging to perform. Default: - Required if ``fileName`` is specified.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            file_asset_source = cdk.FileAssetSource(
                source_hash="sourceHash",
            
                # the properties below are optional
                deploy_time=False,
                display_name="displayName",
                executable=["executable"],
                file_name="fileName",
                packaging=cdk.FileAssetPackaging.ZIP_DIRECTORY
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6a2ffcc71f7081ed98f77dcff398894a56d300191a4893d6469a2218ffed2957)
            check_type(argname="argument source_hash", value=source_hash, expected_type=type_hints["source_hash"])
            check_type(argname="argument deploy_time", value=deploy_time, expected_type=type_hints["deploy_time"])
            check_type(argname="argument display_name", value=display_name, expected_type=type_hints["display_name"])
            check_type(argname="argument executable", value=executable, expected_type=type_hints["executable"])
            check_type(argname="argument file_name", value=file_name, expected_type=type_hints["file_name"])
            check_type(argname="argument packaging", value=packaging, expected_type=type_hints["packaging"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "source_hash": source_hash,
        }
        if deploy_time is not None:
            self._values["deploy_time"] = deploy_time
        if display_name is not None:
            self._values["display_name"] = display_name
        if executable is not None:
            self._values["executable"] = executable
        if file_name is not None:
            self._values["file_name"] = file_name
        if packaging is not None:
            self._values["packaging"] = packaging

    @builtins.property
    def source_hash(self) -> builtins.str:
        '''A hash on the content source.

        This hash is used to uniquely identify this
        asset throughout the system. If this value doesn't change, the asset will
        not be rebuilt or republished.
        '''
        result = self._values.get("source_hash")
        assert result is not None, "Required property 'source_hash' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def deploy_time(self) -> typing.Optional[builtins.bool]:
        '''Whether or not the asset needs to exist beyond deployment time;

        i.e.
        are copied over to a different location and not needed afterwards.
        Setting this property to true has an impact on the lifecycle of the asset,
        because we will assume that it is safe to delete after the CloudFormation
        deployment succeeds.

        For example, Lambda Function assets are copied over to Lambda during
        deployment. Therefore, it is not necessary to store the asset in S3, so
        we consider those deployTime assets.

        :default: false
        '''
        result = self._values.get("deploy_time")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def display_name(self) -> typing.Optional[builtins.str]:
        '''A display name for this asset.

        If supplied, the display name will be used in locations where the asset
        identifier is printed, like in the CLI progress information.

        :default: - The asset hash is used to display the asset
        '''
        result = self._values.get("display_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def executable(self) -> typing.Optional[typing.List[builtins.str]]:
        '''An external command that will produce the packaged asset.

        The command should produce the location of a ZIP file on ``stdout``.

        :default: - Exactly one of ``fileName`` and ``executable`` is required
        '''
        result = self._values.get("executable")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def file_name(self) -> typing.Optional[builtins.str]:
        '''The path, relative to the root of the cloud assembly, in which this asset source resides.

        This can be a path to a file or a directory, depending on the
        packaging type.

        :default: - Exactly one of ``fileName`` and ``executable`` is required
        '''
        result = self._values.get("file_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def packaging(self) -> typing.Optional[FileAssetPackaging]:
        '''Which type of packaging to perform.

        :default: - Required if ``fileName`` is specified.
        '''
        result = self._values.get("packaging")
        return typing.cast(typing.Optional[FileAssetPackaging], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FileAssetSource(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.FileCopyOptions",
    jsii_struct_bases=[],
    name_mapping={
        "exclude": "exclude",
        "follow_symlinks": "followSymlinks",
        "ignore_mode": "ignoreMode",
    },
)
class FileCopyOptions:
    def __init__(
        self,
        *,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional["SymlinkFollowMode"] = None,
        ignore_mode: typing.Optional["IgnoreMode"] = None,
    ) -> None:
        '''Options applied when copying directories into the staging location.

        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            file_copy_options = cdk.FileCopyOptions(
                exclude=["exclude"],
                follow_symlinks=cdk.SymlinkFollowMode.NEVER,
                ignore_mode=cdk.IgnoreMode.GLOB
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5120efac45c641672970adcfed38ee5374d9b0422445a65a257abf9314d99159)
            check_type(argname="argument exclude", value=exclude, expected_type=type_hints["exclude"])
            check_type(argname="argument follow_symlinks", value=follow_symlinks, expected_type=type_hints["follow_symlinks"])
            check_type(argname="argument ignore_mode", value=ignore_mode, expected_type=type_hints["ignore_mode"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if exclude is not None:
            self._values["exclude"] = exclude
        if follow_symlinks is not None:
            self._values["follow_symlinks"] = follow_symlinks
        if ignore_mode is not None:
            self._values["ignore_mode"] = ignore_mode

    @builtins.property
    def exclude(self) -> typing.Optional[typing.List[builtins.str]]:
        '''File paths matching the patterns will be excluded.

        See ``ignoreMode`` to set the matching behavior.
        Has no effect on Assets bundled using the ``bundling`` property.

        :default: - nothing is excluded
        '''
        result = self._values.get("exclude")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def follow_symlinks(self) -> typing.Optional["SymlinkFollowMode"]:
        '''A strategy for how to handle symlinks.

        :default: SymlinkFollowMode.NEVER
        '''
        result = self._values.get("follow_symlinks")
        return typing.cast(typing.Optional["SymlinkFollowMode"], result)

    @builtins.property
    def ignore_mode(self) -> typing.Optional["IgnoreMode"]:
        '''The ignore behavior to use for ``exclude`` patterns.

        :default: IgnoreMode.GLOB
        '''
        result = self._values.get("ignore_mode")
        return typing.cast(typing.Optional["IgnoreMode"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FileCopyOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.FileFingerprintOptions",
    jsii_struct_bases=[FileCopyOptions],
    name_mapping={
        "exclude": "exclude",
        "follow_symlinks": "followSymlinks",
        "ignore_mode": "ignoreMode",
        "extra_hash": "extraHash",
    },
)
class FileFingerprintOptions(FileCopyOptions):
    def __init__(
        self,
        *,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional["SymlinkFollowMode"] = None,
        ignore_mode: typing.Optional["IgnoreMode"] = None,
        extra_hash: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options related to calculating source hash.

        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        :param extra_hash: Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            file_fingerprint_options = cdk.FileFingerprintOptions(
                exclude=["exclude"],
                extra_hash="extraHash",
                follow_symlinks=cdk.SymlinkFollowMode.NEVER,
                ignore_mode=cdk.IgnoreMode.GLOB
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d403764442343602467a17fdcf31336cac62624399e64929394dc7e3f76b1d98)
            check_type(argname="argument exclude", value=exclude, expected_type=type_hints["exclude"])
            check_type(argname="argument follow_symlinks", value=follow_symlinks, expected_type=type_hints["follow_symlinks"])
            check_type(argname="argument ignore_mode", value=ignore_mode, expected_type=type_hints["ignore_mode"])
            check_type(argname="argument extra_hash", value=extra_hash, expected_type=type_hints["extra_hash"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if exclude is not None:
            self._values["exclude"] = exclude
        if follow_symlinks is not None:
            self._values["follow_symlinks"] = follow_symlinks
        if ignore_mode is not None:
            self._values["ignore_mode"] = ignore_mode
        if extra_hash is not None:
            self._values["extra_hash"] = extra_hash

    @builtins.property
    def exclude(self) -> typing.Optional[typing.List[builtins.str]]:
        '''File paths matching the patterns will be excluded.

        See ``ignoreMode`` to set the matching behavior.
        Has no effect on Assets bundled using the ``bundling`` property.

        :default: - nothing is excluded
        '''
        result = self._values.get("exclude")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def follow_symlinks(self) -> typing.Optional["SymlinkFollowMode"]:
        '''A strategy for how to handle symlinks.

        :default: SymlinkFollowMode.NEVER
        '''
        result = self._values.get("follow_symlinks")
        return typing.cast(typing.Optional["SymlinkFollowMode"], result)

    @builtins.property
    def ignore_mode(self) -> typing.Optional["IgnoreMode"]:
        '''The ignore behavior to use for ``exclude`` patterns.

        :default: IgnoreMode.GLOB
        '''
        result = self._values.get("ignore_mode")
        return typing.cast(typing.Optional["IgnoreMode"], result)

    @builtins.property
    def extra_hash(self) -> typing.Optional[builtins.str]:
        '''Extra information to encode into the fingerprint (e.g. build instructions and other inputs).

        :default: - hash is only based on source content
        '''
        result = self._values.get("extra_hash")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FileFingerprintOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class FileSystem(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.FileSystem"):
    '''File system utilities.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        file_system = cdk.FileSystem()
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="copyDirectory")
    @builtins.classmethod
    def copy_directory(
        cls,
        src_dir: builtins.str,
        dest_dir: builtins.str,
        options: typing.Optional[typing.Union[CopyOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        root_dir: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Copies an entire directory structure.

        :param src_dir: Source directory.
        :param dest_dir: Destination directory.
        :param options: options.
        :param root_dir: Root directory to calculate exclusions from.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9bbb07c71d69293035ba08bc48c15ea77be881a28d5910fe8216bad506f5ae0b)
            check_type(argname="argument src_dir", value=src_dir, expected_type=type_hints["src_dir"])
            check_type(argname="argument dest_dir", value=dest_dir, expected_type=type_hints["dest_dir"])
            check_type(argname="argument options", value=options, expected_type=type_hints["options"])
            check_type(argname="argument root_dir", value=root_dir, expected_type=type_hints["root_dir"])
        return typing.cast(None, jsii.sinvoke(cls, "copyDirectory", [src_dir, dest_dir, options, root_dir]))

    @jsii.member(jsii_name="fingerprint")
    @builtins.classmethod
    def fingerprint(
        cls,
        file_or_directory: builtins.str,
        *,
        extra_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional["SymlinkFollowMode"] = None,
        ignore_mode: typing.Optional["IgnoreMode"] = None,
    ) -> builtins.str:
        '''Produces fingerprint based on the contents of a single file or an entire directory tree.

        Line endings are converted from CRLF to LF.

        The fingerprint will also include:

        1. An extra string if defined in ``options.extra``.
        2. The symlink follow mode value.

        :param file_or_directory: The directory or file to fingerprint.
        :param extra_hash: Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a37ffb2c9d929a5fa20dfdff77d653bb37d3d4eb85e896171ee214f5e7a08fae)
            check_type(argname="argument file_or_directory", value=file_or_directory, expected_type=type_hints["file_or_directory"])
        options = FingerprintOptions(
            extra_hash=extra_hash,
            exclude=exclude,
            follow=follow,
            ignore_mode=ignore_mode,
        )

        return typing.cast(builtins.str, jsii.sinvoke(cls, "fingerprint", [file_or_directory, options]))

    @jsii.member(jsii_name="isEmpty")
    @builtins.classmethod
    def is_empty(cls, dir: builtins.str) -> builtins.bool:
        '''Checks whether a directory is empty.

        :param dir: The directory to check.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35c0ba523e1cafab7b097299e6d08618213eb99b7f5c141c19aa07b0016fbeb2)
            check_type(argname="argument dir", value=dir, expected_type=type_hints["dir"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isEmpty", [dir]))

    @jsii.member(jsii_name="mkdtemp")
    @builtins.classmethod
    def mkdtemp(cls, prefix: builtins.str) -> builtins.str:
        '''Creates a unique temporary directory in the **system temp directory**.

        :param prefix: A prefix for the directory name. Six random characters will be generated and appended behind this prefix.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__04cde75c7956e809744e60404a6ac6cbcbafe03a002f260dc69e1acc5e363c54)
            check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "mkdtemp", [prefix]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="tmpdir")
    def tmpdir(cls) -> builtins.str:
        '''The real path of the system temp directory.'''
        return typing.cast(builtins.str, jsii.sget(cls, "tmpdir"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.FingerprintOptions",
    jsii_struct_bases=[CopyOptions],
    name_mapping={
        "exclude": "exclude",
        "follow": "follow",
        "ignore_mode": "ignoreMode",
        "extra_hash": "extraHash",
    },
)
class FingerprintOptions(CopyOptions):
    def __init__(
        self,
        *,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional["SymlinkFollowMode"] = None,
        ignore_mode: typing.Optional["IgnoreMode"] = None,
        extra_hash: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options related to calculating source hash.

        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        :param extra_hash: Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            fingerprint_options = cdk.FingerprintOptions(
                exclude=["exclude"],
                extra_hash="extraHash",
                follow=cdk.SymlinkFollowMode.NEVER,
                ignore_mode=cdk.IgnoreMode.GLOB
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__edf3a1bab5c95a33bfa2c7268d598d07e7f39b3caad1e6134492ef7ecfd63704)
            check_type(argname="argument exclude", value=exclude, expected_type=type_hints["exclude"])
            check_type(argname="argument follow", value=follow, expected_type=type_hints["follow"])
            check_type(argname="argument ignore_mode", value=ignore_mode, expected_type=type_hints["ignore_mode"])
            check_type(argname="argument extra_hash", value=extra_hash, expected_type=type_hints["extra_hash"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if exclude is not None:
            self._values["exclude"] = exclude
        if follow is not None:
            self._values["follow"] = follow
        if ignore_mode is not None:
            self._values["ignore_mode"] = ignore_mode
        if extra_hash is not None:
            self._values["extra_hash"] = extra_hash

    @builtins.property
    def exclude(self) -> typing.Optional[typing.List[builtins.str]]:
        '''File paths matching the patterns will be excluded.

        See ``ignoreMode`` to set the matching behavior.
        Has no effect on Assets bundled using the ``bundling`` property.

        :default: - nothing is excluded
        '''
        result = self._values.get("exclude")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def follow(self) -> typing.Optional["SymlinkFollowMode"]:
        '''A strategy for how to handle symlinks.

        :default: SymlinkFollowMode.NEVER
        '''
        result = self._values.get("follow")
        return typing.cast(typing.Optional["SymlinkFollowMode"], result)

    @builtins.property
    def ignore_mode(self) -> typing.Optional["IgnoreMode"]:
        '''The ignore behavior to use for ``exclude`` patterns.

        :default: IgnoreMode.GLOB
        '''
        result = self._values.get("ignore_mode")
        return typing.cast(typing.Optional["IgnoreMode"], result)

    @builtins.property
    def extra_hash(self) -> typing.Optional[builtins.str]:
        '''Extra information to encode into the fingerprint (e.g. build instructions and other inputs).

        :default: - hash is only based on source content
        '''
        result = self._values.get("extra_hash")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FingerprintOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Fn(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.Fn"):
    '''CloudFormation intrinsic functions.

    http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference.html

    :exampleMetadata: infused

    Example::

        import aws_cdk as cdk
        
        # destination_bucket: s3.Bucket
        
        
        my_bucket_deployment = s3deploy.BucketDeployment(self, "DeployMeWithoutExtractingFilesOnDestination",
            sources=[s3deploy.Source.asset(path.join(__dirname, "my-website"))],
            destination_bucket=destination_bucket,
            extract=False
        )
        
        cdk.CfnOutput(self, "ObjectKey",
            value=cdk.Fn.select(0, my_bucket_deployment.object_keys)
        )
    '''

    @jsii.member(jsii_name="base64")
    @builtins.classmethod
    def base64(cls, data: builtins.str) -> builtins.str:
        '''The intrinsic function ``Fn::Base64`` returns the Base64 representation of the input string.

        This function is typically used to pass encoded data to
        Amazon EC2 instances by way of the UserData property.

        :param data: The string value you want to convert to Base64.

        :return: a token represented as a string
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c968decb283577595fc32888ff002ac60f6b91053af9b1f6a6a8029e5da374f2)
            check_type(argname="argument data", value=data, expected_type=type_hints["data"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "base64", [data]))

    @jsii.member(jsii_name="cidr")
    @builtins.classmethod
    def cidr(
        cls,
        ip_block: builtins.str,
        count: jsii.Number,
        size_mask: typing.Optional[builtins.str] = None,
    ) -> typing.List[builtins.str]:
        '''The intrinsic function ``Fn::Cidr`` returns the specified Cidr address block.

        :param ip_block: The user-specified default Cidr address block.
        :param count: The number of subnets' Cidr block wanted. Count can be 1 to 256.
        :param size_mask: The digit covered in the subnet.

        :return: a token represented as a string
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__40ca73690ff9a87a97ceeeedfb2b490b094581b53978d17824b46f9b936ce441)
            check_type(argname="argument ip_block", value=ip_block, expected_type=type_hints["ip_block"])
            check_type(argname="argument count", value=count, expected_type=type_hints["count"])
            check_type(argname="argument size_mask", value=size_mask, expected_type=type_hints["size_mask"])
        return typing.cast(typing.List[builtins.str], jsii.sinvoke(cls, "cidr", [ip_block, count, size_mask]))

    @jsii.member(jsii_name="conditionAnd")
    @builtins.classmethod
    def condition_and(
        cls,
        *conditions: "ICfnConditionExpression",
    ) -> "ICfnRuleConditionExpression":
        '''Returns true if all the specified conditions evaluate to true, or returns false if any one of the conditions evaluates to false.

        ``Fn::And`` acts as
        an AND operator. The minimum number of conditions that you can include is
        1.

        :param conditions: conditions to AND.

        :return: an FnCondition token
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cc9e2502c6c26e0498b070cc35196c30eb2057975dda1184a51c2db3e90b4753)
            check_type(argname="argument conditions", value=conditions, expected_type=typing.Tuple[type_hints["conditions"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("ICfnRuleConditionExpression", jsii.sinvoke(cls, "conditionAnd", [*conditions]))

    @jsii.member(jsii_name="conditionContains")
    @builtins.classmethod
    def condition_contains(
        cls,
        list_of_strings: typing.Sequence[builtins.str],
        value: builtins.str,
    ) -> "ICfnRuleConditionExpression":
        '''Returns true if a specified string matches at least one value in a list of strings.

        :param list_of_strings: A list of strings, such as "A", "B", "C".
        :param value: A string, such as "A", that you want to compare against a list of strings.

        :return: an FnCondition token
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3db357a7a6ab35e360db1627b2c28e6430e2fd5664045113c4890d2c3ba6cddc)
            check_type(argname="argument list_of_strings", value=list_of_strings, expected_type=type_hints["list_of_strings"])
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("ICfnRuleConditionExpression", jsii.sinvoke(cls, "conditionContains", [list_of_strings, value]))

    @jsii.member(jsii_name="conditionEachMemberEquals")
    @builtins.classmethod
    def condition_each_member_equals(
        cls,
        list_of_strings: typing.Sequence[builtins.str],
        value: builtins.str,
    ) -> "ICfnRuleConditionExpression":
        '''Returns true if a specified string matches all values in a list.

        :param list_of_strings: A list of strings, such as "A", "B", "C".
        :param value: A string, such as "A", that you want to compare against a list of strings.

        :return: an FnCondition token
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cc8412fe8022857a87ccaf556907d92a05dd4ea9f2b262f08b5636872700e404)
            check_type(argname="argument list_of_strings", value=list_of_strings, expected_type=type_hints["list_of_strings"])
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        return typing.cast("ICfnRuleConditionExpression", jsii.sinvoke(cls, "conditionEachMemberEquals", [list_of_strings, value]))

    @jsii.member(jsii_name="conditionEachMemberIn")
    @builtins.classmethod
    def condition_each_member_in(
        cls,
        strings_to_check: typing.Sequence[builtins.str],
        strings_to_match: typing.Sequence[builtins.str],
    ) -> "ICfnRuleConditionExpression":
        '''Returns true if each member in a list of strings matches at least one value in a second list of strings.

        :param strings_to_check: A list of strings, such as "A", "B", "C". AWS CloudFormation checks whether each member in the strings_to_check parameter is in the strings_to_match parameter.
        :param strings_to_match: A list of strings, such as "A", "B", "C". Each member in the strings_to_match parameter is compared against the members of the strings_to_check parameter.

        :return: an FnCondition token
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ca8377c34939a3a0d90b9789cfb349bf16099fab0681dffbfccb91682c2fa582)
            check_type(argname="argument strings_to_check", value=strings_to_check, expected_type=type_hints["strings_to_check"])
            check_type(argname="argument strings_to_match", value=strings_to_match, expected_type=type_hints["strings_to_match"])
        return typing.cast("ICfnRuleConditionExpression", jsii.sinvoke(cls, "conditionEachMemberIn", [strings_to_check, strings_to_match]))

    @jsii.member(jsii_name="conditionEquals")
    @builtins.classmethod
    def condition_equals(
        cls,
        lhs: typing.Any,
        rhs: typing.Any,
    ) -> "ICfnRuleConditionExpression":
        '''Compares if two values are equal.

        Returns true if the two values are equal
        or false if they aren't.

        :param lhs: A value of any type that you want to compare.
        :param rhs: A value of any type that you want to compare.

        :return: an FnCondition token
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8383941eb156981e7ea03108eee708088b51c839a2f32a9dd4b31e29635f1b4b)
            check_type(argname="argument lhs", value=lhs, expected_type=type_hints["lhs"])
            check_type(argname="argument rhs", value=rhs, expected_type=type_hints["rhs"])
        return typing.cast("ICfnRuleConditionExpression", jsii.sinvoke(cls, "conditionEquals", [lhs, rhs]))

    @jsii.member(jsii_name="conditionIf")
    @builtins.classmethod
    def condition_if(
        cls,
        condition_id: builtins.str,
        value_if_true: typing.Any,
        value_if_false: typing.Any,
    ) -> "ICfnRuleConditionExpression":
        '''Returns one value if the specified condition evaluates to true and another value if the specified condition evaluates to false.

        Currently, AWS
        CloudFormation supports the ``Fn::If`` intrinsic function in the metadata
        attribute, update policy attribute, and property values in the Resources
        section and Outputs sections of a template. You can use the AWS::NoValue
        pseudo parameter as a return value to remove the corresponding property.

        :param condition_id: A reference to a condition in the Conditions section. Use the condition's name to reference it.
        :param value_if_true: A value to be returned if the specified condition evaluates to true.
        :param value_if_false: A value to be returned if the specified condition evaluates to false.

        :return: an FnCondition token
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5e2f1e84f8d8eeea64aee71f8c67368db0517adba018dd2479d271d960ce1c80)
            check_type(argname="argument condition_id", value=condition_id, expected_type=type_hints["condition_id"])
            check_type(argname="argument value_if_true", value=value_if_true, expected_type=type_hints["value_if_true"])
            check_type(argname="argument value_if_false", value=value_if_false, expected_type=type_hints["value_if_false"])
        return typing.cast("ICfnRuleConditionExpression", jsii.sinvoke(cls, "conditionIf", [condition_id, value_if_true, value_if_false]))

    @jsii.member(jsii_name="conditionNot")
    @builtins.classmethod
    def condition_not(
        cls,
        condition: "ICfnConditionExpression",
    ) -> "ICfnRuleConditionExpression":
        '''Returns true for a condition that evaluates to false or returns false for a condition that evaluates to true.

        ``Fn::Not`` acts as a NOT operator.

        :param condition: A condition such as ``Fn::Equals`` that evaluates to true or false.

        :return: an FnCondition token
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a7c7c7f156aaf618f5f12f184c737446f140031e927b16eeb7f3568e7f808c34)
            check_type(argname="argument condition", value=condition, expected_type=type_hints["condition"])
        return typing.cast("ICfnRuleConditionExpression", jsii.sinvoke(cls, "conditionNot", [condition]))

    @jsii.member(jsii_name="conditionOr")
    @builtins.classmethod
    def condition_or(
        cls,
        *conditions: "ICfnConditionExpression",
    ) -> "ICfnRuleConditionExpression":
        '''Returns true if any one of the specified conditions evaluate to true, or returns false if all of the conditions evaluates to false.

        ``Fn::Or`` acts
        as an OR operator. The minimum number of conditions that you can include is
        1.

        :param conditions: conditions that evaluates to true or false.

        :return: an FnCondition token
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf13ce3f53862812789696d37d3bb3a3316f1aa9dc789bb622da7ea80e5a6882)
            check_type(argname="argument conditions", value=conditions, expected_type=typing.Tuple[type_hints["conditions"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("ICfnRuleConditionExpression", jsii.sinvoke(cls, "conditionOr", [*conditions]))

    @jsii.member(jsii_name="findInMap")
    @builtins.classmethod
    def find_in_map(
        cls,
        map_name: builtins.str,
        top_level_key: builtins.str,
        second_level_key: builtins.str,
        default_value: typing.Optional[builtins.str] = None,
    ) -> builtins.str:
        '''The intrinsic function ``Fn::FindInMap`` returns the value corresponding to keys in a two-level map that is declared in the Mappings section.

        Warning: do not use with lazy mappings as this function will not guarentee a lazy mapping to render in the template.
        Prefer to use ``CfnMapping.findInMap`` in general.

        :param map_name: -
        :param top_level_key: -
        :param second_level_key: -
        :param default_value: -

        :return: a token represented as a string
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a418601ea806056ab758734464be8a572e937b0cd4ac3c0b0fcc9e22615c2c78)
            check_type(argname="argument map_name", value=map_name, expected_type=type_hints["map_name"])
            check_type(argname="argument top_level_key", value=top_level_key, expected_type=type_hints["top_level_key"])
            check_type(argname="argument second_level_key", value=second_level_key, expected_type=type_hints["second_level_key"])
            check_type(argname="argument default_value", value=default_value, expected_type=type_hints["default_value"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "findInMap", [map_name, top_level_key, second_level_key, default_value]))

    @jsii.member(jsii_name="getAtt")
    @builtins.classmethod
    def get_att(
        cls,
        logical_name_of_resource: builtins.str,
        attribute_name: builtins.str,
    ) -> "IResolvable":
        '''The ``Fn::GetAtt`` intrinsic function returns the value of an attribute from a resource in the template.

        :param logical_name_of_resource: The logical name (also called logical ID) of the resource that contains the attribute that you want.
        :param attribute_name: The name of the resource-specific attribute whose value you want. See the resource's reference page for details about the attributes available for that resource type.

        :return: an IResolvable object
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c799c0078b5925a51301f8dc083f2963b8eceda1d3d8eabd029bdb7a0f09adcf)
            check_type(argname="argument logical_name_of_resource", value=logical_name_of_resource, expected_type=type_hints["logical_name_of_resource"])
            check_type(argname="argument attribute_name", value=attribute_name, expected_type=type_hints["attribute_name"])
        return typing.cast("IResolvable", jsii.sinvoke(cls, "getAtt", [logical_name_of_resource, attribute_name]))

    @jsii.member(jsii_name="getAzs")
    @builtins.classmethod
    def get_azs(
        cls,
        region: typing.Optional[builtins.str] = None,
    ) -> typing.List[builtins.str]:
        '''The intrinsic function ``Fn::GetAZs`` returns an array that lists Availability Zones for a specified region.

        Because customers have access to
        different Availability Zones, the intrinsic function ``Fn::GetAZs`` enables
        template authors to write templates that adapt to the calling user's
        access. That way you don't have to hard-code a full list of Availability
        Zones for a specified region.

        :param region: The name of the region for which you want to get the Availability Zones. You can use the AWS::Region pseudo parameter to specify the region in which the stack is created. Specifying an empty string is equivalent to specifying AWS::Region.

        :return: a token represented as a string array
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ddfeaf20d2c8e6c042c332c2e1beca65310a356288fbbdfcce98bc268f89077e)
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
        return typing.cast(typing.List[builtins.str], jsii.sinvoke(cls, "getAzs", [region]))

    @jsii.member(jsii_name="importListValue")
    @builtins.classmethod
    def import_list_value(
        cls,
        shared_value_to_import: builtins.str,
        assumed_length: jsii.Number,
        delimiter: typing.Optional[builtins.str] = None,
    ) -> typing.List[builtins.str]:
        '''Like ``Fn.importValue``, but import a list with a known length.

        If you explicitly want a list with an unknown length, call ``Fn.split(',', Fn.importValue(exportName))``. See the documentation of ``Fn.split`` to read
        more about the limitations of using lists of unknown length.

        ``Fn.importListValue(exportName, assumedLength)`` is the same as
        ``Fn.split(',', Fn.importValue(exportName), assumedLength)``,
        but easier to read and impossible to forget to pass ``assumedLength``.

        :param shared_value_to_import: -
        :param assumed_length: -
        :param delimiter: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__60fe490fdd623330e77a09e19de30aacd031fb555f9d3c3c923390a43f63aa0a)
            check_type(argname="argument shared_value_to_import", value=shared_value_to_import, expected_type=type_hints["shared_value_to_import"])
            check_type(argname="argument assumed_length", value=assumed_length, expected_type=type_hints["assumed_length"])
            check_type(argname="argument delimiter", value=delimiter, expected_type=type_hints["delimiter"])
        return typing.cast(typing.List[builtins.str], jsii.sinvoke(cls, "importListValue", [shared_value_to_import, assumed_length, delimiter]))

    @jsii.member(jsii_name="importValue")
    @builtins.classmethod
    def import_value(cls, shared_value_to_import: builtins.str) -> builtins.str:
        '''The intrinsic function ``Fn::ImportValue`` returns the value of an output exported by another stack.

        You typically use this function to create
        cross-stack references. In the following example template snippets, Stack A
        exports VPC security group values and Stack B imports them.

        :param shared_value_to_import: The stack output value that you want to import.

        :return: a token represented as a string
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fd722ef1848499dff9f55ac202d6bf91c0660090621f084a5dee4aa20d90ea89)
            check_type(argname="argument shared_value_to_import", value=shared_value_to_import, expected_type=type_hints["shared_value_to_import"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "importValue", [shared_value_to_import]))

    @jsii.member(jsii_name="join")
    @builtins.classmethod
    def join(
        cls,
        delimiter: builtins.str,
        list_of_values: typing.Sequence[builtins.str],
    ) -> builtins.str:
        '''The intrinsic function ``Fn::Join`` appends a set of values into a single value, separated by the specified delimiter.

        If a delimiter is the empty
        string, the set of values are concatenated with no delimiter.

        :param delimiter: The value you want to occur between fragments. The delimiter will occur between fragments only. It will not terminate the final value.
        :param list_of_values: The list of values you want combined.

        :return: a token represented as a string
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d3f2d897ca12a587358275f09aa5c1123d692ed7d97f46e5d44a0904a6fc05c6)
            check_type(argname="argument delimiter", value=delimiter, expected_type=type_hints["delimiter"])
            check_type(argname="argument list_of_values", value=list_of_values, expected_type=type_hints["list_of_values"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "join", [delimiter, list_of_values]))

    @jsii.member(jsii_name="len")
    @builtins.classmethod
    def len(cls, array: typing.Any) -> jsii.Number:
        '''The intrinsic function ``Fn::Length`` returns the number of elements within an array or an intrinsic function that returns an array.

        :param array: The array you want to return the number of elements from.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c80377501125d1e68c6e56e6e615c3db370503b2b5677c4d268a6ae34a43cf8b)
            check_type(argname="argument array", value=array, expected_type=type_hints["array"])
        return typing.cast(jsii.Number, jsii.sinvoke(cls, "len", [array]))

    @jsii.member(jsii_name="parseDomainName")
    @builtins.classmethod
    def parse_domain_name(cls, url: builtins.str) -> builtins.str:
        '''Given an url, parse the domain name.

        :param url: the url to parse.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f282039570caae76378ac471a9c3f15fea8faeef6fbfe93275ddb219a7d8f1f2)
            check_type(argname="argument url", value=url, expected_type=type_hints["url"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "parseDomainName", [url]))

    @jsii.member(jsii_name="ref")
    @builtins.classmethod
    def ref(cls, logical_name: builtins.str) -> builtins.str:
        '''The ``Ref`` intrinsic function returns the value of the specified parameter or resource.

        Note that it doesn't validate the logicalName, it mainly serves parameter/resource reference defined in a ``CfnInclude`` template.

        :param logical_name: The logical name of a parameter/resource for which you want to retrieve its value.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fac172e4890e1fddc047775c4e09d461d27ec4ed444f4a329f95a19b1f5335ae)
            check_type(argname="argument logical_name", value=logical_name, expected_type=type_hints["logical_name"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "ref", [logical_name]))

    @jsii.member(jsii_name="refAll")
    @builtins.classmethod
    def ref_all(cls, parameter_type: builtins.str) -> typing.List[builtins.str]:
        '''Returns all values for a specified parameter type.

        :param parameter_type: An AWS-specific parameter type, such as AWS::EC2::SecurityGroup::Id or AWS::EC2::VPC::Id. For more information, see Parameters in the AWS CloudFormation User Guide.

        :return: a token represented as a string array
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f5f90ea72bf9959e1d7f29acb9fdaaf2f4984a4cfe83fbf01a0b514bb61bc64f)
            check_type(argname="argument parameter_type", value=parameter_type, expected_type=type_hints["parameter_type"])
        return typing.cast(typing.List[builtins.str], jsii.sinvoke(cls, "refAll", [parameter_type]))

    @jsii.member(jsii_name="select")
    @builtins.classmethod
    def select(
        cls,
        index: jsii.Number,
        array: typing.Sequence[builtins.str],
    ) -> builtins.str:
        '''The intrinsic function ``Fn::Select`` returns a single object from a list of objects by index.

        :param index: The index of the object to retrieve. This must be a value from zero to N-1, where N represents the number of elements in the array.
        :param array: The list of objects to select from. This list must not be null, nor can it have null entries.

        :return: a token represented as a string
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__99b22c467ea344548eb5451ccd4d3fc49e35275a667e4169cb4ae15d3e344496)
            check_type(argname="argument index", value=index, expected_type=type_hints["index"])
            check_type(argname="argument array", value=array, expected_type=type_hints["array"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "select", [index, array]))

    @jsii.member(jsii_name="split")
    @builtins.classmethod
    def split(
        cls,
        delimiter: builtins.str,
        source: builtins.str,
        assumed_length: typing.Optional[jsii.Number] = None,
    ) -> typing.List[builtins.str]:
        '''Split a string token into a token list of string values.

        Specify the location of splits with a delimiter such as ',' (a comma).
        Renders to the ``Fn::Split`` intrinsic function.


        Lists with unknown lengths (default)

        Since this function is used to work with deploy-time values, if ``assumedLength``
        is not given the CDK cannot know the length of the resulting list at synthesis time.
        This brings the following restrictions:

        - You must use ``Fn.select(i, list)`` to pick elements out of the list (you must not use
          ``list[i]``).
        - You cannot add elements to the list, remove elements from the list,
          combine two such lists together, or take a slice of the list.
        - You cannot pass the list to constructs that do any of the above.

        The only valid operation with such a tokenized list is to pass it unmodified to a
        CloudFormation Resource construct.


        Lists with assumed lengths

        Pass ``assumedLength`` if you know the length of the list that will be
        produced by splitting. The actual list length at deploy time may be
        *longer* than the number you pass, but not *shorter*.

        The returned list will look like::

           [Fn.select(0, split), Fn.select(1, split), Fn.select(2, split), ...]

        The restrictions from the section "Lists with unknown lengths" will now be lifted,
        at the expense of having to know and fix the length of the list.

        :param delimiter: A string value that determines where the source string is divided.
        :param source: The string value that you want to split.
        :param assumed_length: The length of the list that will be produced by splitting.

        :return: a token represented as a string array
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__771ebb1136f5957558c323f7f913c566739a95ab9c7756fdd720967ec4a76c70)
            check_type(argname="argument delimiter", value=delimiter, expected_type=type_hints["delimiter"])
            check_type(argname="argument source", value=source, expected_type=type_hints["source"])
            check_type(argname="argument assumed_length", value=assumed_length, expected_type=type_hints["assumed_length"])
        return typing.cast(typing.List[builtins.str], jsii.sinvoke(cls, "split", [delimiter, source, assumed_length]))

    @jsii.member(jsii_name="sub")
    @builtins.classmethod
    def sub(
        cls,
        body: builtins.str,
        variables: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> builtins.str:
        '''The intrinsic function ``Fn::Sub`` substitutes variables in an input string with values that you specify.

        In your templates, you can use this function
        to construct commands or outputs that include values that aren't available
        until you create or update a stack.

        :param body: A string with variables that AWS CloudFormation substitutes with their associated values at runtime. Write variables as ${MyVarName}. Variables can be template parameter names, resource logical IDs, resource attributes, or a variable in a key-value map. If you specify only template parameter names, resource logical IDs, and resource attributes, don't specify a key-value map.
        :param variables: The name of a variable that you included in the String parameter. The value that AWS CloudFormation substitutes for the associated variable name at runtime.

        :return: a token represented as a string
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fdd4b480239a845b96d39383e171f4ef1cd1620956c36f99f6c8da1ba432c80d)
            check_type(argname="argument body", value=body, expected_type=type_hints["body"])
            check_type(argname="argument variables", value=variables, expected_type=type_hints["variables"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "sub", [body, variables]))

    @jsii.member(jsii_name="toJsonString")
    @builtins.classmethod
    def to_json_string(cls, object: typing.Any) -> builtins.str:
        '''The ``Fn::ToJsonString`` intrinsic function converts an object or array to its corresponding JSON string.

        :param object: The object or array to stringify.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5e83fb1351911b7a1cb7d94f11c60722b03fedebd632c78a31764d6465735012)
            check_type(argname="argument object", value=object, expected_type=type_hints["object"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "toJsonString", [object]))

    @jsii.member(jsii_name="transform")
    @builtins.classmethod
    def transform(
        cls,
        macro_name: builtins.str,
        parameters: typing.Mapping[builtins.str, typing.Any],
    ) -> "IResolvable":
        '''Creates a token representing the ``Fn::Transform`` expression.

        :param macro_name: The name of the macro to perform the processing.
        :param parameters: The parameters to be passed to the macro.

        :return: a token representing the transform expression

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-transform.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__63d0ddd0b94e9ff3ff01e960b5c1bbc2be177f50e692ffbb8c5790fdc387e384)
            check_type(argname="argument macro_name", value=macro_name, expected_type=type_hints["macro_name"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
        return typing.cast("IResolvable", jsii.sinvoke(cls, "transform", [macro_name, parameters]))

    @jsii.member(jsii_name="valueOf")
    @builtins.classmethod
    def value_of(
        cls,
        parameter_or_logical_id: builtins.str,
        attribute: builtins.str,
    ) -> builtins.str:
        '''Returns an attribute value or list of values for a specific parameter and attribute.

        :param parameter_or_logical_id: The name of a parameter for which you want to retrieve attribute values. The parameter must be declared in the Parameters section of the template.
        :param attribute: The name of an attribute from which you want to retrieve a value.

        :return: a token represented as a string
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d4f7085f1a8686894309a3bd52356001495652b6c1e21c63d614635a59eae91f)
            check_type(argname="argument parameter_or_logical_id", value=parameter_or_logical_id, expected_type=type_hints["parameter_or_logical_id"])
            check_type(argname="argument attribute", value=attribute, expected_type=type_hints["attribute"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "valueOf", [parameter_or_logical_id, attribute]))

    @jsii.member(jsii_name="valueOfAll")
    @builtins.classmethod
    def value_of_all(
        cls,
        parameter_type: builtins.str,
        attribute: builtins.str,
    ) -> typing.List[builtins.str]:
        '''Returns a list of all attribute values for a given parameter type and attribute.

        :param parameter_type: An AWS-specific parameter type, such as AWS::EC2::SecurityGroup::Id or AWS::EC2::VPC::Id. For more information, see Parameters in the AWS CloudFormation User Guide.
        :param attribute: The name of an attribute from which you want to retrieve a value. For more information about attributes, see Supported Attributes.

        :return: a token represented as a string array
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1aa84b6f7c7da29afeaf8cce6ea2a4aee978df497c5cb6d4432224b24a765222)
            check_type(argname="argument parameter_type", value=parameter_type, expected_type=type_hints["parameter_type"])
            check_type(argname="argument attribute", value=attribute, expected_type=type_hints["attribute"])
        return typing.cast(typing.List[builtins.str], jsii.sinvoke(cls, "valueOfAll", [parameter_type, attribute]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.GetContextKeyOptions",
    jsii_struct_bases=[],
    name_mapping={
        "provider": "provider",
        "include_environment": "includeEnvironment",
        "props": "props",
    },
)
class GetContextKeyOptions:
    def __init__(
        self,
        *,
        provider: builtins.str,
        include_environment: typing.Optional[builtins.bool] = None,
        props: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''
        :param provider: The context provider to query.
        :param include_environment: Whether to include the stack's account and region automatically. Default: true
        :param props: Provider-specific properties.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # props: Any
            
            get_context_key_options = cdk.GetContextKeyOptions(
                provider="provider",
            
                # the properties below are optional
                include_environment=False,
                props={
                    "props_key": props
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4a3bc64bd38ef14ad5ebdf5807563d2355b26f4f0c446c315782c9818b76192a)
            check_type(argname="argument provider", value=provider, expected_type=type_hints["provider"])
            check_type(argname="argument include_environment", value=include_environment, expected_type=type_hints["include_environment"])
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "provider": provider,
        }
        if include_environment is not None:
            self._values["include_environment"] = include_environment
        if props is not None:
            self._values["props"] = props

    @builtins.property
    def provider(self) -> builtins.str:
        '''The context provider to query.'''
        result = self._values.get("provider")
        assert result is not None, "Required property 'provider' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def include_environment(self) -> typing.Optional[builtins.bool]:
        '''Whether to include the stack's account and region automatically.

        :default: true
        '''
        result = self._values.get("include_environment")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def props(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Provider-specific properties.'''
        result = self._values.get("props")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GetContextKeyOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.GetContextKeyResult",
    jsii_struct_bases=[],
    name_mapping={"key": "key", "props": "props"},
)
class GetContextKeyResult:
    def __init__(
        self,
        *,
        key: builtins.str,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> None:
        '''
        :param key: 
        :param props: 

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # props: Any
            
            get_context_key_result = cdk.GetContextKeyResult(
                key="key",
                props={
                    "props_key": props
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8870c87947d551a1842549591423e81e319fe3185c0ae81f67b67b1bf05d6690)
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "key": key,
            "props": props,
        }

    @builtins.property
    def key(self) -> builtins.str:
        result = self._values.get("key")
        assert result is not None, "Required property 'key' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def props(self) -> typing.Mapping[builtins.str, typing.Any]:
        result = self._values.get("props")
        assert result is not None, "Required property 'props' is missing"
        return typing.cast(typing.Mapping[builtins.str, typing.Any], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GetContextKeyResult(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.GetContextValueOptions",
    jsii_struct_bases=[GetContextKeyOptions],
    name_mapping={
        "provider": "provider",
        "include_environment": "includeEnvironment",
        "props": "props",
        "dummy_value": "dummyValue",
        "ignore_error_on_missing_context": "ignoreErrorOnMissingContext",
        "must_exist": "mustExist",
    },
)
class GetContextValueOptions(GetContextKeyOptions):
    def __init__(
        self,
        *,
        provider: builtins.str,
        include_environment: typing.Optional[builtins.bool] = None,
        props: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        dummy_value: typing.Any,
        ignore_error_on_missing_context: typing.Optional[builtins.bool] = None,
        must_exist: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param provider: The context provider to query.
        :param include_environment: Whether to include the stack's account and region automatically. Default: true
        :param props: Provider-specific properties.
        :param dummy_value: The value to return if the lookup has not yet been performed. Upon first synthesis, the lookups has not yet been performed. The ``getValue()`` operation returns this value instead, so that synthesis can proceed. After synthesis completes the first time, the actual lookup will be performed and synthesis will run again with the *real* value. Dummy values should preferably have valid shapes so that downstream consumers of lookup values don't throw validation exceptions if they encounter a dummy value (or all possible downstream consumers need to effectively check for the well-known shape of the dummy value); throwing an exception would error out the synthesis operation and prevent the lookup and the second, real, synthesis from happening. Connection to mustExist ``dummyValue`` is also used as the official value to return if the lookup has failed and ``mustExist == false``.
        :param ignore_error_on_missing_context: (deprecated) Ignore a lookup failure and return the ``dummyValue`` instead. ``mustExist`` is the recommended alias for this deprecated property (note that its value is reversed). Default: false
        :param must_exist: Whether the resource must exist. If this is set (the default), the query fails if the value or resource we tried to look up doesn't exist. If this is ``false`` and the value we tried to look up could not be found, the failure is suppressed and ``dummyValue`` is officially returned instead. When this happens, ``dummyValue`` is encoded into cached context and it will never be refreshed anymore until the user runs ``cdk context --reset <key>``. Note that it is not possible for the CDK app code to make a distinction between "the lookup has not been performed yet" and "the lookup didn't find anything and we returned a default value instead". Context providers This feature must explicitly be supported by context providers. It is currently supported by: - KMS key provider - SSM parameter provider Note to implementors The dummy value should not be returned for all SDK lookup failures. For example, "no network" or "no credentials" or "malformed query" should not lead to the dummy value being returned. Only the case of "no such resource" should. Default: true

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # dummy_value: Any
            # props: Any
            
            get_context_value_options = cdk.GetContextValueOptions(
                dummy_value=dummy_value,
                provider="provider",
            
                # the properties below are optional
                ignore_error_on_missing_context=False,
                include_environment=False,
                must_exist=False,
                props={
                    "props_key": props
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__16ea3024419c210903405ca05f7de1bb3fad517c101a7a84926fb5e09ac784ec)
            check_type(argname="argument provider", value=provider, expected_type=type_hints["provider"])
            check_type(argname="argument include_environment", value=include_environment, expected_type=type_hints["include_environment"])
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
            check_type(argname="argument dummy_value", value=dummy_value, expected_type=type_hints["dummy_value"])
            check_type(argname="argument ignore_error_on_missing_context", value=ignore_error_on_missing_context, expected_type=type_hints["ignore_error_on_missing_context"])
            check_type(argname="argument must_exist", value=must_exist, expected_type=type_hints["must_exist"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "provider": provider,
            "dummy_value": dummy_value,
        }
        if include_environment is not None:
            self._values["include_environment"] = include_environment
        if props is not None:
            self._values["props"] = props
        if ignore_error_on_missing_context is not None:
            self._values["ignore_error_on_missing_context"] = ignore_error_on_missing_context
        if must_exist is not None:
            self._values["must_exist"] = must_exist

    @builtins.property
    def provider(self) -> builtins.str:
        '''The context provider to query.'''
        result = self._values.get("provider")
        assert result is not None, "Required property 'provider' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def include_environment(self) -> typing.Optional[builtins.bool]:
        '''Whether to include the stack's account and region automatically.

        :default: true
        '''
        result = self._values.get("include_environment")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def props(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Provider-specific properties.'''
        result = self._values.get("props")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def dummy_value(self) -> typing.Any:
        '''The value to return if the lookup has not yet been performed.

        Upon first synthesis, the lookups has not yet been performed. The
        ``getValue()`` operation returns this value instead, so that synthesis can
        proceed. After synthesis completes the first time, the actual lookup will
        be performed and synthesis will run again with the *real* value.

        Dummy values should preferably have valid shapes so that downstream
        consumers of lookup values don't throw validation exceptions if they
        encounter a dummy value (or all possible downstream consumers need to
        effectively check for the well-known shape of the dummy value); throwing an
        exception would error out the synthesis operation and prevent the lookup
        and the second, real, synthesis from happening.


        Connection to mustExist

        ``dummyValue`` is also used as the official value to return if the lookup has
        failed and ``mustExist == false``.
        '''
        result = self._values.get("dummy_value")
        assert result is not None, "Required property 'dummy_value' is missing"
        return typing.cast(typing.Any, result)

    @builtins.property
    def ignore_error_on_missing_context(self) -> typing.Optional[builtins.bool]:
        '''(deprecated) Ignore a lookup failure and return the ``dummyValue`` instead.

        ``mustExist`` is the recommended alias for this deprecated
        property (note that its value is reversed).

        :default: false

        :deprecated: Use mustExist instead

        :stability: deprecated
        '''
        result = self._values.get("ignore_error_on_missing_context")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def must_exist(self) -> typing.Optional[builtins.bool]:
        '''Whether the resource must exist.

        If this is set (the default), the query fails if the value or resource we
        tried to look up doesn't exist.

        If this is ``false`` and the value we tried to look up could not be found, the
        failure is suppressed and ``dummyValue`` is officially returned instead.

        When this happens, ``dummyValue`` is encoded into cached context and it will
        never be refreshed anymore until the user runs ``cdk context --reset <key>``.

        Note that it is not possible for the CDK app code to make a distinction
        between "the lookup has not been performed yet" and "the lookup didn't
        find anything and we returned a default value instead".


        Context providers

        This feature must explicitly be supported by context providers. It is
        currently supported by:

        - KMS key provider
        - SSM parameter provider



        Note to implementors

        The dummy value should not be returned for all SDK lookup failures. For
        example, "no network" or "no credentials" or "malformed query" should
        not lead to the dummy value being returned. Only the case of "no such
        resource" should.

        :default: true
        '''
        result = self._values.get("must_exist")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GetContextValueOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.GetContextValueResult",
    jsii_struct_bases=[],
    name_mapping={"value": "value"},
)
class GetContextValueResult:
    def __init__(self, *, value: typing.Any = None) -> None:
        '''
        :param value: 

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # value: Any
            
            get_context_value_result = cdk.GetContextValueResult(
                value=value
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__511e716563179c71c9d4244900fd8533b6991157b613101e2f6e7d24bec59465)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if value is not None:
            self._values["value"] = value

    @builtins.property
    def value(self) -> typing.Any:
        result = self._values.get("value")
        return typing.cast(typing.Any, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GetContextValueResult(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="aws-cdk-lib.IAnyProducer")
class IAnyProducer(typing_extensions.Protocol):
    '''Interface for lazy untyped value producers.'''

    @jsii.member(jsii_name="produce")
    def produce(self, context: "IResolveContext") -> typing.Any:
        '''Produce the value.

        :param context: -
        '''
        ...


class _IAnyProducerProxy:
    '''Interface for lazy untyped value producers.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IAnyProducer"

    @jsii.member(jsii_name="produce")
    def produce(self, context: "IResolveContext") -> typing.Any:
        '''Produce the value.

        :param context: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__92276a48d407daf25e6b28030d1b6162cfbd4f06c1c44b55d1b6fec6cf13d391)
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
        return typing.cast(typing.Any, jsii.invoke(self, "produce", [context]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAnyProducer).__jsii_proxy_class__ = lambda : _IAnyProducerProxy


@jsii.interface(jsii_type="aws-cdk-lib.IAspect")
class IAspect(typing_extensions.Protocol):
    '''Represents an Aspect.'''

    @jsii.member(jsii_name="visit")
    def visit(self, node: _constructs_77d1e7e8.IConstruct) -> None:
        '''All aspects can visit an IConstruct.

        :param node: -
        '''
        ...


class _IAspectProxy:
    '''Represents an Aspect.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IAspect"

    @jsii.member(jsii_name="visit")
    def visit(self, node: _constructs_77d1e7e8.IConstruct) -> None:
        '''All aspects can visit an IConstruct.

        :param node: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b02bde979e3a117df05ffe8e33120a48562cfd4e82718d3e201138aaa3680027)
            check_type(argname="argument node", value=node, expected_type=type_hints["node"])
        return typing.cast(None, jsii.invoke(self, "visit", [node]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAspect).__jsii_proxy_class__ = lambda : _IAspectProxy


@jsii.interface(jsii_type="aws-cdk-lib.IAsset")
class IAsset(typing_extensions.Protocol):
    '''Common interface for all assets.'''

    @builtins.property
    @jsii.member(jsii_name="assetHash")
    def asset_hash(self) -> builtins.str:
        '''A hash of this asset, which is available at construction time.

        As this is a plain string, it
        can be used in construct IDs in order to enforce creation of a new resource when the content
        hash has changed.
        '''
        ...


class _IAssetProxy:
    '''Common interface for all assets.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IAsset"

    @builtins.property
    @jsii.member(jsii_name="assetHash")
    def asset_hash(self) -> builtins.str:
        '''A hash of this asset, which is available at construction time.

        As this is a plain string, it
        can be used in construct IDs in order to enforce creation of a new resource when the content
        hash has changed.
        '''
        return typing.cast(builtins.str, jsii.get(self, "assetHash"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAsset).__jsii_proxy_class__ = lambda : _IAssetProxy


@jsii.interface(jsii_type="aws-cdk-lib.ICfnResourceOptions")
class ICfnResourceOptions(typing_extensions.Protocol):
    @builtins.property
    @jsii.member(jsii_name="condition")
    def condition(self) -> typing.Optional["CfnCondition"]:
        '''A condition to associate with this resource.

        This means that only if the condition evaluates to 'true' when the stack
        is deployed, the resource will be included. This is provided to allow CDK projects to produce legacy templates, but normally
        there is no need to use it in CDK projects.
        '''
        ...

    @condition.setter
    def condition(self, value: typing.Optional["CfnCondition"]) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="creationPolicy")
    def creation_policy(self) -> typing.Optional[CfnCreationPolicy]:
        '''Associate the CreationPolicy attribute with a resource to prevent its status from reaching create complete until AWS CloudFormation receives a specified number of success signals or the timeout period is exceeded.

        To signal a
        resource, you can use the cfn-signal helper script or SignalResource API. AWS CloudFormation publishes valid signals
        to the stack events so that you track the number of signals sent.
        '''
        ...

    @creation_policy.setter
    def creation_policy(self, value: typing.Optional[CfnCreationPolicy]) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="deletionPolicy")
    def deletion_policy(self) -> typing.Optional[CfnDeletionPolicy]:
        '''With the DeletionPolicy attribute you can preserve or (in some cases) backup a resource when its stack is deleted.

        You specify a DeletionPolicy attribute for each resource that you want to control. If a resource has no DeletionPolicy
        attribute, AWS CloudFormation deletes the resource by default. Note that this capability also applies to update operations
        that lead to resources being removed.
        '''
        ...

    @deletion_policy.setter
    def deletion_policy(self, value: typing.Optional[CfnDeletionPolicy]) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of this resource.

        Used for informational purposes only, is not processed in any way
        (and stays with the CloudFormation template, is not passed to the underlying resource,
        even if it does have a 'description' property).
        '''
        ...

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="metadata")
    def metadata(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Metadata associated with the CloudFormation resource.

        This is not the same as the construct metadata which can be added
        using construct.addMetadata(), but would not appear in the CloudFormation template automatically.
        '''
        ...

    @metadata.setter
    def metadata(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, typing.Any]],
    ) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="updatePolicy")
    def update_policy(self) -> typing.Optional[CfnUpdatePolicy]:
        '''Use the UpdatePolicy attribute to specify how AWS CloudFormation handles updates to the AWS::AutoScaling::AutoScalingGroup resource.

        AWS CloudFormation invokes one of three update policies depending on the type of change you make or whether a
        scheduled action is associated with the Auto Scaling group.
        '''
        ...

    @update_policy.setter
    def update_policy(self, value: typing.Optional[CfnUpdatePolicy]) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="updateReplacePolicy")
    def update_replace_policy(self) -> typing.Optional[CfnDeletionPolicy]:
        '''Use the UpdateReplacePolicy attribute to retain or (in some cases) backup the existing physical instance of a resource when it is replaced during a stack update operation.'''
        ...

    @update_replace_policy.setter
    def update_replace_policy(self, value: typing.Optional[CfnDeletionPolicy]) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[builtins.str]:
        '''The version of this resource.

        Used only for custom CloudFormation resources.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html
        '''
        ...

    @version.setter
    def version(self, value: typing.Optional[builtins.str]) -> None:
        ...


class _ICfnResourceOptionsProxy:
    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.ICfnResourceOptions"

    @builtins.property
    @jsii.member(jsii_name="condition")
    def condition(self) -> typing.Optional["CfnCondition"]:
        '''A condition to associate with this resource.

        This means that only if the condition evaluates to 'true' when the stack
        is deployed, the resource will be included. This is provided to allow CDK projects to produce legacy templates, but normally
        there is no need to use it in CDK projects.
        '''
        return typing.cast(typing.Optional["CfnCondition"], jsii.get(self, "condition"))

    @condition.setter
    def condition(self, value: typing.Optional["CfnCondition"]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c340bd017ade3179e8e3171e1ad6f8f3ba29242e8489eb7e3c6980ea7628eaa0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "condition", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="creationPolicy")
    def creation_policy(self) -> typing.Optional[CfnCreationPolicy]:
        '''Associate the CreationPolicy attribute with a resource to prevent its status from reaching create complete until AWS CloudFormation receives a specified number of success signals or the timeout period is exceeded.

        To signal a
        resource, you can use the cfn-signal helper script or SignalResource API. AWS CloudFormation publishes valid signals
        to the stack events so that you track the number of signals sent.
        '''
        return typing.cast(typing.Optional[CfnCreationPolicy], jsii.get(self, "creationPolicy"))

    @creation_policy.setter
    def creation_policy(self, value: typing.Optional[CfnCreationPolicy]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__810d44b93f0de1f70deab96318824fa079fd4fad9830b45ed196a06bd2a32899)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "creationPolicy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="deletionPolicy")
    def deletion_policy(self) -> typing.Optional[CfnDeletionPolicy]:
        '''With the DeletionPolicy attribute you can preserve or (in some cases) backup a resource when its stack is deleted.

        You specify a DeletionPolicy attribute for each resource that you want to control. If a resource has no DeletionPolicy
        attribute, AWS CloudFormation deletes the resource by default. Note that this capability also applies to update operations
        that lead to resources being removed.
        '''
        return typing.cast(typing.Optional[CfnDeletionPolicy], jsii.get(self, "deletionPolicy"))

    @deletion_policy.setter
    def deletion_policy(self, value: typing.Optional[CfnDeletionPolicy]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d217244047834b21b19a7d6d13f0a9ee80513ca9adeed4eecb61cc60f4cbc083)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deletionPolicy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of this resource.

        Used for informational purposes only, is not processed in any way
        (and stays with the CloudFormation template, is not passed to the underlying resource,
        even if it does have a 'description' property).
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__85c1cdbe10f838e2728d0e14dd5d307715d9a727821b133204eb5fe64108ae86)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="metadata")
    def metadata(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Metadata associated with the CloudFormation resource.

        This is not the same as the construct metadata which can be added
        using construct.addMetadata(), but would not appear in the CloudFormation template automatically.
        '''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], jsii.get(self, "metadata"))

    @metadata.setter
    def metadata(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, typing.Any]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__36fe2b1cb86bc896248b9d242a84d7203ff331fa096782c69ed2ad15b923ac75)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "metadata", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="updatePolicy")
    def update_policy(self) -> typing.Optional[CfnUpdatePolicy]:
        '''Use the UpdatePolicy attribute to specify how AWS CloudFormation handles updates to the AWS::AutoScaling::AutoScalingGroup resource.

        AWS CloudFormation invokes one of three update policies depending on the type of change you make or whether a
        scheduled action is associated with the Auto Scaling group.
        '''
        return typing.cast(typing.Optional[CfnUpdatePolicy], jsii.get(self, "updatePolicy"))

    @update_policy.setter
    def update_policy(self, value: typing.Optional[CfnUpdatePolicy]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7c725970b5b7c9a0b3b4f52905cab1308f1db7c3e61f492c49e4c25040aca4bf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "updatePolicy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="updateReplacePolicy")
    def update_replace_policy(self) -> typing.Optional[CfnDeletionPolicy]:
        '''Use the UpdateReplacePolicy attribute to retain or (in some cases) backup the existing physical instance of a resource when it is replaced during a stack update operation.'''
        return typing.cast(typing.Optional[CfnDeletionPolicy], jsii.get(self, "updateReplacePolicy"))

    @update_replace_policy.setter
    def update_replace_policy(self, value: typing.Optional[CfnDeletionPolicy]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f38be0d686c18d026164a59d72c296e1e4f3624847d244ce628f7b90984abd80)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "updateReplacePolicy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[builtins.str]:
        '''The version of this resource.

        Used only for custom CloudFormation resources.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "version"))

    @version.setter
    def version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42237fa93d35be34f4782c5d322036259e005ed0e4f435315b820b187245110b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "version", value) # pyright: ignore[reportArgumentType]

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ICfnResourceOptions).__jsii_proxy_class__ = lambda : _ICfnResourceOptionsProxy


@jsii.interface(jsii_type="aws-cdk-lib.IFragmentConcatenator")
class IFragmentConcatenator(typing_extensions.Protocol):
    '''Function used to concatenate symbols in the target document language.

    Interface so it could potentially be exposed over jsii.
    '''

    @jsii.member(jsii_name="join")
    def join(self, left: typing.Any, right: typing.Any) -> typing.Any:
        '''Join the fragment on the left and on the right.

        :param left: -
        :param right: -
        '''
        ...


class _IFragmentConcatenatorProxy:
    '''Function used to concatenate symbols in the target document language.

    Interface so it could potentially be exposed over jsii.
    '''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IFragmentConcatenator"

    @jsii.member(jsii_name="join")
    def join(self, left: typing.Any, right: typing.Any) -> typing.Any:
        '''Join the fragment on the left and on the right.

        :param left: -
        :param right: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bfafa51d440aea5cf865cffef86aaa2ee71eefc8d710c07db95e541828c87596)
            check_type(argname="argument left", value=left, expected_type=type_hints["left"])
            check_type(argname="argument right", value=right, expected_type=type_hints["right"])
        return typing.cast(typing.Any, jsii.invoke(self, "join", [left, right]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IFragmentConcatenator).__jsii_proxy_class__ = lambda : _IFragmentConcatenatorProxy


@jsii.interface(jsii_type="aws-cdk-lib.IInspectable")
class IInspectable(typing_extensions.Protocol):
    '''Interface for examining a construct and exposing metadata.'''

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: "TreeInspector") -> None:
        '''Examines construct.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        ...


class _IInspectableProxy:
    '''Interface for examining a construct and exposing metadata.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IInspectable"

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: "TreeInspector") -> None:
        '''Examines construct.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1a75471862917255fac334b683fed2abafbf6f64092bdde033b04efea7341659)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IInspectable).__jsii_proxy_class__ = lambda : _IInspectableProxy


@jsii.interface(jsii_type="aws-cdk-lib.IListProducer")
class IListProducer(typing_extensions.Protocol):
    '''Interface for lazy list producers.'''

    @jsii.member(jsii_name="produce")
    def produce(
        self,
        context: "IResolveContext",
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''Produce the list value.

        :param context: -
        '''
        ...


class _IListProducerProxy:
    '''Interface for lazy list producers.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IListProducer"

    @jsii.member(jsii_name="produce")
    def produce(
        self,
        context: "IResolveContext",
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''Produce the list value.

        :param context: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a08adacd25e3643b80ce868ab487089be117119da27d733e04baedf154c2ad4)
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.invoke(self, "produce", [context]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IListProducer).__jsii_proxy_class__ = lambda : _IListProducerProxy


@jsii.interface(jsii_type="aws-cdk-lib.ILocalBundling")
class ILocalBundling(typing_extensions.Protocol):
    '''Local bundling.'''

    @jsii.member(jsii_name="tryBundle")
    def try_bundle(
        self,
        output_dir: builtins.str,
        *,
        image: DockerImage,
        bundling_file_access: typing.Optional[BundlingFileAccess] = None,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        local: typing.Optional["ILocalBundling"] = None,
        network: typing.Optional[builtins.str] = None,
        output_type: typing.Optional[BundlingOutput] = None,
        platform: typing.Optional[builtins.str] = None,
        security_opt: typing.Optional[builtins.str] = None,
        user: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Sequence[typing.Union[DockerVolume, typing.Dict[builtins.str, typing.Any]]]] = None,
        volumes_from: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> builtins.bool:
        '''This method is called before attempting docker bundling to allow the bundler to be executed locally.

        If the local bundler exists, and bundling
        was performed locally, return ``true``. Otherwise, return ``false``.

        :param output_dir: the directory where the bundled asset should be output.
        :param image: The Docker image where the command will run.
        :param bundling_file_access: The access mechanism used to make source files available to the bundling container and to return the bundling output back to the host. Default: - BundlingFileAccess.BIND_MOUNT
        :param command: The command to run in the Docker container. Example value: ``['npm', 'install']`` Default: - run the command defined in the image
        :param entrypoint: The entrypoint to run in the Docker container. Example value: ``['/bin/sh', '-c']`` Default: - run the entrypoint defined in the image
        :param environment: The environment variables to pass to the Docker container. Default: - no environment variables.
        :param local: Local bundling provider. The provider implements a method ``tryBundle()`` which should return ``true`` if local bundling was performed. If ``false`` is returned, docker bundling will be done. Default: - bundling will only be performed in a Docker container
        :param network: Docker `Networking options <https://docs.docker.com/engine/reference/commandline/run/#connect-a-container-to-a-network---network>`_. Default: - no networking options
        :param output_type: The type of output that this bundling operation is producing. Default: BundlingOutput.AUTO_DISCOVER
        :param platform: Platform to build for. *Requires Docker Buildx*. Specify this property to build images on a specific platform. Default: - no platform specified (the current machine architecture will be used)
        :param security_opt: `Security configuration <https://docs.docker.com/engine/reference/run/#security-configuration>`_ when running the docker container. Default: - no security options
        :param user: The user to use when running the Docker container. user | user:group | uid | uid:gid | user:gid | uid:group Default: - uid:gid of the current user or 1000:1000 on Windows
        :param volumes: Additional Docker volumes to mount. Default: - no additional volumes are mounted
        :param volumes_from: Where to mount the specified volumes from. Default: - no containers are specified to mount volumes from
        :param working_directory: Working directory inside the Docker container. Default: /asset-input
        '''
        ...


class _ILocalBundlingProxy:
    '''Local bundling.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.ILocalBundling"

    @jsii.member(jsii_name="tryBundle")
    def try_bundle(
        self,
        output_dir: builtins.str,
        *,
        image: DockerImage,
        bundling_file_access: typing.Optional[BundlingFileAccess] = None,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        local: typing.Optional[ILocalBundling] = None,
        network: typing.Optional[builtins.str] = None,
        output_type: typing.Optional[BundlingOutput] = None,
        platform: typing.Optional[builtins.str] = None,
        security_opt: typing.Optional[builtins.str] = None,
        user: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Sequence[typing.Union[DockerVolume, typing.Dict[builtins.str, typing.Any]]]] = None,
        volumes_from: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> builtins.bool:
        '''This method is called before attempting docker bundling to allow the bundler to be executed locally.

        If the local bundler exists, and bundling
        was performed locally, return ``true``. Otherwise, return ``false``.

        :param output_dir: the directory where the bundled asset should be output.
        :param image: The Docker image where the command will run.
        :param bundling_file_access: The access mechanism used to make source files available to the bundling container and to return the bundling output back to the host. Default: - BundlingFileAccess.BIND_MOUNT
        :param command: The command to run in the Docker container. Example value: ``['npm', 'install']`` Default: - run the command defined in the image
        :param entrypoint: The entrypoint to run in the Docker container. Example value: ``['/bin/sh', '-c']`` Default: - run the entrypoint defined in the image
        :param environment: The environment variables to pass to the Docker container. Default: - no environment variables.
        :param local: Local bundling provider. The provider implements a method ``tryBundle()`` which should return ``true`` if local bundling was performed. If ``false`` is returned, docker bundling will be done. Default: - bundling will only be performed in a Docker container
        :param network: Docker `Networking options <https://docs.docker.com/engine/reference/commandline/run/#connect-a-container-to-a-network---network>`_. Default: - no networking options
        :param output_type: The type of output that this bundling operation is producing. Default: BundlingOutput.AUTO_DISCOVER
        :param platform: Platform to build for. *Requires Docker Buildx*. Specify this property to build images on a specific platform. Default: - no platform specified (the current machine architecture will be used)
        :param security_opt: `Security configuration <https://docs.docker.com/engine/reference/run/#security-configuration>`_ when running the docker container. Default: - no security options
        :param user: The user to use when running the Docker container. user | user:group | uid | uid:gid | user:gid | uid:group Default: - uid:gid of the current user or 1000:1000 on Windows
        :param volumes: Additional Docker volumes to mount. Default: - no additional volumes are mounted
        :param volumes_from: Where to mount the specified volumes from. Default: - no containers are specified to mount volumes from
        :param working_directory: Working directory inside the Docker container. Default: /asset-input
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__218c8a03cb3dcf5b235ec3caba6a460cc22efe76513ec926cb85d9399ad214a6)
            check_type(argname="argument output_dir", value=output_dir, expected_type=type_hints["output_dir"])
        options = BundlingOptions(
            image=image,
            bundling_file_access=bundling_file_access,
            command=command,
            entrypoint=entrypoint,
            environment=environment,
            local=local,
            network=network,
            output_type=output_type,
            platform=platform,
            security_opt=security_opt,
            user=user,
            volumes=volumes,
            volumes_from=volumes_from,
            working_directory=working_directory,
        )

        return typing.cast(builtins.bool, jsii.invoke(self, "tryBundle", [output_dir, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ILocalBundling).__jsii_proxy_class__ = lambda : _ILocalBundlingProxy


@jsii.interface(jsii_type="aws-cdk-lib.INumberProducer")
class INumberProducer(typing_extensions.Protocol):
    '''Interface for lazy number producers.'''

    @jsii.member(jsii_name="produce")
    def produce(self, context: "IResolveContext") -> typing.Optional[jsii.Number]:
        '''Produce the number value.

        :param context: -
        '''
        ...


class _INumberProducerProxy:
    '''Interface for lazy number producers.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.INumberProducer"

    @jsii.member(jsii_name="produce")
    def produce(self, context: "IResolveContext") -> typing.Optional[jsii.Number]:
        '''Produce the number value.

        :param context: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30b3c7fb9a08baf4f7cd67be0951b086dfc7fd44085a0a6af2eb9881a9f52a27)
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
        return typing.cast(typing.Optional[jsii.Number], jsii.invoke(self, "produce", [context]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, INumberProducer).__jsii_proxy_class__ = lambda : _INumberProducerProxy


@jsii.interface(jsii_type="aws-cdk-lib.IPolicyValidationContextBeta1")
class IPolicyValidationContextBeta1(typing_extensions.Protocol):
    '''Context available to the validation plugin.'''

    @builtins.property
    @jsii.member(jsii_name="templatePaths")
    def template_paths(self) -> typing.List[builtins.str]:
        '''The absolute path of all templates to be processed.'''
        ...


class _IPolicyValidationContextBeta1Proxy:
    '''Context available to the validation plugin.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IPolicyValidationContextBeta1"

    @builtins.property
    @jsii.member(jsii_name="templatePaths")
    def template_paths(self) -> typing.List[builtins.str]:
        '''The absolute path of all templates to be processed.'''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "templatePaths"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IPolicyValidationContextBeta1).__jsii_proxy_class__ = lambda : _IPolicyValidationContextBeta1Proxy


@jsii.interface(jsii_type="aws-cdk-lib.IPolicyValidationPluginBeta1")
class IPolicyValidationPluginBeta1(typing_extensions.Protocol):
    '''Represents a validation plugin that will be executed during synthesis.

    Example::

        @jsii.implements(IPolicyValidationPluginBeta1)
        class MyPlugin:
        
            def validate(self, context):
                # First read the templates using context.templatePaths...
        
                # ...then perform the validation, and then compose and return the report.
                # Using hard-coded values here for better clarity:
                return PolicyValidationPluginReportBeta1(
                    success=False,
                    violations=[PolicyViolationBeta1(
                        rule_name="CKV_AWS_117",
                        description="Ensure that AWS Lambda function is configured inside a VPC",
                        fix="https://docs.bridgecrew.io/docs/ensure-that-aws-lambda-function-is-configured-inside-a-vpc-1",
                        violating_resources=[PolicyViolatingResourceBeta1(
                            resource_logical_id="MyFunction3BAA72D1",
                            template_path="/home/johndoe/myapp/cdk.out/MyService.template.json",
                            locations=["Properties/VpcConfig"]
                        )]
                    )]
                )
    '''

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the plugin that will be displayed in the validation report.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="ruleIds")
    def rule_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of rule IDs that the plugin will evaluate.

        Used for analytics
        purposes.

        :default: - No rule is reported
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[builtins.str]:
        '''The version of the plugin, following the Semantic Versioning specification (see https://semver.org/). This version is used for analytics purposes, to measure the usage of different plugins and different versions. The value of this property should be kept in sync with the actual version of the software package. If the version is not provided or is not a valid semantic version, it will be reported as ``0.0.0``.'''
        ...

    @jsii.member(jsii_name="validate")
    def validate(
        self,
        context: IPolicyValidationContextBeta1,
    ) -> "PolicyValidationPluginReportBeta1":
        '''The method that will be called by the CDK framework to perform validations.

        This is where the plugin will evaluate the CloudFormation
        templates for compliance and report and violations

        :param context: -
        '''
        ...


class _IPolicyValidationPluginBeta1Proxy:
    '''Represents a validation plugin that will be executed during synthesis.

    Example::

        @jsii.implements(IPolicyValidationPluginBeta1)
        class MyPlugin:
        
            def validate(self, context):
                # First read the templates using context.templatePaths...
        
                # ...then perform the validation, and then compose and return the report.
                # Using hard-coded values here for better clarity:
                return PolicyValidationPluginReportBeta1(
                    success=False,
                    violations=[PolicyViolationBeta1(
                        rule_name="CKV_AWS_117",
                        description="Ensure that AWS Lambda function is configured inside a VPC",
                        fix="https://docs.bridgecrew.io/docs/ensure-that-aws-lambda-function-is-configured-inside-a-vpc-1",
                        violating_resources=[PolicyViolatingResourceBeta1(
                            resource_logical_id="MyFunction3BAA72D1",
                            template_path="/home/johndoe/myapp/cdk.out/MyService.template.json",
                            locations=["Properties/VpcConfig"]
                        )]
                    )]
                )
    '''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IPolicyValidationPluginBeta1"

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the plugin that will be displayed in the validation report.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @builtins.property
    @jsii.member(jsii_name="ruleIds")
    def rule_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of rule IDs that the plugin will evaluate.

        Used for analytics
        purposes.

        :default: - No rule is reported
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "ruleIds"))

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[builtins.str]:
        '''The version of the plugin, following the Semantic Versioning specification (see https://semver.org/). This version is used for analytics purposes, to measure the usage of different plugins and different versions. The value of this property should be kept in sync with the actual version of the software package. If the version is not provided or is not a valid semantic version, it will be reported as ``0.0.0``.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "version"))

    @jsii.member(jsii_name="validate")
    def validate(
        self,
        context: IPolicyValidationContextBeta1,
    ) -> "PolicyValidationPluginReportBeta1":
        '''The method that will be called by the CDK framework to perform validations.

        This is where the plugin will evaluate the CloudFormation
        templates for compliance and report and violations

        :param context: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ff68896e7a1dd6c95e953098d0c8bc64151d6882e77d1d918aef207ff49dbfad)
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
        return typing.cast("PolicyValidationPluginReportBeta1", jsii.invoke(self, "validate", [context]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IPolicyValidationPluginBeta1).__jsii_proxy_class__ = lambda : _IPolicyValidationPluginBeta1Proxy


@jsii.interface(jsii_type="aws-cdk-lib.IPostProcessor")
class IPostProcessor(typing_extensions.Protocol):
    '''A Token that can post-process the complete resolved value, after resolve() has recursed over it.'''

    @jsii.member(jsii_name="postProcess")
    def post_process(self, input: typing.Any, context: "IResolveContext") -> typing.Any:
        '''Process the completely resolved value, after full recursion/resolution has happened.

        :param input: -
        :param context: -
        '''
        ...


class _IPostProcessorProxy:
    '''A Token that can post-process the complete resolved value, after resolve() has recursed over it.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IPostProcessor"

    @jsii.member(jsii_name="postProcess")
    def post_process(self, input: typing.Any, context: "IResolveContext") -> typing.Any:
        '''Process the completely resolved value, after full recursion/resolution has happened.

        :param input: -
        :param context: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__421f2755d7e81609d225eb32560b6f8749147026d0e9c4a8130c5421ad7a46f5)
            check_type(argname="argument input", value=input, expected_type=type_hints["input"])
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
        return typing.cast(typing.Any, jsii.invoke(self, "postProcess", [input, context]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IPostProcessor).__jsii_proxy_class__ = lambda : _IPostProcessorProxy


@jsii.interface(jsii_type="aws-cdk-lib.IResolvable")
class IResolvable(typing_extensions.Protocol):
    '''Interface for values that can be resolvable later.

    Tokens are special objects that participate in synthesis.
    '''

    @builtins.property
    @jsii.member(jsii_name="creationStack")
    def creation_stack(self) -> typing.List[builtins.str]:
        '''The creation stack of this resolvable which will be appended to errors thrown during resolution.

        This may return an array with a single informational element indicating how
        to get this property populated, if it was skipped for performance reasons.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="typeHint")
    def type_hint(self) -> typing.Optional["ResolutionTypeHint"]:
        '''The type that this token will likely resolve to.'''
        ...

    @jsii.member(jsii_name="resolve")
    def resolve(self, context: "IResolveContext") -> typing.Any:
        '''Produce the Token's value at resolution time.

        :param context: -
        '''
        ...

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Return a string representation of this resolvable object.

        Returns a reversible string representation.
        '''
        ...


class _IResolvableProxy:
    '''Interface for values that can be resolvable later.

    Tokens are special objects that participate in synthesis.
    '''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IResolvable"

    @builtins.property
    @jsii.member(jsii_name="creationStack")
    def creation_stack(self) -> typing.List[builtins.str]:
        '''The creation stack of this resolvable which will be appended to errors thrown during resolution.

        This may return an array with a single informational element indicating how
        to get this property populated, if it was skipped for performance reasons.
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "creationStack"))

    @builtins.property
    @jsii.member(jsii_name="typeHint")
    def type_hint(self) -> typing.Optional["ResolutionTypeHint"]:
        '''The type that this token will likely resolve to.'''
        return typing.cast(typing.Optional["ResolutionTypeHint"], jsii.get(self, "typeHint"))

    @jsii.member(jsii_name="resolve")
    def resolve(self, context: "IResolveContext") -> typing.Any:
        '''Produce the Token's value at resolution time.

        :param context: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__09341b9d2d6a06b3937530fa96a1bcb7562ccdf5246e5563a8a552ec1f8c0fa5)
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [context]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Return a string representation of this resolvable object.

        Returns a reversible string representation.
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IResolvable).__jsii_proxy_class__ = lambda : _IResolvableProxy


@jsii.interface(jsii_type="aws-cdk-lib.IResolveContext")
class IResolveContext(typing_extensions.Protocol):
    '''Current resolution context for tokens.'''

    @builtins.property
    @jsii.member(jsii_name="documentPath")
    def document_path(self) -> typing.List[builtins.str]:
        '''Path in the JSON document that is being constructed.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="preparing")
    def preparing(self) -> builtins.bool:
        '''True when we are still preparing, false if we're rendering the final output.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="scope")
    def scope(self) -> _constructs_77d1e7e8.IConstruct:
        '''The scope from which resolution has been initiated.'''
        ...

    @jsii.member(jsii_name="registerPostProcessor")
    def register_post_processor(self, post_processor: IPostProcessor) -> None:
        '''Use this postprocessor after the entire token structure has been resolved.

        :param post_processor: -
        '''
        ...

    @jsii.member(jsii_name="resolve")
    def resolve(
        self,
        x: typing.Any,
        *,
        allow_intrinsic_keys: typing.Optional[builtins.bool] = None,
        remove_empty: typing.Optional[builtins.bool] = None,
    ) -> typing.Any:
        '''Resolve an inner object.

        :param x: -
        :param allow_intrinsic_keys: Change the 'allowIntrinsicKeys' option. Default: - Unchanged
        :param remove_empty: Whether to remove undefined elements from arrays and objects when resolving. Default: - Unchanged
        '''
        ...


class _IResolveContextProxy:
    '''Current resolution context for tokens.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IResolveContext"

    @builtins.property
    @jsii.member(jsii_name="documentPath")
    def document_path(self) -> typing.List[builtins.str]:
        '''Path in the JSON document that is being constructed.'''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "documentPath"))

    @builtins.property
    @jsii.member(jsii_name="preparing")
    def preparing(self) -> builtins.bool:
        '''True when we are still preparing, false if we're rendering the final output.'''
        return typing.cast(builtins.bool, jsii.get(self, "preparing"))

    @builtins.property
    @jsii.member(jsii_name="scope")
    def scope(self) -> _constructs_77d1e7e8.IConstruct:
        '''The scope from which resolution has been initiated.'''
        return typing.cast(_constructs_77d1e7e8.IConstruct, jsii.get(self, "scope"))

    @jsii.member(jsii_name="registerPostProcessor")
    def register_post_processor(self, post_processor: IPostProcessor) -> None:
        '''Use this postprocessor after the entire token structure has been resolved.

        :param post_processor: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4dab78357f67cc2a481b6c04f75e260abebb31e228ac27ad9a3df36fc236679e)
            check_type(argname="argument post_processor", value=post_processor, expected_type=type_hints["post_processor"])
        return typing.cast(None, jsii.invoke(self, "registerPostProcessor", [post_processor]))

    @jsii.member(jsii_name="resolve")
    def resolve(
        self,
        x: typing.Any,
        *,
        allow_intrinsic_keys: typing.Optional[builtins.bool] = None,
        remove_empty: typing.Optional[builtins.bool] = None,
    ) -> typing.Any:
        '''Resolve an inner object.

        :param x: -
        :param allow_intrinsic_keys: Change the 'allowIntrinsicKeys' option. Default: - Unchanged
        :param remove_empty: Whether to remove undefined elements from arrays and objects when resolving. Default: - Unchanged
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__45145027bbaab577db67abc303cbf624b589bc400de57576f4b127e57d47c246)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        options = ResolveChangeContextOptions(
            allow_intrinsic_keys=allow_intrinsic_keys, remove_empty=remove_empty
        )

        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [x, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IResolveContext).__jsii_proxy_class__ = lambda : _IResolveContextProxy


@jsii.interface(jsii_type="aws-cdk-lib.IResource")
class IResource(_constructs_77d1e7e8.IConstruct, typing_extensions.Protocol):
    '''Interface for the Resource construct.'''

    @builtins.property
    @jsii.member(jsii_name="env")
    def env(self) -> "ResourceEnvironment":
        '''The environment this resource belongs to.

        For resources that are created and managed by the CDK
        (generally, those created by creating new class instances like Role, Bucket, etc.),
        this is always the same as the environment of the stack they belong to;
        however, for imported resources
        (those obtained from static methods like fromRoleArn, fromBucketName, etc.),
        that might be different than the stack they were imported into.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="stack")
    def stack(self) -> "Stack":
        '''The stack in which this resource is defined.'''
        ...

    @jsii.member(jsii_name="applyRemovalPolicy")
    def apply_removal_policy(self, policy: "RemovalPolicy") -> None:
        '''Apply the given removal policy to this resource.

        The Removal Policy controls what happens to this resource when it stops
        being managed by CloudFormation, either because you've removed it from the
        CDK application or because you've made a change that requires the resource
        to be replaced.

        The resource can be deleted (``RemovalPolicy.DESTROY``), or left in your AWS
        account for data recovery and cleanup later (``RemovalPolicy.RETAIN``).

        :param policy: -
        '''
        ...


class _IResourceProxy(
    jsii.proxy_for(_constructs_77d1e7e8.IConstruct), # type: ignore[misc]
):
    '''Interface for the Resource construct.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IResource"

    @builtins.property
    @jsii.member(jsii_name="env")
    def env(self) -> "ResourceEnvironment":
        '''The environment this resource belongs to.

        For resources that are created and managed by the CDK
        (generally, those created by creating new class instances like Role, Bucket, etc.),
        this is always the same as the environment of the stack they belong to;
        however, for imported resources
        (those obtained from static methods like fromRoleArn, fromBucketName, etc.),
        that might be different than the stack they were imported into.
        '''
        return typing.cast("ResourceEnvironment", jsii.get(self, "env"))

    @builtins.property
    @jsii.member(jsii_name="stack")
    def stack(self) -> "Stack":
        '''The stack in which this resource is defined.'''
        return typing.cast("Stack", jsii.get(self, "stack"))

    @jsii.member(jsii_name="applyRemovalPolicy")
    def apply_removal_policy(self, policy: "RemovalPolicy") -> None:
        '''Apply the given removal policy to this resource.

        The Removal Policy controls what happens to this resource when it stops
        being managed by CloudFormation, either because you've removed it from the
        CDK application or because you've made a change that requires the resource
        to be replaced.

        The resource can be deleted (``RemovalPolicy.DESTROY``), or left in your AWS
        account for data recovery and cleanup later (``RemovalPolicy.RETAIN``).

        :param policy: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6d5541801f9a5919d69716df13fab30782755f44f837d9460ced46c44db4c571)
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
        return typing.cast(None, jsii.invoke(self, "applyRemovalPolicy", [policy]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IResource).__jsii_proxy_class__ = lambda : _IResourceProxy


@jsii.interface(jsii_type="aws-cdk-lib.IStableAnyProducer")
class IStableAnyProducer(typing_extensions.Protocol):
    '''Interface for (stable) lazy untyped value producers.'''

    @jsii.member(jsii_name="produce")
    def produce(self) -> typing.Any:
        '''Produce the value.'''
        ...


class _IStableAnyProducerProxy:
    '''Interface for (stable) lazy untyped value producers.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IStableAnyProducer"

    @jsii.member(jsii_name="produce")
    def produce(self) -> typing.Any:
        '''Produce the value.'''
        return typing.cast(typing.Any, jsii.invoke(self, "produce", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStableAnyProducer).__jsii_proxy_class__ = lambda : _IStableAnyProducerProxy


@jsii.interface(jsii_type="aws-cdk-lib.IStableListProducer")
class IStableListProducer(typing_extensions.Protocol):
    '''Interface for (stable) lazy list producers.'''

    @jsii.member(jsii_name="produce")
    def produce(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Produce the list value.'''
        ...


class _IStableListProducerProxy:
    '''Interface for (stable) lazy list producers.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IStableListProducer"

    @jsii.member(jsii_name="produce")
    def produce(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Produce the list value.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.invoke(self, "produce", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStableListProducer).__jsii_proxy_class__ = lambda : _IStableListProducerProxy


@jsii.interface(jsii_type="aws-cdk-lib.IStableNumberProducer")
class IStableNumberProducer(typing_extensions.Protocol):
    '''Interface for (stable) lazy number producers.'''

    @jsii.member(jsii_name="produce")
    def produce(self) -> typing.Optional[jsii.Number]:
        '''Produce the number value.'''
        ...


class _IStableNumberProducerProxy:
    '''Interface for (stable) lazy number producers.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IStableNumberProducer"

    @jsii.member(jsii_name="produce")
    def produce(self) -> typing.Optional[jsii.Number]:
        '''Produce the number value.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.invoke(self, "produce", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStableNumberProducer).__jsii_proxy_class__ = lambda : _IStableNumberProducerProxy


@jsii.interface(jsii_type="aws-cdk-lib.IStableStringProducer")
class IStableStringProducer(typing_extensions.Protocol):
    '''Interface for (stable) lazy string producers.'''

    @jsii.member(jsii_name="produce")
    def produce(self) -> typing.Optional[builtins.str]:
        '''Produce the string value.'''
        ...


class _IStableStringProducerProxy:
    '''Interface for (stable) lazy string producers.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IStableStringProducer"

    @jsii.member(jsii_name="produce")
    def produce(self) -> typing.Optional[builtins.str]:
        '''Produce the string value.'''
        return typing.cast(typing.Optional[builtins.str], jsii.invoke(self, "produce", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStableStringProducer).__jsii_proxy_class__ = lambda : _IStableStringProducerProxy


@jsii.interface(jsii_type="aws-cdk-lib.IStackSynthesizer")
class IStackSynthesizer(typing_extensions.Protocol):
    '''Encodes information how a certain Stack should be deployed.'''

    @builtins.property
    @jsii.member(jsii_name="bootstrapQualifier")
    def bootstrap_qualifier(self) -> typing.Optional[builtins.str]:
        '''The qualifier used to bootstrap this stack.

        :default: - no qualifier
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="lookupRole")
    def lookup_role(self) -> typing.Optional[builtins.str]:
        '''The role used to lookup for this stack.

        :default: - no role
        '''
        ...

    @jsii.member(jsii_name="addDockerImageAsset")
    def add_docker_image_asset(
        self,
        *,
        source_hash: builtins.str,
        asset_name: typing.Optional[builtins.str] = None,
        directory_name: typing.Optional[builtins.str] = None,
        display_name: typing.Optional[builtins.str] = None,
        docker_build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_build_secrets: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_build_ssh: typing.Optional[builtins.str] = None,
        docker_build_target: typing.Optional[builtins.str] = None,
        docker_cache_disabled: typing.Optional[builtins.bool] = None,
        docker_cache_from: typing.Optional[typing.Sequence[typing.Union[DockerCacheOption, typing.Dict[builtins.str, typing.Any]]]] = None,
        docker_cache_to: typing.Optional[typing.Union[DockerCacheOption, typing.Dict[builtins.str, typing.Any]]] = None,
        docker_file: typing.Optional[builtins.str] = None,
        docker_outputs: typing.Optional[typing.Sequence[builtins.str]] = None,
        executable: typing.Optional[typing.Sequence[builtins.str]] = None,
        network_mode: typing.Optional[builtins.str] = None,
        platform: typing.Optional[builtins.str] = None,
    ) -> DockerImageAssetLocation:
        '''Register a Docker Image Asset.

        Returns the parameters that can be used to refer to the asset inside the template.

        :param source_hash: The hash of the contents of the docker build context. This hash is used throughout the system to identify this image and avoid duplicate work in case the source did not change. NOTE: this means that if you wish to update your docker image, you must make a modification to the source (e.g. add some metadata to your Dockerfile).
        :param asset_name: Unique identifier of the docker image asset and its potential revisions. Required if using AppScopedStagingSynthesizer. Default: - no asset name
        :param directory_name: The directory where the Dockerfile is stored, must be relative to the cloud assembly root. Default: - Exactly one of ``directoryName`` and ``executable`` is required
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. Default: - The asset hash is used to display the asset
        :param docker_build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Only allowed when ``directoryName`` is specified. Default: - no build args are passed
        :param docker_build_secrets: Build secrets to pass to the ``docker build`` command. Since Docker build secrets are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Only allowed when ``directoryName`` is specified. Default: - no build secrets are passed
        :param docker_build_ssh: SSH agent socket or keys to pass to the ``docker buildx`` command. Default: - no ssh arg is passed
        :param docker_build_target: Docker target to build to. Only allowed when ``directoryName`` is specified. Default: - no target
        :param docker_cache_disabled: Disable the cache and pass ``--no-cache`` to the ``docker build`` command. Default: - cache is used
        :param docker_cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from args are passed
        :param docker_cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to args are passed
        :param docker_file: Path to the Dockerfile (relative to the directory). Only allowed when ``directoryName`` is specified. Default: - no file
        :param docker_outputs: Outputs to pass to the ``docker build`` command. Default: - no build args are passed
        :param executable: An external command that will produce the packaged asset. The command should produce the name of a local Docker image on ``stdout``. Default: - Exactly one of ``directoryName`` and ``executable`` is required
        :param network_mode: Networking mode for the RUN commands during build. *Requires Docker Engine API v1.25+*. Specify this property to build images on a specific networking mode. Default: - no networking mode specified
        :param platform: Platform to build for. *Requires Docker Buildx*. Specify this property to build images on a specific platform. Default: - no platform specified (the current machine architecture will be used)
        '''
        ...

    @jsii.member(jsii_name="addFileAsset")
    def add_file_asset(
        self,
        *,
        source_hash: builtins.str,
        deploy_time: typing.Optional[builtins.bool] = None,
        display_name: typing.Optional[builtins.str] = None,
        executable: typing.Optional[typing.Sequence[builtins.str]] = None,
        file_name: typing.Optional[builtins.str] = None,
        packaging: typing.Optional[FileAssetPackaging] = None,
    ) -> FileAssetLocation:
        '''Register a File Asset.

        Returns the parameters that can be used to refer to the asset inside the template.

        :param source_hash: A hash on the content source. This hash is used to uniquely identify this asset throughout the system. If this value doesn't change, the asset will not be rebuilt or republished.
        :param deploy_time: Whether or not the asset needs to exist beyond deployment time; i.e. are copied over to a different location and not needed afterwards. Setting this property to true has an impact on the lifecycle of the asset, because we will assume that it is safe to delete after the CloudFormation deployment succeeds. For example, Lambda Function assets are copied over to Lambda during deployment. Therefore, it is not necessary to store the asset in S3, so we consider those deployTime assets. Default: false
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. Default: - The asset hash is used to display the asset
        :param executable: An external command that will produce the packaged asset. The command should produce the location of a ZIP file on ``stdout``. Default: - Exactly one of ``fileName`` and ``executable`` is required
        :param file_name: The path, relative to the root of the cloud assembly, in which this asset source resides. This can be a path to a file or a directory, depending on the packaging type. Default: - Exactly one of ``fileName`` and ``executable`` is required
        :param packaging: Which type of packaging to perform. Default: - Required if ``fileName`` is specified.
        '''
        ...

    @jsii.member(jsii_name="bind")
    def bind(self, stack: "Stack") -> None:
        '''Bind to the stack this environment is going to be used on.

        Must be called before any of the other methods are called, and can only be called once.

        :param stack: -
        '''
        ...

    @jsii.member(jsii_name="synthesize")
    def synthesize(self, session: "ISynthesisSession") -> None:
        '''Synthesize the associated stack to the session.

        :param session: -
        '''
        ...


class _IStackSynthesizerProxy:
    '''Encodes information how a certain Stack should be deployed.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IStackSynthesizer"

    @builtins.property
    @jsii.member(jsii_name="bootstrapQualifier")
    def bootstrap_qualifier(self) -> typing.Optional[builtins.str]:
        '''The qualifier used to bootstrap this stack.

        :default: - no qualifier
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "bootstrapQualifier"))

    @builtins.property
    @jsii.member(jsii_name="lookupRole")
    def lookup_role(self) -> typing.Optional[builtins.str]:
        '''The role used to lookup for this stack.

        :default: - no role
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "lookupRole"))

    @jsii.member(jsii_name="addDockerImageAsset")
    def add_docker_image_asset(
        self,
        *,
        source_hash: builtins.str,
        asset_name: typing.Optional[builtins.str] = None,
        directory_name: typing.Optional[builtins.str] = None,
        display_name: typing.Optional[builtins.str] = None,
        docker_build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_build_secrets: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_build_ssh: typing.Optional[builtins.str] = None,
        docker_build_target: typing.Optional[builtins.str] = None,
        docker_cache_disabled: typing.Optional[builtins.bool] = None,
        docker_cache_from: typing.Optional[typing.Sequence[typing.Union[DockerCacheOption, typing.Dict[builtins.str, typing.Any]]]] = None,
        docker_cache_to: typing.Optional[typing.Union[DockerCacheOption, typing.Dict[builtins.str, typing.Any]]] = None,
        docker_file: typing.Optional[builtins.str] = None,
        docker_outputs: typing.Optional[typing.Sequence[builtins.str]] = None,
        executable: typing.Optional[typing.Sequence[builtins.str]] = None,
        network_mode: typing.Optional[builtins.str] = None,
        platform: typing.Optional[builtins.str] = None,
    ) -> DockerImageAssetLocation:
        '''Register a Docker Image Asset.

        Returns the parameters that can be used to refer to the asset inside the template.

        :param source_hash: The hash of the contents of the docker build context. This hash is used throughout the system to identify this image and avoid duplicate work in case the source did not change. NOTE: this means that if you wish to update your docker image, you must make a modification to the source (e.g. add some metadata to your Dockerfile).
        :param asset_name: Unique identifier of the docker image asset and its potential revisions. Required if using AppScopedStagingSynthesizer. Default: - no asset name
        :param directory_name: The directory where the Dockerfile is stored, must be relative to the cloud assembly root. Default: - Exactly one of ``directoryName`` and ``executable`` is required
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. Default: - The asset hash is used to display the asset
        :param docker_build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Only allowed when ``directoryName`` is specified. Default: - no build args are passed
        :param docker_build_secrets: Build secrets to pass to the ``docker build`` command. Since Docker build secrets are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Only allowed when ``directoryName`` is specified. Default: - no build secrets are passed
        :param docker_build_ssh: SSH agent socket or keys to pass to the ``docker buildx`` command. Default: - no ssh arg is passed
        :param docker_build_target: Docker target to build to. Only allowed when ``directoryName`` is specified. Default: - no target
        :param docker_cache_disabled: Disable the cache and pass ``--no-cache`` to the ``docker build`` command. Default: - cache is used
        :param docker_cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from args are passed
        :param docker_cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to args are passed
        :param docker_file: Path to the Dockerfile (relative to the directory). Only allowed when ``directoryName`` is specified. Default: - no file
        :param docker_outputs: Outputs to pass to the ``docker build`` command. Default: - no build args are passed
        :param executable: An external command that will produce the packaged asset. The command should produce the name of a local Docker image on ``stdout``. Default: - Exactly one of ``directoryName`` and ``executable`` is required
        :param network_mode: Networking mode for the RUN commands during build. *Requires Docker Engine API v1.25+*. Specify this property to build images on a specific networking mode. Default: - no networking mode specified
        :param platform: Platform to build for. *Requires Docker Buildx*. Specify this property to build images on a specific platform. Default: - no platform specified (the current machine architecture will be used)
        '''
        asset = DockerImageAssetSource(
            source_hash=source_hash,
            asset_name=asset_name,
            directory_name=directory_name,
            display_name=display_name,
            docker_build_args=docker_build_args,
            docker_build_secrets=docker_build_secrets,
            docker_build_ssh=docker_build_ssh,
            docker_build_target=docker_build_target,
            docker_cache_disabled=docker_cache_disabled,
            docker_cache_from=docker_cache_from,
            docker_cache_to=docker_cache_to,
            docker_file=docker_file,
            docker_outputs=docker_outputs,
            executable=executable,
            network_mode=network_mode,
            platform=platform,
        )

        return typing.cast(DockerImageAssetLocation, jsii.invoke(self, "addDockerImageAsset", [asset]))

    @jsii.member(jsii_name="addFileAsset")
    def add_file_asset(
        self,
        *,
        source_hash: builtins.str,
        deploy_time: typing.Optional[builtins.bool] = None,
        display_name: typing.Optional[builtins.str] = None,
        executable: typing.Optional[typing.Sequence[builtins.str]] = None,
        file_name: typing.Optional[builtins.str] = None,
        packaging: typing.Optional[FileAssetPackaging] = None,
    ) -> FileAssetLocation:
        '''Register a File Asset.

        Returns the parameters that can be used to refer to the asset inside the template.

        :param source_hash: A hash on the content source. This hash is used to uniquely identify this asset throughout the system. If this value doesn't change, the asset will not be rebuilt or republished.
        :param deploy_time: Whether or not the asset needs to exist beyond deployment time; i.e. are copied over to a different location and not needed afterwards. Setting this property to true has an impact on the lifecycle of the asset, because we will assume that it is safe to delete after the CloudFormation deployment succeeds. For example, Lambda Function assets are copied over to Lambda during deployment. Therefore, it is not necessary to store the asset in S3, so we consider those deployTime assets. Default: false
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. Default: - The asset hash is used to display the asset
        :param executable: An external command that will produce the packaged asset. The command should produce the location of a ZIP file on ``stdout``. Default: - Exactly one of ``fileName`` and ``executable`` is required
        :param file_name: The path, relative to the root of the cloud assembly, in which this asset source resides. This can be a path to a file or a directory, depending on the packaging type. Default: - Exactly one of ``fileName`` and ``executable`` is required
        :param packaging: Which type of packaging to perform. Default: - Required if ``fileName`` is specified.
        '''
        asset = FileAssetSource(
            source_hash=source_hash,
            deploy_time=deploy_time,
            display_name=display_name,
            executable=executable,
            file_name=file_name,
            packaging=packaging,
        )

        return typing.cast(FileAssetLocation, jsii.invoke(self, "addFileAsset", [asset]))

    @jsii.member(jsii_name="bind")
    def bind(self, stack: "Stack") -> None:
        '''Bind to the stack this environment is going to be used on.

        Must be called before any of the other methods are called, and can only be called once.

        :param stack: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76602730e22afc5fda286d991c9f104d35c9545fbadd2863c9497b98b7592ebb)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
        return typing.cast(None, jsii.invoke(self, "bind", [stack]))

    @jsii.member(jsii_name="synthesize")
    def synthesize(self, session: "ISynthesisSession") -> None:
        '''Synthesize the associated stack to the session.

        :param session: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3b3965dd30316829afbc88f685d046d1d1fe91f253fc244eef72dcf7111981ca)
            check_type(argname="argument session", value=session, expected_type=type_hints["session"])
        return typing.cast(None, jsii.invoke(self, "synthesize", [session]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStackSynthesizer).__jsii_proxy_class__ = lambda : _IStackSynthesizerProxy


@jsii.interface(jsii_type="aws-cdk-lib.IStringProducer")
class IStringProducer(typing_extensions.Protocol):
    '''Interface for lazy string producers.'''

    @jsii.member(jsii_name="produce")
    def produce(self, context: IResolveContext) -> typing.Optional[builtins.str]:
        '''Produce the string value.

        :param context: -
        '''
        ...


class _IStringProducerProxy:
    '''Interface for lazy string producers.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.IStringProducer"

    @jsii.member(jsii_name="produce")
    def produce(self, context: IResolveContext) -> typing.Optional[builtins.str]:
        '''Produce the string value.

        :param context: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9351a4ff13506cce5153011386184952bdb6db44580e204cfe4b7bfd16e659f3)
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
        return typing.cast(typing.Optional[builtins.str], jsii.invoke(self, "produce", [context]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStringProducer).__jsii_proxy_class__ = lambda : _IStringProducerProxy


@jsii.interface(jsii_type="aws-cdk-lib.ISynthesisSession")
class ISynthesisSession(typing_extensions.Protocol):
    '''Represents a single session of synthesis.

    Passed into ``Construct.synthesize()`` methods.
    '''

    @builtins.property
    @jsii.member(jsii_name="assembly")
    def assembly(self) -> _CloudAssemblyBuilder_c90cccf3:
        '''Cloud assembly builder.'''
        ...

    @assembly.setter
    def assembly(self, value: _CloudAssemblyBuilder_c90cccf3) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="outdir")
    def outdir(self) -> builtins.str:
        '''The output directory for this synthesis session.'''
        ...

    @outdir.setter
    def outdir(self, value: builtins.str) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="validateOnSynth")
    def validate_on_synth(self) -> typing.Optional[builtins.bool]:
        '''Whether the stack should be validated after synthesis to check for error metadata.

        :default: - false
        '''
        ...

    @validate_on_synth.setter
    def validate_on_synth(self, value: typing.Optional[builtins.bool]) -> None:
        ...


class _ISynthesisSessionProxy:
    '''Represents a single session of synthesis.

    Passed into ``Construct.synthesize()`` methods.
    '''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.ISynthesisSession"

    @builtins.property
    @jsii.member(jsii_name="assembly")
    def assembly(self) -> _CloudAssemblyBuilder_c90cccf3:
        '''Cloud assembly builder.'''
        return typing.cast(_CloudAssemblyBuilder_c90cccf3, jsii.get(self, "assembly"))

    @assembly.setter
    def assembly(self, value: _CloudAssemblyBuilder_c90cccf3) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__46911aaa2f35f64b5c8d1cad71cbd1206b72081a1063b06d7ea031c371fdd641)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "assembly", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="outdir")
    def outdir(self) -> builtins.str:
        '''The output directory for this synthesis session.'''
        return typing.cast(builtins.str, jsii.get(self, "outdir"))

    @outdir.setter
    def outdir(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__22d70f5ea7ea04eb8f4a37057f361ddffad9030cd0ae9f9ca1b37feeb53bddf5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "outdir", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="validateOnSynth")
    def validate_on_synth(self) -> typing.Optional[builtins.bool]:
        '''Whether the stack should be validated after synthesis to check for error metadata.

        :default: - false
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "validateOnSynth"))

    @validate_on_synth.setter
    def validate_on_synth(self, value: typing.Optional[builtins.bool]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__430bfb7b63050c3533b372b1ee6579b19303be2daa47188ff3e88d2926b3156c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "validateOnSynth", value) # pyright: ignore[reportArgumentType]

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ISynthesisSession).__jsii_proxy_class__ = lambda : _ISynthesisSessionProxy


@jsii.interface(jsii_type="aws-cdk-lib.ITaggable")
class ITaggable(typing_extensions.Protocol):
    '''Interface to implement tags.'''

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> "TagManager":
        '''TagManager to set, remove and format tags.'''
        ...


class _ITaggableProxy:
    '''Interface to implement tags.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.ITaggable"

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> "TagManager":
        '''TagManager to set, remove and format tags.'''
        return typing.cast("TagManager", jsii.get(self, "tags"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ITaggable).__jsii_proxy_class__ = lambda : _ITaggableProxy


@jsii.interface(jsii_type="aws-cdk-lib.ITaggableV2")
class ITaggableV2(typing_extensions.Protocol):
    '''Modernized version of ITaggable.

    ``ITaggable`` has a problem: for a number of L1 resources, we failed to generate
    ``tags: TagManager``, and generated ``tags: CfnSomeResource.TagProperty[]`` instead.

    To mark these resources as taggable, we need to put the ``TagManager`` in a new property
    whose name is unlikely to conflict with any existing properties. Hence, a new interface
    for that purpose. All future resources will implement ``ITaggableV2``.
    '''

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> "TagManager":
        '''TagManager to set, remove and format tags.'''
        ...


class _ITaggableV2Proxy:
    '''Modernized version of ITaggable.

    ``ITaggable`` has a problem: for a number of L1 resources, we failed to generate
    ``tags: TagManager``, and generated ``tags: CfnSomeResource.TagProperty[]`` instead.

    To mark these resources as taggable, we need to put the ``TagManager`` in a new property
    whose name is unlikely to conflict with any existing properties. Hence, a new interface
    for that purpose. All future resources will implement ``ITaggableV2``.
    '''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.ITaggableV2"

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> "TagManager":
        '''TagManager to set, remove and format tags.'''
        return typing.cast("TagManager", jsii.get(self, "cdkTagManager"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ITaggableV2).__jsii_proxy_class__ = lambda : _ITaggableV2Proxy


@jsii.interface(jsii_type="aws-cdk-lib.ITemplateOptions")
class ITemplateOptions(typing_extensions.Protocol):
    '''CloudFormation template options for a stack.'''

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''Gets or sets the description of this stack.

        If provided, it will be included in the CloudFormation template's "Description" attribute.
        '''
        ...

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="metadata")
    def metadata(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Metadata associated with the CloudFormation template.'''
        ...

    @metadata.setter
    def metadata(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, typing.Any]],
    ) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="templateFormatVersion")
    def template_format_version(self) -> typing.Optional[builtins.str]:
        '''Gets or sets the AWSTemplateFormatVersion field of the CloudFormation template.'''
        ...

    @template_format_version.setter
    def template_format_version(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property
    @jsii.member(jsii_name="transforms")
    def transforms(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Gets or sets the top-level template transform(s) for this stack (e.g. ``["AWS::Serverless-2016-10-31"]``).'''
        ...

    @transforms.setter
    def transforms(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        ...


class _ITemplateOptionsProxy:
    '''CloudFormation template options for a stack.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.ITemplateOptions"

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''Gets or sets the description of this stack.

        If provided, it will be included in the CloudFormation template's "Description" attribute.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f0d6e5bd1a069c4761813858df94c7552e68e02237fa641508f5ed8ebe319b67)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="metadata")
    def metadata(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Metadata associated with the CloudFormation template.'''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], jsii.get(self, "metadata"))

    @metadata.setter
    def metadata(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, typing.Any]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6bcc681ac84509a3a0616141f17d6581bd269eca64bc60741b547a56ed1e85a1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "metadata", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="templateFormatVersion")
    def template_format_version(self) -> typing.Optional[builtins.str]:
        '''Gets or sets the AWSTemplateFormatVersion field of the CloudFormation template.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "templateFormatVersion"))

    @template_format_version.setter
    def template_format_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ac3a835d9d8a0cf07a444f7b9a14b82dfdd549d911dcb27d0164d87229852292)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "templateFormatVersion", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="transforms")
    def transforms(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Gets or sets the top-level template transform(s) for this stack (e.g. ``["AWS::Serverless-2016-10-31"]``).'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "transforms"))

    @transforms.setter
    def transforms(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6177de085e2d8887b29294dfdd9d70a505769acece57f422e90bd9865e27266d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "transforms", value) # pyright: ignore[reportArgumentType]

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ITemplateOptions).__jsii_proxy_class__ = lambda : _ITemplateOptionsProxy


@jsii.interface(jsii_type="aws-cdk-lib.ITokenMapper")
class ITokenMapper(typing_extensions.Protocol):
    '''Interface to apply operation to tokens in a string.

    Interface so it can be exported via jsii.
    '''

    @jsii.member(jsii_name="mapToken")
    def map_token(self, t: IResolvable) -> typing.Any:
        '''Replace a single token.

        :param t: -
        '''
        ...


class _ITokenMapperProxy:
    '''Interface to apply operation to tokens in a string.

    Interface so it can be exported via jsii.
    '''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.ITokenMapper"

    @jsii.member(jsii_name="mapToken")
    def map_token(self, t: IResolvable) -> typing.Any:
        '''Replace a single token.

        :param t: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__afad57955bb17ca5bd56eaabe68f45accf4f551ce1f14b7747c3a078d0c0ab69)
            check_type(argname="argument t", value=t, expected_type=type_hints["t"])
        return typing.cast(typing.Any, jsii.invoke(self, "mapToken", [t]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ITokenMapper).__jsii_proxy_class__ = lambda : _ITokenMapperProxy


@jsii.interface(jsii_type="aws-cdk-lib.ITokenResolver")
class ITokenResolver(typing_extensions.Protocol):
    '''How to resolve tokens.'''

    @jsii.member(jsii_name="resolveList")
    def resolve_list(
        self,
        l: typing.Sequence[builtins.str],
        context: IResolveContext,
    ) -> typing.Any:
        '''Resolve a tokenized list.

        :param l: -
        :param context: -
        '''
        ...

    @jsii.member(jsii_name="resolveString")
    def resolve_string(
        self,
        s: "TokenizedStringFragments",
        context: IResolveContext,
    ) -> typing.Any:
        '''Resolve a string with at least one stringified token in it.

        (May use concatenation)

        :param s: -
        :param context: -
        '''
        ...

    @jsii.member(jsii_name="resolveToken")
    def resolve_token(
        self,
        t: IResolvable,
        context: IResolveContext,
        post_processor: IPostProcessor,
    ) -> typing.Any:
        '''Resolve a single token.

        :param t: -
        :param context: -
        :param post_processor: -
        '''
        ...


class _ITokenResolverProxy:
    '''How to resolve tokens.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.ITokenResolver"

    @jsii.member(jsii_name="resolveList")
    def resolve_list(
        self,
        l: typing.Sequence[builtins.str],
        context: IResolveContext,
    ) -> typing.Any:
        '''Resolve a tokenized list.

        :param l: -
        :param context: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cadfbbcbc9807f77890b67d7e296ee35d1a66321b338be2d2481d57dbeaa47c0)
            check_type(argname="argument l", value=l, expected_type=type_hints["l"])
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolveList", [l, context]))

    @jsii.member(jsii_name="resolveString")
    def resolve_string(
        self,
        s: "TokenizedStringFragments",
        context: IResolveContext,
    ) -> typing.Any:
        '''Resolve a string with at least one stringified token in it.

        (May use concatenation)

        :param s: -
        :param context: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0f63e96d2ad8aac1d8518c41835e8931e3a21c6205fe7b02bf15d5a89253d4ea)
            check_type(argname="argument s", value=s, expected_type=type_hints["s"])
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolveString", [s, context]))

    @jsii.member(jsii_name="resolveToken")
    def resolve_token(
        self,
        t: IResolvable,
        context: IResolveContext,
        post_processor: IPostProcessor,
    ) -> typing.Any:
        '''Resolve a single token.

        :param t: -
        :param context: -
        :param post_processor: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__53aadffeb9f088ccaa612e3090e7ef0814fbd9a5f8fd4c684365e5ecf0adc63e)
            check_type(argname="argument t", value=t, expected_type=type_hints["t"])
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
            check_type(argname="argument post_processor", value=post_processor, expected_type=type_hints["post_processor"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolveToken", [t, context, post_processor]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ITokenResolver).__jsii_proxy_class__ = lambda : _ITokenResolverProxy


@jsii.enum(jsii_type="aws-cdk-lib.IgnoreMode")
class IgnoreMode(enum.Enum):
    '''Determines the ignore behavior to use.

    :exampleMetadata: infused

    Example::

        lambda_.Function(self, "Function",
            code=lambda_.Code.from_asset(path.join(__dirname, "my-python-handler"),
                exclude=["*.ignore"],
                ignore_mode=IgnoreMode.DOCKER
            ),
            runtime=lambda_.Runtime.PYTHON_3_9,
            handler="index.handler"
        )
    '''

    GLOB = "GLOB"
    '''Ignores file paths based on simple glob patterns.

    This is the default for file assets.

    It is also the default for Docker image assets, unless the '@aws-cdk/aws-ecr-assets:dockerIgnoreSupport'
    context flag is set.
    '''
    GIT = "GIT"
    '''Ignores file paths based on the ```.gitignore specification`` <https://git-scm.com/docs/gitignore>`_.'''
    DOCKER = "DOCKER"
    '''Ignores file paths based on the ```.dockerignore specification`` <https://docs.docker.com/engine/reference/builder/#dockerignore-file>`_.

    This is the default for Docker image assets if the '@aws-cdk/aws-ecr-assets:dockerIgnoreSupport'
    context flag is set.
    '''


class IgnoreStrategy(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.IgnoreStrategy",
):
    '''Represents file path ignoring behavior.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        ignore_strategy = cdk.IgnoreStrategy.from_copy_options(cdk.CopyOptions(
            exclude=["exclude"],
            follow=cdk.SymlinkFollowMode.NEVER,
            ignore_mode=cdk.IgnoreMode.GLOB
        ), "absoluteRootPath")
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="docker")
    @builtins.classmethod
    def docker(
        cls,
        absolute_root_path: builtins.str,
        patterns: typing.Sequence[builtins.str],
    ) -> "DockerIgnoreStrategy":
        '''Ignores file paths based on the ```.dockerignore specification`` <https://docs.docker.com/engine/reference/builder/#dockerignore-file>`_.

        :param absolute_root_path: the absolute path to the root directory of the paths to be considered.
        :param patterns: -

        :return: ``DockerIgnorePattern`` associated with the given patterns.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cdc10917b064e113453c6c338848cdd2ca58bcd257b97a100498364fc95a4208)
            check_type(argname="argument absolute_root_path", value=absolute_root_path, expected_type=type_hints["absolute_root_path"])
            check_type(argname="argument patterns", value=patterns, expected_type=type_hints["patterns"])
        return typing.cast("DockerIgnoreStrategy", jsii.sinvoke(cls, "docker", [absolute_root_path, patterns]))

    @jsii.member(jsii_name="fromCopyOptions")
    @builtins.classmethod
    def from_copy_options(
        cls,
        options: typing.Union[CopyOptions, typing.Dict[builtins.str, typing.Any]],
        absolute_root_path: builtins.str,
    ) -> "IgnoreStrategy":
        '''Creates an IgnoreStrategy based on the ``ignoreMode`` and ``exclude`` in a ``CopyOptions``.

        :param options: the ``CopyOptions`` to create the ``IgnoreStrategy`` from.
        :param absolute_root_path: the absolute path to the root directory of the paths to be considered.

        :return: ``IgnoreStrategy`` based on the ``CopyOptions``
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6e47fba8c3c050212ae55fd57905efc0e009d182e5de256fd7afd1837c1871b0)
            check_type(argname="argument options", value=options, expected_type=type_hints["options"])
            check_type(argname="argument absolute_root_path", value=absolute_root_path, expected_type=type_hints["absolute_root_path"])
        return typing.cast("IgnoreStrategy", jsii.sinvoke(cls, "fromCopyOptions", [options, absolute_root_path]))

    @jsii.member(jsii_name="git")
    @builtins.classmethod
    def git(
        cls,
        absolute_root_path: builtins.str,
        patterns: typing.Sequence[builtins.str],
    ) -> "GitIgnoreStrategy":
        '''Ignores file paths based on the ```.gitignore specification`` <https://git-scm.com/docs/gitignore>`_.

        :param absolute_root_path: the absolute path to the root directory of the paths to be considered.
        :param patterns: -

        :return: ``GitIgnorePattern`` associated with the given patterns.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c8c92e95c1fd4f104e347f9bc9a221e466687a6a52e765284a357724eff94500)
            check_type(argname="argument absolute_root_path", value=absolute_root_path, expected_type=type_hints["absolute_root_path"])
            check_type(argname="argument patterns", value=patterns, expected_type=type_hints["patterns"])
        return typing.cast("GitIgnoreStrategy", jsii.sinvoke(cls, "git", [absolute_root_path, patterns]))

    @jsii.member(jsii_name="glob")
    @builtins.classmethod
    def glob(
        cls,
        absolute_root_path: builtins.str,
        patterns: typing.Sequence[builtins.str],
    ) -> "GlobIgnoreStrategy":
        '''Ignores file paths based on simple glob patterns.

        :param absolute_root_path: the absolute path to the root directory of the paths to be considered.
        :param patterns: -

        :return: ``GlobIgnorePattern`` associated with the given patterns.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eece2ad7e2f3cbe035a92302343ee656a4e8438489e1b15a43554b124e4ab0d2)
            check_type(argname="argument absolute_root_path", value=absolute_root_path, expected_type=type_hints["absolute_root_path"])
            check_type(argname="argument patterns", value=patterns, expected_type=type_hints["patterns"])
        return typing.cast("GlobIgnoreStrategy", jsii.sinvoke(cls, "glob", [absolute_root_path, patterns]))

    @jsii.member(jsii_name="add")
    @abc.abstractmethod
    def add(self, pattern: builtins.str) -> None:
        '''Adds another pattern.

        :param pattern: -

        :params: pattern the pattern to add
        '''
        ...

    @jsii.member(jsii_name="ignores")
    @abc.abstractmethod
    def ignores(self, absolute_file_path: builtins.str) -> builtins.bool:
        '''Determines whether a given file path should be ignored or not.

        :param absolute_file_path: absolute file path to be assessed against the pattern.

        :return: ``true`` if the file should be ignored
        '''
        ...


class _IgnoreStrategyProxy(IgnoreStrategy):
    @jsii.member(jsii_name="add")
    def add(self, pattern: builtins.str) -> None:
        '''Adds another pattern.

        :param pattern: -

        :params: pattern the pattern to add
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30e8e458c385d1db329cfc0c45f966d2d15c1b0166b3e7cfe8a57c7809196bf5)
            check_type(argname="argument pattern", value=pattern, expected_type=type_hints["pattern"])
        return typing.cast(None, jsii.invoke(self, "add", [pattern]))

    @jsii.member(jsii_name="ignores")
    def ignores(self, absolute_file_path: builtins.str) -> builtins.bool:
        '''Determines whether a given file path should be ignored or not.

        :param absolute_file_path: absolute file path to be assessed against the pattern.

        :return: ``true`` if the file should be ignored
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__639820bc3ab9fb78a31dec5d3d916a2befafc70af8c041b6e444ac075904e2f5)
            check_type(argname="argument absolute_file_path", value=absolute_file_path, expected_type=type_hints["absolute_file_path"])
        return typing.cast(builtins.bool, jsii.invoke(self, "ignores", [absolute_file_path]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, IgnoreStrategy).__jsii_proxy_class__ = lambda : _IgnoreStrategyProxy


@jsii.implements(IResolvable)
class Intrinsic(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.Intrinsic"):
    '''Token subclass that represents values intrinsic to the target document language.

    WARNING: this class should not be externally exposed, but is currently visible
    because of a limitation of jsii (https://github.com/aws/jsii/issues/524).

    This class will disappear in a future release and should not be used.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        # value: Any
        
        intrinsic = cdk.Intrinsic(value,
            stack_trace=False,
            type_hint=cdk.ResolutionTypeHint.STRING
        )
    '''

    def __init__(
        self,
        value: typing.Any,
        *,
        stack_trace: typing.Optional[builtins.bool] = None,
        type_hint: typing.Optional["ResolutionTypeHint"] = None,
    ) -> None:
        '''
        :param value: -
        :param stack_trace: Capture the stack trace of where this token is created. Default: true
        :param type_hint: Type that this token is expected to evaluate to. Default: ResolutionTypeHint.STRING
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4fdfa4eddae987229a9f6236b6c0b78b4d7e291c17d32a624ad710cd632917ff)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        options = IntrinsicProps(stack_trace=stack_trace, type_hint=type_hint)

        jsii.create(self.__class__, self, [value, options])

    @jsii.member(jsii_name="newError")
    def _new_error(self, message: builtins.str) -> typing.Any:
        '''Creates a throwable Error object that contains the token creation stack trace.

        :param message: Error message.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c20b1d83a93d00dcee2affc768f1ef97b9bb74531862804b989e173228cb00f7)
            check_type(argname="argument message", value=message, expected_type=type_hints["message"])
        return typing.cast(typing.Any, jsii.invoke(self, "newError", [message]))

    @jsii.member(jsii_name="resolve")
    def resolve(self, _context: IResolveContext) -> typing.Any:
        '''Produce the Token's value at resolution time.

        :param _context: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__50acf50c393d626484e54803d52c208f8b31db97b9731240ea6a6f226d851b17)
            check_type(argname="argument _context", value=_context, expected_type=type_hints["_context"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [_context]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''Turn this Token into JSON.

        Called automatically when JSON.stringify() is called on a Token.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Convert an instance of this Token to a string.

        This method will be called implicitly by language runtimes if the object
        is embedded into a string. We treat it the same as an explicit
        stringification.
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.member(jsii_name="toStringList")
    def to_string_list(self) -> typing.List[builtins.str]:
        '''Convert an instance of this Token to a string list.

        This method will be called implicitly by language runtimes if the object
        is embedded into a list. We treat it the same as an explicit
        stringification.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "toStringList", []))

    @builtins.property
    @jsii.member(jsii_name="creationStack")
    def creation_stack(self) -> typing.List[builtins.str]:
        '''The captured stack trace which represents the location in which this token was created.'''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "creationStack"))

    @builtins.property
    @jsii.member(jsii_name="typeHint")
    def type_hint(self) -> typing.Optional["ResolutionTypeHint"]:
        '''Type that the Intrinsic is expected to evaluate to.'''
        return typing.cast(typing.Optional["ResolutionTypeHint"], jsii.get(self, "typeHint"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.IntrinsicProps",
    jsii_struct_bases=[],
    name_mapping={"stack_trace": "stackTrace", "type_hint": "typeHint"},
)
class IntrinsicProps:
    def __init__(
        self,
        *,
        stack_trace: typing.Optional[builtins.bool] = None,
        type_hint: typing.Optional["ResolutionTypeHint"] = None,
    ) -> None:
        '''Customization properties for an Intrinsic token.

        :param stack_trace: Capture the stack trace of where this token is created. Default: true
        :param type_hint: Type that this token is expected to evaluate to. Default: ResolutionTypeHint.STRING

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            intrinsic_props = cdk.IntrinsicProps(
                stack_trace=False,
                type_hint=cdk.ResolutionTypeHint.STRING
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f67de7630019d8a4f78026ec610cbf53cd2c609aa298b0e06234c9c363f62262)
            check_type(argname="argument stack_trace", value=stack_trace, expected_type=type_hints["stack_trace"])
            check_type(argname="argument type_hint", value=type_hint, expected_type=type_hints["type_hint"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if stack_trace is not None:
            self._values["stack_trace"] = stack_trace
        if type_hint is not None:
            self._values["type_hint"] = type_hint

    @builtins.property
    def stack_trace(self) -> typing.Optional[builtins.bool]:
        '''Capture the stack trace of where this token is created.

        :default: true
        '''
        result = self._values.get("stack_trace")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def type_hint(self) -> typing.Optional["ResolutionTypeHint"]:
        '''Type that this token is expected to evaluate to.

        :default: ResolutionTypeHint.STRING
        '''
        result = self._values.get("type_hint")
        return typing.cast(typing.Optional["ResolutionTypeHint"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "IntrinsicProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IResolvable)
class JsonNull(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.JsonNull"):
    '''An object which serializes to the JSON ``null`` literal, and which can safely be passed across languages where ``undefined`` and ``null`` are not different.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        json_null = cdk.JsonNull.INSTANCE
    '''

    @jsii.member(jsii_name="resolve")
    def resolve(self, _ctx: IResolveContext) -> typing.Any:
        '''Produce the Token's value at resolution time.

        :param _ctx: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__da15a1f43436705565f12902e6cf1add27c14cb5e266f2224d7871f88d3c75b3)
            check_type(argname="argument _ctx", value=_ctx, expected_type=type_hints["_ctx"])
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [_ctx]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''Obtains the JSON representation of this object (``null``).'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Obtains the string representation of this object (``'null'``).'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="INSTANCE")
    def INSTANCE(cls) -> "JsonNull":
        '''The canonical instance of ``JsonNull``.'''
        return typing.cast("JsonNull", jsii.sget(cls, "INSTANCE"))

    @builtins.property
    @jsii.member(jsii_name="creationStack")
    def creation_stack(self) -> typing.List[builtins.str]:
        '''The creation stack of this resolvable which will be appended to errors thrown during resolution.

        This may return an array with a single informational element indicating how
        to get this property populated, if it was skipped for performance reasons.
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "creationStack"))


class Lazy(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.Lazy"):
    '''Lazily produce a value.

    Can be used to return a string, list or numeric value whose actual value
    will only be calculated later, during synthesis.
    '''

    @jsii.member(jsii_name="any")
    @builtins.classmethod
    def any(
        cls,
        producer: IStableAnyProducer,
        *,
        display_hint: typing.Optional[builtins.str] = None,
        omit_empty_array: typing.Optional[builtins.bool] = None,
    ) -> IResolvable:
        '''Defer the one-time calculation of an arbitrarily typed value to synthesis time.

        Use this if you want to render an object to a template whose actual value depends on
        some state mutation that may happen after the construct has been created.

        The inner function will only be invoked one time and cannot depend on
        resolution context.

        :param producer: -
        :param display_hint: Use the given name as a display hint. Default: - No hint
        :param omit_empty_array: If the produced value is an array and it is empty, return 'undefined' instead. Default: false
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__99716a72c06950a9f33565328da874cc4227956a9c0ed645df15681cfca7329b)
            check_type(argname="argument producer", value=producer, expected_type=type_hints["producer"])
        options = LazyAnyValueOptions(
            display_hint=display_hint, omit_empty_array=omit_empty_array
        )

        return typing.cast(IResolvable, jsii.sinvoke(cls, "any", [producer, options]))

    @jsii.member(jsii_name="list")
    @builtins.classmethod
    def list(
        cls,
        producer: IStableListProducer,
        *,
        display_hint: typing.Optional[builtins.str] = None,
        omit_empty: typing.Optional[builtins.bool] = None,
    ) -> typing.List[builtins.str]:
        '''Defer the one-time calculation of a list value to synthesis time.

        Use this if you want to render a list to a template whose actual value depends on
        some state mutation that may happen after the construct has been created.

        If you are simply looking to force a value to a ``string[]`` type and don't need
        the calculation to be deferred, use ``Token.asList()`` instead.

        The inner function will only be invoked once, and the resolved value
        cannot depend on the Stack the Token is used in.

        :param producer: -
        :param display_hint: Use the given name as a display hint. Default: - No hint
        :param omit_empty: If the produced list is empty, return 'undefined' instead. Default: false
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__128d00ae0712e626b97c783d2ab35c65b5f2216e0246e0fc9a0841a877c1c1e3)
            check_type(argname="argument producer", value=producer, expected_type=type_hints["producer"])
        options = LazyListValueOptions(
            display_hint=display_hint, omit_empty=omit_empty
        )

        return typing.cast(typing.List[builtins.str], jsii.sinvoke(cls, "list", [producer, options]))

    @jsii.member(jsii_name="number")
    @builtins.classmethod
    def number(cls, producer: IStableNumberProducer) -> jsii.Number:
        '''Defer the one-time calculation of a number value to synthesis time.

        Use this if you want to render a number to a template whose actual value depends on
        some state mutation that may happen after the construct has been created.

        If you are simply looking to force a value to a ``number`` type and don't need
        the calculation to be deferred, use ``Token.asNumber()`` instead.

        The inner function will only be invoked once, and the resolved value
        cannot depend on the Stack the Token is used in.

        :param producer: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5a405ac7d4eec08bc7c4c7e98538ae24cc3a627456a70dc33b3f2eba3a5c9222)
            check_type(argname="argument producer", value=producer, expected_type=type_hints["producer"])
        return typing.cast(jsii.Number, jsii.sinvoke(cls, "number", [producer]))

    @jsii.member(jsii_name="string")
    @builtins.classmethod
    def string(
        cls,
        producer: IStableStringProducer,
        *,
        display_hint: typing.Optional[builtins.str] = None,
    ) -> builtins.str:
        '''Defer the one-time calculation of a string value to synthesis time.

        Use this if you want to render a string to a template whose actual value depends on
        some state mutation that may happen after the construct has been created.

        If you are simply looking to force a value to a ``string`` type and don't need
        the calculation to be deferred, use ``Token.asString()`` instead.

        The inner function will only be invoked once, and the resolved value
        cannot depend on the Stack the Token is used in.

        :param producer: -
        :param display_hint: Use the given name as a display hint. Default: - No hint
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b2c13f41d7c6ff99c6b15563d19d8fa6842d285d26b2fc1702b58a9b6a07b298)
            check_type(argname="argument producer", value=producer, expected_type=type_hints["producer"])
        options = LazyStringValueOptions(display_hint=display_hint)

        return typing.cast(builtins.str, jsii.sinvoke(cls, "string", [producer, options]))

    @jsii.member(jsii_name="uncachedAny")
    @builtins.classmethod
    def uncached_any(
        cls,
        producer: IAnyProducer,
        *,
        display_hint: typing.Optional[builtins.str] = None,
        omit_empty_array: typing.Optional[builtins.bool] = None,
    ) -> IResolvable:
        '''Defer the calculation of an untyped value to synthesis time.

        Use of this function is not recommended; unless you know you need it for sure, you
        probably don't. Use ``Lazy.any()`` instead.

        The inner function may be invoked multiple times during synthesis. You
        should only use this method if the returned value depends on variables
        that may change during the Aspect application phase of synthesis, or if
        the value depends on the Stack the value is being used in. Both of these
        cases are rare, and only ever occur for AWS Construct Library authors.

        :param producer: -
        :param display_hint: Use the given name as a display hint. Default: - No hint
        :param omit_empty_array: If the produced value is an array and it is empty, return 'undefined' instead. Default: false
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3886f36ff33f624162274925263286300879bdaa1bb8c3ad400f8f37bf5a7915)
            check_type(argname="argument producer", value=producer, expected_type=type_hints["producer"])
        options = LazyAnyValueOptions(
            display_hint=display_hint, omit_empty_array=omit_empty_array
        )

        return typing.cast(IResolvable, jsii.sinvoke(cls, "uncachedAny", [producer, options]))

    @jsii.member(jsii_name="uncachedList")
    @builtins.classmethod
    def uncached_list(
        cls,
        producer: IListProducer,
        *,
        display_hint: typing.Optional[builtins.str] = None,
        omit_empty: typing.Optional[builtins.bool] = None,
    ) -> typing.List[builtins.str]:
        '''Defer the calculation of a list value to synthesis time.

        Use of this function is not recommended; unless you know you need it for sure, you
        probably don't. Use ``Lazy.list()`` instead.

        The inner function may be invoked multiple times during synthesis. You
        should only use this method if the returned value depends on variables
        that may change during the Aspect application phase of synthesis, or if
        the value depends on the Stack the value is being used in. Both of these
        cases are rare, and only ever occur for AWS Construct Library authors.

        :param producer: -
        :param display_hint: Use the given name as a display hint. Default: - No hint
        :param omit_empty: If the produced list is empty, return 'undefined' instead. Default: false
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bc6c14dd408a30b6bfeff8f08575ca0d990c50b7835a670a5cf9897c08de7072)
            check_type(argname="argument producer", value=producer, expected_type=type_hints["producer"])
        options = LazyListValueOptions(
            display_hint=display_hint, omit_empty=omit_empty
        )

        return typing.cast(typing.List[builtins.str], jsii.sinvoke(cls, "uncachedList", [producer, options]))

    @jsii.member(jsii_name="uncachedNumber")
    @builtins.classmethod
    def uncached_number(cls, producer: INumberProducer) -> jsii.Number:
        '''Defer the calculation of a number value to synthesis time.

        Use of this function is not recommended; unless you know you need it for sure, you
        probably don't. Use ``Lazy.number()`` instead.

        The inner function may be invoked multiple times during synthesis. You
        should only use this method if the returned value depends on variables
        that may change during the Aspect application phase of synthesis, or if
        the value depends on the Stack the value is being used in. Both of these
        cases are rare, and only ever occur for AWS Construct Library authors.

        :param producer: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__971b9262740edcd95cd59615124cc5579573ad5f62ddcfe273e1db624ba86e83)
            check_type(argname="argument producer", value=producer, expected_type=type_hints["producer"])
        return typing.cast(jsii.Number, jsii.sinvoke(cls, "uncachedNumber", [producer]))

    @jsii.member(jsii_name="uncachedString")
    @builtins.classmethod
    def uncached_string(
        cls,
        producer: IStringProducer,
        *,
        display_hint: typing.Optional[builtins.str] = None,
    ) -> builtins.str:
        '''Defer the calculation of a string value to synthesis time.

        Use of this function is not recommended; unless you know you need it for sure, you
        probably don't. Use ``Lazy.string()`` instead.

        The inner function may be invoked multiple times during synthesis. You
        should only use this method if the returned value depends on variables
        that may change during the Aspect application phase of synthesis, or if
        the value depends on the Stack the value is being used in. Both of these
        cases are rare, and only ever occur for AWS Construct Library authors.

        :param producer: -
        :param display_hint: Use the given name as a display hint. Default: - No hint
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4f56cac5754fbaa934b21f0ae649b635a42de210ff0606443e3e58f519a99600)
            check_type(argname="argument producer", value=producer, expected_type=type_hints["producer"])
        options = LazyStringValueOptions(display_hint=display_hint)

        return typing.cast(builtins.str, jsii.sinvoke(cls, "uncachedString", [producer, options]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.LazyAnyValueOptions",
    jsii_struct_bases=[],
    name_mapping={"display_hint": "displayHint", "omit_empty_array": "omitEmptyArray"},
)
class LazyAnyValueOptions:
    def __init__(
        self,
        *,
        display_hint: typing.Optional[builtins.str] = None,
        omit_empty_array: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for creating lazy untyped tokens.

        :param display_hint: Use the given name as a display hint. Default: - No hint
        :param omit_empty_array: If the produced value is an array and it is empty, return 'undefined' instead. Default: false

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            lazy_any_value_options = cdk.LazyAnyValueOptions(
                display_hint="displayHint",
                omit_empty_array=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a87993e8b54334ed0acf91c5866729e50fe95de71ae78139ad1629915ad63fdc)
            check_type(argname="argument display_hint", value=display_hint, expected_type=type_hints["display_hint"])
            check_type(argname="argument omit_empty_array", value=omit_empty_array, expected_type=type_hints["omit_empty_array"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if display_hint is not None:
            self._values["display_hint"] = display_hint
        if omit_empty_array is not None:
            self._values["omit_empty_array"] = omit_empty_array

    @builtins.property
    def display_hint(self) -> typing.Optional[builtins.str]:
        '''Use the given name as a display hint.

        :default: - No hint
        '''
        result = self._values.get("display_hint")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def omit_empty_array(self) -> typing.Optional[builtins.bool]:
        '''If the produced value is an array and it is empty, return 'undefined' instead.

        :default: false
        '''
        result = self._values.get("omit_empty_array")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LazyAnyValueOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.LazyListValueOptions",
    jsii_struct_bases=[],
    name_mapping={"display_hint": "displayHint", "omit_empty": "omitEmpty"},
)
class LazyListValueOptions:
    def __init__(
        self,
        *,
        display_hint: typing.Optional[builtins.str] = None,
        omit_empty: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for creating a lazy list token.

        :param display_hint: Use the given name as a display hint. Default: - No hint
        :param omit_empty: If the produced list is empty, return 'undefined' instead. Default: false

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            lazy_list_value_options = cdk.LazyListValueOptions(
                display_hint="displayHint",
                omit_empty=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d241cd00f4455dadba9174848189f54276c3c7f745aff63ba354b97d87f8e765)
            check_type(argname="argument display_hint", value=display_hint, expected_type=type_hints["display_hint"])
            check_type(argname="argument omit_empty", value=omit_empty, expected_type=type_hints["omit_empty"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if display_hint is not None:
            self._values["display_hint"] = display_hint
        if omit_empty is not None:
            self._values["omit_empty"] = omit_empty

    @builtins.property
    def display_hint(self) -> typing.Optional[builtins.str]:
        '''Use the given name as a display hint.

        :default: - No hint
        '''
        result = self._values.get("display_hint")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def omit_empty(self) -> typing.Optional[builtins.bool]:
        '''If the produced list is empty, return 'undefined' instead.

        :default: false
        '''
        result = self._values.get("omit_empty")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LazyListValueOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.LazyStringValueOptions",
    jsii_struct_bases=[],
    name_mapping={"display_hint": "displayHint"},
)
class LazyStringValueOptions:
    def __init__(self, *, display_hint: typing.Optional[builtins.str] = None) -> None:
        '''Options for creating a lazy string token.

        :param display_hint: Use the given name as a display hint. Default: - No hint

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            lazy_string_value_options = cdk.LazyStringValueOptions(
                display_hint="displayHint"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__283fa3218deb31c702af8ec0f581964e58afe471191dd394e1567cb4d5429509)
            check_type(argname="argument display_hint", value=display_hint, expected_type=type_hints["display_hint"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if display_hint is not None:
            self._values["display_hint"] = display_hint

    @builtins.property
    def display_hint(self) -> typing.Optional[builtins.str]:
        '''Use the given name as a display hint.

        :default: - No hint
        '''
        result = self._values.get("display_hint")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LazyStringValueOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class MissingRemovalPolicies(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.MissingRemovalPolicies",
):
    '''Manages removal policies for resources without existing policies within a construct scope.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        missing_removal_policies = cdk.MissingRemovalPolicies.of(self)
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, scope: _constructs_77d1e7e8.IConstruct) -> "MissingRemovalPolicies":
        '''Returns the missing removal policies API for the given scope.

        :param scope: The scope.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__53b49b542fe09ce9b0aa527b531ee0b1e7f57e7e0c6e23c2483a0b347d813ebb)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("MissingRemovalPolicies", jsii.sinvoke(cls, "of", [scope]))

    @jsii.member(jsii_name="apply")
    def apply(
        self,
        policy: "RemovalPolicy",
        *,
        apply_to_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Apply a removal policy only to resources without existing policies within this scope.

        :param policy: The removal policy to apply.
        :param apply_to_resource_types: Apply the removal policy only to specific resource types. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - apply to all resources
        :param exclude_resource_types: Exclude specific resource types from the removal policy. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - no exclusions
        :param priority: The priority to use when applying this policy. The priority affects only the order in which aspects are applied during synthesis. For RemovalPolicies, the last applied policy will override previous ones. NOTE: Priority does NOT determine which policy "wins" when there are conflicts. The order of application determines the final policy, with later policies overriding earlier ones. Default: - AspectPriority.MUTATING
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3f1229a84cb5fdb052e8db135be3d25670d9793cf44ddc71d8134240bed0efc)
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
        props = RemovalPolicyProps(
            apply_to_resource_types=apply_to_resource_types,
            exclude_resource_types=exclude_resource_types,
            priority=priority,
        )

        return typing.cast(None, jsii.invoke(self, "apply", [policy, props]))

    @jsii.member(jsii_name="destroy")
    def destroy(
        self,
        *,
        apply_to_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Apply DESTROY removal policy only to resources without existing policies within this scope.

        :param apply_to_resource_types: Apply the removal policy only to specific resource types. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - apply to all resources
        :param exclude_resource_types: Exclude specific resource types from the removal policy. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - no exclusions
        :param priority: The priority to use when applying this policy. The priority affects only the order in which aspects are applied during synthesis. For RemovalPolicies, the last applied policy will override previous ones. NOTE: Priority does NOT determine which policy "wins" when there are conflicts. The order of application determines the final policy, with later policies overriding earlier ones. Default: - AspectPriority.MUTATING
        '''
        props = RemovalPolicyProps(
            apply_to_resource_types=apply_to_resource_types,
            exclude_resource_types=exclude_resource_types,
            priority=priority,
        )

        return typing.cast(None, jsii.invoke(self, "destroy", [props]))

    @jsii.member(jsii_name="retain")
    def retain(
        self,
        *,
        apply_to_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Apply RETAIN removal policy only to resources without existing policies within this scope.

        :param apply_to_resource_types: Apply the removal policy only to specific resource types. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - apply to all resources
        :param exclude_resource_types: Exclude specific resource types from the removal policy. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - no exclusions
        :param priority: The priority to use when applying this policy. The priority affects only the order in which aspects are applied during synthesis. For RemovalPolicies, the last applied policy will override previous ones. NOTE: Priority does NOT determine which policy "wins" when there are conflicts. The order of application determines the final policy, with later policies overriding earlier ones. Default: - AspectPriority.MUTATING
        '''
        props = RemovalPolicyProps(
            apply_to_resource_types=apply_to_resource_types,
            exclude_resource_types=exclude_resource_types,
            priority=priority,
        )

        return typing.cast(None, jsii.invoke(self, "retain", [props]))

    @jsii.member(jsii_name="retainOnUpdateOrDelete")
    def retain_on_update_or_delete(
        self,
        *,
        apply_to_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Apply RETAIN_ON_UPDATE_OR_DELETE removal policy only to resources without existing policies within this scope.

        :param apply_to_resource_types: Apply the removal policy only to specific resource types. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - apply to all resources
        :param exclude_resource_types: Exclude specific resource types from the removal policy. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - no exclusions
        :param priority: The priority to use when applying this policy. The priority affects only the order in which aspects are applied during synthesis. For RemovalPolicies, the last applied policy will override previous ones. NOTE: Priority does NOT determine which policy "wins" when there are conflicts. The order of application determines the final policy, with later policies overriding earlier ones. Default: - AspectPriority.MUTATING
        '''
        props = RemovalPolicyProps(
            apply_to_resource_types=apply_to_resource_types,
            exclude_resource_types=exclude_resource_types,
            priority=priority,
        )

        return typing.cast(None, jsii.invoke(self, "retainOnUpdateOrDelete", [props]))

    @jsii.member(jsii_name="snapshot")
    def snapshot(
        self,
        *,
        apply_to_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Apply SNAPSHOT removal policy only to resources without existing policies within this scope.

        :param apply_to_resource_types: Apply the removal policy only to specific resource types. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - apply to all resources
        :param exclude_resource_types: Exclude specific resource types from the removal policy. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - no exclusions
        :param priority: The priority to use when applying this policy. The priority affects only the order in which aspects are applied during synthesis. For RemovalPolicies, the last applied policy will override previous ones. NOTE: Priority does NOT determine which policy "wins" when there are conflicts. The order of application determines the final policy, with later policies overriding earlier ones. Default: - AspectPriority.MUTATING
        '''
        props = RemovalPolicyProps(
            apply_to_resource_types=apply_to_resource_types,
            exclude_resource_types=exclude_resource_types,
            priority=priority,
        )

        return typing.cast(None, jsii.invoke(self, "snapshot", [props]))


class Names(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.Names"):
    '''Functions for devising unique names for constructs.

    For example, those can be
    used to allocate unique physical names for resources.
    '''

    @jsii.member(jsii_name="nodeUniqueId")
    @builtins.classmethod
    def node_unique_id(cls, node: _constructs_77d1e7e8.Node) -> builtins.str:
        '''Returns a CloudFormation-compatible unique identifier for a construct based on its path.

        The identifier includes a human readable portion rendered
        from the path components and a hash suffix.

        TODO (v2): replace with API to use ``constructs.Node``.

        :param node: The construct node.

        :return: a unique id based on the construct path
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__569ac5c6adbac9575b4137a0343aea0625df007d849957bfacf2e65a01153962)
            check_type(argname="argument node", value=node, expected_type=type_hints["node"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "nodeUniqueId", [node]))

    @jsii.member(jsii_name="stackRelativeConstructPath")
    @builtins.classmethod
    def stack_relative_construct_path(
        cls,
        construct: _constructs_77d1e7e8.IConstruct,
    ) -> builtins.str:
        '''Return the construct path of the given construct, starting at the nearest enclosing Stack.

        Skips over Nested Stacks, in other words Nested Stacks are included in the construct
        paths.

        :param construct: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__307d610951ef0b232f9222edc6ac5421fa347060bc21fb75e850a338e9aa77c7)
            check_type(argname="argument construct", value=construct, expected_type=type_hints["construct"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "stackRelativeConstructPath", [construct]))

    @jsii.member(jsii_name="uniqueId")
    @builtins.classmethod
    def unique_id(cls, construct: _constructs_77d1e7e8.IConstruct) -> builtins.str:
        '''Returns a CloudFormation-compatible unique identifier for a construct based on its path.

        The identifier includes a human readable portion rendered
        from the path components and a hash suffix. uniqueId is not unique if multiple
        copies of the stack are deployed. Prefer using uniqueResourceName().

        :param construct: The construct.

        :return: a unique id based on the construct path
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8c1ba4bae2d17f227472b42e48bc60d8c4a5dd7a3c60793d3b332fc432589917)
            check_type(argname="argument construct", value=construct, expected_type=type_hints["construct"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "uniqueId", [construct]))

    @jsii.member(jsii_name="uniqueResourceName")
    @builtins.classmethod
    def unique_resource_name(
        cls,
        construct: _constructs_77d1e7e8.IConstruct,
        *,
        allowed_special_characters: typing.Optional[builtins.str] = None,
        max_length: typing.Optional[jsii.Number] = None,
        separator: typing.Optional[builtins.str] = None,
    ) -> builtins.str:
        '''Returns a CloudFormation-compatible unique identifier for a construct based on its path.

        This function finds the stackName of the parent stack (non-nested)
        to the construct, and the ids of the components in the construct path.

        The user can define allowed special characters, a separator between the elements,
        and the maximum length of the resource name. The name includes a human readable portion rendered
        from the path components, with or without user defined separators, and a hash suffix.
        If the resource name is longer than the maximum length, it is trimmed in the middle.

        :param construct: The construct.
        :param allowed_special_characters: Non-alphanumeric characters allowed in the unique resource name. Default: - none
        :param max_length: The maximum length of the unique resource name. Default: - 256
        :param separator: The separator used between the path components. Default: - none

        :return: a unique resource name based on the construct path
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe58b3647205c016b529f2592c2456221c7c16e7f42c500d5e0d372010be9fc5)
            check_type(argname="argument construct", value=construct, expected_type=type_hints["construct"])
        options = UniqueResourceNameOptions(
            allowed_special_characters=allowed_special_characters,
            max_length=max_length,
            separator=separator,
        )

        return typing.cast(builtins.str, jsii.sinvoke(cls, "uniqueResourceName", [construct, options]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.NestedStackProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "notification_arns": "notificationArns",
        "parameters": "parameters",
        "removal_policy": "removalPolicy",
        "timeout": "timeout",
    },
)
class NestedStackProps:
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        notification_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        removal_policy: typing.Optional["RemovalPolicy"] = None,
        timeout: typing.Optional[Duration] = None,
    ) -> None:
        '''Initialization props for the ``NestedStack`` construct.

        :param description: A description of the stack. Default: - No description.
        :param notification_arns: The Simple Notification Service (SNS) topics to publish stack related events. Default: - notifications are not sent for this stack.
        :param parameters: The set value pairs that represent the parameters passed to CloudFormation when this nested stack is created. Each parameter has a name corresponding to a parameter defined in the embedded template and a value representing the value that you want to set for the parameter. The nested stack construct will automatically synthesize parameters in order to bind references from the parent stack(s) into the nested stack. Default: - no user-defined parameters are passed to the nested stack
        :param removal_policy: Policy to apply when the nested stack is removed. The default is ``Destroy``, because all Removal Policies of resources inside the Nested Stack should already have been set correctly. You normally should not need to set this value. Default: RemovalPolicy.DESTROY
        :param timeout: The length of time that CloudFormation waits for the nested stack to reach the CREATE_COMPLETE state. When CloudFormation detects that the nested stack has reached the CREATE_COMPLETE state, it marks the nested stack resource as CREATE_COMPLETE in the parent stack and resumes creating the parent stack. If the timeout period expires before the nested stack reaches CREATE_COMPLETE, CloudFormation marks the nested stack as failed and rolls back both the nested stack and parent stack. Default: - no timeout

        :exampleMetadata: lit=aws-apigateway/test/integ.restapi-import.lit.ts infused

        Example::

            from aws_cdk.aws_apigateway import IntegrationResponse, MethodResponse, IntegrationResponse, MethodResponse
            from constructs import Construct
            from aws_cdk import App, CfnOutput, NestedStack, NestedStackProps, Stack
            from aws_cdk.aws_apigateway import Deployment, Method, MockIntegration, PassthroughBehavior, RestApi, Stage
            
            #
            # This file showcases how to split up a RestApi's Resources and Methods across nested stacks.
            #
            # The root stack 'RootStack' first defines a RestApi.
            # Two nested stacks BooksStack and PetsStack, create corresponding Resources '/books' and '/pets'.
            # They are then deployed to a 'prod' Stage via a third nested stack - DeployStack.
            #
            # To verify this worked, go to the APIGateway
            #
            
            class RootStack(Stack):
                def __init__(self, scope):
                    super().__init__(scope, "integ-restapi-import-RootStack")
            
                    rest_api = RestApi(self, "RestApi",
                        cloud_watch_role=True,
                        deploy=False
                    )
                    rest_api.root.add_method("ANY")
            
                    pets_stack = PetsStack(self,
                        rest_api_id=rest_api.rest_api_id,
                        root_resource_id=rest_api.rest_api_root_resource_id
                    )
                    books_stack = BooksStack(self,
                        rest_api_id=rest_api.rest_api_id,
                        root_resource_id=rest_api.rest_api_root_resource_id
                    )
                    DeployStack(self,
                        rest_api_id=rest_api.rest_api_id,
                        methods=pets_stack.methods.concat(books_stack.methods)
                    )
            
                    CfnOutput(self, "PetsURL",
                        value=f"https://{restApi.restApiId}.execute-api.{this.region}.amazonaws.com/prod/pets"
                    )
            
                    CfnOutput(self, "BooksURL",
                        value=f"https://{restApi.restApiId}.execute-api.{this.region}.amazonaws.com/prod/books"
                    )
            
            class PetsStack(NestedStack):
            
                def __init__(self, scope, *, restApiId, rootResourceId, parameters=None, timeout=None, notificationArns=None, removalPolicy=None, description=None):
                    super().__init__(scope, "integ-restapi-import-PetsStack", restApiId=restApiId, rootResourceId=rootResourceId, parameters=parameters, timeout=timeout, notificationArns=notificationArns, removalPolicy=removalPolicy, description=description)
            
                    api = RestApi.from_rest_api_attributes(self, "RestApi",
                        rest_api_id=rest_api_id,
                        root_resource_id=root_resource_id
                    )
            
                    method = api.root.add_resource("pets").add_method("GET", MockIntegration(
                        integration_responses=[IntegrationResponse(
                            status_code="200"
                        )],
                        passthrough_behavior=PassthroughBehavior.NEVER,
                        request_templates={
                            "application/json": "{ \"statusCode\": 200 }"
                        }
                    ),
                        method_responses=[MethodResponse(status_code="200")]
                    )
            
                    self.methods.push(method)
            
            class BooksStack(NestedStack):
            
                def __init__(self, scope, *, restApiId, rootResourceId, parameters=None, timeout=None, notificationArns=None, removalPolicy=None, description=None):
                    super().__init__(scope, "integ-restapi-import-BooksStack", restApiId=restApiId, rootResourceId=rootResourceId, parameters=parameters, timeout=timeout, notificationArns=notificationArns, removalPolicy=removalPolicy, description=description)
            
                    api = RestApi.from_rest_api_attributes(self, "RestApi",
                        rest_api_id=rest_api_id,
                        root_resource_id=root_resource_id
                    )
            
                    method = api.root.add_resource("books").add_method("GET", MockIntegration(
                        integration_responses=[IntegrationResponse(
                            status_code="200"
                        )],
                        passthrough_behavior=PassthroughBehavior.NEVER,
                        request_templates={
                            "application/json": "{ \"statusCode\": 200 }"
                        }
                    ),
                        method_responses=[MethodResponse(status_code="200")]
                    )
            
                    self.methods.push(method)
            
            class DeployStack(NestedStack):
                def __init__(self, scope, *, restApiId, methods=None, parameters=None, timeout=None, notificationArns=None, removalPolicy=None, description=None):
                    super().__init__(scope, "integ-restapi-import-DeployStack", restApiId=restApiId, methods=methods, parameters=parameters, timeout=timeout, notificationArns=notificationArns, removalPolicy=removalPolicy, description=description)
            
                    deployment = Deployment(self, "Deployment",
                        api=RestApi.from_rest_api_id(self, "RestApi", rest_api_id)
                    )
                    if methods:
                        for method in methods:
                            deployment.node.add_dependency(method)
                    Stage(self, "Stage", deployment=deployment)
            
            RootStack(App())
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__68736f67b0cf5effa29d0315b62c690a91db0932ebc56b25811b40ec08a95ba9)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument notification_arns", value=notification_arns, expected_type=type_hints["notification_arns"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument removal_policy", value=removal_policy, expected_type=type_hints["removal_policy"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if notification_arns is not None:
            self._values["notification_arns"] = notification_arns
        if parameters is not None:
            self._values["parameters"] = parameters
        if removal_policy is not None:
            self._values["removal_policy"] = removal_policy
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the stack.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def notification_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The Simple Notification Service (SNS) topics to publish stack related events.

        :default: - notifications are not sent for this stack.
        '''
        result = self._values.get("notification_arns")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The set value pairs that represent the parameters passed to CloudFormation when this nested stack is created.

        Each parameter has a name corresponding
        to a parameter defined in the embedded template and a value representing
        the value that you want to set for the parameter.

        The nested stack construct will automatically synthesize parameters in order
        to bind references from the parent stack(s) into the nested stack.

        :default: - no user-defined parameters are passed to the nested stack
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def removal_policy(self) -> typing.Optional["RemovalPolicy"]:
        '''Policy to apply when the nested stack is removed.

        The default is ``Destroy``, because all Removal Policies of resources inside the
        Nested Stack should already have been set correctly. You normally should
        not need to set this value.

        :default: RemovalPolicy.DESTROY
        '''
        result = self._values.get("removal_policy")
        return typing.cast(typing.Optional["RemovalPolicy"], result)

    @builtins.property
    def timeout(self) -> typing.Optional[Duration]:
        '''The length of time that CloudFormation waits for the nested stack to reach the CREATE_COMPLETE state.

        When CloudFormation detects that the nested stack has reached the
        CREATE_COMPLETE state, it marks the nested stack resource as
        CREATE_COMPLETE in the parent stack and resumes creating the parent stack.
        If the timeout period expires before the nested stack reaches
        CREATE_COMPLETE, CloudFormation marks the nested stack as failed and rolls
        back both the nested stack and parent stack.

        :default: - no timeout
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[Duration], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "NestedStackProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class PermissionsBoundary(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.PermissionsBoundary",
):
    '''Apply a permissions boundary to all IAM Roles and Users within a specific scope.

    A permissions boundary is typically applied at the ``Stage`` scope.
    This allows setting different permissions boundaries per Stage. For
    example, you may *not* apply a boundary to the ``Dev`` stage which deploys
    to a personal dev account, but you *do* apply the default boundary to the
    ``Prod`` stage.

    It is possible to apply different permissions boundaries to different scopes
    within your app. In this case the most specifically applied one wins

    Example::

        # no permissions boundary for dev stage
        Stage(app, "DevStage")
        
        # default boundary for prod stage
        prod_stage = Stage(app, "ProdStage",
            permissions_boundary=PermissionsBoundary.from_name("prod-pb")
        )
        
        # overriding the pb applied for this stack
        Stack(prod_stage, "ProdStack1",
            permissions_boundary=PermissionsBoundary.from_name("stack-pb")
        )
        
        # will inherit the permissions boundary from the stage
        Stack(prod_stage, "ProdStack2")
    '''

    @jsii.member(jsii_name="fromArn")
    @builtins.classmethod
    def from_arn(cls, arn: builtins.str) -> "PermissionsBoundary":
        '''Apply a permissions boundary with the given ARN to all IAM Roles and Users created within a scope.

        The arn can include placeholders for the partition, region, qualifier, and account
        These placeholders will be replaced with the actual values if available. This requires
        that the Stack has the environment specified, it does not work with environment
        agnostic stacks.

        - '${AWS::Partition}'
        - '${AWS::Region}'
        - '${AWS::AccountId}'
        - '${Qualifier}'

        :param arn: the ARN of the permissions boundary policy.

        Example::

            Stage(app, "ProdStage",
                permissions_boundary=PermissionsBoundary.from_arn("arn:aws:iam::${AWS::AccountId}:policy/my-custom-permissions-boundary")
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c099a69d5d9bcb07edbf91e3a92296eca8e89a908d05b7e721238b7564a6dfc6)
            check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
        return typing.cast("PermissionsBoundary", jsii.sinvoke(cls, "fromArn", [arn]))

    @jsii.member(jsii_name="fromName")
    @builtins.classmethod
    def from_name(cls, name: builtins.str) -> "PermissionsBoundary":
        '''Apply a permissions boundary with the given name to all IAM Roles and Users created within a scope.

        The name can include placeholders for the partition, region, qualifier, and account
        These placeholders will be replaced with the actual values if available. This requires
        that the Stack has the environment specified, it does not work with environment
        agnostic stacks.

        - '${AWS::Partition}'
        - '${AWS::Region}'
        - '${AWS::AccountId}'
        - '${Qualifier}'

        :param name: the name of the permissions boundary policy.

        Example::

            Stage(app, "ProdStage",
                permissions_boundary=PermissionsBoundary.from_name("my-custom-permissions-boundary")
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf7e2a64613c569c1b642bf8637f52fe0045cfdf4ca0f963af822a6040d0b3c9)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        return typing.cast("PermissionsBoundary", jsii.sinvoke(cls, "fromName", [name]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.PermissionsBoundaryBindOptions",
    jsii_struct_bases=[],
    name_mapping={},
)
class PermissionsBoundaryBindOptions:
    def __init__(self) -> None:
        '''Options for binding a Permissions Boundary to a construct scope.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            permissions_boundary_bind_options = cdk.PermissionsBoundaryBindOptions()
        '''
        self._values: typing.Dict[builtins.str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PermissionsBoundaryBindOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class PhysicalName(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.PhysicalName"):
    '''Includes special markers for automatic generation of physical names.'''

    @jsii.python.classproperty
    @jsii.member(jsii_name="GENERATE_IF_NEEDED")
    def GENERATE_IF_NEEDED(cls) -> builtins.str:
        '''Use this to automatically generate a physical name for an AWS resource only if the resource is referenced across environments (account/region).

        Otherwise, the name will be allocated during deployment by CloudFormation.

        If you are certain that a resource will be referenced across environments,
        you may also specify an explicit physical name for it. This option is
        mostly designed for reusable constructs which may or may not be referenced
        across environments.
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "GENERATE_IF_NEEDED"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.PolicyValidationPluginReportBeta1",
    jsii_struct_bases=[],
    name_mapping={
        "success": "success",
        "violations": "violations",
        "metadata": "metadata",
        "plugin_version": "pluginVersion",
    },
)
class PolicyValidationPluginReportBeta1:
    def __init__(
        self,
        *,
        success: builtins.bool,
        violations: typing.Sequence[typing.Union["PolicyViolationBeta1", typing.Dict[builtins.str, typing.Any]]],
        metadata: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        plugin_version: typing.Optional[builtins.str] = None,
    ) -> None:
        '''The report emitted by the plugin after evaluation.

        :param success: Whether or not the report was successful.
        :param violations: List of violations in the report.
        :param metadata: Arbitrary information about the report. Default: - no metadata
        :param plugin_version: The version of the plugin that created the report. Default: - no version

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            policy_validation_plugin_report_beta1 = cdk.PolicyValidationPluginReportBeta1(
                success=False,
                violations=[cdk.PolicyViolationBeta1(
                    description="description",
                    rule_name="ruleName",
                    violating_resources=[cdk.PolicyViolatingResourceBeta1(
                        locations=["locations"],
                        resource_logical_id="resourceLogicalId",
                        template_path="templatePath"
                    )],
            
                    # the properties below are optional
                    fix="fix",
                    rule_metadata={
                        "rule_metadata_key": "ruleMetadata"
                    },
                    severity="severity"
                )],
            
                # the properties below are optional
                metadata={
                    "metadata_key": "metadata"
                },
                plugin_version="pluginVersion"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__53f32baa53504563bf348cb26d2308de0b3c94f4fea93a1c0384fb0aed6ab2f1)
            check_type(argname="argument success", value=success, expected_type=type_hints["success"])
            check_type(argname="argument violations", value=violations, expected_type=type_hints["violations"])
            check_type(argname="argument metadata", value=metadata, expected_type=type_hints["metadata"])
            check_type(argname="argument plugin_version", value=plugin_version, expected_type=type_hints["plugin_version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "success": success,
            "violations": violations,
        }
        if metadata is not None:
            self._values["metadata"] = metadata
        if plugin_version is not None:
            self._values["plugin_version"] = plugin_version

    @builtins.property
    def success(self) -> builtins.bool:
        '''Whether or not the report was successful.'''
        result = self._values.get("success")
        assert result is not None, "Required property 'success' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def violations(self) -> typing.List["PolicyViolationBeta1"]:
        '''List of violations in the report.'''
        result = self._values.get("violations")
        assert result is not None, "Required property 'violations' is missing"
        return typing.cast(typing.List["PolicyViolationBeta1"], result)

    @builtins.property
    def metadata(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Arbitrary information about the report.

        :default: - no metadata
        '''
        result = self._values.get("metadata")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def plugin_version(self) -> typing.Optional[builtins.str]:
        '''The version of the plugin that created the report.

        :default: - no version
        '''
        result = self._values.get("plugin_version")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PolicyValidationPluginReportBeta1(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.PolicyValidationReportStatusBeta1")
class PolicyValidationReportStatusBeta1(enum.Enum):
    '''The final status of the validation report.'''

    SUCCESS = "SUCCESS"
    '''No violations were found.'''
    FAILURE = "FAILURE"
    '''At least one violation was found.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.PolicyViolatingResourceBeta1",
    jsii_struct_bases=[],
    name_mapping={
        "locations": "locations",
        "resource_logical_id": "resourceLogicalId",
        "template_path": "templatePath",
    },
)
class PolicyViolatingResourceBeta1:
    def __init__(
        self,
        *,
        locations: typing.Sequence[builtins.str],
        resource_logical_id: builtins.str,
        template_path: builtins.str,
    ) -> None:
        '''Resource violating a specific rule.

        :param locations: The locations in the CloudFormation template that pose the violations.
        :param resource_logical_id: The logical ID of the resource in the CloudFormation template.
        :param template_path: The path to the CloudFormation template that contains this resource.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            policy_violating_resource_beta1 = cdk.PolicyViolatingResourceBeta1(
                locations=["locations"],
                resource_logical_id="resourceLogicalId",
                template_path="templatePath"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d6f720d14d413c17b7efbaec187c8ce7e76ca4e581d69e43119ae9751d6e5529)
            check_type(argname="argument locations", value=locations, expected_type=type_hints["locations"])
            check_type(argname="argument resource_logical_id", value=resource_logical_id, expected_type=type_hints["resource_logical_id"])
            check_type(argname="argument template_path", value=template_path, expected_type=type_hints["template_path"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "locations": locations,
            "resource_logical_id": resource_logical_id,
            "template_path": template_path,
        }

    @builtins.property
    def locations(self) -> typing.List[builtins.str]:
        '''The locations in the CloudFormation template that pose the violations.'''
        result = self._values.get("locations")
        assert result is not None, "Required property 'locations' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def resource_logical_id(self) -> builtins.str:
        '''The logical ID of the resource in the CloudFormation template.'''
        result = self._values.get("resource_logical_id")
        assert result is not None, "Required property 'resource_logical_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def template_path(self) -> builtins.str:
        '''The path to the CloudFormation template that contains this resource.'''
        result = self._values.get("template_path")
        assert result is not None, "Required property 'template_path' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PolicyViolatingResourceBeta1(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.PolicyViolationBeta1",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "rule_name": "ruleName",
        "violating_resources": "violatingResources",
        "fix": "fix",
        "rule_metadata": "ruleMetadata",
        "severity": "severity",
    },
)
class PolicyViolationBeta1:
    def __init__(
        self,
        *,
        description: builtins.str,
        rule_name: builtins.str,
        violating_resources: typing.Sequence[typing.Union[PolicyViolatingResourceBeta1, typing.Dict[builtins.str, typing.Any]]],
        fix: typing.Optional[builtins.str] = None,
        rule_metadata: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        severity: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Violation produced by the validation plugin.

        :param description: The description of the violation.
        :param rule_name: The name of the rule.
        :param violating_resources: The resources violating this rule.
        :param fix: How to fix the violation. Default: - no fix is provided
        :param rule_metadata: Additional metadata to include with the rule results. This can be used to provide additional information that is plugin specific. The data provided here will be rendered as is. Default: - no rule metadata
        :param severity: The severity of the violation, only used for reporting purposes. This is useful for helping the user discriminate between warnings, errors, information, etc. Default: - no severity

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            policy_violation_beta1 = cdk.PolicyViolationBeta1(
                description="description",
                rule_name="ruleName",
                violating_resources=[cdk.PolicyViolatingResourceBeta1(
                    locations=["locations"],
                    resource_logical_id="resourceLogicalId",
                    template_path="templatePath"
                )],
            
                # the properties below are optional
                fix="fix",
                rule_metadata={
                    "rule_metadata_key": "ruleMetadata"
                },
                severity="severity"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2f965066f077908d4839609b519c39167a0b14e83374fbf4bb1601ab56f9728f)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument rule_name", value=rule_name, expected_type=type_hints["rule_name"])
            check_type(argname="argument violating_resources", value=violating_resources, expected_type=type_hints["violating_resources"])
            check_type(argname="argument fix", value=fix, expected_type=type_hints["fix"])
            check_type(argname="argument rule_metadata", value=rule_metadata, expected_type=type_hints["rule_metadata"])
            check_type(argname="argument severity", value=severity, expected_type=type_hints["severity"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "description": description,
            "rule_name": rule_name,
            "violating_resources": violating_resources,
        }
        if fix is not None:
            self._values["fix"] = fix
        if rule_metadata is not None:
            self._values["rule_metadata"] = rule_metadata
        if severity is not None:
            self._values["severity"] = severity

    @builtins.property
    def description(self) -> builtins.str:
        '''The description of the violation.'''
        result = self._values.get("description")
        assert result is not None, "Required property 'description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rule_name(self) -> builtins.str:
        '''The name of the rule.'''
        result = self._values.get("rule_name")
        assert result is not None, "Required property 'rule_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def violating_resources(self) -> typing.List[PolicyViolatingResourceBeta1]:
        '''The resources violating this rule.'''
        result = self._values.get("violating_resources")
        assert result is not None, "Required property 'violating_resources' is missing"
        return typing.cast(typing.List[PolicyViolatingResourceBeta1], result)

    @builtins.property
    def fix(self) -> typing.Optional[builtins.str]:
        '''How to fix the violation.

        :default: - no fix is provided
        '''
        result = self._values.get("fix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def rule_metadata(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Additional metadata to include with the rule results.

        This can be used to provide additional information that is
        plugin specific. The data provided here will be rendered as is.

        :default: - no rule metadata
        '''
        result = self._values.get("rule_metadata")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def severity(self) -> typing.Optional[builtins.str]:
        '''The severity of the violation, only used for reporting purposes.

        This is useful for helping the user discriminate between warnings,
        errors, information, etc.

        :default: - no severity
        '''
        result = self._values.get("severity")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PolicyViolationBeta1(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Reference(
    Intrinsic,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.Reference",
):
    '''An intrinsic Token that represents a reference to a construct.

    References are recorded.
    '''

    def __init__(
        self,
        value: typing.Any,
        target: _constructs_77d1e7e8.IConstruct,
        display_name: typing.Optional[builtins.str] = None,
        type_hint: typing.Optional["ResolutionTypeHint"] = None,
    ) -> None:
        '''
        :param value: -
        :param target: -
        :param display_name: -
        :param type_hint: Type that the Intrinsic is expected to evaluate to.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__151f083e00bfac139cbf164b9069ab02924a37639eb1b7b58edc3dee3fdf3279)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
            check_type(argname="argument display_name", value=display_name, expected_type=type_hints["display_name"])
            check_type(argname="argument type_hint", value=type_hint, expected_type=type_hints["type_hint"])
        jsii.create(self.__class__, self, [value, target, display_name, type_hint])

    @jsii.member(jsii_name="isReference")
    @builtins.classmethod
    def is_reference(cls, x: typing.Any) -> builtins.bool:
        '''Check whether this is actually a Reference.

        :param x: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__46271c1d495743d8b70cfe27766a83a4dfd7bc3e4cf9af80b7823e48a6e15bd0)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isReference", [x]))

    @builtins.property
    @jsii.member(jsii_name="displayName")
    def display_name(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "displayName"))

    @builtins.property
    @jsii.member(jsii_name="target")
    def target(self) -> _constructs_77d1e7e8.IConstruct:
        return typing.cast(_constructs_77d1e7e8.IConstruct, jsii.get(self, "target"))


class _ReferenceProxy(Reference):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Reference).__jsii_proxy_class__ = lambda : _ReferenceProxy


class RemovalPolicies(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.RemovalPolicies"):
    '''Manages removal policies for all resources within a construct scope, overriding any existing policies by default.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        removal_policies = cdk.RemovalPolicies.of(self)
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, scope: _constructs_77d1e7e8.IConstruct) -> "RemovalPolicies":
        '''Returns the removal policies API for the given scope.

        :param scope: The scope.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__16fdd678fafc26c9e1ca2d269cac449744dcfd65a4b8fce227879a53407f64f2)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("RemovalPolicies", jsii.sinvoke(cls, "of", [scope]))

    @jsii.member(jsii_name="apply")
    def apply(
        self,
        policy: "RemovalPolicy",
        *,
        apply_to_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Apply a removal policy to all resources within this scope, overriding any existing policies.

        :param policy: The removal policy to apply.
        :param apply_to_resource_types: Apply the removal policy only to specific resource types. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - apply to all resources
        :param exclude_resource_types: Exclude specific resource types from the removal policy. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - no exclusions
        :param priority: The priority to use when applying this policy. The priority affects only the order in which aspects are applied during synthesis. For RemovalPolicies, the last applied policy will override previous ones. NOTE: Priority does NOT determine which policy "wins" when there are conflicts. The order of application determines the final policy, with later policies overriding earlier ones. Default: - AspectPriority.MUTATING
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d1a6f380a312346d77f5859f328e14cef3d77e7eb99ba0dffc2dd7da58b9b0ca)
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
        props = RemovalPolicyProps(
            apply_to_resource_types=apply_to_resource_types,
            exclude_resource_types=exclude_resource_types,
            priority=priority,
        )

        return typing.cast(None, jsii.invoke(self, "apply", [policy, props]))

    @jsii.member(jsii_name="destroy")
    def destroy(
        self,
        *,
        apply_to_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Apply DESTROY removal policy to all resources within this scope.

        :param apply_to_resource_types: Apply the removal policy only to specific resource types. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - apply to all resources
        :param exclude_resource_types: Exclude specific resource types from the removal policy. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - no exclusions
        :param priority: The priority to use when applying this policy. The priority affects only the order in which aspects are applied during synthesis. For RemovalPolicies, the last applied policy will override previous ones. NOTE: Priority does NOT determine which policy "wins" when there are conflicts. The order of application determines the final policy, with later policies overriding earlier ones. Default: - AspectPriority.MUTATING
        '''
        props = RemovalPolicyProps(
            apply_to_resource_types=apply_to_resource_types,
            exclude_resource_types=exclude_resource_types,
            priority=priority,
        )

        return typing.cast(None, jsii.invoke(self, "destroy", [props]))

    @jsii.member(jsii_name="retain")
    def retain(
        self,
        *,
        apply_to_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Apply RETAIN removal policy to all resources within this scope.

        :param apply_to_resource_types: Apply the removal policy only to specific resource types. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - apply to all resources
        :param exclude_resource_types: Exclude specific resource types from the removal policy. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - no exclusions
        :param priority: The priority to use when applying this policy. The priority affects only the order in which aspects are applied during synthesis. For RemovalPolicies, the last applied policy will override previous ones. NOTE: Priority does NOT determine which policy "wins" when there are conflicts. The order of application determines the final policy, with later policies overriding earlier ones. Default: - AspectPriority.MUTATING
        '''
        props = RemovalPolicyProps(
            apply_to_resource_types=apply_to_resource_types,
            exclude_resource_types=exclude_resource_types,
            priority=priority,
        )

        return typing.cast(None, jsii.invoke(self, "retain", [props]))

    @jsii.member(jsii_name="retainOnUpdateOrDelete")
    def retain_on_update_or_delete(
        self,
        *,
        apply_to_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Apply RETAIN_ON_UPDATE_OR_DELETE removal policy to all resources within this scope.

        :param apply_to_resource_types: Apply the removal policy only to specific resource types. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - apply to all resources
        :param exclude_resource_types: Exclude specific resource types from the removal policy. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - no exclusions
        :param priority: The priority to use when applying this policy. The priority affects only the order in which aspects are applied during synthesis. For RemovalPolicies, the last applied policy will override previous ones. NOTE: Priority does NOT determine which policy "wins" when there are conflicts. The order of application determines the final policy, with later policies overriding earlier ones. Default: - AspectPriority.MUTATING
        '''
        props = RemovalPolicyProps(
            apply_to_resource_types=apply_to_resource_types,
            exclude_resource_types=exclude_resource_types,
            priority=priority,
        )

        return typing.cast(None, jsii.invoke(self, "retainOnUpdateOrDelete", [props]))

    @jsii.member(jsii_name="snapshot")
    def snapshot(
        self,
        *,
        apply_to_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Apply SNAPSHOT removal policy to all resources within this scope.

        :param apply_to_resource_types: Apply the removal policy only to specific resource types. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - apply to all resources
        :param exclude_resource_types: Exclude specific resource types from the removal policy. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - no exclusions
        :param priority: The priority to use when applying this policy. The priority affects only the order in which aspects are applied during synthesis. For RemovalPolicies, the last applied policy will override previous ones. NOTE: Priority does NOT determine which policy "wins" when there are conflicts. The order of application determines the final policy, with later policies overriding earlier ones. Default: - AspectPriority.MUTATING
        '''
        props = RemovalPolicyProps(
            apply_to_resource_types=apply_to_resource_types,
            exclude_resource_types=exclude_resource_types,
            priority=priority,
        )

        return typing.cast(None, jsii.invoke(self, "snapshot", [props]))


@jsii.enum(jsii_type="aws-cdk-lib.RemovalPolicy")
class RemovalPolicy(enum.Enum):
    '''Possible values for a resource's Removal Policy.

    The removal policy controls what happens to the resource if it stops being
    managed by CloudFormation. This can happen in one of three situations:

    - The resource is removed from the template, so CloudFormation stops managing it;
    - A change to the resource is made that requires it to be replaced, so CloudFormation stops
      managing it;
    - The stack is deleted, so CloudFormation stops managing all resources in it.

    The Removal Policy applies to all above cases.

    Many stateful resources in the AWS Construct Library will accept a
    ``removalPolicy`` as a property, typically defaulting it to ``RETAIN``.

    If the AWS Construct Library resource does not accept a ``removalPolicy``
    argument, you can always configure it by using the escape hatch mechanism,
    as shown in the following example::

       # bucket: s3.Bucket


       cfn_bucket = bucket.node.find_child("Resource")
       cfn_bucket.apply_removal_policy(RemovalPolicy.DESTROY)

    :exampleMetadata: infused

    Example::

        # my_role: iam.Role
        
        cr.AwsCustomResource(self, "Customized",
            role=my_role,  # must be assumable by the `lambda.amazonaws.com` service principal
            timeout=Duration.minutes(10),  # defaults to 2 minutes
            memory_size=1025,  # defaults to 512 if installLatestAwsSdk is true
            log_group=logs.LogGroup(self, "AwsCustomResourceLogs",
                retention=logs.RetentionDays.ONE_DAY
            ),
            function_name="my-custom-name",  # defaults to a CloudFormation generated name
            removal_policy=RemovalPolicy.RETAIN,  # defaults to `RemovalPolicy.DESTROY`
            policy=cr.AwsCustomResourcePolicy.from_sdk_calls(
                resources=cr.AwsCustomResourcePolicy.ANY_RESOURCE
            )
        )
    '''

    DESTROY = "DESTROY"
    '''This is the default removal policy.

    It means that when the resource is
    removed from the app, it will be physically destroyed.
    '''
    RETAIN = "RETAIN"
    '''This uses the 'Retain' DeletionPolicy, which will cause the resource to be retained in the account, but orphaned from the stack.'''
    SNAPSHOT = "SNAPSHOT"
    '''This retention policy deletes the resource, but saves a snapshot of its data before deleting, so that it can be re-created later.

    Only available for some stateful resources,
    like databases, EC2 volumes, etc.

    :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html#aws-attribute-deletionpolicy-options
    '''
    RETAIN_ON_UPDATE_OR_DELETE = "RETAIN_ON_UPDATE_OR_DELETE"
    '''Resource will be retained when they are requested to be deleted during a stack delete request or need to be replaced due to a stack update request.

    Resource are not retained, if the creation is rolled back.

    The result is that new, empty, and unused resources are deleted,
    while in-use resources and their data are retained.

    This uses the 'RetainExceptOnCreate' DeletionPolicy,
    and the 'Retain' UpdateReplacePolicy, when ``applyToUpdateReplacePolicy`` is set.

    :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html#aws-attribute-deletionpolicy-options
    '''


@jsii.data_type(
    jsii_type="aws-cdk-lib.RemovalPolicyOptions",
    jsii_struct_bases=[],
    name_mapping={
        "apply_to_update_replace_policy": "applyToUpdateReplacePolicy",
        "default": "default",
    },
)
class RemovalPolicyOptions:
    def __init__(
        self,
        *,
        apply_to_update_replace_policy: typing.Optional[builtins.bool] = None,
        default: typing.Optional[RemovalPolicy] = None,
    ) -> None:
        '''
        :param apply_to_update_replace_policy: Apply the same deletion policy to the resource's "UpdateReplacePolicy". Default: true
        :param default: The default policy to apply in case the removal policy is not defined. Default: - Default value is resource specific. To determine the default value for a resource, please consult that specific resource's documentation.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            removal_policy_options = cdk.RemovalPolicyOptions(
                apply_to_update_replace_policy=False,
                default=cdk.RemovalPolicy.DESTROY
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cf59d09830cb04cd8d888f7fcdf115e700dad2a3650ac4a9beaed76aea901790)
            check_type(argname="argument apply_to_update_replace_policy", value=apply_to_update_replace_policy, expected_type=type_hints["apply_to_update_replace_policy"])
            check_type(argname="argument default", value=default, expected_type=type_hints["default"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if apply_to_update_replace_policy is not None:
            self._values["apply_to_update_replace_policy"] = apply_to_update_replace_policy
        if default is not None:
            self._values["default"] = default

    @builtins.property
    def apply_to_update_replace_policy(self) -> typing.Optional[builtins.bool]:
        '''Apply the same deletion policy to the resource's "UpdateReplacePolicy".

        :default: true
        '''
        result = self._values.get("apply_to_update_replace_policy")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def default(self) -> typing.Optional[RemovalPolicy]:
        '''The default policy to apply in case the removal policy is not defined.

        :default:

        - Default value is resource specific. To determine the default value for a resource,
        please consult that specific resource's documentation.
        '''
        result = self._values.get("default")
        return typing.cast(typing.Optional[RemovalPolicy], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RemovalPolicyOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.RemovalPolicyProps",
    jsii_struct_bases=[],
    name_mapping={
        "apply_to_resource_types": "applyToResourceTypes",
        "exclude_resource_types": "excludeResourceTypes",
        "priority": "priority",
    },
)
class RemovalPolicyProps:
    def __init__(
        self,
        *,
        apply_to_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for applying a removal policy.

        :param apply_to_resource_types: Apply the removal policy only to specific resource types. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - apply to all resources
        :param exclude_resource_types: Exclude specific resource types from the removal policy. Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket'). Default: - no exclusions
        :param priority: The priority to use when applying this policy. The priority affects only the order in which aspects are applied during synthesis. For RemovalPolicies, the last applied policy will override previous ones. NOTE: Priority does NOT determine which policy "wins" when there are conflicts. The order of application determines the final policy, with later policies overriding earlier ones. Default: - AspectPriority.MUTATING

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            removal_policy_props = cdk.RemovalPolicyProps(
                apply_to_resource_types=["applyToResourceTypes"],
                exclude_resource_types=["excludeResourceTypes"],
                priority=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c5c7fd6f5b115619c0c625e32cc15eb9bdbfb4931f25b97d8db03b7037bf1a39)
            check_type(argname="argument apply_to_resource_types", value=apply_to_resource_types, expected_type=type_hints["apply_to_resource_types"])
            check_type(argname="argument exclude_resource_types", value=exclude_resource_types, expected_type=type_hints["exclude_resource_types"])
            check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if apply_to_resource_types is not None:
            self._values["apply_to_resource_types"] = apply_to_resource_types
        if exclude_resource_types is not None:
            self._values["exclude_resource_types"] = exclude_resource_types
        if priority is not None:
            self._values["priority"] = priority

    @builtins.property
    def apply_to_resource_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Apply the removal policy only to specific resource types.

        Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket').

        :default: - apply to all resources
        '''
        result = self._values.get("apply_to_resource_types")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def exclude_resource_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Exclude specific resource types from the removal policy.

        Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket').

        :default: - no exclusions
        '''
        result = self._values.get("exclude_resource_types")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def priority(self) -> typing.Optional[jsii.Number]:
        '''The priority to use when applying this policy.

        The priority affects only the order in which aspects are applied during synthesis.
        For RemovalPolicies, the last applied policy will override previous ones.

        NOTE: Priority does NOT determine which policy "wins" when there are conflicts.
        The order of application determines the final policy, with later policies
        overriding earlier ones.

        :default: - AspectPriority.MUTATING
        '''
        result = self._values.get("priority")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RemovalPolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IAspect)
class RemoveTag(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.RemoveTag"):
    '''The RemoveTag Aspect will handle removing tags from this node and children.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        remove_tag = cdk.RemoveTag("key",
            apply_to_launched_instances=False,
            exclude_resource_types=["excludeResourceTypes"],
            include_resource_types=["includeResourceTypes"],
            priority=123
        )
    '''

    def __init__(
        self,
        key: builtins.str,
        *,
        apply_to_launched_instances: typing.Optional[builtins.bool] = None,
        exclude_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        include_resource_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param key: The string key for the tag.
        :param apply_to_launched_instances: Whether the tag should be applied to instances in an AutoScalingGroup. Default: true
        :param exclude_resource_types: An array of Resource Types that will not receive this tag. An empty array will allow this tag to be applied to all resources. A non-empty array will apply this tag only if the Resource type is not in this array. Default: []
        :param include_resource_types: An array of Resource Types that will receive this tag. An empty array will match any Resource. A non-empty array will apply this tag only to Resource types that are included in this array. Default: []
        :param priority: Priority of the tag operation. Higher or equal priority tags will take precedence. Setting priority will enable the user to control tags when they need to not follow the default precedence pattern of last applied and closest to the construct in the tree. Default: Default priorities: - 100 for ``SetTag`` - 200 for ``RemoveTag`` - 50 for tags added directly to CloudFormation resources
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__944c89fd9e400a7d6e966cc04588226988866735f42049e619d446afc77d54c6)
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
        props = TagProps(
            apply_to_launched_instances=apply_to_launched_instances,
            exclude_resource_types=exclude_resource_types,
            include_resource_types=include_resource_types,
            priority=priority,
        )

        jsii.create(self.__class__, self, [key, props])

    @jsii.member(jsii_name="applyTag")
    def _apply_tag(self, resource: ITaggable) -> None:
        '''
        :param resource: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2f8978b8bc5c9737805c5858f1441a5870dddeb85f06a4d696d24aeee615baec)
            check_type(argname="argument resource", value=resource, expected_type=type_hints["resource"])
        return typing.cast(None, jsii.invoke(self, "applyTag", [resource]))

    @jsii.member(jsii_name="applyTagV2")
    def _apply_tag_v2(self, resource: ITaggableV2) -> None:
        '''
        :param resource: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1653e2686a9854afa401d3c57a2fa1ca443a2f8ad16ead4fb21328dbbcf3e4f4)
            check_type(argname="argument resource", value=resource, expected_type=type_hints["resource"])
        return typing.cast(None, jsii.invoke(self, "applyTagV2", [resource]))

    @jsii.member(jsii_name="visit")
    def visit(self, construct: _constructs_77d1e7e8.IConstruct) -> None:
        '''All aspects can visit an IConstruct.

        :param construct: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__87d826eb9a59857ea8d0e99f91f6f35db7f178187ddb0ba7a65f8cdabc502a39)
            check_type(argname="argument construct", value=construct, expected_type=type_hints["construct"])
        return typing.cast(None, jsii.invoke(self, "visit", [construct]))

    @builtins.property
    @jsii.member(jsii_name="key")
    def key(self) -> builtins.str:
        '''The string key for the tag.'''
        return typing.cast(builtins.str, jsii.get(self, "key"))

    @builtins.property
    @jsii.member(jsii_name="props")
    def _props(self) -> "TagProps":
        return typing.cast("TagProps", jsii.get(self, "props"))


@jsii.enum(jsii_type="aws-cdk-lib.ResolutionTypeHint")
class ResolutionTypeHint(enum.Enum):
    '''Type hints for resolved values.'''

    STRING = "STRING"
    '''This value is expected to resolve to a String.'''
    NUMBER = "NUMBER"
    '''This value is expected to resolve to a Number.'''
    STRING_LIST = "STRING_LIST"
    '''This value is expected to resolve to a String List.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.ResolveChangeContextOptions",
    jsii_struct_bases=[],
    name_mapping={
        "allow_intrinsic_keys": "allowIntrinsicKeys",
        "remove_empty": "removeEmpty",
    },
)
class ResolveChangeContextOptions:
    def __init__(
        self,
        *,
        allow_intrinsic_keys: typing.Optional[builtins.bool] = None,
        remove_empty: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options that can be changed while doing a recursive resolve.

        :param allow_intrinsic_keys: Change the 'allowIntrinsicKeys' option. Default: - Unchanged
        :param remove_empty: Whether to remove undefined elements from arrays and objects when resolving. Default: - Unchanged

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            resolve_change_context_options = cdk.ResolveChangeContextOptions(
                allow_intrinsic_keys=False,
                remove_empty=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e759a4813367bf33ce09bc5ad1a3183d48997597c4145a75820c8c623577c913)
            check_type(argname="argument allow_intrinsic_keys", value=allow_intrinsic_keys, expected_type=type_hints["allow_intrinsic_keys"])
            check_type(argname="argument remove_empty", value=remove_empty, expected_type=type_hints["remove_empty"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allow_intrinsic_keys is not None:
            self._values["allow_intrinsic_keys"] = allow_intrinsic_keys
        if remove_empty is not None:
            self._values["remove_empty"] = remove_empty

    @builtins.property
    def allow_intrinsic_keys(self) -> typing.Optional[builtins.bool]:
        '''Change the 'allowIntrinsicKeys' option.

        :default: - Unchanged
        '''
        result = self._values.get("allow_intrinsic_keys")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def remove_empty(self) -> typing.Optional[builtins.bool]:
        '''Whether to remove undefined elements from arrays and objects when resolving.

        :default: - Unchanged
        '''
        result = self._values.get("remove_empty")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ResolveChangeContextOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.ResolveOptions",
    jsii_struct_bases=[],
    name_mapping={
        "resolver": "resolver",
        "scope": "scope",
        "preparing": "preparing",
        "remove_empty": "removeEmpty",
    },
)
class ResolveOptions:
    def __init__(
        self,
        *,
        resolver: ITokenResolver,
        scope: _constructs_77d1e7e8.IConstruct,
        preparing: typing.Optional[builtins.bool] = None,
        remove_empty: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options to the resolve() operation.

        NOT the same as the ResolveContext; ResolveContext is exposed to Token
        implementors and resolution hooks, whereas this struct is just to bundle
        a number of things that would otherwise be arguments to resolve() in a
        readable way.

        :param resolver: The resolver to apply to any resolvable tokens found.
        :param scope: The scope from which resolution is performed.
        :param preparing: Whether the resolution is being executed during the prepare phase or not. Default: false
        :param remove_empty: Whether to remove undefined elements from arrays and objects when resolving. Default: true

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            import constructs as constructs
            
            # construct: constructs.Construct
            # token_resolver: cdk.ITokenResolver
            
            resolve_options = cdk.ResolveOptions(
                resolver=token_resolver,
                scope=construct,
            
                # the properties below are optional
                preparing=False,
                remove_empty=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cca95bfb48af9f56980b5278429eed5b79f1526370cdd73c6a79640ece0a5c1f)
            check_type(argname="argument resolver", value=resolver, expected_type=type_hints["resolver"])
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument preparing", value=preparing, expected_type=type_hints["preparing"])
            check_type(argname="argument remove_empty", value=remove_empty, expected_type=type_hints["remove_empty"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "resolver": resolver,
            "scope": scope,
        }
        if preparing is not None:
            self._values["preparing"] = preparing
        if remove_empty is not None:
            self._values["remove_empty"] = remove_empty

    @builtins.property
    def resolver(self) -> ITokenResolver:
        '''The resolver to apply to any resolvable tokens found.'''
        result = self._values.get("resolver")
        assert result is not None, "Required property 'resolver' is missing"
        return typing.cast(ITokenResolver, result)

    @builtins.property
    def scope(self) -> _constructs_77d1e7e8.IConstruct:
        '''The scope from which resolution is performed.'''
        result = self._values.get("scope")
        assert result is not None, "Required property 'scope' is missing"
        return typing.cast(_constructs_77d1e7e8.IConstruct, result)

    @builtins.property
    def preparing(self) -> typing.Optional[builtins.bool]:
        '''Whether the resolution is being executed during the prepare phase or not.

        :default: false
        '''
        result = self._values.get("preparing")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def remove_empty(self) -> typing.Optional[builtins.bool]:
        '''Whether to remove undefined elements from arrays and objects when resolving.

        :default: true
        '''
        result = self._values.get("remove_empty")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ResolveOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IResource)
class Resource(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.Resource",
):
    '''A construct which represents an AWS resource.

    :exampleMetadata: infused

    Example::

        @jsii.implements(ITaggable)
        class MyConstruct(Resource):
        
            def __init__(self, scope, id):
                super().__init__(scope, id)
        
                CfnResource(self, "Resource",
                    type="Whatever::The::Type",
                    properties={
                        # ...
                        "Tags": self.tags.rendered_tags
                    }
                )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        environment_from_arn: typing.Optional[builtins.str] = None,
        physical_name: typing.Optional[builtins.str] = None,
        region: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param account: The AWS account ID this resource belongs to. Default: - the resource is in the same account as the stack it belongs to
        :param environment_from_arn: ARN to deduce region and account from. The ARN is parsed and the account and region are taken from the ARN. This should be used for imported resources. Cannot be supplied together with either ``account`` or ``region``. Default: - take environment from ``account``, ``region`` parameters, or use Stack environment.
        :param physical_name: The value passed in by users to the physical name prop of the resource. - ``undefined`` implies that a physical name will be allocated by CloudFormation during deployment. - a concrete value implies a specific physical name - ``PhysicalName.GENERATE_IF_NEEDED`` is a marker that indicates that a physical will only be generated by the CDK if it is needed for cross-environment references. Otherwise, it will be allocated by CloudFormation. Default: - The physical name will be allocated by CloudFormation at deployment time
        :param region: The AWS region this resource belongs to. Default: - the resource is in the same region as the stack it belongs to
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f38d604b7651ff46ec22b40c01fb8155f8c79decce9dcc3739431012b7093c59)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ResourceProps(
            account=account,
            environment_from_arn=environment_from_arn,
            physical_name=physical_name,
            region=region,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="isOwnedResource")
    @builtins.classmethod
    def is_owned_resource(
        cls,
        construct: _constructs_77d1e7e8.IConstruct,
    ) -> builtins.bool:
        '''Returns true if the construct was created by CDK, and false otherwise.

        :param construct: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b98d028f2c2f5fc2fc282f5d2c0db05cfaf9a952f78b921ddbad309dd32bb693)
            check_type(argname="argument construct", value=construct, expected_type=type_hints["construct"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isOwnedResource", [construct]))

    @jsii.member(jsii_name="isResource")
    @builtins.classmethod
    def is_resource(cls, construct: _constructs_77d1e7e8.IConstruct) -> builtins.bool:
        '''Check whether the given construct is a Resource.

        :param construct: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f6205fdb97a6e809ad8be1edd5047aba48fa457be4543da186027361b3415004)
            check_type(argname="argument construct", value=construct, expected_type=type_hints["construct"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isResource", [construct]))

    @jsii.member(jsii_name="applyRemovalPolicy")
    def apply_removal_policy(self, policy: RemovalPolicy) -> None:
        '''Apply the given removal policy to this resource.

        The Removal Policy controls what happens to this resource when it stops
        being managed by CloudFormation, either because you've removed it from the
        CDK application or because you've made a change that requires the resource
        to be replaced.

        The resource can be deleted (``RemovalPolicy.DESTROY``), or left in your AWS
        account for data recovery and cleanup later (``RemovalPolicy.RETAIN``).

        :param policy: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6695af80c1fedde1bce659ecfd4adaa992f8d80a9991731fd956c6031773a81e)
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
        return typing.cast(None, jsii.invoke(self, "applyRemovalPolicy", [policy]))

    @jsii.member(jsii_name="generatePhysicalName")
    def _generate_physical_name(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "generatePhysicalName", []))

    @jsii.member(jsii_name="getResourceArnAttribute")
    def _get_resource_arn_attribute(
        self,
        arn_attr: builtins.str,
        *,
        resource: builtins.str,
        service: builtins.str,
        account: typing.Optional[builtins.str] = None,
        arn_format: typing.Optional[ArnFormat] = None,
        partition: typing.Optional[builtins.str] = None,
        region: typing.Optional[builtins.str] = None,
        resource_name: typing.Optional[builtins.str] = None,
    ) -> builtins.str:
        '''Returns an environment-sensitive token that should be used for the resource's "ARN" attribute (e.g. ``bucket.bucketArn``).

        Normally, this token will resolve to ``arnAttr``, but if the resource is
        referenced across environments, ``arnComponents`` will be used to synthesize
        a concrete ARN with the resource's physical name. Make sure to reference
        ``this.physicalName`` in ``arnComponents``.

        :param arn_attr: The CFN attribute which resolves to the ARN of the resource. Commonly it will be called "Arn" (e.g. ``resource.attrArn``), but sometimes it's the CFN resource's ``ref``.
        :param resource: Resource type (e.g. "table", "autoScalingGroup", "certificate"). For some resource types, e.g. S3 buckets, this field defines the bucket name.
        :param service: The service namespace that identifies the AWS product (for example, 's3', 'iam', 'codepipeline').
        :param account: The ID of the AWS account that owns the resource, without the hyphens. For example, 123456789012. Note that the ARNs for some resources don't require an account number, so this component might be omitted. Default: The account the stack is deployed to.
        :param arn_format: The specific ARN format to use for this ARN value. Default: - uses value of ``sep`` as the separator for formatting, ``ArnFormat.SLASH_RESOURCE_NAME`` if that property was also not provided
        :param partition: The partition that the resource is in. For standard AWS regions, the partition is aws. If you have resources in other partitions, the partition is aws-partitionname. For example, the partition for resources in the China (Beijing) region is aws-cn. Default: The AWS partition the stack is deployed to.
        :param region: The region the resource resides in. Note that the ARNs for some resources do not require a region, so this component might be omitted. Default: The region the stack is deployed to.
        :param resource_name: Resource name or path within the resource (i.e. S3 bucket object key) or a wildcard such as ``"*"``. This is service-dependent.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__81a274b9bddb7d2d27126243d1f51bbc1a0b5a35ad11c3871538c25c42a9147d)
            check_type(argname="argument arn_attr", value=arn_attr, expected_type=type_hints["arn_attr"])
        arn_components = ArnComponents(
            resource=resource,
            service=service,
            account=account,
            arn_format=arn_format,
            partition=partition,
            region=region,
            resource_name=resource_name,
        )

        return typing.cast(builtins.str, jsii.invoke(self, "getResourceArnAttribute", [arn_attr, arn_components]))

    @jsii.member(jsii_name="getResourceNameAttribute")
    def _get_resource_name_attribute(self, name_attr: builtins.str) -> builtins.str:
        '''Returns an environment-sensitive token that should be used for the resource's "name" attribute (e.g. ``bucket.bucketName``).

        Normally, this token will resolve to ``nameAttr``, but if the resource is
        referenced across environments, it will be resolved to ``this.physicalName``,
        which will be a concrete name.

        :param name_attr: The CFN attribute which resolves to the resource's name. Commonly this is the resource's ``ref``.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e152963ad0a4f5f40b2ee753c3fe470e49b2c37b56c79a40576ebacd92139d4f)
            check_type(argname="argument name_attr", value=name_attr, expected_type=type_hints["name_attr"])
        return typing.cast(builtins.str, jsii.invoke(self, "getResourceNameAttribute", [name_attr]))

    @builtins.property
    @jsii.member(jsii_name="env")
    def env(self) -> "ResourceEnvironment":
        '''The environment this resource belongs to.

        For resources that are created and managed by the CDK
        (generally, those created by creating new class instances like Role, Bucket, etc.),
        this is always the same as the environment of the stack they belong to;
        however, for imported resources
        (those obtained from static methods like fromRoleArn, fromBucketName, etc.),
        that might be different than the stack they were imported into.
        '''
        return typing.cast("ResourceEnvironment", jsii.get(self, "env"))

    @builtins.property
    @jsii.member(jsii_name="physicalName")
    def _physical_name(self) -> builtins.str:
        '''Returns a string-encoded token that resolves to the physical name that should be passed to the CloudFormation resource.

        This value will resolve to one of the following:

        - a concrete value (e.g. ``"my-awesome-bucket"``)
        - ``undefined``, when a name should be generated by CloudFormation
        - a concrete name generated automatically during synthesis, in
          cross-environment scenarios.
        '''
        return typing.cast(builtins.str, jsii.get(self, "physicalName"))

    @builtins.property
    @jsii.member(jsii_name="stack")
    def stack(self) -> "Stack":
        '''The stack in which this resource is defined.'''
        return typing.cast("Stack", jsii.get(self, "stack"))


class _ResourceProxy(Resource):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Resource).__jsii_proxy_class__ = lambda : _ResourceProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.ResourceEnvironment",
    jsii_struct_bases=[],
    name_mapping={"account": "account", "region": "region"},
)
class ResourceEnvironment:
    def __init__(self, *, account: builtins.str, region: builtins.str) -> None:
        '''Represents the environment a given resource lives in.

        Used as the return value for the ``IResource.env`` property.

        :param account: The AWS account ID that this resource belongs to. Since this can be a Token (for example, when the account is CloudFormation's AWS::AccountId intrinsic), make sure to use Token.compareStrings() instead of just comparing the values for equality.
        :param region: The AWS region that this resource belongs to. Since this can be a Token (for example, when the region is CloudFormation's AWS::Region intrinsic), make sure to use Token.compareStrings() instead of just comparing the values for equality.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            resource_environment = cdk.ResourceEnvironment(
                account="account",
                region="region"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cf7fa86b31dd6c8ec358d9877964c9388a3b3135c168d4af11dbeb917590695d)
            check_type(argname="argument account", value=account, expected_type=type_hints["account"])
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "account": account,
            "region": region,
        }

    @builtins.property
    def account(self) -> builtins.str:
        '''The AWS account ID that this resource belongs to.

        Since this can be a Token
        (for example, when the account is CloudFormation's AWS::AccountId intrinsic),
        make sure to use Token.compareStrings()
        instead of just comparing the values for equality.
        '''
        result = self._values.get("account")
        assert result is not None, "Required property 'account' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def region(self) -> builtins.str:
        '''The AWS region that this resource belongs to.

        Since this can be a Token
        (for example, when the region is CloudFormation's AWS::Region intrinsic),
        make sure to use Token.compareStrings()
        instead of just comparing the values for equality.
        '''
        result = self._values.get("region")
        assert result is not None, "Required property 'region' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ResourceEnvironment(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.ResourceProps",
    jsii_struct_bases=[],
    name_mapping={
        "account": "account",
        "environment_from_arn": "environmentFromArn",
        "physical_name": "physicalName",
        "region": "region",
    },
)
class ResourceProps:
    def __init__(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        environment_from_arn: typing.Optional[builtins.str] = None,
        physical_name: typing.Optional[builtins.str] = None,
        region: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Construction properties for ``Resource``.

        :param account: The AWS account ID this resource belongs to. Default: - the resource is in the same account as the stack it belongs to
        :param environment_from_arn: ARN to deduce region and account from. The ARN is parsed and the account and region are taken from the ARN. This should be used for imported resources. Cannot be supplied together with either ``account`` or ``region``. Default: - take environment from ``account``, ``region`` parameters, or use Stack environment.
        :param physical_name: The value passed in by users to the physical name prop of the resource. - ``undefined`` implies that a physical name will be allocated by CloudFormation during deployment. - a concrete value implies a specific physical name - ``PhysicalName.GENERATE_IF_NEEDED`` is a marker that indicates that a physical will only be generated by the CDK if it is needed for cross-environment references. Otherwise, it will be allocated by CloudFormation. Default: - The physical name will be allocated by CloudFormation at deployment time
        :param region: The AWS region this resource belongs to. Default: - the resource is in the same region as the stack it belongs to

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            resource_props = cdk.ResourceProps(
                account="account",
                environment_from_arn="environmentFromArn",
                physical_name="physicalName",
                region="region"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__addea20f6555c6b1cb134b415dbf6769d4ce77cc07c0cce9135a8d648d78b8a7)
            check_type(argname="argument account", value=account, expected_type=type_hints["account"])
            check_type(argname="argument environment_from_arn", value=environment_from_arn, expected_type=type_hints["environment_from_arn"])
            check_type(argname="argument physical_name", value=physical_name, expected_type=type_hints["physical_name"])
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if account is not None:
            self._values["account"] = account
        if environment_from_arn is not None:
            self._values["environment_from_arn"] = environment_from_arn
        if physical_name is not None:
            self._values["physical_name"] = physical_name
        if region is not None:
            self._values["region"] = region

    @builtins.property
    def account(self) -> typing.Optional[builtins.str]:
        '''The AWS account ID this resource belongs to.

        :default: - the resource is in the same account as the stack it belongs to
        '''
        result = self._values.get("account")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment_from_arn(self) -> typing.Optional[builtins.str]:
        '''ARN to deduce region and account from.

        The ARN is parsed and the account and region are taken from the ARN.
        This should be used for imported resources.

        Cannot be supplied together with either ``account`` or ``region``.

        :default: - take environment from ``account``, ``region`` parameters, or use Stack environment.
        '''
        result = self._values.get("environment_from_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def physical_name(self) -> typing.Optional[builtins.str]:
        '''The value passed in by users to the physical name prop of the resource.

        - ``undefined`` implies that a physical name will be allocated by
          CloudFormation during deployment.
        - a concrete value implies a specific physical name
        - ``PhysicalName.GENERATE_IF_NEEDED`` is a marker that indicates that a physical will only be generated
          by the CDK if it is needed for cross-environment references. Otherwise, it will be allocated by CloudFormation.

        :default: - The physical name will be allocated by CloudFormation at deployment time
        '''
        result = self._values.get("physical_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def region(self) -> typing.Optional[builtins.str]:
        '''The AWS region this resource belongs to.

        :default: - the resource is in the same region as the stack it belongs to
        '''
        result = self._values.get("region")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ResourceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.ReverseOptions",
    jsii_struct_bases=[],
    name_mapping={"fail_concat": "failConcat"},
)
class ReverseOptions:
    def __init__(self, *, fail_concat: typing.Optional[builtins.bool] = None) -> None:
        '''Options for the 'reverse()' operation.

        :param fail_concat: Fail if the given string is a concatenation. If ``false``, just return ``undefined``. Default: true

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            reverse_options = cdk.ReverseOptions(
                fail_concat=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__55afe4d64fb4d5ea0191f49597eab03d77d7784d61bfdf374b068db656601802)
            check_type(argname="argument fail_concat", value=fail_concat, expected_type=type_hints["fail_concat"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if fail_concat is not None:
            self._values["fail_concat"] = fail_concat

    @builtins.property
    def fail_concat(self) -> typing.Optional[builtins.bool]:
        '''Fail if the given string is a concatenation.

        If ``false``, just return ``undefined``.

        :default: true
        '''
        result = self._values.get("fail_concat")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ReverseOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.RoleOptions",
    jsii_struct_bases=[],
    name_mapping={
        "assume_role_arn": "assumeRoleArn",
        "assume_role_additional_options": "assumeRoleAdditionalOptions",
        "assume_role_external_id": "assumeRoleExternalId",
    },
)
class RoleOptions:
    def __init__(
        self,
        *,
        assume_role_arn: builtins.str,
        assume_role_additional_options: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        assume_role_external_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options for specifying a role.

        :param assume_role_arn: ARN of the role to assume.
        :param assume_role_additional_options: Additional options to pass to STS when assuming the role for cloudformation deployments. - ``RoleArn`` should not be used. Use the dedicated ``assumeRoleArn`` property instead. - ``ExternalId`` should not be used. Use the dedicated ``assumeRoleExternalId`` instead. - ``TransitiveTagKeys`` defaults to use all keys (if any) specified in ``Tags``. E.g, all tags are transitive by default. Default: - No additional options.
        :param assume_role_external_id: External ID to use when assuming the role. Default: - No external ID

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            
            # assume_role_additional_options: Any
            
            role_options = cdk.RoleOptions(
                assume_role_arn="assumeRoleArn",
            
                # the properties below are optional
                assume_role_additional_options={
                    "assume_role_additional_options_key": assume_role_additional_options
                },
                assume_role_external_id="assumeRoleExternalId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__350c3eeb771368884765de415edb2b16931d742fbdb5253af2222e0e84a150c0)
            check_type(argname="argument assume_role_arn", value=assume_role_arn, expected_type=type_hints["assume_role_arn"])
            check_type(argname="argument assume_role_additional_options", value=assume_role_additional_options, expected_type=type_hints["assume_role_additional_options"])
            check_type(argname="argument assume_role_external_id", value=assume_role_external_id, expected_type=type_hints["assume_role_external_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "assume_role_arn": assume_role_arn,
        }
        if assume_role_additional_options is not None:
            self._values["assume_role_additional_options"] = assume_role_additional_options
        if assume_role_external_id is not None:
            self._values["assume_role_external_id"] = assume_role_external_id

    @builtins.property
    def assume_role_arn(self) -> builtins.str:
        '''ARN of the role to assume.'''
        result = self._values.get("assume_role_arn")
        assert result is not None, "Required property 'assume_role_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def assume_role_additional_options(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Additional options to pass to STS when assuming the role for cloudformation deployments.

        - ``RoleArn`` should not be used. Use the dedicated ``assumeRoleArn`` property instead.
        - ``ExternalId`` should not be used. Use the dedicated ``assumeRoleExternalId`` instead.
        - ``TransitiveTagKeys`` defaults to use all keys (if any) specified in ``Tags``. E.g, all tags are transitive by default.

        :default: - No additional options.

        :see: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/STS.html#assumeRole-property
        '''
        result = self._values.get("assume_role_additional_options")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def assume_role_external_id(self) -> typing.Optional[builtins.str]:
        '''External ID to use when assuming the role.

        :default: - No external ID
        '''
        result = self._values.get("assume_role_external_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RoleOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ScopedAws(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.ScopedAws"):
    '''Accessor for scoped pseudo parameters.

    These pseudo parameters are anchored to a stack somewhere in the construct
    tree, and their values will be exported automatically.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        
        scoped_aws = cdk.ScopedAws(self)
    '''

    def __init__(self, scope: _constructs_77d1e7e8.Construct) -> None:
        '''
        :param scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__689f321e7e4b904582ab3274360f95cc0da0af0d900d2026491edbacc5078af7)
            check_type(argname="argument scope", value=scope, expected_