import { Construct } from 'constructs';
import { LogOptions } from './waiter-state-machine';
import '../../../aws-cloudformation';
import * as ec2 from '../../../aws-ec2';
import * as iam from '../../../aws-iam';
import * as kms from '../../../aws-kms';
import * as lambda from '../../../aws-lambda';
import * as logs from '../../../aws-logs';
import { Duration } from '../../../core';
/**
 * Initialization properties for the `Provider` construct.
 */
export interface ProviderProps {
    /**
     * The AWS Lambda function to invoke for all resource lifecycle operations
     * (CREATE/UPDATE/DELETE).
     *
     * This function is responsible to begin the requested resource operation
     * (CREATE/UPDATE/DELETE) and return any additional properties to add to the
     * event, which will later be passed to `isComplete`. The `PhysicalResourceId`
     * property must be included in the response.
     */
    readonly onEventHandler: lambda.IFunction;
    /**
     * The AWS Lambda function to invoke in order to determine if the operation is
     * complete.
     *
     * This function will be called immediately after `onEvent` and then
     * periodically based on the configured query interval as long as it returns
     * `false`. If the function still returns `false` and the alloted timeout has
     * passed, the operation will fail.
     *
     * @default - provider is synchronous. This means that the `onEvent` handler
     * is expected to finish all lifecycle operations within the initial invocation.
     */
    readonly isCompleteHandler?: lambda.IFunction;
    /**
     * Time between calls to the `isComplete` handler which determines if the
     * resource has been stabilized.
     *
     * The first `isComplete` will be called immediately after `handler` and then
     * every `queryInterval` seconds, and until `timeout` has been reached or until
     * `isComplete` returns `true`.
     *
     * @default Duration.seconds(5)
     */
    readonly queryInterval?: Duration;
    /**
     * Total timeout for the entire operation.
     *
     * The maximum timeout is 1 hour (yes, it can exceed the AWS Lambda 15 minutes)
     *
     * @default Duration.minutes(30)
     */
    readonly totalTimeout?: Duration;
    /**
     * The number of days framework log events are kept in CloudWatch Logs. When
     * updating this property, unsetting it doesn't remove the log retention policy.
     * To remove the retention policy, set the value to `INFINITE`.
     *
     * This is a legacy API and we strongly recommend you migrate to `logGroup` if you can.
     * `logGroup` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it.
     *
     * @default logs.RetentionDays.INFINITE
     */
    readonly logRetention?: logs.RetentionDays;
    /**
     * The Log Group used for logging of events emitted by the custom resource's lambda function.
     *
     * Providing a user-controlled log group was rolled out to commercial regions on 2023-11-16.
     * If you are deploying to another type of region, please check regional availability first.
     *
     * @default - a default log group created by AWS Lambda
     */
    readonly logGroup?: logs.ILogGroup;
    /**
     * The vpc to provision the lambda functions in.
     *
     * @default - functions are not provisioned inside a vpc.
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Which subnets from the VPC to place the lambda functions in.
     *
     * Only used if 'vpc' is supplied. Note: internet access for Lambdas
     * requires a NAT gateway, so picking Public subnets is not allowed.
     *
     * @default - the Vpc default strategy if not specified
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * Security groups to attach to the provider functions.
     *
     * Only used if 'vpc' is supplied
     *
     * @default - If `vpc` is not supplied, no security groups are attached. Otherwise, a dedicated security
     * group is created for each function.
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * AWS Lambda execution role.
     *
     * The role is shared by provider framework's onEvent, isComplete lambda, and onTimeout Lambda functions.
     * This role will be assumed by the AWS Lambda, so it must be assumable by the 'lambda.amazonaws.com'
     * service principal.
     *
     * @default - A default role will be created.
     * @deprecated - Use frameworkOnEventLambdaRole, frameworkIsCompleteLambdaRole, frameworkOnTimeoutLambdaRole
     */
    readonly role?: iam.IRole;
    /**
     * Lambda execution role for provider framework's onEvent Lambda function. Note that this role must be assumed
     * by the 'lambda.amazonaws.com' service principal.
     *
     * This property cannot be used with 'role' property
     *
     * @default - A default role will be created.
     */
    readonly frameworkOnEventRole?: iam.IRole;
    /**
     * Lambda execution role for provider framework's isComplete/onTimeout Lambda function. Note that this role
     * must be assumed by the 'lambda.amazonaws.com' service principal. To prevent circular dependency problem
     * in the provider framework, please ensure you specify a different IAM Role for 'frameworkCompleteAndTimeoutRole'
     * from 'frameworkOnEventRole'.
     *
     * This property cannot be used with 'role' property
     *
     * @default - A default role will be created.
     */
    readonly frameworkCompleteAndTimeoutRole?: iam.IRole;
    /**
     * Provider Lambda name.
     *
     * The provider lambda function name.
     *
     * @default -  CloudFormation default name from unique physical ID
     */
    readonly providerFunctionName?: string;
    /**
     * AWS KMS key used to encrypt provider lambda's environment variables.
     *
     * @default -  AWS Lambda creates and uses an AWS managed customer master key (CMK)
     */
    readonly providerFunctionEnvEncryption?: kms.IKey;
    /**
     * Defines what execution history events of the waiter state machine are logged and where they are logged.
     *
     * @default - A default log group will be created if logging for the waiter state machine is enabled.
     */
    readonly waiterStateMachineLogOptions?: LogOptions;
    /**
     * Whether logging for the waiter state machine is disabled.
     *
     * @default - false
     */
    readonly disableWaiterStateMachineLogging?: boolean;
}
/**
 * Defines an AWS CloudFormation custom resource provider.
 */
export declare class Provider extends Construct {
    /**
     * The user-defined AWS Lambda function which is invoked for all resource
     * lifecycle operations (CREATE/UPDATE/DELETE).
     */
    readonly onEventHandler: lambda.IFunction;
    /**
     * The user-defined AWS Lambda function which is invoked asynchronously in
     * order to determine if the operation is complete.
     */
    readonly isCompleteHandler?: lambda.IFunction;
    /**
     * The service token to use in order to define custom resources that are
     * backed by this provider.
     */
    readonly serviceToken: string;
    private readonly entrypoint;
    private readonly logRetention?;
    private readonly logGroup?;
    private readonly vpc?;
    private readonly vpcSubnets?;
    private readonly securityGroups?;
    private readonly role?;
    private readonly providerFunctionEnvEncryption?;
    constructor(scope: Construct, id: string, props: ProviderProps);
    private addPermissions;
    private createFunction;
}
