import { ISynthesisSession } from './types';
import * as cxschema from '../../../cloud-assembly-schema';
import { FileAssetSource, DockerImageAssetSource } from '../assets';
import { Stack } from '../stack';
/**
 * Build an asset manifest from assets added to a stack
 *
 * This class does not need to be used by app builders; it is only necessary for building Stack Synthesizers.
 */
export declare class AssetManifestBuilder {
    private readonly files;
    private readonly dockerImages;
    /**
     * Add a file asset to the manifest with default settings
     *
     * Derive the region from the stack, use the asset hash as the key, copy the
     * file extension over, and set the prefix.
     */
    defaultAddFileAsset(stack: Stack, asset: FileAssetSource, target: AssetManifestFileDestination, options?: AddFileAssetOptions): cxschema.FileDestination;
    /**
     * Add a docker image asset to the manifest with default settings
     *
     * Derive the region from the stack, use the asset hash as the key, and set the prefix.
     */
    defaultAddDockerImageAsset(stack: Stack, asset: DockerImageAssetSource, target: AssetManifestDockerImageDestination, options?: AddDockerImageAssetOptions): cxschema.DockerImageDestination;
    /**
     * Add a file asset source and destination to the manifest
     *
     * sourceHash should be unique for every source.
     */
    addFileAsset(stack: Stack, sourceHash: string, source: cxschema.FileSource, dest: cxschema.FileDestination, options?: AddFileAssetOptions): cxschema.FileDestination;
    /**
     * Add a docker asset source and destination to the manifest
     *
     * sourceHash should be unique for every source.
     */
    addDockerImageAsset(stack: Stack, sourceHash: string, source: cxschema.DockerImageSource, dest: cxschema.DockerImageDestination, options?: AddDockerImageAssetOptions): cxschema.DockerImageDestination;
    /**
     * Whether there are any assets registered in the manifest
     */
    get hasAssets(): boolean;
    /**
     * Write the manifest to disk, and add it to the synthesis session
     *
     * Return the artifact id, which should be added to the `additionalDependencies`
     * field of the stack artifact.
     */
    emitManifest(stack: Stack, session: ISynthesisSession, options?: cxschema.AssetManifestOptions, dependencies?: string[]): string;
    private manifestEnvName;
}
/**
 * Options for the addFileAsset operation
 */
export interface AddFileAssetOptions {
    /**
     * A display name to associate with the asset
     *
     * @default - No display name
     */
    readonly displayName?: string;
}
/**
 * Options for the addDockerImageAsset operation
 */
export interface AddDockerImageAssetOptions {
    /**
     * A display name to associate with the asset
     *
     * @default - No display name
     */
    readonly displayName?: string;
}
/**
 * The destination for a file asset, when it is given to the AssetManifestBuilder
 */
export interface AssetManifestFileDestination {
    /**
     * Bucket name where the file asset should be written
     */
    readonly bucketName: string;
    /**
     * Prefix to prepend to the asset hash
     *
     * @default ''
     */
    readonly bucketPrefix?: string;
    /**
     * Role to use for uploading
     *
     * @default - current role
     */
    readonly role?: RoleOptions;
}
/**
 * The destination for a docker image asset, when it is given to the AssetManifestBuilder
 */
export interface AssetManifestDockerImageDestination {
    /**
     * Repository name where the docker image asset should be written
     */
    readonly repositoryName: string;
    /**
     * Prefix to add to the asset hash to make the Docker image tag
     *
     * @default ''
     */
    readonly dockerTagPrefix?: string;
    /**
     * Role to use to perform the upload
     *
     * @default - No role
     */
    readonly role?: RoleOptions;
}
/**
 * Options for specifying a role
 */
export interface RoleOptions {
    /**
     * ARN of the role to assume
     */
    readonly assumeRoleArn: string;
    /**
     * External ID to use when assuming the role
     *
     * @default - No external ID
     */
    readonly assumeRoleExternalId?: string;
    /**
     * Additional options to pass to STS when assuming the role for cloudformation deployments.
     *
     * - `RoleArn` should not be used. Use the dedicated `assumeRoleArn` property instead.
     * - `ExternalId` should not be used. Use the dedicated `assumeRoleExternalId` instead.
     * - `TransitiveTagKeys` defaults to use all keys (if any) specified in `Tags`. E.g, all tags are transitive by default.
     *
     * @see https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/STS.html#assumeRole-property
     * @default - No additional options.
     */
    readonly assumeRoleAdditionalOptions?: {
        [key: string]: any;
    };
}
