import { IConstruct } from 'constructs';
import { RemovalPolicy } from './removal-policy';
/**
 * Properties for applying a removal policy
 */
export interface RemovalPolicyProps {
    /**
     * Apply the removal policy only to specific resource types.
     * Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket').
     * @default - apply to all resources
     */
    readonly applyToResourceTypes?: string[];
    /**
     * Exclude specific resource types from the removal policy.
     * Can be a CloudFormation resource type string (e.g., 'AWS::S3::Bucket').
     * @default - no exclusions
     */
    readonly excludeResourceTypes?: string[];
    /**
     * The priority to use when applying this policy.
     *
     * The priority affects only the order in which aspects are applied during synthesis.
     * For RemovalPolicies, the last applied policy will override previous ones.
     *
     * NOTE: Priority does NOT determine which policy "wins" when there are conflicts.
     * The order of application determines the final policy, with later policies
     * overriding earlier ones.
     *
     * @default - AspectPriority.MUTATING
     */
    readonly priority?: number;
}
/**
 * Manages removal policies for all resources within a construct scope,
 * overriding any existing policies by default
 */
export declare class RemovalPolicies {
    private readonly scope;
    /**
     * Returns the removal policies API for the given scope
     * @param scope The scope
     */
    static of(scope: IConstruct): RemovalPolicies;
    private constructor();
    /**
     * Apply a removal policy to all resources within this scope,
     * overriding any existing policies
     *
     * @param policy The removal policy to apply
     * @param props Configuration options
     */
    apply(policy: RemovalPolicy, props?: RemovalPolicyProps): void;
    /**
     * Apply DESTROY removal policy to all resources within this scope
     *
     * @param props Configuration options
     */
    destroy(props?: RemovalPolicyProps): void;
    /**
     * Apply RETAIN removal policy to all resources within this scope
     *
     * @param props Configuration options
     */
    retain(props?: RemovalPolicyProps): void;
    /**
     * Apply SNAPSHOT removal policy to all resources within this scope
     *
     * @param props Configuration options
     */
    snapshot(props?: RemovalPolicyProps): void;
    /**
     * Apply RETAIN_ON_UPDATE_OR_DELETE removal policy to all resources within this scope
     *
     * @param props Configuration options
     */
    retainOnUpdateOrDelete(props?: RemovalPolicyProps): void;
}
/**
 * Manages removal policies for resources without existing policies within a construct scope
 */
export declare class MissingRemovalPolicies {
    private readonly scope;
    /**
     * Returns the missing removal policies API for the given scope
     * @param scope The scope
     */
    static of(scope: IConstruct): MissingRemovalPolicies;
    private constructor();
    /**
     * Apply a removal policy only to resources without existing policies within this scope
     *
     * @param policy The removal policy to apply
     * @param props Configuration options
     */
    apply(policy: RemovalPolicy, props?: RemovalPolicyProps): void;
    /**
     * Apply DESTROY removal policy only to resources without existing policies within this scope
     *
     * @param props Configuration options
     */
    destroy(props?: RemovalPolicyProps): void;
    /**
     * Apply RETAIN removal policy only to resources without existing policies within this scope
     *
     * @param props Configuration options
     */
    retain(props?: RemovalPolicyProps): void;
    /**
     * Apply SNAPSHOT removal policy only to resources without existing policies within this scope
     *
     * @param props Configuration options
     */
    snapshot(props?: RemovalPolicyProps): void;
    /**
     * Apply RETAIN_ON_UPDATE_OR_DELETE removal policy only to resources without existing policies within this scope
     *
     * @param props Configuration options
     */
    retainOnUpdateOrDelete(props?: RemovalPolicyProps): void;
}
