import { IConstruct } from 'constructs';
/**
 * Represents an Aspect
 */
export interface IAspect {
    /**
     * All aspects can visit an IConstruct
     */
    visit(node: IConstruct): void;
}
/**
 * Default Priority values for Aspects.
 */
export declare class AspectPriority {
    /**
     * Suggested priority for Aspects that mutate the construct tree.
     */
    static readonly MUTATING: number;
    /**
     * Suggested priority for Aspects that only read the construct tree.
     */
    static readonly READONLY: number;
    /**
     * Default priority for Aspects that are applied without a priority.
     */
    static readonly DEFAULT: number;
}
/**
 * Options when Applying an Aspect.
 */
export interface AspectOptions {
    /**
     * The priority value to apply on an Aspect.
     * Priority must be a non-negative integer.
     *
     * Aspects that have same priority value are not guaranteed to be
     * executed in a consistent order.
     *
     * @default AspectPriority.DEFAULT
     */
    readonly priority?: number;
}
/**
 * Aspects can be applied to CDK tree scopes and can operate on the tree before
 * synthesis.
 */
export declare class Aspects {
    /**
     * Returns the `Aspects` object associated with a construct scope.
     * @param scope The scope for which these aspects will apply.
     */
    static of(scope: IConstruct): Aspects;
    private readonly _scope;
    private readonly _appliedAspects;
    private constructor();
    /**
     * Adds an aspect to apply this scope before synthesis.
     * @param aspect The aspect to add.
     * @param options Options to apply on this aspect.
     */
    add(aspect: IAspect, options?: AspectOptions): void;
    /**
     * The list of aspects which were directly applied on this scope.
     */
    get all(): IAspect[];
    /**
     * The list of aspects with priority which were directly applied on this scope.
     *
     * Also returns inherited Aspects of this node.
     */
    get applied(): AspectApplication[];
}
/**
 * Return the aspect revision of the tree that the given construct is part of
 *
 * The revision number is changed every time an aspect is added to the tree.
 *
 * Instead of returning the version, returns a function that returns the
 * version: we can cache the root lookup inside the function; this saves
 * crawling the tree on every read, which is frequent.
 *
 * @internal
 */
export declare function _aspectTreeRevisionReader(construct: IConstruct): () => number;
/**
 * Object respresenting an Aspect application. Stores the Aspect, the pointer to the construct it was applied
 * to, and the priority value of that Aspect.
 */
export declare class AspectApplication {
    /**
     * The construct that the Aspect was applied to.
     */
    readonly construct: IConstruct;
    /**
     * The Aspect that was applied.
     */
    readonly aspect: IAspect;
    /**
     * The priority value of this Aspect. Must be non-negative integer.
     */
    private _priority;
    /**
     * Initializes AspectApplication object
     */
    constructor(construct: IConstruct, aspect: IAspect, priority: number);
    /**
     * Gets the priority value.
     */
    get priority(): number;
    /**
     * Sets the priority value.
     */
    set priority(priority: number);
}
