import { Construct } from 'constructs';
import { ScheduleExpression } from './schedule-expression';
import { IScheduleGroup } from './schedule-group';
import { IScheduleTarget } from './target';
import * as cloudwatch from '../../aws-cloudwatch';
import * as kms from '../../aws-kms';
import { Duration, IResource, Resource } from '../../core';
/**
 * Interface representing a created or an imported `Schedule`.
 */
export interface ISchedule extends IResource {
    /**
     * The arn of the schedule.
     * @attribute
     */
    readonly scheduleArn: string;
    /**
     * The name of the schedule.
     * @attribute
     */
    readonly scheduleName: string;
    /**
     * The schedule group associated with this schedule.
     */
    readonly scheduleGroup?: IScheduleGroup;
}
/**
 * A time window during which EventBridge Scheduler invokes the schedule.
 */
export declare class TimeWindow {
    /**
     * TimeWindow is disabled.
     */
    static off(): TimeWindow;
    /**
     * TimeWindow is enabled.
     */
    static flexible(maxWindow: Duration): TimeWindow;
    /**
     * Determines whether the schedule is invoked within a flexible time window.
     */
    readonly mode: 'OFF' | 'FLEXIBLE';
    /**
     * The maximum time window during which the schedule can be invoked.
     *
     * Must be between 1 to 1440 minutes.
     *
     * @default - no value
     */
    readonly maxWindow?: Duration;
    private constructor();
}
/**
 * Construction properties for `Schedule`.
 */
export interface ScheduleProps {
    /**
     * The expression that defines when the schedule runs. Can be either a `at`, `rate`
     * or `cron` expression.
     */
    readonly schedule: ScheduleExpression;
    /**
     * The schedule's target details.
     */
    readonly target: IScheduleTarget;
    /**
     * The name of the schedule.
     *
     * Up to 64 letters (uppercase and lowercase), numbers, hyphens, underscores and dots are allowed.
     *
     * @default - A unique name will be generated
     */
    readonly scheduleName?: string;
    /**
     * The description you specify for the schedule.
     *
     * @default - no value
     */
    readonly description?: string;
    /**
     * The schedule's group.
     *
     * @default - By default a schedule will be associated with the `default` group.
     */
    readonly scheduleGroup?: IScheduleGroup;
    /**
     * Indicates whether the schedule is enabled.
     *
     * @default true
     */
    readonly enabled?: boolean;
    /**
     * The customer managed KMS key that EventBridge Scheduler will use to encrypt and decrypt your data.
     *
     * @default - All events in Scheduler are encrypted with a key that AWS owns and manages.
     */
    readonly key?: kms.IKey;
    /**
     * A time window during which EventBridge Scheduler invokes the schedule.
     *
     * @see https://docs.aws.amazon.com/scheduler/latest/UserGuide/managing-schedule-flexible-time-windows.html
     *
     * @default TimeWindow.off()
     */
    readonly timeWindow?: TimeWindow;
    /**
     * The date, in UTC, after which the schedule can begin invoking its target.
     * EventBridge Scheduler ignores start for one-time schedules.
     *
     * @default - no value
     */
    readonly start?: Date;
    /**
     * The date, in UTC, before which the schedule can invoke its target.
     * EventBridge Scheduler ignores end for one-time schedules.
     *
     * @default - no value
     */
    readonly end?: Date;
}
/**
 * An EventBridge Schedule
 */
export declare class Schedule extends Resource implements ISchedule {
    /**
     * Return the given named metric for all schedules.
     *
     * @default - sum over 5 minutes
     */
    static metricAll(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of invocations that were throttled across all schedules.
     *
     * @see https://docs.aws.amazon.com/scheduler/latest/UserGuide/scheduler-quotas.html
     *
     * @default - sum over 5 minutes
     */
    static metricAllThrottled(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for all invocation attempts across all schedules.
     *
     * @default - sum over 5 minutes
     */
    static metricAllAttempts(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Emitted when the target returns an exception after EventBridge Scheduler calls the target API across all schedules.
     *
     * @default - sum over 5 minutes
     */
    static metricAllErrors(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for invocation failures due to API throttling by the target across all schedules.
     *
     * @default - sum over 5 minutes
     */
    static metricAllTargetThrottled(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for dropped invocations when EventBridge Scheduler stops attempting to invoke the target after a schedule's retry policy has been exhausted.
     * Metric is calculated for all schedules.
     *
     * @default - sum over 5 minutes
     */
    static metricAllDropped(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for invocations delivered to the DLQ across all schedules.
     *
     * @default - sum over 5 minutes
     */
    static metricAllSentToDLQ(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for failed invocations that also failed to deliver to DLQ across all schedules.
     *
     * @default - sum over 5 minutes
     */
    static metricAllFailedToBeSentToDLQ(errorCode?: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for delivery of failed invocations to DLQ when the payload of the event sent to the DLQ exceeds the maximum size allowed by Amazon SQS.
     * Metric is calculated for all schedules.
     *
     * @default - sum over 5 minutes
     */
    static metricAllSentToDLQTruncated(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Import an existing schedule using the ARN.
     */
    static fromScheduleArn(scope: Construct, id: string, scheduleArn: string): ISchedule;
    /**
     * The schedule group associated with this schedule.
     */
    readonly scheduleGroup?: IScheduleGroup;
    /**
     * The arn of the schedule.
     */
    readonly scheduleArn: string;
    /**
     * The name of the schedule.
     */
    readonly scheduleName: string;
    /**
     * The customer managed KMS key that EventBridge Scheduler will use to encrypt and decrypt your data.
     */
    readonly key?: kms.IKey;
    /**
     * A `RetryPolicy` object that includes information about the retry policy settings.
     */
    private readonly retryPolicy?;
    constructor(scope: Construct, id: string, props: ScheduleProps);
    private renderRetryPolicy;
    private validateTimeFrame;
}
