import { ScheduleTargetBase, ScheduleTargetBaseProps } from './target';
import * as ec2 from '../../aws-ec2';
import * as ecs from '../../aws-ecs';
import { IRole } from '../../aws-iam';
import { ISchedule, IScheduleTarget, ScheduleTargetConfig } from '../../aws-scheduler';
/**
 * Metadata that you apply to a resource to help categorize and organize the resource. Each tag consists of a key and an optional value, both of which you define.
 */
export interface Tag {
    /**
     * Key is the name of the tag
     */
    readonly key: string;
    /**
     * Value is the metadata contents of the tag
     */
    readonly value: string;
}
/**
 * Parameters for scheduling ECS Run Task (common to EC2 and Fargate launch types).
 */
export interface EcsRunTaskBaseProps extends ScheduleTargetBaseProps {
    /**
     * The task definition to use for scheduled tasks.
     *
     * Note: this must be TaskDefinition, and not ITaskDefinition,
     * as it requires properties that are not known for imported task definitions
     * If you want to run a RunTask with an imported task definition,
     * consider using a Universal target.
     */
    readonly taskDefinition: ecs.TaskDefinition;
    /**
     * The capacity provider strategy to use for the task.
     *
     * @default - No capacity provider strategy
     */
    readonly capacityProviderStrategies?: ecs.CapacityProviderStrategy[];
    /**
     * The subnets associated with the task. These subnets must all be in the same VPC.
     * The task will be launched in these subnets.
     *
     * @default - all private subnets of the VPC are selected.
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * The security groups associated with the task. These security groups must all be in the same VPC.
     * Controls inbound and outbound network access for the task.
     *
     * @default - The security group for the VPC is used.
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * Specifies whether to enable Amazon ECS managed tags for the task.
     * @default - false
     */
    readonly enableEcsManagedTags?: boolean;
    /**
     * Whether to enable execute command functionality for the containers in this task.
     * If true, this enables execute command functionality on all containers in the task.
     *
     * @default - false
     */
    readonly enableExecuteCommand?: boolean;
    /**
     * Specifies an ECS task group for the task.
     *
     * @default - No group
     */
    readonly group?: string;
    /**
     * Specifies whether to propagate the tags from the task definition to the task.
     * If no value is specified, the tags are not propagated.
     *
     * @default - No tag propagation
     */
    readonly propagateTags?: boolean;
    /**
     * The reference ID to use for the task.
     *
     * @default - No reference ID.
     */
    readonly referenceId?: string;
    /**
     * The metadata that you apply to the task to help you categorize and organize them.
     * Each tag consists of a key and an optional value, both of which you define.
     *
     * @default - No tags
     */
    readonly tags?: Tag[];
    /**
     * The number of tasks to create based on TaskDefinition.
     *
     * @default 1
     */
    readonly taskCount?: number;
}
/**
 * Properties for scheduling an ECS Fargate Task.
 */
export interface FargateTaskProps extends EcsRunTaskBaseProps {
    /**
     * Specifies whether the task's elastic network interface receives a public IP address.
     * If true, the task will receive a public IP address and be accessible from the internet.
     * Should only be set to true when using public subnets.
     *
     * @default - true if the subnet type is PUBLIC, otherwise false
     */
    readonly assignPublicIp?: boolean;
    /**
     * Specifies the platform version for the task.
     * Specify only the numeric portion of the platform version, such as 1.1.0.
     * Platform versions determine the underlying runtime environment for the task.
     *
     * @default - LATEST
     */
    readonly platformVersion?: ecs.FargatePlatformVersion;
}
/**
 * Properties for scheduling an ECS Task on EC2.
 */
export interface Ec2TaskProps extends EcsRunTaskBaseProps {
    /**
     * The rules that must be met in order to place a task on a container instance.
     *
     * @default - No placement constraints.
     */
    readonly placementConstraints?: ecs.PlacementConstraint[];
    /**
     * The algorithm for selecting container instances for task placement.
     *
     * @default - No placement strategies.
     */
    readonly placementStrategies?: ecs.PlacementStrategy[];
}
/**
 * Schedule an ECS Task using AWS EventBridge Scheduler.
 */
export declare abstract class EcsRunTask extends ScheduleTargetBase implements IScheduleTarget {
    protected readonly cluster: ecs.ICluster;
    protected readonly props: EcsRunTaskBaseProps;
    constructor(cluster: ecs.ICluster, props: EcsRunTaskBaseProps);
    protected addTargetActionToRole(role: IRole): void;
    protected bindBaseTargetConfig(_schedule: ISchedule): ScheduleTargetConfig;
}
/**
 * Schedule an ECS Task on Fargate using AWS EventBridge Scheduler.
 */
export declare class EcsRunFargateTask extends EcsRunTask {
    private readonly subnetSelection?;
    private readonly assignPublicIp?;
    private readonly platformVersion?;
    private readonly capacityProviderStrategies?;
    constructor(cluster: ecs.ICluster, props: FargateTaskProps);
    protected bindBaseTargetConfig(_schedule: ISchedule): ScheduleTargetConfig;
}
/**
 * Schedule an ECS Task on EC2 using AWS EventBridge Scheduler.
 */
export declare class EcsRunEc2Task extends EcsRunTask {
    private readonly capacityProviderStrategies?;
    private readonly placementConstraints?;
    private readonly placementStrategies?;
    constructor(cluster: ecs.ICluster, props: Ec2TaskProps);
    protected bindBaseTargetConfig(_schedule: ISchedule): ScheduleTargetConfig;
}
