import * as constructs from 'constructs';
import { StreamEventSource, StreamEventSourceProps } from './stream';
import * as iam from '../../aws-iam';
import * as kinesis from '../../aws-kinesis';
import * as lambda from '../../aws-lambda';
export interface KinesisEventSourceProps extends StreamEventSourceProps {
    /**
     * The time from which to start reading, in Unix time seconds.
     *
     * @default - no timestamp
     */
    readonly startingPositionTimestamp?: number;
}
/**
 * Props for use with {@link KinesisEventSourceBase}
 */
interface KinesisSource {
    readonly node: constructs.Node;
    readonly sourceArn: string;
    readonly eventSourceName: string;
    grantRead(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Base class for {@link KinesisEventSource} and {@link KinesisConsumerEventSource}
 */
declare abstract class KinesisEventSourceBase extends StreamEventSource {
    readonly source: KinesisSource;
    private _eventSourceMappingId?;
    private _eventSourceMappingArn?;
    private startingPositionTimestamp?;
    constructor(source: KinesisSource, props: KinesisEventSourceProps);
    bind(target: lambda.IFunction): void;
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId(): string;
    /**
     * The ARN for this EventSourceMapping
     */
    get eventSourceMappingArn(): string;
}
/**
 * Use an Amazon Kinesis stream as an event source for AWS Lambda.
 */
export declare class KinesisEventSource extends KinesisEventSourceBase {
    readonly stream: kinesis.IStream;
    constructor(stream: kinesis.IStream, props: KinesisEventSourceProps);
}
/**
 * Use an Amazon Kinesis stream consumer as an event source for AWS Lambda.
 */
export declare class KinesisConsumerEventSource extends KinesisEventSourceBase {
    readonly streamConsumer: kinesis.IStreamConsumer;
    constructor(streamConsumer: kinesis.IStreamConsumer, props: KinesisEventSourceProps);
}
export {};
