import { Construct } from 'constructs';
import { IDestination } from './destination';
import { StreamEncryption } from './encryption';
import { ISource } from './source';
import * as cloudwatch from '../../aws-cloudwatch';
import * as ec2 from '../../aws-ec2';
import * as iam from '../../aws-iam';
import * as cdk from '../../core';
/**
 * Represents an Amazon Data Firehose delivery stream.
 */
export interface IDeliveryStream extends cdk.IResource, iam.IGrantable, ec2.IConnectable {
    /**
     * The ARN of the delivery stream.
     *
     * @attribute
     */
    readonly deliveryStreamArn: string;
    /**
     * The name of the delivery stream.
     *
     * @attribute
     */
    readonly deliveryStreamName: string;
    /**
     * Grant the `grantee` identity permissions to perform `actions`.
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the `grantee` identity permissions to perform `firehose:PutRecord` and `firehose:PutRecordBatch` actions on this delivery stream.
     */
    grantPutRecords(grantee: iam.IGrantable): iam.Grant;
    /**
     * Return the given named metric for this delivery stream.
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of bytes ingested successfully into the delivery stream over the specified time period after throttling.
     *
     * By default, this metric will be calculated as an average over a period of 5 minutes.
     */
    metricIncomingBytes(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of records ingested successfully into the delivery stream over the specified time period after throttling.
     *
     * By default, this metric will be calculated as an average over a period of 5 minutes.
     */
    metricIncomingRecords(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of bytes delivered to Amazon S3 for backup over the specified time period.
     *
     * By default, this metric will be calculated as an average over a period of 5 minutes.
     */
    metricBackupToS3Bytes(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the age (from getting into Amazon Data Firehose to now) of the oldest record in Amazon Data Firehose.
     *
     * Any record older than this age has been delivered to the Amazon S3 bucket for backup.
     *
     * By default, this metric will be calculated as an average over a period of 5 minutes.
     */
    metricBackupToS3DataFreshness(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of records delivered to Amazon S3 for backup over the specified time period.
     *
     * By default, this metric will be calculated as an average over a period of 5 minutes.
     */
    metricBackupToS3Records(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * Base class for new and imported Amazon Data Firehose delivery streams.
 */
declare abstract class DeliveryStreamBase extends cdk.Resource implements IDeliveryStream {
    abstract readonly deliveryStreamName: string;
    abstract readonly deliveryStreamArn: string;
    abstract readonly grantPrincipal: iam.IPrincipal;
    /**
     * Network connections between Amazon Data Firehose and other resources, i.e. Redshift cluster.
     */
    readonly connections: ec2.Connections;
    constructor(scope: Construct, id: string, props?: cdk.ResourceProps);
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    grantPutRecords(grantee: iam.IGrantable): iam.Grant;
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    metricIncomingBytes(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    metricIncomingRecords(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    metricBackupToS3Bytes(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    metricBackupToS3DataFreshness(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    metricBackupToS3Records(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    private cannedMetric;
}
/**
 * Options for server-side encryption of a delivery stream.
 */
export declare enum StreamEncryptionType {
    /**
     * Data in the stream is stored unencrypted.
     */
    UNENCRYPTED = 0,
    /**
     * Data in the stream is stored encrypted by a KMS key managed by the customer.
     */
    CUSTOMER_MANAGED = 1,
    /**
     * Data in the stream is stored encrypted by a KMS key owned by AWS and managed for use in multiple AWS accounts.
     */
    AWS_OWNED = 2
}
/**
 * Properties for a new delivery stream.
 */
export interface DeliveryStreamProps {
    /**
     * The destination that this delivery stream will deliver data to.
     */
    readonly destination: IDestination;
    /**
     * A name for the delivery stream.
     *
     * @default - a name is generated by CloudFormation.
     */
    readonly deliveryStreamName?: string;
    /**
     * The Kinesis data stream to use as a source for this delivery stream.
     *
     * @default - data must be written to the delivery stream via a direct put.
     */
    readonly source?: ISource;
    /**
     * The IAM role associated with this delivery stream.
     *
     * Assumed by Amazon Data Firehose to read from sources and encrypt data server-side.
     *
     * @default - a role will be created with default permissions.
     */
    readonly role?: iam.IRole;
    /**
     * Indicates the type of customer master key (CMK) to use for server-side encryption, if any.
     *
     * @default StreamEncryption.unencrypted()
     */
    readonly encryption?: StreamEncryption;
}
/**
 * A full specification of a delivery stream that can be used to import it fluently into the CDK application.
 */
export interface DeliveryStreamAttributes {
    /**
     * The ARN of the delivery stream.
     *
     * At least one of deliveryStreamArn and deliveryStreamName must be provided.
     *
     * @default - derived from `deliveryStreamName`.
     */
    readonly deliveryStreamArn?: string;
    /**
     * The name of the delivery stream
     *
     * At least one of deliveryStreamName and deliveryStreamArn  must be provided.
     *
     * @default - derived from `deliveryStreamArn`.
     */
    readonly deliveryStreamName?: string;
    /**
     * The IAM role associated with this delivery stream.
     *
     * Assumed by Amazon Data Firehose to read from sources and encrypt data server-side.
     *
     * @default - the imported stream cannot be granted access to other resources as an `iam.IGrantable`.
     */
    readonly role?: iam.IRole;
}
/**
 * Create a Amazon Data Firehose delivery stream
 *
 * @resource AWS::KinesisFirehose::DeliveryStream
 */
export declare class DeliveryStream extends DeliveryStreamBase {
    /**
     * Import an existing delivery stream from its name.
     */
    static fromDeliveryStreamName(scope: Construct, id: string, deliveryStreamName: string): IDeliveryStream;
    /**
     * Import an existing delivery stream from its ARN.
     */
    static fromDeliveryStreamArn(scope: Construct, id: string, deliveryStreamArn: string): IDeliveryStream;
    /**
     * Import an existing delivery stream from its attributes.
     */
    static fromDeliveryStreamAttributes(scope: Construct, id: string, attrs: DeliveryStreamAttributes): IDeliveryStream;
    readonly deliveryStreamName: string;
    readonly deliveryStreamArn: string;
    private _role?;
    get grantPrincipal(): iam.IPrincipal;
    constructor(scope: Construct, id: string, props: DeliveryStreamProps);
}
export {};
