import { Construct } from 'constructs';
import { IRole } from './role';
import { Resource, IResource } from '../../core';
/**
 * Represents an IAM Instance Profile
 */
export interface IInstanceProfile extends IResource {
    /**
     * The InstanceProfile's name.
     * @attribute
     */
    readonly instanceProfileName: string;
    /**
     * The InstanceProfile's ARN.
     * @attribute
     */
    readonly instanceProfileArn: string;
    /**
     * The role associated with the InstanceProfile.
     */
    readonly role?: IRole;
}
/**
 * Properties of an Instance Profile
 */
export interface InstanceProfileProps {
    /**
     * An IAM role to associate with the instance profile that is used by EC2 instances.
     *
     * The role must be assumable by the service principal `ec2.amazonaws.com`:
     *
     * @example
     * const role = new iam.Role(this, 'MyRole', {
     *   assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
     * });
     *
     * @default - a role will be automatically created, it can be accessed via the `role` property
     */
    readonly role?: IRole;
    /**
     * The name of the InstanceProfile to create.
     *
     * @default - generated by CloudFormation
     */
    readonly instanceProfileName?: string;
    /**
     * The path to the InstanceProfile.
     *
     * @default /
     */
    readonly path?: string;
}
/**
 * Attributes of an Instance Profile
 */
export interface InstanceProfileAttributes {
    /**
     * The ARN of the InstanceProfile.
     *
     * Format: arn:<partition>:iam::<account-id>:instance-profile/<instance-profile-name-with-path>
     */
    readonly instanceProfileArn: string;
    /**
     * The role associated with the InstanceProfile.
     *
     * @default - no role
     */
    readonly role?: IRole;
}
/**
 * Base class for an Instance Profile
 */
declare abstract class InstanceProfileBase extends Resource implements IInstanceProfile {
    abstract readonly instanceProfileName: string;
    abstract readonly instanceProfileArn: string;
    /**
     * The role associated with the InstanceProfile.
     * @internal
     */
    protected _role?: IRole;
    /**
     * Returns the role associated with this InstanceProfile.
     */
    get role(): IRole | undefined;
}
/**
 * IAM Instance Profile
 */
export declare class InstanceProfile extends InstanceProfileBase {
    /**
     * Import an existing InstanceProfile from an InstanceProfile name.
     *
     * @param scope construct scope
     * @param id construct id
     * @param instanceProfileName the name of the existing InstanceProfile to import
     */
    static fromInstanceProfileName(scope: Construct, id: string, instanceProfileName: string): IInstanceProfile;
    /**
     * Import an existing InstanceProfile from an InstanceProfile ARN.
     *
     * If the ARN comes from a Token, the InstanceProfile cannot have a path; if so, any attempt
     * to reference its instanceProfileName will fail.
     *
     * @param scope construct scope
     * @param id construct id
     * @param instanceProfileArn the ARN of the exiting InstanceProfile to import
     */
    static fromInstanceProfileArn(scope: Construct, id: string, instanceProfileArn: string): IInstanceProfile;
    /**
     * Import an existing InstanceProfile from given InstanceProfile attributes.
     *
     * If the ARN comes from a Token, the InstanceProfile cannot have a path; if so, any attempt
     * to reference its instanceProfileName will fail.
     *
     * @param scope construct scope
     * @param id construct id
     * @param attrs the attributes of the InstanceProfile to import
     */
    static fromInstanceProfileAttributes(scope: Construct, id: string, attrs: InstanceProfileAttributes): IInstanceProfile;
    /**
     * Returns the name of this InstanceProfile.
     */
    readonly instanceProfileName: string;
    /**
     * Returns the ARN of this InstanceProfile.
     */
    readonly instanceProfileArn: string;
    constructor(scope: Construct, id: string, props?: InstanceProfileProps);
}
export {};
