import { Construct } from 'constructs';
import * as ec2 from '../../../aws-ec2';
import * as iam from '../../../aws-iam';
import * as s3 from '../../../aws-s3';
import * as cxschema from '../../../cloud-assembly-schema';
import { IResource, Resource } from '../../../core';
import * as cxapi from '../../../cx-api';
/**
 * Shared properties of both Application and Network Load Balancers
 */
export interface BaseLoadBalancerProps {
    /**
     * Name of the load balancer
     *
     * @default - Automatically generated name.
     */
    readonly loadBalancerName?: string;
    /**
     * The VPC network to place the load balancer in
     */
    readonly vpc: ec2.IVpc;
    /**
     * Whether the load balancer has an internet-routable address
     *
     * @default false
     */
    readonly internetFacing?: boolean;
    /**
     * Which subnets place the load balancer in
     *
     * @default - the Vpc default strategy.
     *
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * Indicates whether deletion protection is enabled.
     *
     * @default false
     */
    readonly deletionProtection?: boolean;
    /**
     * Indicates whether cross-zone load balancing is enabled.
     *
     * @default - false for Network Load Balancers and true for Application Load Balancers.
     * This can not be `false` for Application Load Balancers.
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-elasticloadbalancingv2-loadbalancer-loadbalancerattribute.html
     */
    readonly crossZoneEnabled?: boolean;
    /**
     * Indicates whether the load balancer blocks traffic through the Internet Gateway (IGW).
     *
     * @default - false for internet-facing load balancers and true for internal load balancers
     */
    readonly denyAllIgwTraffic?: boolean;
    /**
     * The minimum capacity (LCU) for a load balancer.
     *
     * @default undefined - ELB default is 0 LCU
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/capacity-unit-reservation.html
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/network/capacity-unit-reservation.html
     * @see https://exampleloadbalancer.com/ondemand_capacity_reservation_calculator.html
     */
    readonly minimumCapacityUnit?: number;
}
export interface ILoadBalancerV2 extends IResource {
    /**
     * The canonical hosted zone ID of this load balancer
     *
     * Example value: `Z2P70J7EXAMPLE`
     *
     * @attribute
     */
    readonly loadBalancerCanonicalHostedZoneId: string;
    /**
     * The DNS name of this load balancer
     *
     * Example value: `my-load-balancer-424835706.us-west-2.elb.amazonaws.com`
     *
     * @attribute
     */
    readonly loadBalancerDnsName: string;
}
/**
 * Options for looking up load balancers
 */
export interface BaseLoadBalancerLookupOptions {
    /**
     * Find by load balancer's ARN
     * @default - does not search by load balancer arn
     */
    readonly loadBalancerArn?: string;
    /**
     * Match load balancer tags.
     * @default - does not match load balancers by tags
     */
    readonly loadBalancerTags?: Record<string, string>;
}
/**
 * Options for query context provider
 * @internal
 */
export interface LoadBalancerQueryContextProviderOptions {
    /**
     * User's lookup options
     */
    readonly userOptions: BaseLoadBalancerLookupOptions;
    /**
     * Type of load balancer
     */
    readonly loadBalancerType: cxschema.LoadBalancerType;
}
/**
 * Base class for both Application and Network Load Balancers
 */
export declare abstract class BaseLoadBalancer extends Resource {
    /**
     * Queries the load balancer context provider for load balancer info.
     * @internal
     */
    protected static _queryContextProvider(scope: Construct, options: LoadBalancerQueryContextProviderOptions): cxapi.LoadBalancerContextResponse;
    /**
     * The canonical hosted zone ID of this load balancer
     *
     * Example value: `Z2P70J7EXAMPLE`
     *
     * @attribute
     */
    readonly loadBalancerCanonicalHostedZoneId: string;
    /**
     * The DNS name of this load balancer
     *
     * Example value: `my-load-balancer-424835706.us-west-2.elb.amazonaws.com`
     *
     * @attribute
     */
    readonly loadBalancerDnsName: string;
    /**
     * The full name of this load balancer
     *
     * Example value: `app/my-load-balancer/50dc6c495c0c9188`
     *
     * @attribute
     */
    readonly loadBalancerFullName: string;
    /**
     * The name of this load balancer
     *
     * Example value: `my-load-balancer`
     *
     * @attribute
     */
    readonly loadBalancerName: string;
    /**
     * The ARN of this load balancer
     *
     * Example value: `arn:aws:elasticloadbalancing:us-west-2:123456789012:loadbalancer/app/my-internal-load-balancer/50dc6c495c0c9188`
     *
     * @attribute
     */
    readonly loadBalancerArn: string;
    /**
     * @attribute
     */
    readonly loadBalancerSecurityGroups: string[];
    /**
     * The VPC this load balancer has been created in.
     *
     * This property is always defined (not `null` or `undefined`) for sub-classes of `BaseLoadBalancer`.
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Attributes set on this load balancer
     */
    private readonly attributes;
    constructor(scope: Construct, id: string, baseProps: BaseLoadBalancerProps, additionalProps: any);
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    logAccessLogs(bucket: s3.IBucket, prefix?: string): void;
    /**
     * Set a non-standard attribute on the load balancer
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/application-load-balancers.html#load-balancer-attributes
     */
    setAttribute(key: string, value: string | undefined): void;
    /**
     * Remove an attribute from the load balancer
     */
    removeAttribute(key: string): void;
    protected resourcePolicyPrincipal(): iam.IPrincipal;
    protected validateLoadBalancer(): string[];
}
