import { Construct } from 'constructs';
import { IUserPoolAuthenticationProvider } from './identitypool-user-pool-authentication-provider';
import { IUserPool, IUserPoolClient } from '../../aws-cognito';
import { IOpenIdConnectProvider, ISamlProvider, IRole } from '../../aws-iam';
import { Resource, IResource } from '../../core';
/**
 * Represents a Cognito Identity Pool
 */
export interface IIdentityPool extends IResource {
    /**
     * The ID of the Identity Pool in the format REGION:GUID
     * @attribute
     */
    readonly identityPoolId: string;
    /**
     * The ARN of the Identity Pool
     * @attribute
     */
    readonly identityPoolArn: string;
    /**
     * Name of the Identity Pool
     * @attribute
     */
    readonly identityPoolName: string;
}
/**
 * Props for the Identity Pool construct
 */
export interface IdentityPoolProps {
    /**
     * The name of the Identity Pool
     * @default - Automatically generated name by CloudFormation at deploy time
     */
    readonly identityPoolName?: string;
    /**
     * The default Role to be assumed by authenticated users
     * @default - A default authenticated Role will be added
     */
    readonly authenticatedRole?: IRole;
    /**
     * The default Role to be assumed by unauthenticated users
     * @default - A default unauthenticated Role will be added
     */
    readonly unauthenticatedRole?: IRole;
    /**
     * Whether the Identity Pool supports unauthenticated logins
     * @default - false
     */
    readonly allowUnauthenticatedIdentities?: boolean;
    /**
     * Rules for mapping roles to users
     * @default - no role mappings
     */
    readonly roleMappings?: IdentityPoolRoleMapping[];
    /**
     * Enables the Basic (Classic) authentication flow
     * @default - Classic Flow not allowed
     */
    readonly allowClassicFlow?: boolean;
    /**
     * Authentication Providers for using in Identity Pool
     * @default - No Authentication Providers passed directly to Identity Pool
     */
    readonly authenticationProviders?: IdentityPoolAuthenticationProviders;
}
/**
 * Types of Identity Pool Login Providers
 */
export declare enum IdentityPoolProviderType {
    /** Facebook provider type */
    FACEBOOK = "Facebook",
    /** Google provider type */
    GOOGLE = "Google",
    /** Amazon provider type */
    AMAZON = "Amazon",
    /** Apple provider type */
    APPLE = "Apple",
    /** Twitter provider type */
    TWITTER = "Twitter",
    /** Open Id provider type */
    OPEN_ID = "OpenId",
    /** Saml provider type */
    SAML = "Saml",
    /** User Pool provider type */
    USER_POOL = "UserPool",
    /** Custom provider type */
    CUSTOM = "Custom"
}
/**
 * Keys for Login Providers - each correspond to the client IDs of their respective federation Identity Providers
 */
export declare class IdentityPoolProviderUrl {
    /**
     * The type of Identity Pool Provider
     */
    readonly type: IdentityPoolProviderType;
    /**
     * The value of the Identity Pool Provider
     */
    readonly value: string;
    /** Facebook Provider url */
    static readonly FACEBOOK: IdentityPoolProviderUrl;
    /** Google Provider url */
    static readonly GOOGLE: IdentityPoolProviderUrl;
    /** Amazon Provider url */
    static readonly AMAZON: IdentityPoolProviderUrl;
    /** Apple Provider url */
    static readonly APPLE: IdentityPoolProviderUrl;
    /** Twitter Provider url */
    static readonly TWITTER: IdentityPoolProviderUrl;
    /** OpenId Provider url */
    static openId(url: string): IdentityPoolProviderUrl;
    /** Saml Provider url */
    static saml(url: string): IdentityPoolProviderUrl;
    /** User Pool Provider Url */
    static userPool(userPool: IUserPool, userPoolClient: IUserPoolClient): IdentityPoolProviderUrl;
    /** Custom Provider url */
    static custom(url: string): IdentityPoolProviderUrl;
    constructor(
    /**
     * The type of Identity Pool Provider
     */
    type: IdentityPoolProviderType, 
    /**
     * The value of the Identity Pool Provider
     */
    value: string);
}
/**
 * Login Provider for identity federation using Amazon credentials
 */
export interface IdentityPoolAmazonLoginProvider {
    /**
     * App ID for Amazon identity federation
     */
    readonly appId: string;
}
/**
 * Login Provider for identity federation using Facebook credentials
 */
export interface IdentityPoolFacebookLoginProvider {
    /**
     * App ID for Facebook identity federation
     */
    readonly appId: string;
}
/**
 * Login Provider for identity federation using Apple credentials
 */
export interface IdentityPoolAppleLoginProvider {
    /**
     * Services ID for Apple identity federation
     */
    readonly servicesId: string;
}
/**
 * Login Provider for identity federation using Google credentials
 */
export interface IdentityPoolGoogleLoginProvider {
    /**
     * Client ID for Google identity federation
     */
    readonly clientId: string;
}
/**
 * Login Provider for identity federation using Twitter credentials
 */
export interface IdentityPoolTwitterLoginProvider {
    /**
     * Consumer key for Twitter identity federation
     */
    readonly consumerKey: string;
    /**
     * Consumer secret for identity federation
     */
    readonly consumerSecret: string;
}
/**
 * External Authentication Providers for usage in Identity Pool.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/external-identity-providers.html
 */
export interface IdentityPoolAuthenticationProviders {
    /**
     * The Facebook Authentication Provider associated with this Identity Pool
     * @default - No Facebook Authentication Provider used without OpenIdConnect or a User Pool
     */
    readonly facebook?: IdentityPoolFacebookLoginProvider;
    /**
     * The Google Authentication Provider associated with this Identity Pool
     * @default - No Google Authentication Provider used without OpenIdConnect or a User Pool
     */
    readonly google?: IdentityPoolGoogleLoginProvider;
    /**
     * The Amazon Authentication Provider associated with this Identity Pool
     * @default -  No Amazon Authentication Provider used without OpenIdConnect or a User Pool
     */
    readonly amazon?: IdentityPoolAmazonLoginProvider;
    /**
     * The Apple Authentication Provider associated with this Identity Pool
     * @default - No Apple Authentication Provider used without OpenIdConnect or a User Pool
     */
    readonly apple?: IdentityPoolAppleLoginProvider;
    /**
     * The Twitter Authentication Provider associated with this Identity Pool
     * @default - No Twitter Authentication Provider used without OpenIdConnect or a User Pool
     */
    readonly twitter?: IdentityPoolTwitterLoginProvider;
    /**
     * The User Pool Authentication Providers associated with this Identity Pool
     * @default - no User Pools associated
     */
    readonly userPools?: IUserPoolAuthenticationProvider[];
    /**
     * The OpenIdConnect Provider associated with this Identity Pool
     * @default - no OpenIdConnectProvider
     */
    readonly openIdConnectProviders?: IOpenIdConnectProvider[];
    /**
     * The Security Assertion Markup Language provider associated with this Identity Pool
     * @default - no SamlProvider
     */
    readonly samlProviders?: ISamlProvider[];
    /**
     * The developer provider name to associate with this Identity Pool
     * @default - no custom provider
     */
    readonly customProvider?: string;
}
/**
 * Map roles to users in the Identity Pool based on claims from the Identity Provider
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolroleattachment.html
 */
export interface IdentityPoolRoleMapping {
    /**
     * The url of the Provider for which the role is mapped
     */
    readonly providerUrl: IdentityPoolProviderUrl;
    /**
     * The key used for the role mapping in the role mapping hash. Required if the providerUrl is a token.
     * @default - The provided providerUrl
     */
    readonly mappingKey?: string;
    /**
     * If true then mapped roles must be passed through the cognito:roles or cognito:preferred_role claims from Identity Provider.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/role-based-access-control.html#using-tokens-to-assign-roles-to-users
     *
     * @default false
     */
    readonly useToken?: boolean;
    /**
     * Allow for role assumption when results of role mapping are ambiguous
     * @default false - Ambiguous role resolutions will lead to requester being denied
     */
    readonly resolveAmbiguousRoles?: boolean;
    /**
     * The claim and value that must be matched in order to assume the role. Required if useToken is false
     * @default - No role mapping rule
     */
    readonly rules?: RoleMappingRule[];
}
/**
 * Types of matches allowed for role mapping
 */
export declare enum RoleMappingMatchType {
    /**
     * The claim from the token must equal the given value in order for a match
     */
    EQUALS = "Equals",
    /**
     * The claim from the token must contain the given value in order for a match
     */
    CONTAINS = "Contains",
    /**
     * The claim from the token must start with the given value in order for a match
     */
    STARTS_WITH = "StartsWith",
    /**
     * The claim from the token must not equal the given value in order for a match
     */
    NOTEQUAL = "NotEqual"
}
/**
 * Represents an Identity Pool Role Attachment role mapping rule
 */
export interface RoleMappingRule {
    /**
     * The key sent in the token by the federated Identity Provider
     */
    readonly claim: string;
    /**
     * The role to be assumed when the claim value is matched
     */
    readonly mappedRole: IRole;
    /**
     * The value of the claim that must be matched
     */
    readonly claimValue: string;
    /**
     * How to match with the claim value
     *
     * @default RoleMappingMatchType.EQUALS
     */
    readonly matchType?: RoleMappingMatchType;
}
/**
 * Define a Cognito Identity Pool
 *
 * @resource AWS::Cognito::IdentityPool
 */
export declare class IdentityPool extends Resource implements IIdentityPool {
    /**
     * Import an existing Identity Pool from its ID
     */
    static fromIdentityPoolId(scope: Construct, id: string, identityPoolId: string): IIdentityPool;
    /**
     * Import an existing Identity Pool from its ARN
     */
    static fromIdentityPoolArn(scope: Construct, id: string, identityPoolArn: string): IIdentityPool;
    /**
     * The ID of the Identity Pool in the format REGION:GUID
     * @attribute
     */
    readonly identityPoolId: string;
    /**
     * The ARN of the Identity Pool
     * @attribute
     */
    readonly identityPoolArn: string;
    /**
     * The name of the Identity Pool
     * @attribute
     */
    readonly identityPoolName: string;
    /**
     * Default Role for authenticated users
     */
    readonly authenticatedRole: IRole;
    /**
     * Default Role for unauthenticated users
     */
    readonly unauthenticatedRole: IRole;
    /**
     * Role Provider for the default Role for authenticated users
     */
    private readonly roleAttachment;
    /**
     * List of Identity Providers added in constructor for use with property overrides
     */
    private cognitoIdentityProviders;
    constructor(scope: Construct, id: string, props?: IdentityPoolProps);
    /**
     * Add a User Pool to the Identity Pool and configure the User Pool client to handle identities
     */
    addUserPoolAuthentication(userPool: IUserPoolAuthenticationProvider): void;
    /**
     * Configure default Roles for Identity Pool
     */
    private configureDefaultRole;
    private configureDefaultGrantPrincipal;
}
