import * as cdk from "../../core";
import * as constructs from "constructs";
import * as cfn_parse from "../../core/lib/helpers-internal";
/**
 * The `AWS::CodeBuild::Project` resource configures how AWS CodeBuild builds your source code.
 *
 * For example, it tells CodeBuild where to get the source code and which build environment to use.
 *
 * > To unset or remove a project value via CFN, explicitly provide the attribute with value as empty input.
 *
 * @cloudformationResource AWS::CodeBuild::Project
 * @stability external
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html
 */
export declare class CfnProject extends cdk.CfnResource implements cdk.IInspectable, cdk.ITaggable {
    /**
     * The CloudFormation resource type name for this resource class.
     */
    static readonly CFN_RESOURCE_TYPE_NAME: string;
    /**
     * Build a CfnProject from CloudFormation properties
     *
     * A factory method that creates a new instance of this class from an object
     * containing the CloudFormation properties of this resource.
     * Used in the @aws-cdk/cloudformation-include module.
     *
     * @internal
     */
    static _fromCloudFormation(scope: constructs.Construct, id: string, resourceAttributes: any, options: cfn_parse.FromCloudFormationOptions): CfnProject;
    /**
     * The ARN of the AWS CodeBuild project, such as `arn:aws:codebuild:us-west-2:123456789012:project/myProjectName` .
     *
     * @cloudformationAttribute Arn
     */
    readonly attrArn: string;
    /**
     * @cloudformationAttribute Id
     */
    readonly attrId: string;
    /**
     * `Artifacts` is a property of the [AWS::CodeBuild::Project](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html) resource that specifies output settings for artifacts generated by an AWS CodeBuild build.
     */
    artifacts: CfnProject.ArtifactsProperty | cdk.IResolvable;
    autoRetryLimit?: number;
    /**
     * Indicates whether AWS CodeBuild generates a publicly accessible URL for your project's build badge.
     */
    badgeEnabled?: boolean | cdk.IResolvable;
    /**
     * A `ProjectBuildBatchConfig` object that defines the batch build options for the project.
     */
    buildBatchConfig?: cdk.IResolvable | CfnProject.ProjectBuildBatchConfigProperty;
    /**
     * Settings that AWS CodeBuild uses to store and reuse build dependencies.
     */
    cache?: cdk.IResolvable | CfnProject.ProjectCacheProperty;
    /**
     * The maximum number of concurrent builds that are allowed for this project.
     */
    concurrentBuildLimit?: number;
    /**
     * A description that makes the build project easy to identify.
     */
    description?: string;
    /**
     * The AWS Key Management Service customer master key (CMK) to be used for encrypting the build output artifacts.
     */
    encryptionKey?: string;
    /**
     * The build environment settings for the project, such as the environment type or the environment variables to use for the build environment.
     */
    environment: CfnProject.EnvironmentProperty | cdk.IResolvable;
    /**
     * An array of `ProjectFileSystemLocation` objects for a CodeBuild build project.
     */
    fileSystemLocations?: Array<cdk.IResolvable | CfnProject.ProjectFileSystemLocationProperty> | cdk.IResolvable;
    /**
     * Information about logs for the build project.
     */
    logsConfig?: cdk.IResolvable | CfnProject.LogsConfigProperty;
    /**
     * The name of the build project.
     */
    name?: string;
    /**
     * The number of minutes a build is allowed to be queued before it times out.
     */
    queuedTimeoutInMinutes?: number;
    /**
     * The ARN of the IAM role that enables CodeBuild to access the CloudWatch Logs and Amazon S3 artifacts for the project's builds.
     */
    resourceAccessRole?: string;
    /**
     * A list of `Artifacts` objects.
     */
    secondaryArtifacts?: Array<CfnProject.ArtifactsProperty | cdk.IResolvable> | cdk.IResolvable;
    /**
     * An array of `ProjectSource` objects.
     */
    secondarySources?: Array<cdk.IResolvable | CfnProject.SourceProperty> | cdk.IResolvable;
    /**
     * An array of `ProjectSourceVersion` objects.
     */
    secondarySourceVersions?: Array<cdk.IResolvable | CfnProject.ProjectSourceVersionProperty> | cdk.IResolvable;
    /**
     * The ARN of the IAM role that enables AWS CodeBuild to interact with dependent AWS services on behalf of the AWS account.
     */
    serviceRole: string;
    /**
     * The source code settings for the project, such as the source code's repository type and location.
     */
    source: cdk.IResolvable | CfnProject.SourceProperty;
    /**
     * A version of the build input to be built for this project.
     */
    sourceVersion?: string;
    /**
     * Tag Manager which manages the tags for this resource
     */
    readonly tags: cdk.TagManager;
    /**
     * An arbitrary set of tags (key-value pairs) for the AWS CodeBuild project.
     */
    tagsRaw?: Array<cdk.CfnTag>;
    /**
     * How long, in minutes, from 5 to 2160 (36 hours), for AWS CodeBuild to wait before timing out any related build that did not get marked as completed.
     */
    timeoutInMinutes?: number;
    /**
     * For an existing AWS CodeBuild build project that has its source code stored in a GitHub repository, enables AWS CodeBuild to begin automatically rebuilding the source code every time a code change is pushed to the repository.
     */
    triggers?: cdk.IResolvable | CfnProject.ProjectTriggersProperty;
    /**
     * Specifies the visibility of the project's builds. Possible values are:.
     */
    visibility?: string;
    /**
     * `VpcConfig` specifies settings that enable AWS CodeBuild to access resources in an Amazon VPC.
     */
    vpcConfig?: cdk.IResolvable | CfnProject.VpcConfigProperty;
    /**
     * @param scope Scope in which this resource is defined
     * @param id Construct identifier for this resource (unique in its scope)
     * @param props Resource properties
     */
    constructor(scope: constructs.Construct, id: string, props: CfnProjectProps);
    protected get cfnProperties(): Record<string, any>;
    /**
     * Examines the CloudFormation resource and discloses attributes
     *
     * @param inspector tree inspector to collect and process attributes
     */
    inspect(inspector: cdk.TreeInspector): void;
    protected renderProperties(props: Record<string, any>): Record<string, any>;
}
export declare namespace CfnProject {
    /**
     * `VpcConfig` is a property of the [AWS::CodeBuild::Project](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html) resource that enable AWS CodeBuild to access resources in an Amazon VPC. For more information, see [Use AWS CodeBuild with Amazon Virtual Private Cloud](https://docs.aws.amazon.com/codebuild/latest/userguide/vpc-support.html) in the *AWS CodeBuild User Guide* .
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-vpcconfig.html
     */
    interface VpcConfigProperty {
        /**
         * A list of one or more security groups IDs in your Amazon VPC.
         *
         * The maximum count is 5.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-vpcconfig.html#cfn-codebuild-project-vpcconfig-securitygroupids
         */
        readonly securityGroupIds?: Array<string>;
        /**
         * A list of one or more subnet IDs in your Amazon VPC.
         *
         * The maximum count is 16.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-vpcconfig.html#cfn-codebuild-project-vpcconfig-subnets
         */
        readonly subnets?: Array<string>;
        /**
         * The ID of the Amazon VPC.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-vpcconfig.html#cfn-codebuild-project-vpcconfig-vpcid
         */
        readonly vpcId?: string;
    }
    /**
     * `Source` is a property of the [AWS::CodeBuild::Project](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html) resource that specifies the source code settings for the project, such as the source code's repository type and location.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html
     */
    interface SourceProperty {
        /**
         * Information about the authorization settings for AWS CodeBuild to access the source code to be built.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html#cfn-codebuild-project-source-auth
         */
        readonly auth?: cdk.IResolvable | CfnProject.SourceAuthProperty;
        /**
         * The build specification for the project.
         *
         * If this value is not provided, then the source code must contain a buildspec file named `buildspec.yml` at the root level. If this value is provided, it can be either a single string containing the entire build specification, or the path to an alternate buildspec file relative to the value of the built-in environment variable `CODEBUILD_SRC_DIR` . The alternate buildspec file can have a name other than `buildspec.yml` , for example `myspec.yml` or `build_spec_qa.yml` or similar. For more information, see the [Build Spec Reference](https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-example) in the *AWS CodeBuild User Guide* .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html#cfn-codebuild-project-source-buildspec
         */
        readonly buildSpec?: string;
        /**
         * Contains information that defines how the build project reports the build status to the source provider.
         *
         * This option is only used when the source provider is `GITHUB` , `GITHUB_ENTERPRISE` , or `BITBUCKET` .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html#cfn-codebuild-project-source-buildstatusconfig
         */
        readonly buildStatusConfig?: CfnProject.BuildStatusConfigProperty | cdk.IResolvable;
        /**
         * The depth of history to download.
         *
         * Minimum value is 0. If this value is 0, greater than 25, or not provided, then the full history is downloaded with each build project. If your source type is Amazon S3, this value is not supported.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html#cfn-codebuild-project-source-gitclonedepth
         */
        readonly gitCloneDepth?: number;
        /**
         * Information about the Git submodules configuration for the build project.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html#cfn-codebuild-project-source-gitsubmodulesconfig
         */
        readonly gitSubmodulesConfig?: CfnProject.GitSubmodulesConfigProperty | cdk.IResolvable;
        /**
         * This is used with GitHub Enterprise only.
         *
         * Set to true to ignore SSL warnings while connecting to your GitHub Enterprise project repository. The default value is `false` . `InsecureSsl` should be used for testing purposes only. It should not be used in a production environment.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html#cfn-codebuild-project-source-insecuressl
         */
        readonly insecureSsl?: boolean | cdk.IResolvable;
        /**
         * Information about the location of the source code to be built. Valid values include:.
         *
         * - For source code settings that are specified in the source action of a pipeline in CodePipeline, `location` should not be specified. If it is specified, CodePipeline ignores it. This is because CodePipeline uses the settings in a pipeline's source action instead of this value.
         * - For source code in an CodeCommit repository, the HTTPS clone URL to the repository that contains the source code and the buildspec file (for example, `https://git-codecommit.<region-ID>.amazonaws.com/v1/repos/<repo-name>` ).
         * - For source code in an Amazon S3 input bucket, one of the following.
         *
         * - The path to the ZIP file that contains the source code (for example, `<bucket-name>/<path>/<object-name>.zip` ).
         * - The path to the folder that contains the source code (for example, `<bucket-name>/<path-to-source-code>/<folder>/` ).
         * - For source code in a GitHub repository, the HTTPS clone URL to the repository that contains the source and the buildspec file. You must connect your AWS account to your GitHub account. Use the AWS CodeBuild console to start creating a build project. When you use the console to connect (or reconnect) with GitHub, on the GitHub *Authorize application* page, for *Organization access* , choose *Request access* next to each repository you want to allow AWS CodeBuild to have access to, and then choose *Authorize application* . (After you have connected to your GitHub account, you do not need to finish creating the build project. You can leave the AWS CodeBuild console.) To instruct AWS CodeBuild to use this connection, in the `source` object, set the `auth` object's `type` value to `OAUTH` .
         * - For source code in an GitLab or self-managed GitLab repository, the HTTPS clone URL to the repository that contains the source and the buildspec file. You must connect your AWS account to your GitLab account. Use the AWS CodeBuild console to start creating a build project. When you use the console to connect (or reconnect) with GitLab, on the Connections *Authorize application* page, choose *Authorize* . Then on the AWS CodeConnections *Create GitLab connection* page, choose *Connect to GitLab* . (After you have connected to your GitLab account, you do not need to finish creating the build project. You can leave the AWS CodeBuild console.) To instruct AWS CodeBuild to override the default connection and use this connection instead, set the `auth` object's `type` value to `CODECONNECTIONS` in the `source` object.
         * - For source code in a Bitbucket repository, the HTTPS clone URL to the repository that contains the source and the buildspec file. You must connect your AWS account to your Bitbucket account. Use the AWS CodeBuild console to start creating a build project. When you use the console to connect (or reconnect) with Bitbucket, on the Bitbucket *Confirm access to your account* page, choose *Grant access* . (After you have connected to your Bitbucket account, you do not need to finish creating the build project. You can leave the AWS CodeBuild console.) To instruct AWS CodeBuild to use this connection, in the `source` object, set the `auth` object's `type` value to `OAUTH` .
         *
         * If you specify `CODEPIPELINE` for the `Type` property, don't specify this property. For all of the other types, you must specify `Location` .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html#cfn-codebuild-project-source-location
         */
        readonly location?: string;
        /**
         * Set to true to report the status of a build's start and finish to your source provider.
         *
         * This option is valid only when your source provider is GitHub, GitHub Enterprise, GitLab, GitLab Self Managed, or Bitbucket. If this is set and you use a different source provider, an `invalidInputException` is thrown.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html#cfn-codebuild-project-source-reportbuildstatus
         */
        readonly reportBuildStatus?: boolean | cdk.IResolvable;
        /**
         * An identifier for this project source.
         *
         * The identifier can only contain alphanumeric characters and underscores, and must be less than 128 characters in length.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html#cfn-codebuild-project-source-sourceidentifier
         */
        readonly sourceIdentifier?: string;
        /**
         * The type of repository that contains the source code to be built. Valid values include:.
         *
         * - `BITBUCKET` : The source code is in a Bitbucket repository.
         * - `CODECOMMIT` : The source code is in an CodeCommit repository.
         * - `CODEPIPELINE` : The source code settings are specified in the source action of a pipeline in CodePipeline.
         * - `GITHUB` : The source code is in a GitHub repository.
         * - `GITHUB_ENTERPRISE` : The source code is in a GitHub Enterprise Server repository.
         * - `GITLAB` : The source code is in a GitLab repository.
         * - `GITLAB_SELF_MANAGED` : The source code is in a self-managed GitLab repository.
         * - `NO_SOURCE` : The project does not have input source code.
         * - `S3` : The source code is in an Amazon S3 bucket.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html#cfn-codebuild-project-source-type
         */
        readonly type: string;
    }
    /**
     * `SourceAuth` is a property of the [AWS CodeBuild Project Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html) property type that specifies authorization settings for AWS CodeBuild to access the source code to be built.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-sourceauth.html
     */
    interface SourceAuthProperty {
        /**
         * The resource value that applies to the specified authorization type.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-sourceauth.html#cfn-codebuild-project-sourceauth-resource
         */
        readonly resource?: string;
        /**
         * The authorization type to use.
         *
         * Valid options are OAUTH, CODECONNECTIONS, or SECRETS_MANAGER.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-sourceauth.html#cfn-codebuild-project-sourceauth-type
         */
        readonly type: string;
    }
    /**
     * Contains information that defines how the AWS CodeBuild build project reports the build status to the source provider.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-buildstatusconfig.html
     */
    interface BuildStatusConfigProperty {
        /**
         * Specifies the context of the build status CodeBuild sends to the source provider.
         *
         * The usage of this parameter depends on the source provider.
         *
         * - **Bitbucket** - This parameter is used for the `name` parameter in the Bitbucket commit status. For more information, see [build](https://docs.aws.amazon.com/https://developer.atlassian.com/bitbucket/api/2/reference/resource/repositories/%7Bworkspace%7D/%7Brepo_slug%7D/commit/%7Bnode%7D/statuses/build) in the Bitbucket API documentation.
         * - **GitHub/GitHub Enterprise Server** - This parameter is used for the `context` parameter in the GitHub commit status. For more information, see [Create a commit status](https://docs.aws.amazon.com/https://developer.github.com/v3/repos/statuses/#create-a-commit-status) in the GitHub developer guide.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-buildstatusconfig.html#cfn-codebuild-project-buildstatusconfig-context
         */
        readonly context?: string;
        /**
         * Specifies the target url of the build status CodeBuild sends to the source provider.
         *
         * The usage of this parameter depends on the source provider.
         *
         * - **Bitbucket** - This parameter is used for the `url` parameter in the Bitbucket commit status. For more information, see [build](https://docs.aws.amazon.com/https://developer.atlassian.com/bitbucket/api/2/reference/resource/repositories/%7Bworkspace%7D/%7Brepo_slug%7D/commit/%7Bnode%7D/statuses/build) in the Bitbucket API documentation.
         * - **GitHub/GitHub Enterprise Server** - This parameter is used for the `target_url` parameter in the GitHub commit status. For more information, see [Create a commit status](https://docs.aws.amazon.com/https://developer.github.com/v3/repos/statuses/#create-a-commit-status) in the GitHub developer guide.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-buildstatusconfig.html#cfn-codebuild-project-buildstatusconfig-targeturl
         */
        readonly targetUrl?: string;
    }
    /**
     * `GitSubmodulesConfig` is a property of the [AWS CodeBuild Project Source](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-source.html) property type that specifies information about the Git submodules configuration for the build project.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-gitsubmodulesconfig.html
     */
    interface GitSubmodulesConfigProperty {
        /**
         * Set to true to fetch Git submodules for your AWS CodeBuild build project.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-gitsubmodulesconfig.html#cfn-codebuild-project-gitsubmodulesconfig-fetchsubmodules
         */
        readonly fetchSubmodules: boolean | cdk.IResolvable;
    }
    /**
     * `ProjectTriggers` is a property of the [AWS CodeBuild Project](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html) resource that specifies webhooks that trigger an AWS CodeBuild build.
     *
     * > The Webhook feature isn't available in AWS CloudFormation for GitHub Enterprise projects. Use the AWS CLI or AWS CodeBuild console to create the webhook.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projecttriggers.html
     */
    interface ProjectTriggersProperty {
        /**
         * Specifies the type of build this webhook will trigger. Allowed values are:.
         *
         * - **BUILD** - A single build
         * - **BUILD_BATCH** - A batch build
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projecttriggers.html#cfn-codebuild-project-projecttriggers-buildtype
         */
        readonly buildType?: string;
        /**
         * A list of lists of `WebhookFilter` objects used to determine which webhook events are triggered.
         *
         * At least one `WebhookFilter` in the array must specify `EVENT` as its type.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projecttriggers.html#cfn-codebuild-project-projecttriggers-filtergroups
         */
        readonly filterGroups?: Array<Array<cdk.IResolvable | CfnProject.WebhookFilterProperty> | cdk.IResolvable> | cdk.IResolvable;
        /**
         * Contains configuration information about the scope for a webhook.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projecttriggers.html#cfn-codebuild-project-projecttriggers-scopeconfiguration
         */
        readonly scopeConfiguration?: cdk.IResolvable | CfnProject.ScopeConfigurationProperty;
        /**
         * Specifies whether or not to begin automatically rebuilding the source code every time a code change is pushed to the repository.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projecttriggers.html#cfn-codebuild-project-projecttriggers-webhook
         */
        readonly webhook?: boolean | cdk.IResolvable;
    }
    /**
     * `WebhookFilter` is a structure of the `FilterGroups` property on the [AWS CodeBuild Project ProjectTriggers](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projecttriggers.html) property type that specifies which webhooks trigger an AWS CodeBuild build.
     *
     * > The Webhook feature isn't available in AWS CloudFormation for GitHub Enterprise projects. Use the AWS CLI or AWS CodeBuild console to create the webhook.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-webhookfilter.html
     */
    interface WebhookFilterProperty {
        /**
         * Used to indicate that the `pattern` determines which webhook events do not trigger a build.
         *
         * If true, then a webhook event that does not match the `pattern` triggers a build. If false, then a webhook event that matches the `pattern` triggers a build.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-webhookfilter.html#cfn-codebuild-project-webhookfilter-excludematchedpattern
         */
        readonly excludeMatchedPattern?: boolean | cdk.IResolvable;
        /**
         * For a `WebHookFilter` that uses `EVENT` type, a comma-separated string that specifies one or more events.
         *
         * For example, the webhook filter `PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED` allows all push, pull request created, and pull request updated events to trigger a build.
         *
         * For a `WebHookFilter` that uses any of the other filter types, a regular expression pattern. For example, a `WebHookFilter` that uses `HEAD_REF` for its `type` and the pattern `^refs/heads/` triggers a build when the head reference is a branch with a reference name `refs/heads/branch-name` .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-webhookfilter.html#cfn-codebuild-project-webhookfilter-pattern
         */
        readonly pattern: string;
        /**
         * The type of webhook filter.
         *
         * There are 11 webhook filter types: `EVENT` , `ACTOR_ACCOUNT_ID` , `HEAD_REF` , `BASE_REF` , `FILE_PATH` , `COMMIT_MESSAGE` , `TAG_NAME` , `RELEASE_NAME` , `REPOSITORY_NAME` , `ORGANIZATION_NAME` , and `WORKFLOW_NAME` .
         *
         * - EVENT
         *
         * - A webhook event triggers a build when the provided `pattern` matches one of nine event types: `PUSH` , `PULL_REQUEST_CREATED` , `PULL_REQUEST_UPDATED` , `PULL_REQUEST_CLOSED` , `PULL_REQUEST_REOPENED` , `PULL_REQUEST_MERGED` , `RELEASED` , `PRERELEASED` , and `WORKFLOW_JOB_QUEUED` . The `EVENT` patterns are specified as a comma-separated string. For example, `PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED` filters all push, pull request created, and pull request updated events.
         *
         * > Types `PULL_REQUEST_REOPENED` and `WORKFLOW_JOB_QUEUED` work with GitHub and GitHub Enterprise only. Types `RELEASED` and `PRERELEASED` work with GitHub only.
         * - ACTOR_ACCOUNT_ID
         *
         * - A webhook event triggers a build when a GitHub, GitHub Enterprise, or Bitbucket account ID matches the regular expression `pattern` .
         * - HEAD_REF
         *
         * - A webhook event triggers a build when the head reference matches the regular expression `pattern` . For example, `refs/heads/branch-name` and `refs/tags/tag-name` .
         *
         * > Works with GitHub and GitHub Enterprise push, GitHub and GitHub Enterprise pull request, Bitbucket push, and Bitbucket pull request events.
         * - BASE_REF
         *
         * - A webhook event triggers a build when the base reference matches the regular expression `pattern` . For example, `refs/heads/branch-name` .
         *
         * > Works with pull request events only.
         * - FILE_PATH
         *
         * - A webhook triggers a build when the path of a changed file matches the regular expression `pattern` .
         *
         * > Works with push and pull request events only.
         * - COMMIT_MESSAGE
         *
         * - A webhook triggers a build when the head commit message matches the regular expression `pattern` .
         *
         * > Works with push and pull request events only.
         * - TAG_NAME
         *
         * - A webhook triggers a build when the tag name of the release matches the regular expression `pattern` .
         *
         * > Works with `RELEASED` and `PRERELEASED` events only.
         * - RELEASE_NAME
         *
         * - A webhook triggers a build when the release name matches the regular expression `pattern` .
         *
         * > Works with `RELEASED` and `PRERELEASED` events only.
         * - REPOSITORY_NAME
         *
         * - A webhook triggers a build when the repository name matches the regular expression `pattern` .
         *
         * > Works with GitHub global or organization webhooks only.
         * - ORGANIZATION_NAME
         *
         * - A webhook triggers a build when the organization name matches the regular expression `pattern` .
         *
         * > Works with GitHub global webhooks only.
         * - WORKFLOW_NAME
         *
         * - A webhook triggers a build when the workflow name matches the regular expression `pattern` .
         *
         * > Works with `WORKFLOW_JOB_QUEUED` events only. > For CodeBuild-hosted Buildkite runner builds, WORKFLOW_NAME filters will filter by pipeline name.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-webhookfilter.html#cfn-codebuild-project-webhookfilter-type
         */
        readonly type: string;
    }
    /**
     * Contains configuration information about the scope for a webhook.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-scopeconfiguration.html
     */
    interface ScopeConfigurationProperty {
        /**
         * The name of either the enterprise or organization that will send webhook events to CodeBuild , depending on if the webhook is a global or organization webhook respectively.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-scopeconfiguration.html#cfn-codebuild-project-scopeconfiguration-name
         */
        readonly name: string;
    }
    /**
     * `Artifacts` is a property of the [AWS::CodeBuild::Project](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html) resource that specifies output settings for artifacts generated by an AWS CodeBuild build.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-artifacts.html
     */
    interface ArtifactsProperty {
        /**
         * An identifier for this artifact definition.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-artifacts.html#cfn-codebuild-project-artifacts-artifactidentifier
         */
        readonly artifactIdentifier?: string;
        /**
         * Set to true if you do not want your output artifacts encrypted.
         *
         * This option is valid only if your artifacts type is Amazon Simple Storage Service (Amazon S3). If this is set with another artifacts type, an `invalidInputException` is thrown.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-artifacts.html#cfn-codebuild-project-artifacts-encryptiondisabled
         */
        readonly encryptionDisabled?: boolean | cdk.IResolvable;
        /**
         * Information about the build output artifact location:.
         *
         * - If `type` is set to `CODEPIPELINE` , AWS CodePipeline ignores this value if specified. This is because CodePipeline manages its build output locations instead of CodeBuild .
         * - If `type` is set to `NO_ARTIFACTS` , this value is ignored if specified, because no build output is produced.
         * - If `type` is set to `S3` , this is the name of the output bucket.
         *
         * If you specify `CODEPIPELINE` or `NO_ARTIFACTS` for the `Type` property, don't specify this property. For all of the other types, you must specify this property.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-artifacts.html#cfn-codebuild-project-artifacts-location
         */
        readonly location?: string;
        /**
         * Along with `path` and `namespaceType` , the pattern that AWS CodeBuild uses to name and store the output artifact:.
         *
         * - If `type` is set to `CODEPIPELINE` , AWS CodePipeline ignores this value if specified. This is because CodePipeline manages its build output names instead of AWS CodeBuild .
         * - If `type` is set to `NO_ARTIFACTS` , this value is ignored if specified, because no build output is produced.
         * - If `type` is set to `S3` , this is the name of the output artifact object. If you set the name to be a forward slash ("/"), the artifact is stored in the root of the output bucket.
         *
         * For example:
         *
         * - If `path` is set to `MyArtifacts` , `namespaceType` is set to `BUILD_ID` , and `name` is set to `MyArtifact.zip` , then the output artifact is stored in `MyArtifacts/ *build-ID* /MyArtifact.zip` .
         * - If `path` is empty, `namespaceType` is set to `NONE` , and `name` is set to " `/` ", the output artifact is stored in the root of the output bucket.
         * - If `path` is set to `MyArtifacts` , `namespaceType` is set to `BUILD_ID` , and `name` is set to " `/` ", the output artifact is stored in `MyArtifacts/ *build-ID*` .
         *
         * If you specify `CODEPIPELINE` or `NO_ARTIFACTS` for the `Type` property, don't specify this property. For all of the other types, you must specify this property.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-artifacts.html#cfn-codebuild-project-artifacts-name
         */
        readonly name?: string;
        /**
         * Along with `path` and `name` , the pattern that AWS CodeBuild uses to determine the name and location to store the output artifact:  - If `type` is set to `CODEPIPELINE` , CodePipeline ignores this value if specified.
         *
         * This is because CodePipeline manages its build output names instead of AWS CodeBuild .
         * - If `type` is set to `NO_ARTIFACTS` , this value is ignored if specified, because no build output is produced.
         * - If `type` is set to `S3` , valid values include:
         *
         * - `BUILD_ID` : Include the build ID in the location of the build output artifact.
         * - `NONE` : Do not include the build ID. This is the default if `namespaceType` is not specified.
         *
         * For example, if `path` is set to `MyArtifacts` , `namespaceType` is set to `BUILD_ID` , and `name` is set to `MyArtifact.zip` , the output artifact is stored in `MyArtifacts/<build-ID>/MyArtifact.zip` .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-artifacts.html#cfn-codebuild-project-artifacts-namespacetype
         */
        readonly namespaceType?: string;
        /**
         * If set to true a name specified in the buildspec file overrides the artifact name.
         *
         * The name specified in a buildspec file is calculated at build time and uses the Shell command language. For example, you can append a date and time to your artifact name so that it is always unique.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-artifacts.html#cfn-codebuild-project-artifacts-overrideartifactname
         */
        readonly overrideArtifactName?: boolean | cdk.IResolvable;
        /**
         * The type of build output artifact to create:.
         *
         * - If `type` is set to `CODEPIPELINE` , CodePipeline ignores this value if specified. This is because CodePipeline manages its build output artifacts instead of AWS CodeBuild .
         * - If `type` is set to `NO_ARTIFACTS` , this value is ignored if specified, because no build output is produced.
         * - If `type` is set to `S3` , valid values include:
         *
         * - `NONE` : AWS CodeBuild creates in the output bucket a folder that contains the build output. This is the default if `packaging` is not specified.
         * - `ZIP` : AWS CodeBuild creates in the output bucket a ZIP file that contains the build output.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-artifacts.html#cfn-codebuild-project-artifacts-packaging
         */
        readonly packaging?: string;
        /**
         * Along with `namespaceType` and `name` , the pattern that AWS CodeBuild uses to name and store the output artifact:.
         *
         * - If `type` is set to `CODEPIPELINE` , CodePipeline ignores this value if specified. This is because CodePipeline manages its build output names instead of AWS CodeBuild .
         * - If `type` is set to `NO_ARTIFACTS` , this value is ignored if specified, because no build output is produced.
         * - If `type` is set to `S3` , this is the path to the output artifact. If `path` is not specified, `path` is not used.
         *
         * For example, if `path` is set to `MyArtifacts` , `namespaceType` is set to `NONE` , and `name` is set to `MyArtifact.zip` , the output artifact is stored in the output bucket at `MyArtifacts/MyArtifact.zip` .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-artifacts.html#cfn-codebuild-project-artifacts-path
         */
        readonly path?: string;
        /**
         * The type of build output artifact. Valid values include:.
         *
         * - `CODEPIPELINE` : The build project has build output generated through CodePipeline.
         *
         * > The `CODEPIPELINE` type is not supported for `secondaryArtifacts` .
         * - `NO_ARTIFACTS` : The build project does not produce any build output.
         * - `S3` : The build project stores build output in Amazon S3.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-artifacts.html#cfn-codebuild-project-artifacts-type
         */
        readonly type: string;
    }
    /**
     * `LogsConfig` is a property of the [AWS CodeBuild Project](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html) resource that specifies information about logs for a build project. These can be logs in Amazon CloudWatch Logs, built in a specified S3 bucket, or both.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-logsconfig.html
     */
    interface LogsConfigProperty {
        /**
         * Information about CloudWatch Logs for a build project.
         *
         * CloudWatch Logs are enabled by default.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-logsconfig.html#cfn-codebuild-project-logsconfig-cloudwatchlogs
         */
        readonly cloudWatchLogs?: CfnProject.CloudWatchLogsConfigProperty | cdk.IResolvable;
        /**
         * Information about logs built to an S3 bucket for a build project.
         *
         * S3 logs are not enabled by default.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-logsconfig.html#cfn-codebuild-project-logsconfig-s3logs
         */
        readonly s3Logs?: cdk.IResolvable | CfnProject.S3LogsConfigProperty;
    }
    /**
     * `CloudWatchLogs` is a property of the [AWS CodeBuild Project LogsConfig](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-logsconfig.html) property type that specifies settings for CloudWatch logs generated by an AWS CodeBuild build.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-cloudwatchlogsconfig.html
     */
    interface CloudWatchLogsConfigProperty {
        /**
         * The group name of the logs in CloudWatch Logs.
         *
         * For more information, see [Working with Log Groups and Log Streams](https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/Working-with-log-groups-and-streams.html) .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-cloudwatchlogsconfig.html#cfn-codebuild-project-cloudwatchlogsconfig-groupname
         */
        readonly groupName?: string;
        /**
         * The current status of the logs in CloudWatch Logs for a build project. Valid values are:.
         *
         * - `ENABLED` : CloudWatch Logs are enabled for this build project.
         * - `DISABLED` : CloudWatch Logs are not enabled for this build project.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-cloudwatchlogsconfig.html#cfn-codebuild-project-cloudwatchlogsconfig-status
         */
        readonly status: string;
        /**
         * The prefix of the stream name of the CloudWatch Logs.
         *
         * For more information, see [Working with Log Groups and Log Streams](https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/Working-with-log-groups-and-streams.html) .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-cloudwatchlogsconfig.html#cfn-codebuild-project-cloudwatchlogsconfig-streamname
         */
        readonly streamName?: string;
    }
    /**
     * `S3Logs` is a property of the [AWS CodeBuild Project LogsConfig](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-logsconfig.html) property type that specifies settings for logs generated by an AWS CodeBuild build in an S3 bucket.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-s3logsconfig.html
     */
    interface S3LogsConfigProperty {
        /**
         * Set to true if you do not want your S3 build log output encrypted.
         *
         * By default S3 build logs are encrypted.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-s3logsconfig.html#cfn-codebuild-project-s3logsconfig-encryptiondisabled
         */
        readonly encryptionDisabled?: boolean | cdk.IResolvable;
        /**
         * The ARN of an S3 bucket and the path prefix for S3 logs.
         *
         * If your Amazon S3 bucket name is `my-bucket` , and your path prefix is `build-log` , then acceptable formats are `my-bucket/build-log` or `arn:aws:s3:::my-bucket/build-log` .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-s3logsconfig.html#cfn-codebuild-project-s3logsconfig-location
         */
        readonly location?: string;
        /**
         * The current status of the S3 build logs. Valid values are:.
         *
         * - `ENABLED` : S3 build logs are enabled for this build project.
         * - `DISABLED` : S3 build logs are not enabled for this build project.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-s3logsconfig.html#cfn-codebuild-project-s3logsconfig-status
         */
        readonly status: string;
    }
    /**
     * Information about a file system created by Amazon Elastic File System (EFS).
     *
     * For more information, see [What Is Amazon Elastic File System?](https://docs.aws.amazon.com/efs/latest/ug/whatisefs.html)
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectfilesystemlocation.html
     */
    interface ProjectFileSystemLocationProperty {
        /**
         * The name used to access a file system created by Amazon EFS.
         *
         * CodeBuild creates an environment variable by appending the `identifier` in all capital letters to `CODEBUILD_` . For example, if you specify `my_efs` for `identifier` , a new environment variable is create named `CODEBUILD_MY_EFS` .
         *
         * The `identifier` is used to mount your file system.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectfilesystemlocation.html#cfn-codebuild-project-projectfilesystemlocation-identifier
         */
        readonly identifier: string;
        /**
         * A string that specifies the location of the file system created by Amazon EFS.
         *
         * Its format is `efs-dns-name:/directory-path` . You can find the DNS name of file system when you view it in the Amazon EFS console. The directory path is a path to a directory in the file system that CodeBuild mounts. For example, if the DNS name of a file system is `fs-abcd1234.efs.us-west-2.amazonaws.com` , and its mount directory is `my-efs-mount-directory` , then the `location` is `fs-abcd1234.efs.us-west-2.amazonaws.com:/my-efs-mount-directory` .
         *
         * The directory path in the format `efs-dns-name:/directory-path` is optional. If you do not specify a directory path, the location is only the DNS name and CodeBuild mounts the entire file system.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectfilesystemlocation.html#cfn-codebuild-project-projectfilesystemlocation-location
         */
        readonly location: string;
        /**
         * The mount options for a file system created by Amazon EFS.
         *
         * The default mount options used by CodeBuild are `nfsvers=4.1,rsize=1048576,wsize=1048576,hard,timeo=600,retrans=2` . For more information, see [Recommended NFS Mount Options](https://docs.aws.amazon.com/efs/latest/ug/mounting-fs-nfs-mount-settings.html) .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectfilesystemlocation.html#cfn-codebuild-project-projectfilesystemlocation-mountoptions
         */
        readonly mountOptions?: string;
        /**
         * The location in the container where you mount the file system.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectfilesystemlocation.html#cfn-codebuild-project-projectfilesystemlocation-mountpoint
         */
        readonly mountPoint: string;
        /**
         * The type of the file system.
         *
         * The one supported type is `EFS` .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectfilesystemlocation.html#cfn-codebuild-project-projectfilesystemlocation-type
         */
        readonly type: string;
    }
    /**
     * `Environment` is a property of the [AWS::CodeBuild::Project](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html) resource that specifies the environment for an AWS CodeBuild project.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html
     */
    interface EnvironmentProperty {
        /**
         * The ARN of the Amazon S3 bucket, path prefix, and object key that contains the PEM-encoded certificate for the build project.
         *
         * For more information, see [certificate](https://docs.aws.amazon.com/codebuild/latest/userguide/create-project-cli.html#cli.environment.certificate) in the *AWS CodeBuild User Guide* .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html#cfn-codebuild-project-environment-certificate
         */
        readonly certificate?: string;
        /**
         * The type of compute environment.
         *
         * This determines the number of CPU cores and memory the build environment uses. Available values include:
         *
         * - `ATTRIBUTE_BASED_COMPUTE` : Specify the amount of vCPUs, memory, disk space, and the type of machine.
         *
         * > If you use `ATTRIBUTE_BASED_COMPUTE` , you must define your attributes by using `computeConfiguration` . AWS CodeBuild will select the cheapest instance that satisfies your specified attributes. For more information, see [Reserved capacity environment types](https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html#environment-reserved-capacity.types) in the *AWS CodeBuild User Guide* .
         * - `BUILD_GENERAL1_SMALL` : Use up to 4 GiB memory and 2 vCPUs for builds.
         * - `BUILD_GENERAL1_MEDIUM` : Use up to 8 GiB memory and 4 vCPUs for builds.
         * - `BUILD_GENERAL1_LARGE` : Use up to 16 GiB memory and 8 vCPUs for builds, depending on your environment type.
         * - `BUILD_GENERAL1_XLARGE` : Use up to 72 GiB memory and 36 vCPUs for builds, depending on your environment type.
         * - `BUILD_GENERAL1_2XLARGE` : Use up to 144 GiB memory, 72 vCPUs, and 824 GB of SSD storage for builds. This compute type supports Docker images up to 100 GB uncompressed.
         * - `BUILD_LAMBDA_1GB` : Use up to 1 GiB memory for builds. Only available for environment type `LINUX_LAMBDA_CONTAINER` and `ARM_LAMBDA_CONTAINER` .
         * - `BUILD_LAMBDA_2GB` : Use up to 2 GiB memory for builds. Only available for environment type `LINUX_LAMBDA_CONTAINER` and `ARM_LAMBDA_CONTAINER` .
         * - `BUILD_LAMBDA_4GB` : Use up to 4 GiB memory for builds. Only available for environment type `LINUX_LAMBDA_CONTAINER` and `ARM_LAMBDA_CONTAINER` .
         * - `BUILD_LAMBDA_8GB` : Use up to 8 GiB memory for builds. Only available for environment type `LINUX_LAMBDA_CONTAINER` and `ARM_LAMBDA_CONTAINER` .
         * - `BUILD_LAMBDA_10GB` : Use up to 10 GiB memory for builds. Only available for environment type `LINUX_LAMBDA_CONTAINER` and `ARM_LAMBDA_CONTAINER` .
         *
         * If you use `BUILD_GENERAL1_SMALL` :
         *
         * - For environment type `LINUX_CONTAINER` , you can use up to 4 GiB memory and 2 vCPUs for builds.
         * - For environment type `LINUX_GPU_CONTAINER` , you can use up to 16 GiB memory, 4 vCPUs, and 1 NVIDIA A10G Tensor Core GPU for builds.
         * - For environment type `ARM_CONTAINER` , you can use up to 4 GiB memory and 2 vCPUs on ARM-based processors for builds.
         *
         * If you use `BUILD_GENERAL1_LARGE` :
         *
         * - For environment type `LINUX_CONTAINER` , you can use up to 16 GiB memory and 8 vCPUs for builds.
         * - For environment type `LINUX_GPU_CONTAINER` , you can use up to 255 GiB memory, 32 vCPUs, and 4 NVIDIA Tesla V100 GPUs for builds.
         * - For environment type `ARM_CONTAINER` , you can use up to 16 GiB memory and 8 vCPUs on ARM-based processors for builds.
         *
         * For more information, see [On-demand environment types](https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html#environment.types) in the *AWS CodeBuild User Guide.*
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html#cfn-codebuild-project-environment-computetype
         */
        readonly computeType: string;
        /**
         * A set of environment variables to make available to builds for this build project.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html#cfn-codebuild-project-environment-environmentvariables
         */
        readonly environmentVariables?: Array<CfnProject.EnvironmentVariableProperty | cdk.IResolvable> | cdk.IResolvable;
        /**
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html#cfn-codebuild-project-environment-fleet
         */
        readonly fleet?: cdk.IResolvable | CfnProject.ProjectFleetProperty;
        /**
         * The image tag or image digest that identifies the Docker image to use for this build project.
         *
         * Use the following formats:
         *
         * - For an image tag: `<registry>/<repository>:<tag>` . For example, in the Docker repository that CodeBuild uses to manage its Docker images, this would be `aws/codebuild/standard:4.0` .
         * - For an image digest: `<registry>/<repository>@<digest>` . For example, to specify an image with the digest "sha256:cbbf2f9a99b47fc460d422812b6a5adff7dfee951d8fa2e4a98caa0382cfbdbf," use `<registry>/<repository>@sha256:cbbf2f9a99b47fc460d422812b6a5adff7dfee951d8fa2e4a98caa0382cfbdbf` .
         *
         * For more information, see [Docker images provided by CodeBuild](https://docs.aws.amazon.com//codebuild/latest/userguide/build-env-ref-available.html) in the *AWS CodeBuild user guide* .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html#cfn-codebuild-project-environment-image
         */
        readonly image: string;
        /**
         * The type of credentials AWS CodeBuild uses to pull images in your build. There are two valid values:.
         *
         * - `CODEBUILD` specifies that AWS CodeBuild uses its own credentials. This requires that you modify your ECR repository policy to trust AWS CodeBuild service principal.
         * - `SERVICE_ROLE` specifies that AWS CodeBuild uses your build project's service role.
         *
         * When you use a cross-account or private registry image, you must use SERVICE_ROLE credentials. When you use an AWS CodeBuild curated image, you must use CODEBUILD credentials.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html#cfn-codebuild-project-environment-imagepullcredentialstype
         */
        readonly imagePullCredentialsType?: string;
        /**
         * Enables running the Docker daemon inside a Docker container.
         *
         * Set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails. The default setting is `false` .
         *
         * You can initialize the Docker daemon during the install phase of your build by adding one of the following sets of commands to the install phase of your buildspec file:
         *
         * If the operating system's base image is Ubuntu Linux:
         *
         * `- nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://0.0.0.0:2375 --storage-driver=overlay&`
         *
         * `- timeout 15 sh -c "until docker info; do echo .; sleep 1; done"`
         *
         * If the operating system's base image is Alpine Linux and the previous command does not work, add the `-t` argument to `timeout` :
         *
         * `- nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://0.0.0.0:2375 --storage-driver=overlay&`
         *
         * `- timeout -t 15 sh -c "until docker info; do echo .; sleep 1; done"`
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html#cfn-codebuild-project-environment-privilegedmode
         */
        readonly privilegedMode?: boolean | cdk.IResolvable;
        /**
         * `RegistryCredential` is a property of the [AWS::CodeBuild::Project Environment](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-environment) property that specifies information about credentials that provide access to a private Docker registry. When this is set:.
         *
         * - `imagePullCredentialsType` must be set to `SERVICE_ROLE` .
         * - images cannot be curated or an Amazon ECR image.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html#cfn-codebuild-project-environment-registrycredential
         */
        readonly registryCredential?: cdk.IResolvable | CfnProject.RegistryCredentialProperty;
        /**
         * The type of build environment to use for related builds.
         *
         * > If you're using compute fleets during project creation, `type` will be ignored.
         *
         * For more information, see [Build environment compute types](https://docs.aws.amazon.com//codebuild/latest/userguide/build-env-ref-compute-types.html) in the *AWS CodeBuild user guide* .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html#cfn-codebuild-project-environment-type
         */
        readonly type: string;
    }
    /**
     * `EnvironmentVariable` is a property of the [AWS CodeBuild Project Environment](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html) property type that specifies the name and value of an environment variable for an AWS CodeBuild project environment. When you use the environment to run a build, these variables are available for your builds to use. `EnvironmentVariable` contains a list of `EnvironmentVariable` property types.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environmentvariable.html
     */
    interface EnvironmentVariableProperty {
        /**
         * The name or key of the environment variable.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environmentvariable.html#cfn-codebuild-project-environmentvariable-name
         */
        readonly name: string;
        /**
         * The type of environment variable. Valid values include:.
         *
         * - `PARAMETER_STORE` : An environment variable stored in Systems Manager Parameter Store. For environment variables of this type, specify the name of the parameter as the `value` of the EnvironmentVariable. The parameter value will be substituted for the name at runtime. You can also define Parameter Store environment variables in the buildspec. To learn how to do so, see [env/parameter-store](https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.parameter-store) in the *AWS CodeBuild User Guide* .
         * - `PLAINTEXT` : An environment variable in plain text format. This is the default value.
         * - `SECRETS_MANAGER` : An environment variable stored in AWS Secrets Manager . For environment variables of this type, specify the name of the secret as the `value` of the EnvironmentVariable. The secret value will be substituted for the name at runtime. You can also define AWS Secrets Manager environment variables in the buildspec. To learn how to do so, see [env/secrets-manager](https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager) in the *AWS CodeBuild User Guide* .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environmentvariable.html#cfn-codebuild-project-environmentvariable-type
         */
        readonly type?: string;
        /**
         * The value of the environment variable.
         *
         * > We strongly discourage the use of `PLAINTEXT` environment variables to store sensitive values, especially AWS secret key IDs. `PLAINTEXT` environment variables can be displayed in plain text using the AWS CodeBuild console and the AWS CLI . For sensitive values, we recommend you use an environment variable of type `PARAMETER_STORE` or `SECRETS_MANAGER` .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environmentvariable.html#cfn-codebuild-project-environmentvariable-value
         */
        readonly value: string;
    }
    /**
     * `RegistryCredential` is a property of the [AWS CodeBuild Project Environment](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-environment.html) property type that specifies information about credentials that provide access to a private Docker registry. When this is set:.
     *
     * - `imagePullCredentialsType` must be set to `SERVICE_ROLE` .
     * - images cannot be curated or an Amazon ECR image.
     *
     * For more information, see [Private Registry with AWS Secrets Manager Sample for AWS CodeBuild](https://docs.aws.amazon.com/codebuild/latest/userguide/sample-private-registry.html) .
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-registrycredential.html
     */
    interface RegistryCredentialProperty {
        /**
         * The Amazon Resource Name (ARN) or name of credentials created using AWS Secrets Manager .
         *
         * > The `credential` can use the name of the credentials only if they exist in your current AWS Region .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-registrycredential.html#cfn-codebuild-project-registrycredential-credential
         */
        readonly credential: string;
        /**
         * The service that created the credentials to access a private Docker registry.
         *
         * The valid value, SECRETS_MANAGER, is for AWS Secrets Manager .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-registrycredential.html#cfn-codebuild-project-registrycredential-credentialprovider
         */
        readonly credentialProvider: string;
    }
    /**
     * Information about the compute fleet of the build project.
     *
     * For more information, see [Working with reserved capacity in AWS CodeBuild](https://docs.aws.amazon.com/codebuild/latest/userguide/fleets.html) .
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectfleet.html
     */
    interface ProjectFleetProperty {
        /**
         * Specifies the compute fleet ARN for the build project.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectfleet.html#cfn-codebuild-project-projectfleet-fleetarn
         */
        readonly fleetArn?: string;
    }
    /**
     * A source identifier and its corresponding version.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectsourceversion.html
     */
    interface ProjectSourceVersionProperty {
        /**
         * An identifier for a source in the build project.
         *
         * The identifier can only contain alphanumeric characters and underscores, and must be less than 128 characters in length.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectsourceversion.html#cfn-codebuild-project-projectsourceversion-sourceidentifier
         */
        readonly sourceIdentifier: string;
        /**
         * The source version for the corresponding source identifier. If specified, must be one of:.
         *
         * - For CodeCommit: the commit ID, branch, or Git tag to use.
         * - For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of the source code you want to build. If a pull request ID is specified, it must use the format `pr/pull-request-ID` (for example, `pr/25` ). If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         * - For GitLab: the commit ID, branch, or Git tag to use.
         * - For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code you want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         * - For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
         *
         * For more information, see [Source Version Sample with CodeBuild](https://docs.aws.amazon.com/codebuild/latest/userguide/sample-source-version.html) in the *AWS CodeBuild User Guide* .
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectsourceversion.html#cfn-codebuild-project-projectsourceversion-sourceversion
         */
        readonly sourceVersion?: string;
    }
    /**
     * Contains configuration information about a batch build project.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectbuildbatchconfig.html
     */
    interface ProjectBuildBatchConfigProperty {
        /**
         * Specifies how build status reports are sent to the source provider for the batch build.
         *
         * This property is only used when the source provider for your project is Bitbucket, GitHub, or GitHub Enterprise, and your project is configured to report build statuses to the source provider.
         *
         * - **REPORT_AGGREGATED_BATCH** - (Default) Aggregate all of the build statuses into a single status report.
         * - **REPORT_INDIVIDUAL_BUILDS** - Send a separate status report for each individual build.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectbuildbatchconfig.html#cfn-codebuild-project-projectbuildbatchconfig-batchreportmode
         */
        readonly batchReportMode?: string;
        /**
         * Specifies if the build artifacts for the batch build should be combined into a single artifact location.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectbuildbatchconfig.html#cfn-codebuild-project-projectbuildbatchconfig-combineartifacts
         */
        readonly combineArtifacts?: boolean | cdk.IResolvable;
        /**
         * A `BatchRestrictions` object that specifies the restrictions for the batch build.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectbuildbatchconfig.html#cfn-codebuild-project-projectbuildbatchconfig-restrictions
         */
        readonly restrictions?: CfnProject.BatchRestrictionsProperty | cdk.IResolvable;
        /**
         * Specifies the service role ARN for the batch build project.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectbuildbatchconfig.html#cfn-codebuild-project-projectbuildbatchconfig-servicerole
         */
        readonly serviceRole?: string;
        /**
         * Specifies the maximum amount of time, in minutes, that the batch build must be completed in.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectbuildbatchconfig.html#cfn-codebuild-project-projectbuildbatchconfig-timeoutinmins
         */
        readonly timeoutInMins?: number;
    }
    /**
     * Specifies restrictions for the batch build.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-batchrestrictions.html
     */
    interface BatchRestrictionsProperty {
        /**
         * An array of strings that specify the compute types that are allowed for the batch build.
         *
         * See [Build environment compute types](https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html) in the *AWS CodeBuild User Guide* for these values.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-batchrestrictions.html#cfn-codebuild-project-batchrestrictions-computetypesallowed
         */
        readonly computeTypesAllowed?: Array<string>;
        /**
         * Specifies the maximum number of builds allowed.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-batchrestrictions.html#cfn-codebuild-project-batchrestrictions-maximumbuildsallowed
         */
        readonly maximumBuildsAllowed?: number;
    }
    /**
     * `ProjectCache` is a property of the [AWS CodeBuild Project](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html) resource that specifies information about the cache for the build project. If `ProjectCache` is not specified, then both of its properties default to `NO_CACHE` .
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectcache.html
     */
    interface ProjectCacheProperty {
        /**
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectcache.html#cfn-codebuild-project-projectcache-cachenamespace
         */
        readonly cacheNamespace?: string;
        /**
         * Information about the cache location:.
         *
         * - `NO_CACHE` or `LOCAL` : This value is ignored.
         * - `S3` : This is the S3 bucket name/prefix.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectcache.html#cfn-codebuild-project-projectcache-location
         */
        readonly location?: string;
        /**
         * An array of strings that specify the local cache modes.
         *
         * You can use one or more local cache modes at the same time. This is only used for `LOCAL` cache types.
         *
         * Possible values are:
         *
         * - **LOCAL_SOURCE_CACHE** - Caches Git metadata for primary and secondary sources. After the cache is created, subsequent builds pull only the change between commits. This mode is a good choice for projects with a clean working directory and a source that is a large Git repository. If you choose this option and your project does not use a Git repository (GitHub, GitHub Enterprise, or Bitbucket), the option is ignored.
         * - **LOCAL_DOCKER_LAYER_CACHE** - Caches existing Docker layers. This mode is a good choice for projects that build or pull large Docker images. It can prevent the performance issues caused by pulling large Docker images down from the network.
         *
         * > - You can use a Docker layer cache in the Linux environment only.
         * > - The `privileged` flag must be set so that your project has the required Docker permissions.
         * > - You should consider the security implications before you use a Docker layer cache.
         * - **LOCAL_CUSTOM_CACHE** - Caches directories you specify in the buildspec file. This mode is a good choice if your build scenario is not suited to one of the other three local cache modes. If you use a custom cache:
         *
         * - Only directories can be specified for caching. You cannot specify individual files.
         * - Symlinks are used to reference cached directories.
         * - Cached directories are linked to your build before it downloads its project sources. Cached items are overridden if a source item has the same name. Directories are specified using cache paths in the buildspec file.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectcache.html#cfn-codebuild-project-projectcache-modes
         */
        readonly modes?: Array<string>;
        /**
         * The type of cache used by the build project. Valid values include:.
         *
         * - `NO_CACHE` : The build project does not use any cache.
         * - `S3` : The build project reads and writes from and to S3.
         * - `LOCAL` : The build project stores a cache locally on a build host that is only available to that build host.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-project-projectcache.html#cfn-codebuild-project-projectcache-type
         */
        readonly type: string;
    }
}
/**
 * Properties for defining a `CfnProject`
 *
 * @struct
 * @stability external
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html
 */
export interface CfnProjectProps {
    /**
     * `Artifacts` is a property of the [AWS::CodeBuild::Project](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html) resource that specifies output settings for artifacts generated by an AWS CodeBuild build.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-artifacts
     */
    readonly artifacts: CfnProject.ArtifactsProperty | cdk.IResolvable;
    /**
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-autoretrylimit
     */
    readonly autoRetryLimit?: number;
    /**
     * Indicates whether AWS CodeBuild generates a publicly accessible URL for your project's build badge.
     *
     * For more information, see [Build Badges Sample](https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-badges.html) in the *AWS CodeBuild User Guide* .
     *
     * > Including build badges with your project is currently not supported if the source type is CodePipeline. If you specify `CODEPIPELINE` for the `Source` property, do not specify the `BadgeEnabled` property.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-badgeenabled
     */
    readonly badgeEnabled?: boolean | cdk.IResolvable;
    /**
     * A `ProjectBuildBatchConfig` object that defines the batch build options for the project.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-buildbatchconfig
     */
    readonly buildBatchConfig?: cdk.IResolvable | CfnProject.ProjectBuildBatchConfigProperty;
    /**
     * Settings that AWS CodeBuild uses to store and reuse build dependencies.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-cache
     */
    readonly cache?: cdk.IResolvable | CfnProject.ProjectCacheProperty;
    /**
     * The maximum number of concurrent builds that are allowed for this project.
     *
     * New builds are only started if the current number of builds is less than or equal to this limit. If the current build count meets this limit, new builds are throttled and are not run.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-concurrentbuildlimit
     */
    readonly concurrentBuildLimit?: number;
    /**
     * A description that makes the build project easy to identify.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-description
     */
    readonly description?: string;
    /**
     * The AWS Key Management Service customer master key (CMK) to be used for encrypting the build output artifacts.
     *
     * > You can use a cross-account KMS key to encrypt the build output artifacts if your service role has permission to that key.
     *
     * You can specify either the Amazon Resource Name (ARN) of the CMK or, if available, the CMK's alias (using the format `alias/<alias-name>` ). If you don't specify a value, CodeBuild uses the managed CMK for Amazon Simple Storage Service (Amazon S3).
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-encryptionkey
     */
    readonly encryptionKey?: string;
    /**
     * The build environment settings for the project, such as the environment type or the environment variables to use for the build environment.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-environment
     */
    readonly environment: CfnProject.EnvironmentProperty | cdk.IResolvable;
    /**
     * An array of `ProjectFileSystemLocation` objects for a CodeBuild build project.
     *
     * A `ProjectFileSystemLocation` object specifies the `identifier` , `location` , `mountOptions` , `mountPoint` , and `type` of a file system created using Amazon Elastic File System.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-filesystemlocations
     */
    readonly fileSystemLocations?: Array<cdk.IResolvable | CfnProject.ProjectFileSystemLocationProperty> | cdk.IResolvable;
    /**
     * Information about logs for the build project.
     *
     * A project can create logs in CloudWatch Logs, an S3 bucket, or both.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-logsconfig
     */
    readonly logsConfig?: cdk.IResolvable | CfnProject.LogsConfigProperty;
    /**
     * The name of the build project.
     *
     * The name must be unique across all of the projects in your AWS account .
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-name
     */
    readonly name?: string;
    /**
     * The number of minutes a build is allowed to be queued before it times out.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-queuedtimeoutinminutes
     */
    readonly queuedTimeoutInMinutes?: number;
    /**
     * The ARN of the IAM role that enables CodeBuild to access the CloudWatch Logs and Amazon S3 artifacts for the project's builds.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-resourceaccessrole
     */
    readonly resourceAccessRole?: string;
    /**
     * A list of `Artifacts` objects.
     *
     * Each artifacts object specifies output settings that the project generates during a build.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-secondaryartifacts
     */
    readonly secondaryArtifacts?: Array<CfnProject.ArtifactsProperty | cdk.IResolvable> | cdk.IResolvable;
    /**
     * An array of `ProjectSource` objects.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-secondarysources
     */
    readonly secondarySources?: Array<cdk.IResolvable | CfnProject.SourceProperty> | cdk.IResolvable;
    /**
     * An array of `ProjectSourceVersion` objects.
     *
     * If `secondarySourceVersions` is specified at the build level, then they take over these `secondarySourceVersions` (at the project level).
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-secondarysourceversions
     */
    readonly secondarySourceVersions?: Array<cdk.IResolvable | CfnProject.ProjectSourceVersionProperty> | cdk.IResolvable;
    /**
     * The ARN of the IAM role that enables AWS CodeBuild to interact with dependent AWS services on behalf of the AWS account.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-servicerole
     */
    readonly serviceRole: string;
    /**
     * The source code settings for the project, such as the source code's repository type and location.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-source
     */
    readonly source: cdk.IResolvable | CfnProject.SourceProperty;
    /**
     * A version of the build input to be built for this project.
     *
     * If not specified, the latest version is used. If specified, it must be one of:
     *
     * - For CodeCommit: the commit ID, branch, or Git tag to use.
     * - For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of the source code you want to build. If a pull request ID is specified, it must use the format `pr/pull-request-ID` (for example `pr/25` ). If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
     * - For GitLab: the commit ID, branch, or Git tag to use.
     * - For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code you want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
     * - For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
     *
     * If `sourceVersion` is specified at the build level, then that version takes precedence over this `sourceVersion` (at the project level).
     *
     * For more information, see [Source Version Sample with CodeBuild](https://docs.aws.amazon.com/codebuild/latest/userguide/sample-source-version.html) in the *AWS CodeBuild User Guide* .
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-sourceversion
     */
    readonly sourceVersion?: string;
    /**
     * An arbitrary set of tags (key-value pairs) for the AWS CodeBuild project.
     *
     * These tags are available for use by AWS services that support AWS CodeBuild build project tags.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-tags
     */
    readonly tags?: Array<cdk.CfnTag>;
    /**
     * How long, in minutes, from 5 to 2160 (36 hours), for AWS CodeBuild to wait before timing out any related build that did not get marked as completed.
     *
     * The default is 60 minutes.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-timeoutinminutes
     */
    readonly timeoutInMinutes?: number;
    /**
     * For an existing AWS CodeBuild build project that has its source code stored in a GitHub repository, enables AWS CodeBuild to begin automatically rebuilding the source code every time a code change is pushed to the repository.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-triggers
     */
    readonly triggers?: cdk.IResolvable | CfnProject.ProjectTriggersProperty;
    /**
     * Specifies the visibility of the project's builds. Possible values are:.
     *
     * - **PUBLIC_READ** - The project builds are visible to the public.
     * - **PRIVATE** - The project builds are not visible to the public.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-visibility
     */
    readonly visibility?: string;
    /**
     * `VpcConfig` specifies settings that enable AWS CodeBuild to access resources in an Amazon VPC.
     *
     * For more information, see [Use AWS CodeBuild with Amazon Virtual Private Cloud](https://docs.aws.amazon.com/codebuild/latest/userguide/vpc-support.html) in the *AWS CodeBuild User Guide* .
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-project.html#cfn-codebuild-project-vpcconfig
     */
    readonly vpcConfig?: cdk.IResolvable | CfnProject.VpcConfigProperty;
}
/**
 * Represents a report group.
 *
 * A report group contains a collection of reports.
 *
 * @cloudformationResource AWS::CodeBuild::ReportGroup
 * @stability external
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-reportgroup.html
 */
export declare class CfnReportGroup extends cdk.CfnResource implements cdk.IInspectable, cdk.ITaggable {
    /**
     * The CloudFormation resource type name for this resource class.
     */
    static readonly CFN_RESOURCE_TYPE_NAME: string;
    /**
     * Build a CfnReportGroup from CloudFormation properties
     *
     * A factory method that creates a new instance of this class from an object
     * containing the CloudFormation properties of this resource.
     * Used in the @aws-cdk/cloudformation-include module.
     *
     * @internal
     */
    static _fromCloudFormation(scope: constructs.Construct, id: string, resourceAttributes: any, options: cfn_parse.FromCloudFormationOptions): CfnReportGroup;
    /**
     * The ARN of the AWS CodeBuild report group, such as `arn:aws:codebuild:region:123456789012:report-group/myReportGroupName` .
     *
     * @cloudformationAttribute Arn
     */
    readonly attrArn: string;
    /**
     * @cloudformationAttribute Id
     */
    readonly attrId: string;
    /**
     * When deleting a report group, specifies if reports within the report group should be deleted.
     */
    deleteReports?: boolean | cdk.IResolvable;
    /**
     * Information about the destination where the raw data of this `ReportGroup` is exported.
     */
    exportConfig: cdk.IResolvable | CfnReportGroup.ReportExportConfigProperty;
    /**
     * The name of the `ReportGroup` .
     */
    name?: string;
    /**
     * Tag Manager which manages the tags for this resource
     */
    readonly tags: cdk.TagManager;
    /**
     * A list of tag key and value pairs associated with this report group.
     */
    tagsRaw?: Array<cdk.CfnTag>;
    /**
     * The type of the `ReportGroup` . This can be one of the following values:.
     */
    type: string;
    /**
     * @param scope Scope in which this resource is defined
     * @param id Construct identifier for this resource (unique in its scope)
     * @param props Resource properties
     */
    constructor(scope: constructs.Construct, id: string, props: CfnReportGroupProps);
    protected get cfnProperties(): Record<string, any>;
    /**
     * Examines the CloudFormation resource and discloses attributes
     *
     * @param inspector tree inspector to collect and process attributes
     */
    inspect(inspector: cdk.TreeInspector): void;
    protected renderProperties(props: Record<string, any>): Record<string, any>;
}
export declare namespace CfnReportGroup {
    /**
     * Information about the location where the run of a report is exported.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-reportgroup-reportexportconfig.html
     */
    interface ReportExportConfigProperty {
        /**
         * The export configuration type. Valid values are:.
         *
         * - `S3` : The report results are exported to an S3 bucket.
         * - `NO_EXPORT` : The report results are not exported.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-reportgroup-reportexportconfig.html#cfn-codebuild-reportgroup-reportexportconfig-exportconfigtype
         */
        readonly exportConfigType: string;
        /**
         * A `S3ReportExportConfig` object that contains information about the S3 bucket where the run of a report is exported.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-reportgroup-reportexportconfig.html#cfn-codebuild-reportgroup-reportexportconfig-s3destination
         */
        readonly s3Destination?: cdk.IResolvable | CfnReportGroup.S3ReportExportConfigProperty;
    }
    /**
     * Information about the S3 bucket where the raw data of a report are exported.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-reportgroup-s3reportexportconfig.html
     */
    interface S3ReportExportConfigProperty {
        /**
         * The name of the S3 bucket where the raw data of a report are exported.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-reportgroup-s3reportexportconfig.html#cfn-codebuild-reportgroup-s3reportexportconfig-bucket
         */
        readonly bucket: string;
        /**
         * The AWS account identifier of the owner of the Amazon S3 bucket.
         *
         * This allows report data to be exported to an Amazon S3 bucket that is owned by an account other than the account running the build.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-reportgroup-s3reportexportconfig.html#cfn-codebuild-reportgroup-s3reportexportconfig-bucketowner
         */
        readonly bucketOwner?: string;
        /**
         * A boolean value that specifies if the results of a report are encrypted.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-reportgroup-s3reportexportconfig.html#cfn-codebuild-reportgroup-s3reportexportconfig-encryptiondisabled
         */
        readonly encryptionDisabled?: boolean | cdk.IResolvable;
        /**
         * The encryption key for the report's encrypted raw data.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-reportgroup-s3reportexportconfig.html#cfn-codebuild-reportgroup-s3reportexportconfig-encryptionkey
         */
        readonly encryptionKey?: string;
        /**
         * The type of build output artifact to create. Valid values include:.
         *
         * - `NONE` : CodeBuild creates the raw data in the output bucket. This is the default if packaging is not specified.
         * - `ZIP` : CodeBuild creates a ZIP file with the raw data in the output bucket.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-reportgroup-s3reportexportconfig.html#cfn-codebuild-reportgroup-s3reportexportconfig-packaging
         */
        readonly packaging?: string;
        /**
         * The path to the exported report's raw data results.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-reportgroup-s3reportexportconfig.html#cfn-codebuild-reportgroup-s3reportexportconfig-path
         */
        readonly path?: string;
    }
}
/**
 * Properties for defining a `CfnReportGroup`
 *
 * @struct
 * @stability external
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-reportgroup.html
 */
export interface CfnReportGroupProps {
    /**
     * When deleting a report group, specifies if reports within the report group should be deleted.
     *
     * - **true** - Deletes any reports that belong to the report group before deleting the report group.
     * - **false** - You must delete any reports in the report group. This is the default value. If you delete a report group that contains one or more reports, an exception is thrown.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-reportgroup.html#cfn-codebuild-reportgroup-deletereports
     */
    readonly deleteReports?: boolean | cdk.IResolvable;
    /**
     * Information about the destination where the raw data of this `ReportGroup` is exported.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-reportgroup.html#cfn-codebuild-reportgroup-exportconfig
     */
    readonly exportConfig: cdk.IResolvable | CfnReportGroup.ReportExportConfigProperty;
    /**
     * The name of the `ReportGroup` .
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-reportgroup.html#cfn-codebuild-reportgroup-name
     */
    readonly name?: string;
    /**
     * A list of tag key and value pairs associated with this report group.
     *
     * These tags are available for use by AWS services that support AWS CodeBuild report group tags.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-reportgroup.html#cfn-codebuild-reportgroup-tags
     */
    readonly tags?: Array<cdk.CfnTag>;
    /**
     * The type of the `ReportGroup` . This can be one of the following values:.
     *
     * - **CODE_COVERAGE** - The report group contains code coverage reports.
     * - **TEST** - The report group contains test reports.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-reportgroup.html#cfn-codebuild-reportgroup-type
     */
    readonly type: string;
}
/**
 * Information about the credentials for a GitHub, GitHub Enterprise, or Bitbucket repository.
 *
 * We strongly recommend that you use AWS Secrets Manager to store your credentials. If you use Secrets Manager , you must have secrets in your secrets manager. For more information, see [Using Dynamic References to Specify Template Values](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/dynamic-references.html#dynamic-references-secretsmanager) .
 *
 * > For security purposes, do not use plain text in your AWS CloudFormation template to store your credentials.
 *
 * @cloudformationResource AWS::CodeBuild::SourceCredential
 * @stability external
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-sourcecredential.html
 */
export declare class CfnSourceCredential extends cdk.CfnResource implements cdk.IInspectable {
    /**
     * The CloudFormation resource type name for this resource class.
     */
    static readonly CFN_RESOURCE_TYPE_NAME: string;
    /**
     * Build a CfnSourceCredential from CloudFormation properties
     *
     * A factory method that creates a new instance of this class from an object
     * containing the CloudFormation properties of this resource.
     * Used in the @aws-cdk/cloudformation-include module.
     *
     * @internal
     */
    static _fromCloudFormation(scope: constructs.Construct, id: string, resourceAttributes: any, options: cfn_parse.FromCloudFormationOptions): CfnSourceCredential;
    /**
     * @cloudformationAttribute Id
     */
    readonly attrId: string;
    /**
     * The type of authentication used by the credentials.
     */
    authType: string;
    /**
     * The type of source provider.
     */
    serverType: string;
    /**
     * For GitHub or GitHub Enterprise, this is the personal access token.
     */
    token: string;
    /**
     * The Bitbucket username when the `authType` is BASIC_AUTH.
     */
    username?: string;
    /**
     * @param scope Scope in which this resource is defined
     * @param id Construct identifier for this resource (unique in its scope)
     * @param props Resource properties
     */
    constructor(scope: constructs.Construct, id: string, props: CfnSourceCredentialProps);
    protected get cfnProperties(): Record<string, any>;
    /**
     * Examines the CloudFormation resource and discloses attributes
     *
     * @param inspector tree inspector to collect and process attributes
     */
    inspect(inspector: cdk.TreeInspector): void;
    protected renderProperties(props: Record<string, any>): Record<string, any>;
}
/**
 * Properties for defining a `CfnSourceCredential`
 *
 * @struct
 * @stability external
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-sourcecredential.html
 */
export interface CfnSourceCredentialProps {
    /**
     * The type of authentication used by the credentials.
     *
     * Valid options are OAUTH, BASIC_AUTH, PERSONAL_ACCESS_TOKEN, CODECONNECTIONS, or SECRETS_MANAGER.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-sourcecredential.html#cfn-codebuild-sourcecredential-authtype
     */
    readonly authType: string;
    /**
     * The type of source provider.
     *
     * The valid options are GITHUB, GITHUB_ENTERPRISE, GITLAB, GITLAB_SELF_MANAGED, or BITBUCKET.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-sourcecredential.html#cfn-codebuild-sourcecredential-servertype
     */
    readonly serverType: string;
    /**
     * For GitHub or GitHub Enterprise, this is the personal access token.
     *
     * For Bitbucket, this is either the access token or the app password. For the `authType` CODECONNECTIONS, this is the `connectionArn` . For the `authType` SECRETS_MANAGER, this is the `secretArn` .
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-sourcecredential.html#cfn-codebuild-sourcecredential-token
     */
    readonly token: string;
    /**
     * The Bitbucket username when the `authType` is BASIC_AUTH.
     *
     * This parameter is not valid for other types of source providers or connections.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-sourcecredential.html#cfn-codebuild-sourcecredential-username
     */
    readonly username?: string;
}
/**
 * The `AWS::CodeBuild::Fleet` resource configures a compute fleet, a set of dedicated instances for your build environment.
 *
 * @cloudformationResource AWS::CodeBuild::Fleet
 * @stability external
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html
 */
export declare class CfnFleet extends cdk.CfnResource implements cdk.IInspectable, cdk.ITaggableV2 {
    /**
     * The CloudFormation resource type name for this resource class.
     */
    static readonly CFN_RESOURCE_TYPE_NAME: string;
    /**
     * Build a CfnFleet from CloudFormation properties
     *
     * A factory method that creates a new instance of this class from an object
     * containing the CloudFormation properties of this resource.
     * Used in the @aws-cdk/cloudformation-include module.
     *
     * @internal
     */
    static _fromCloudFormation(scope: constructs.Construct, id: string, resourceAttributes: any, options: cfn_parse.FromCloudFormationOptions): CfnFleet;
    /**
     * The ARN of the compute fleet.
     *
     * @cloudformationAttribute Arn
     */
    readonly attrArn: string;
    /**
     * The initial number of machines allocated to the compute ﬂeet, which deﬁnes the number of builds that can run in parallel.
     */
    baseCapacity?: number;
    /**
     * Tag Manager which manages the tags for this resource
     */
    readonly cdkTagManager: cdk.TagManager;
    /**
     * The compute configuration of the compute fleet.
     */
    computeConfiguration?: CfnFleet.ComputeConfigurationProperty | cdk.IResolvable;
    /**
     * Information about the compute resources the compute fleet uses. Available values include:.
     */
    computeType?: string;
    /**
     * The environment type of the compute fleet.
     */
    environmentType?: string;
    /**
     * Information about the proxy configurations that apply network access control to your reserved capacity instances.
     */
    fleetProxyConfiguration?: cdk.IResolvable | CfnFleet.ProxyConfigurationProperty;
    /**
     * The service role associated with the compute fleet.
     */
    fleetServiceRole?: string;
    /**
     * Information about the VPC configuration that AWS CodeBuild accesses.
     */
    fleetVpcConfig?: cdk.IResolvable | CfnFleet.VpcConfigProperty;
    /**
     * The Amazon Machine Image (AMI) of the compute fleet.
     */
    imageId?: string;
    /**
     * The name of the compute fleet.
     */
    name?: string;
    /**
     * The compute fleet overflow behavior.
     */
    overflowBehavior?: string;
    /**
     * The scaling configuration of the compute fleet.
     */
    scalingConfiguration?: cdk.IResolvable | CfnFleet.ScalingConfigurationInputProperty;
    /**
     * A list of tag key and value pairs associated with this compute fleet.
     */
    tags?: Array<cdk.CfnTag>;
    /**
     * @param scope Scope in which this resource is defined
     * @param id Construct identifier for this resource (unique in its scope)
     * @param props Resource properties
     */
    constructor(scope: constructs.Construct, id: string, props?: CfnFleetProps);
    protected get cfnProperties(): Record<string, any>;
    /**
     * Examines the CloudFormation resource and discloses attributes
     *
     * @param inspector tree inspector to collect and process attributes
     */
    inspect(inspector: cdk.TreeInspector): void;
    protected renderProperties(props: Record<string, any>): Record<string, any>;
}
export declare namespace CfnFleet {
    /**
     * Information about the VPC configuration that AWS CodeBuild accesses.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-vpcconfig.html
     */
    interface VpcConfigProperty {
        /**
         * A list of one or more security groups IDs in your Amazon VPC.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-vpcconfig.html#cfn-codebuild-fleet-vpcconfig-securitygroupids
         */
        readonly securityGroupIds?: Array<string>;
        /**
         * A list of one or more subnet IDs in your Amazon VPC.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-vpcconfig.html#cfn-codebuild-fleet-vpcconfig-subnets
         */
        readonly subnets?: Array<string>;
        /**
         * The ID of the Amazon VPC.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-vpcconfig.html#cfn-codebuild-fleet-vpcconfig-vpcid
         */
        readonly vpcId?: string;
    }
    /**
     * Information about the proxy configurations that apply network access control to your reserved capacity instances.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-proxyconfiguration.html
     */
    interface ProxyConfigurationProperty {
        /**
         * The default behavior of outgoing traffic.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-proxyconfiguration.html#cfn-codebuild-fleet-proxyconfiguration-defaultbehavior
         */
        readonly defaultBehavior?: string;
        /**
         * An array of `FleetProxyRule` objects that represent the specified destination domains or IPs to allow or deny network access control to.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-proxyconfiguration.html#cfn-codebuild-fleet-proxyconfiguration-orderedproxyrules
         */
        readonly orderedProxyRules?: Array<CfnFleet.FleetProxyRuleProperty | cdk.IResolvable> | cdk.IResolvable;
    }
    /**
     * Information about the proxy rule for your reserved capacity instances.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-fleetproxyrule.html
     */
    interface FleetProxyRuleProperty {
        /**
         * The behavior of the proxy rule.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-fleetproxyrule.html#cfn-codebuild-fleet-fleetproxyrule-effect
         */
        readonly effect?: string;
        /**
         * The destination of the proxy rule.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-fleetproxyrule.html#cfn-codebuild-fleet-fleetproxyrule-entities
         */
        readonly entities?: Array<string>;
        /**
         * The type of proxy rule.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-fleetproxyrule.html#cfn-codebuild-fleet-fleetproxyrule-type
         */
        readonly type?: string;
    }
    /**
     * The scaling configuration input of a compute fleet.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-scalingconfigurationinput.html
     */
    interface ScalingConfigurationInputProperty {
        /**
         * The maximum number of instances in the ﬂeet when auto-scaling.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-scalingconfigurationinput.html#cfn-codebuild-fleet-scalingconfigurationinput-maxcapacity
         */
        readonly maxCapacity?: number;
        /**
         * The scaling type for a compute fleet.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-scalingconfigurationinput.html#cfn-codebuild-fleet-scalingconfigurationinput-scalingtype
         */
        readonly scalingType?: string;
        /**
         * A list of `TargetTrackingScalingConfiguration` objects.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-scalingconfigurationinput.html#cfn-codebuild-fleet-scalingconfigurationinput-targettrackingscalingconfigs
         */
        readonly targetTrackingScalingConfigs?: Array<cdk.IResolvable | CfnFleet.TargetTrackingScalingConfigurationProperty> | cdk.IResolvable;
    }
    /**
     * Defines when a new instance is auto-scaled into the compute fleet.
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-targettrackingscalingconfiguration.html
     */
    interface TargetTrackingScalingConfigurationProperty {
        /**
         * The metric type to determine auto-scaling.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-targettrackingscalingconfiguration.html#cfn-codebuild-fleet-targettrackingscalingconfiguration-metrictype
         */
        readonly metricType?: string;
        /**
         * The value of `metricType` when to start scaling.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-targettrackingscalingconfiguration.html#cfn-codebuild-fleet-targettrackingscalingconfiguration-targetvalue
         */
        readonly targetValue?: number;
    }
    /**
     * Contains compute attributes.
     *
     * These attributes only need be specified when your project's or fleet's `computeType` is set to `ATTRIBUTE_BASED_COMPUTE` .
     *
     * @struct
     * @stability external
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-computeconfiguration.html
     */
    interface ComputeConfigurationProperty {
        /**
         * The amount of disk space of the instance type included in your fleet.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-computeconfiguration.html#cfn-codebuild-fleet-computeconfiguration-disk
         */
        readonly disk?: number;
        /**
         * The machine type of the instance type included in your fleet.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-computeconfiguration.html#cfn-codebuild-fleet-computeconfiguration-machinetype
         */
        readonly machineType?: string;
        /**
         * The amount of memory of the instance type included in your fleet.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-computeconfiguration.html#cfn-codebuild-fleet-computeconfiguration-memory
         */
        readonly memory?: number;
        /**
         * The number of vCPUs of the instance type included in your fleet.
         *
         * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-codebuild-fleet-computeconfiguration.html#cfn-codebuild-fleet-computeconfiguration-vcpu
         */
        readonly vCpu?: number;
    }
}
/**
 * Properties for defining a `CfnFleet`
 *
 * @struct
 * @stability external
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html
 */
export interface CfnFleetProps {
    /**
     * The initial number of machines allocated to the compute ﬂeet, which deﬁnes the number of builds that can run in parallel.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-basecapacity
     */
    readonly baseCapacity?: number;
    /**
     * The compute configuration of the compute fleet.
     *
     * This is only required if `computeType` is set to `ATTRIBUTE_BASED_COMPUTE` .
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-computeconfiguration
     */
    readonly computeConfiguration?: CfnFleet.ComputeConfigurationProperty | cdk.IResolvable;
    /**
     * Information about the compute resources the compute fleet uses. Available values include:.
     *
     * - `ATTRIBUTE_BASED_COMPUTE` : Specify the amount of vCPUs, memory, disk space, and the type of machine.
     *
     * > If you use `ATTRIBUTE_BASED_COMPUTE` , you must define your attributes by using `computeConfiguration` . AWS CodeBuild will select the cheapest instance that satisfies your specified attributes. For more information, see [Reserved capacity environment types](https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html#environment-reserved-capacity.types) in the *AWS CodeBuild User Guide* .
     * - `BUILD_GENERAL1_SMALL` : Use up to 4 GiB memory and 2 vCPUs for builds.
     * - `BUILD_GENERAL1_MEDIUM` : Use up to 8 GiB memory and 4 vCPUs for builds.
     * - `BUILD_GENERAL1_LARGE` : Use up to 16 GiB memory and 8 vCPUs for builds, depending on your environment type.
     * - `BUILD_GENERAL1_XLARGE` : Use up to 72 GiB memory and 36 vCPUs for builds, depending on your environment type.
     * - `BUILD_GENERAL1_2XLARGE` : Use up to 144 GiB memory, 72 vCPUs, and 824 GB of SSD storage for builds. This compute type supports Docker images up to 100 GB uncompressed.
     * - `BUILD_LAMBDA_1GB` : Use up to 1 GiB memory for builds. Only available for environment type `LINUX_LAMBDA_CONTAINER` and `ARM_LAMBDA_CONTAINER` .
     * - `BUILD_LAMBDA_2GB` : Use up to 2 GiB memory for builds. Only available for environment type `LINUX_LAMBDA_CONTAINER` and `ARM_LAMBDA_CONTAINER` .
     * - `BUILD_LAMBDA_4GB` : Use up to 4 GiB memory for builds. Only available for environment type `LINUX_LAMBDA_CONTAINER` and `ARM_LAMBDA_CONTAINER` .
     * - `BUILD_LAMBDA_8GB` : Use up to 8 GiB memory for builds. Only available for environment type `LINUX_LAMBDA_CONTAINER` and `ARM_LAMBDA_CONTAINER` .
     * - `BUILD_LAMBDA_10GB` : Use up to 10 GiB memory for builds. Only available for environment type `LINUX_LAMBDA_CONTAINER` and `ARM_LAMBDA_CONTAINER` .
     *
     * If you use `BUILD_GENERAL1_SMALL` :
     *
     * - For environment type `LINUX_CONTAINER` , you can use up to 4 GiB memory and 2 vCPUs for builds.
     * - For environment type `LINUX_GPU_CONTAINER` , you can use up to 16 GiB memory, 4 vCPUs, and 1 NVIDIA A10G Tensor Core GPU for builds.
     * - For environment type `ARM_CONTAINER` , you can use up to 4 GiB memory and 2 vCPUs on ARM-based processors for builds.
     *
     * If you use `BUILD_GENERAL1_LARGE` :
     *
     * - For environment type `LINUX_CONTAINER` , you can use up to 16 GiB memory and 8 vCPUs for builds.
     * - For environment type `LINUX_GPU_CONTAINER` , you can use up to 255 GiB memory, 32 vCPUs, and 4 NVIDIA Tesla V100 GPUs for builds.
     * - For environment type `ARM_CONTAINER` , you can use up to 16 GiB memory and 8 vCPUs on ARM-based processors for builds.
     *
     * For more information, see [On-demand environment types](https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html#environment.types) in the *AWS CodeBuild User Guide.*
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-computetype
     */
    readonly computeType?: string;
    /**
     * The environment type of the compute fleet.
     *
     * - The environment type `ARM_CONTAINER` is available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), EU (Ireland), Asia Pacific (Mumbai), Asia Pacific (Tokyo), Asia Pacific (Singapore), Asia Pacific (Sydney), EU (Frankfurt), and South America (São Paulo).
     * - The environment type `ARM_EC2` is available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), EU (Ireland), EU (Frankfurt), Asia Pacific (Tokyo), Asia Pacific (Singapore), Asia Pacific (Sydney), South America (São Paulo), and Asia Pacific (Mumbai).
     * - The environment type `LINUX_CONTAINER` is available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), EU (Ireland), EU (Frankfurt), Asia Pacific (Tokyo), Asia Pacific (Singapore), Asia Pacific (Sydney), South America (São Paulo), and Asia Pacific (Mumbai).
     * - The environment type `LINUX_EC2` is available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), EU (Ireland), EU (Frankfurt), Asia Pacific (Tokyo), Asia Pacific (Singapore), Asia Pacific (Sydney), South America (São Paulo), and Asia Pacific (Mumbai).
     * - The environment type `LINUX_GPU_CONTAINER` is available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), EU (Ireland), EU (Frankfurt), Asia Pacific (Tokyo), and Asia Pacific (Sydney).
     * - The environment type `MAC_ARM` is available only in regions US East (Ohio), US East (N. Virginia), US West (Oregon), Europe (Frankfurt), and Asia Pacific (Sydney).
     * - The environment type `WINDOWS_EC2` is available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), EU (Ireland), EU (Frankfurt), Asia Pacific (Tokyo), Asia Pacific (Singapore), Asia Pacific (Sydney), South America (São Paulo), and Asia Pacific (Mumbai).
     * - The environment type `WINDOWS_SERVER_2019_CONTAINER` is available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), Asia Pacific (Sydney), Asia Pacific (Tokyo), Asia Pacific (Mumbai) and EU (Ireland).
     * - The environment type `WINDOWS_SERVER_2022_CONTAINER` is available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), EU (Ireland), EU (Frankfurt), Asia Pacific (Sydney), Asia Pacific (Singapore), Asia Pacific (Tokyo), South America (São Paulo) and Asia Pacific (Mumbai).
     *
     * For more information, see [Build environment compute types](https://docs.aws.amazon.com//codebuild/latest/userguide/build-env-ref-compute-types.html) in the *AWS CodeBuild user guide* .
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-environmenttype
     */
    readonly environmentType?: string;
    /**
     * Information about the proxy configurations that apply network access control to your reserved capacity instances.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-fleetproxyconfiguration
     */
    readonly fleetProxyConfiguration?: cdk.IResolvable | CfnFleet.ProxyConfigurationProperty;
    /**
     * The service role associated with the compute fleet.
     *
     * For more information, see [Allow a user to add a permission policy for a fleet service role](https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#customer-managed-policies-example-permission-policy-fleet-service-role.html) in the *AWS CodeBuild User Guide* .
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-fleetservicerole
     */
    readonly fleetServiceRole?: string;
    /**
     * Information about the VPC configuration that AWS CodeBuild accesses.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-fleetvpcconfig
     */
    readonly fleetVpcConfig?: cdk.IResolvable | CfnFleet.VpcConfigProperty;
    /**
     * The Amazon Machine Image (AMI) of the compute fleet.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-imageid
     */
    readonly imageId?: string;
    /**
     * The name of the compute fleet.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-name
     */
    readonly name?: string;
    /**
     * The compute fleet overflow behavior.
     *
     * - For overflow behavior `QUEUE` , your overflow builds need to wait on the existing fleet instance to become available.
     * - For overflow behavior `ON_DEMAND` , your overflow builds run on CodeBuild on-demand.
     *
     * > If you choose to set your overflow behavior to on-demand while creating a VPC-connected fleet, make sure that you add the required VPC permissions to your project service role. For more information, see [Example policy statement to allow CodeBuild access to AWS services required to create a VPC network interface](https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#customer-managed-policies-example-create-vpc-network-interface) .
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-overflowbehavior
     */
    readonly overflowBehavior?: string;
    /**
     * The scaling configuration of the compute fleet.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-scalingconfiguration
     */
    readonly scalingConfiguration?: cdk.IResolvable | CfnFleet.ScalingConfigurationInputProperty;
    /**
     * A list of tag key and value pairs associated with this compute fleet.
     *
     * These tags are available for use by AWS services that support AWS CodeBuild compute fleet tags.
     *
     * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codebuild-fleet.html#cfn-codebuild-fleet-tags
     */
    readonly tags?: Array<cdk.CfnTag>;
}
