import { Construct } from 'constructs';
import { OriginProtocolPolicy, OriginSslPolicy } from '../';
import { IInstance } from '../../aws-ec2';
import { IApplicationLoadBalancer, INetworkLoadBalancer } from '../../aws-elasticloadbalancingv2';
import { IResource, ITaggableV2, Resource, TagManager } from '../../core';
/**
 * Represents a VPC origin.
 */
export interface IVpcOrigin extends IResource {
    /**
     * The VPC origin ARN.
     * @attribute
     */
    readonly vpcOriginArn: string;
    /**
     * The VPC origin ID.
     * @attribute
     */
    readonly vpcOriginId: string;
    /**
     * The domain name of the CloudFront VPC origin endpoint configuration.
     */
    readonly domainName?: string;
}
/**
 * VPC origin endpoint configuration.
 */
export interface VpcOriginOptions {
    /**
     * The HTTP port for the CloudFront VPC origin endpoint configuration.
     * @default 80
     */
    readonly httpPort?: number;
    /**
     * The HTTPS port of the CloudFront VPC origin endpoint configuration.
     * @default 443
     */
    readonly httpsPort?: number;
    /**
     * The name of the CloudFront VPC origin endpoint configuration.
     * @default - generated from the `id`
     */
    readonly vpcOriginName?: string;
    /**
     * The origin protocol policy for the CloudFront VPC origin endpoint configuration.
     * @default OriginProtocolPolicy.MATCH_VIEWER
     */
    readonly protocolPolicy?: OriginProtocolPolicy;
    /**
     * A list that contains allowed SSL/TLS protocols for this distribution.
     * @default - TLSv1.2
     */
    readonly originSslProtocols?: OriginSslPolicy[];
}
/**
 * VPC origin endpoint configuration.
 */
export interface VpcOriginProps extends VpcOriginOptions {
    /**
     * The VPC origin endpoint.
     */
    readonly endpoint: VpcOriginEndpoint;
}
/**
 * The properties to import from the VPC origin
 */
export interface VpcOriginAttributes {
    /**
     * The ARN of the VPC origin.
     *
     * At least one of vpcOriginArn and vpcOriginId must be provided.
     *
     * @default - derived from `vpcOriginId`.
     */
    readonly vpcOriginArn?: string;
    /**
     * The ID of the VPC origin.
     *
     * At least one of vpcOriginArn and vpcOriginId must be provided.
     *
     * @default - derived from `vpcOriginArn`.
     */
    readonly vpcOriginId?: string;
    /**
     * The domain name of the CloudFront VPC origin endpoint configuration.
     * @default - No domain name configured
     */
    readonly domainName?: string;
}
/**
 * Represents the VPC origin endpoint.
 */
export declare abstract class VpcOriginEndpoint {
    /**
     * A VPC origin endpoint from an EC2 instance.
     */
    static ec2Instance(instance: IInstance): VpcOriginEndpoint;
    /**
     * A VPC origin endpoint from an Application Load Balancer.
     */
    static applicationLoadBalancer(alb: IApplicationLoadBalancer): VpcOriginEndpoint;
    /**
     * A VPC origin endpoint from an Network Load Balancer.
     */
    static networkLoadBalancer(nlb: INetworkLoadBalancer): VpcOriginEndpoint;
    /**
     * The ARN of the CloudFront VPC origin endpoint configuration.
     */
    abstract readonly endpointArn: string;
    /**
     * The domain name of the CloudFront VPC origin endpoint configuration.
     * @default - No domain name configured
     */
    abstract readonly domainName?: string;
}
/**
 * A CloudFront VPC Origin configuration.
 *
 * @resource AWS::CloudFront::VpcOrigin
 */
export declare class VpcOrigin extends Resource implements IVpcOrigin, ITaggableV2 {
    /**
     * Import an existing VPC origin from its ID.
     */
    static fromVpcOriginId(scope: Construct, id: string, vpcOriginId: string): IVpcOrigin;
    /**
     * Import an existing VPC origin from its ARN.
     */
    static fromVpcOriginArn(scope: Construct, id: string, vpcOriginArn: string): IVpcOrigin;
    /**
     * Import an existing VPC origin from its attributes.
     */
    static fromVpcOriginAttributes(scope: Construct, id: string, attrs: VpcOriginAttributes): IVpcOrigin;
    /**
     * The VPC origin ARN.
     * @attribute
     */
    readonly vpcOriginArn: string;
    /**
     * The VPC origin ID.
     * @attribute
     */
    readonly vpcOriginId: string;
    /**
     * The domain name of the CloudFront VPC origin endpoint configuration.
     */
    readonly domainName?: string;
    readonly cdkTagManager: TagManager;
    constructor(scope: Construct, id: string, props: VpcOriginProps);
    private validatePortNumber;
}
